/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qfile.h>
#include <qstringlist.h>

#include <settings.h>
#include <element.h>
#include <fileexporterbibtex.h>
#include "fileexporterpdf.h"

namespace BibTeX
{

    FileExporterPDF::FileExporterPDF()
            : FileExporterToolchain(), m_latexLanguage( "english" ), m_latexBibStyle( "plain" )
    {
        laTeXFilename = QString( workingDir ).append( "/bibtex-to-pdf.tex" );
        bibTeXFilename = QString( workingDir ).append( "/bibtex-to-pdf.bib" );
        outputFilename = QString( workingDir ).append( "/bibtex-to-pdf.pdf" );
    }

    FileExporterPDF::~FileExporterPDF()
    {
        // nothing
    }

    bool FileExporterPDF::save( QIODevice* iodevice, File* bibtexfile, QStringList *errorLog )
    {
        bool result = FALSE;

        QFile bibtexFile( bibTeXFilename );
        if ( bibtexFile.open( IO_WriteOnly ) )
        {
            FileExporter * bibtexExporter = new FileExporterBibTeX();
            result = bibtexExporter->save( &bibtexFile, bibtexfile, errorLog );
            bibtexFile.close();
            delete bibtexExporter;
        }

        if ( result )
            result = generatePDF( iodevice, errorLog );

        return result;
    }

    bool FileExporterPDF::save( QIODevice* iodevice, Element* element, QStringList *errorLog )
    {
        bool result = FALSE;

        QFile bibtexFile( bibTeXFilename );
        if ( bibtexFile.open( IO_WriteOnly ) )
        {
            FileExporter * bibtexExporter = new FileExporterBibTeX();
            result = bibtexExporter->save( &bibtexFile, element, errorLog );
            bibtexFile.close();
            delete bibtexExporter;
        }

        if ( result )
            result = generatePDF( iodevice, errorLog );

        return result;
    }

    void FileExporterPDF::setLaTeXLanguage( const QString& language )
    {
        m_latexLanguage = language;
    }

    void FileExporterPDF::setLaTeXBibliographyStyle( const QString& bibStyle )
    {
        m_latexBibStyle = bibStyle;
    }

    bool FileExporterPDF::generatePDF( QIODevice* iodevice, QStringList *errorLog )
    {
        QStringList cmdLines = QStringList::split( '|', "pdflatex bibtex-to-pdf.tex|bibtex bibtex-to-pdf|pdflatex bibtex-to-pdf.tex|pdflatex bibtex-to-pdf.tex" );

        if ( writeLatexFile( laTeXFilename ) && runProcesses( cmdLines, errorLog ) && writeFileToIODevice( outputFilename, iodevice ) )
            return TRUE;
        else
            return FALSE;
    }

    bool FileExporterPDF::writeLatexFile( const QString &filename )
    {
        QFile latexFile( filename );
        if ( latexFile.open( IO_WriteOnly ) )
        {
            QTextStream ts( &latexFile );
            ts.setEncoding( QTextStream::UnicodeUTF8 );
            ts << "\\documentclass{article}\n";
            ts << "\\usepackage[" << m_latexLanguage << "]{babel}\n";
            ts << "\\usepackage{url}\n";
            ts << "\\bibliographystyle{" << m_latexBibStyle << "}\n";
            ts << "\\begin{document}\n";
            ts << "\\nocite{*}\n";
            ts << "\\bibliography{bibtex-to-pdf}\n";
            ts << "\\end{document}\n";
            latexFile.close();
            return TRUE;
        }
        else
            return FALSE;

    }

}
