/***************************************************************************
                           kdetvcpudetection.h
                           -------------------
    begin                : Sat Jul 10 2004
    copyright            : (C) 2004 by Dirk Ziegelmeier
    email                : dziegel@gmx.de
 ***************************************************************************/

/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kdebug.h>

#include "kdetvcpudetection.h"

static KdetvCpuDetection* _instance = 0;

#ifdef ARCH_386

// http://www.sandpile.org/ia32/cpuid.htm

// save ebx (-fPIC may be used)
#define CPUID(cmd, eax, ebx, ecx, edx) \
    __asm__ __volatile__               \
        (                              \
         "pushl %%ebx       \n\t"      \
	     "cpuid             \n\t"      \
	     "movl  %%ebx, %1   \n\t"      \
	     "popl  %%ebx       \n\t"      \
	     : "=a"(eax),                  \
	       "=r"(ebx),                  \
	       "=c"(ecx),                  \
	       "=d"(edx)                   \
	     : "a"(cmd)                    \
	     : "cc")

static struct {
    unsigned int eax, ebx, edx, ecx;
    unsigned int stringterm;
} cpuid;

static void queryCapabilities386(KdetvCpuDetection::Capabilities* cap)
{
    unsigned int cpuidSupported = 0;
    cpuid.stringterm = 0;

    kdDebug() << "KdetvCpuDetection::queryCapabilities386()" << endl;

    // Find out wether CPUID command is supported
	__asm__ __volatile__
        (
         "pushfl                 \n\t"
         "popl   %%ecx           \n\t"
         "movl   %%ecx,     %%eax\n\t"
         "xorl   $0x200000, %%eax\n\t"
         "pushl  %%eax           \n\t"
         "popfl                  \n\t"
         "pushfl                 \n\t"
         "popl   %%eax           \n\t"
         "pushl  %%ecx           \n\t"
         "popfl                  \n\t"
         "xorl   %%ecx,     %%eax\n\t"
         "andl   $0x200000, %%eax\n\t"
         "jz     1f              \n\t"
         "movl   $1,%%eax        \n\t"
         "1:                     \n\t"

         : "=a"(cpuidSupported)
         : /* no inputs */
         : "ecx", "cc");

    if(!cpuidSupported) {
        kdDebug() << "    CPUID not supported." << endl;
        return;
    }

    // Maximum standart level
    CPUID(0, cpuid.eax, cpuid.ebx, cpuid.ecx, cpuid.edx);
    if(!cpuid.eax) {
        kdDebug() << "    Only vendor string is supported." << endl;
        return;
    }
    kdDebug() << "    Vendor Id: " << (char*)&cpuid.ebx << endl;

    // Standart level 1
    CPUID(1, cpuid.eax, cpuid.ebx, cpuid.ecx, cpuid.edx);
    if(cpuid.edx & (1<<23)) {
        kdDebug() << "    CPU supports MMX." << endl;
        *cap = (KdetvCpuDetection::Capabilities)(*cap | KdetvCpuDetection::Cap_MMX);
    }
    if(cpuid.edx & (1<<25)) {
        kdDebug() << "    CPU supports SSE." << endl;
        *cap = (KdetvCpuDetection::Capabilities)(*cap | KdetvCpuDetection::Cap_SSE);
    }
    if(cpuid.edx & (1<<26)) {
        kdDebug() << "    CPU supports SSE2." << endl;
        *cap = (KdetvCpuDetection::Capabilities)(*cap | KdetvCpuDetection::Cap_SSE2);
    }
    
    // Check for AMD extended level
    CPUID(0x80000000, cpuid.eax, cpuid.ebx, cpuid.ecx, cpuid.edx);
    if(cpuid.eax < 0x80000001) {
        return;
    }

    CPUID(0x80000001, cpuid.eax, cpuid.ebx, cpuid.ecx, cpuid.edx);
    if(cpuid.edx & (1<<31)) {
        kdDebug() << "    CPU supports 3DNOW." << endl;
        *cap = (KdetvCpuDetection::Capabilities)(*cap | KdetvCpuDetection::Cap_3DNOW);
    }
}

#undef CPUID

#endif

KdetvCpuDetection::KdetvCpuDetection()
{
#if defined (ARCH_386)

    _architecture = Arch_386;
    queryCapabilities386(&_capabilities);

#elif defined (ARCH_X86_64)

    _architecture = Arch_X86_64;
    _capabilities = (KdetvCpuDetection::Capabilities)
        (KdetvCpuDetection::Cap_MMX   |
         KdetvCpuDetection::Cap_SSE   |
         KdetvCpuDetection::Cap_SSE2  |
         KdetvCpuDetection::Cap_3DNOW   );

#else

    _architecture = Arch_Unknown;
    _capabilities = Cap_Unknown;

#endif
}

KdetvCpuDetection* KdetvCpuDetection::instance()
{
    if(_instance == 0) {
        _instance = new KdetvCpuDetection();
    }
    return _instance;
}
