/***************************************************************************
                          kickpimrepository.cpp  -  description
                             -------------------
    begin                : Mon Jan 13 2003
    copyright            : (C) 2003 by Bert Speckels
    email                : bert.speckels@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kstandarddirs.h>

#include "kickpim.h"

#include "kpevent.h"
#include "kporgevent.h"
#include "kpeventreader.h"
#include "kpcontactevent.h"
#include "kpcontacteventreader.h"


#include "kpcontact.h"
#include "kpcontactreader.h"
#include "kpkabcontact.h"
#include "kpkabcontactreader.h"
 
#include "kickpimrepository.h"

#include "mailing/kpmailurl.h"
#include "mailing/kpmailaccount.h"
#include "mailing/kickpimmailmonitor.h"

#include "LogService.h"

KickPimRepository::KickPimRepository(QObject* owner)
{
  if (LogService::doLogConstruct) LogService::construct( "KickPimRepository" );

  m_options = new KickPimOptions(owner);

  // contact (event) reader
  m_eventReader        = 0;
  m_contactReader      = 0;
  m_contactEventReader = 0;

  // get important directories
  KStandardDirs dirs;
  QString baseDir = dirs.findResourceDir("data","kickpim/");
  m_dirIcons = baseDir + "kickpim/icons/";
  m_dirInfo  = baseDir + "kickpim/info/";
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_REPOSITORY,"KickPimRepository: Directory (Icons): " + m_dirIcons);
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_REPOSITORY,"KickPimRepository: Directory (Info):  " + m_dirInfo );

  m_yearSign = i18n("year").left(1);
}

KickPimRepository::~KickPimRepository()
{
  if (LogService::doLogConstruct) LogService::destruct("KickPimRepository");

  finishMailMonitorThreads();

  delete m_eventReader;          m_eventReader        = 0;
  delete m_contactEventReader;   m_contactEventReader = 0;
  delete m_contactReader;        m_contactReader      = 0;
  delete m_options;              m_options            = 0;
}



void KickPimRepository::initData()
{
  //** Contacts **//
  m_contactReader      = new KPKabContactReader();
  m_contactEventReader = new KPContactEventReader();

  if ( m_contactReader!=0 ) {
    // the following call reads contacts and some events concerning these contacts (e.g.birthdays)
    onContactListChanged ( m_contactReader );
    connect( m_contactReader, SIGNAL(contactsChanged(KPContactReader*) ),
             this,            SLOT  (onContactListChanged(KPContactReader*) ) );
  }
  //** Start emails monitors **/
  createMailMonitorThreads( );
}

void KickPimRepository::mailMonitors_Check() {
  if (LogService::doLogCall) LogService::call("KickPimRepository","mailMonitors_Check");
  QPtrListIterator<KickPimMailMonitorThread> itor( m_mailMonitorThreads );
  KickPimMailMonitorThread* thread = 0;
  QString name;
  while ( itor.current()!=0 ) {
    thread = (KickPimMailMonitorThread*)itor.current();
    thread->monitor()->checkMailNow();
    ++itor;
  }
}

/**
 * Creates and starts the mail monitor threads
 */
void KickPimRepository::createMailMonitorThreads( ) {
  if (LogService::doLogCall) LogService::call("KickPimRepository","createMailMonitorThreads");

  KickPimMailMonitorThread* thread = 0;
  KPMailAccountList&  list = m_options->mailAccounts;
  QString name;
  // iterate all accounts
  KPMailAccount* acc  = list.first();
  while (acc) {
    // create monitor and thread
    thread = mailMonitors_CreateThread( acc );
    // start the (initially paused) thread
    thread->setSkipMailchecks(false);
    thread->start();
    if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_REPOSITORY,"KickPimRepository: New MailMonitor for Account '" + thread->monitor()->account()->name() + "' started.");
    // next
    acc = list.next();
  }
  onEmailAcountsChanged(); // send signal that one or more accounts changed  
}


void KickPimRepository::finishMailMonitorThreads() {
  if (LogService::doLogCall) LogService::call("KickPimRepository","finishMailMonitorThreads");
  QPtrListIterator<KickPimMailMonitorThread> itor( m_mailMonitorThreads );
  KickPimMailMonitorThread* thread = 0;
  QString name;
  while ( itor.current()!=0 ) {
    thread = (KickPimMailMonitorThread*)itor.current();
    thread->terminateThread(); // returns if successfully terminated
    delete thread;thread=0;
    ++itor;
  }
  m_mailMonitorThreads.clear();
  onEmailAcountsChanged(); // send signal that one or more accounts changed
}




KickPimMailMonitorThread* KickPimRepository::mailMonitors_CreateThread( KPMailAccount* acc )
{
  KickPimMailMonitor*       monitor = new KickPimMailMonitor( acc, this );
  KickPimMailMonitorThread* thread  = new KickPimMailMonitorThread( monitor );
  m_mailMonitorThreads.append( thread );
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL, "Created Mail Monitor Thread '"+acc->name()+"'");
  return thread;
}


void KickPimRepository::mailMonitors_FinishThread( KPMailAccount* acc )
{
  QPtrListIterator<KickPimMailMonitorThread> itor( m_mailMonitorThreads );
  KickPimMailMonitorThread* thread = 0;
  while ( thread==0 && itor.current()!=0 ) {
    thread = (KickPimMailMonitorThread*)itor.current();
    if (thread->monitor()->account() != acc) thread=0;
    ++itor;
  }
  if (thread != 0) {
    thread->terminateThread();
    m_mailMonitorThreads.remove( thread );
    delete thread;thread=0;
    if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL, "Terminated Mail Monitor Thread '"+acc->name()+"'");
  }
}

bool KickPimRepository::event(QEvent* event)
{
  if (event->type() == MailMonitorEvent::TYPE) {
    if (LogService::doLogInfo) LogService::logInfo( LogService::CAT_MAIL, "Received an event from a mail monitor");
    MailMonitorEvent* mailEvent = (MailMonitorEvent*)event;
    switch ( mailEvent->id() ) {
      case MailMonitorEvent::NEW_MAIL:
      case MailMonitorEvent::OLD_MAIL:
      case MailMonitorEvent::NO_MAIL:
      case MailMonitorEvent::NO_CONNECTION:
      case MailMonitorEvent::INVALID_LOGIN:
        emit emailsChanged();
        return true;
        break;
    }
  }
  return false;
}


void KickPimRepository::reload()
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","reload");

  clearEvents();   // first, clear the events (referencing contacts)
  clearContacts(); // then clear all contacts

  // reading contact data
  readContacts();
  readDistributionLists();

  // reading contact event data
  // problem: clears also normal events from the event-list
  readContactEvents();
  m_events.sort();

  // tell everybody: contacts and events have changed!
  emit contactListChanged();
  emit eventListChanged();
}

QString KickPimRepository::yearSign()
{
  return m_yearSign;
}

//
// Slots
//


void KickPimRepository::onContactListChanged( KPContactReader* )
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","onContactListChanged");

  reload();
}


void KickPimRepository::onEventListChanged( KPEventReader* )
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","onEventListChanged");
  // read events
  // problems: clears also contactEvents from the event-list
  readEvents();
  m_events.sort();

  // tell everybody: events have changed!
  emit eventListChanged();
}

void KickPimRepository::onEmailAcountsChanged( )
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","onEmailAcountsChanged");

  emit emailAccountsChanged(); // send signal that one or more accounts changed
}


void KickPimRepository::readContactEvents()
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","readContactEvents");

  m_events.clear();

  if (!m_contactEventReader) return;
  QDate date;
  int   count=0;

  KPContactEvent* event = (KPContactEvent*)m_contactEventReader->first();
  while (event)
  {
    date = event->date();
    if( date.isValid() )
    {
      int distance = distanceToDate(date,true);
      if (!m_options->listAnniLimit || distance<=m_options->listAnniFuture)
      {
        m_events.append( event );
        count++;
      }
    }
    event = (KPContactEvent*)m_contactEventReader->next();
  }
}

void KickPimRepository::readEvents()
{
  if (LogService::doLogCall) LogService::call(" KickPimRepository","readEvents");
}


//
// Options
//

KickPimOptions* KickPimRepository::options()
{
  return m_options;
}

//
// Checking ...
//




//
// Protected Methods
//

//
// Services
//

int KickPimRepository::compareEventDate( KPEvent* event1, KPEvent* event2 )
{
  if (event1==event2) return 0;
  if (event1==0) return -1;
  if (event2==0) return 1;
  
  QDate today =QDate().currentDate();  
  QDate date1 = event1->date();
  QDate date2 = event2->date();
  if (event1->ignoreYear()) date1.setYMD(  today.year(), date1.month(), date1.day() );
  if (event2->ignoreYear()) date2.setYMD(  today.year(), date2.month(), date2.day() );
  
  if (date1==date2) return 0;
  else if (date1>date2) return 1;
  else return -1;  
}

int KickPimRepository::dateIsNear( QDate date, bool ignoreYear, int future, int past )
{
  int distance = distanceToDate(date,ignoreYear);
  if      ( distance     <= future ) return 1;
  else if ( distance-365 >= -past  ) return -1;
  return 0;
}

int KickPimRepository::distanceToDate(QDate date, bool ignoreYear)
{
  QDate today = QDate().currentDate();

  // transform event into actual year
  if (ignoreYear)
  {
    // for leap years
    if ( !QDate::leapYear(today.year()) && date.month()==2 && date.day()==29 )
    {
      date.setYMD(today.year(), 3, 1);
    }
    else
    {
      date.setYMD(today.year(), date.month(), date.day() );
    }
  }

  int distance = today.daysTo(date);
  if (ignoreYear && distance<0)
  {
    date = date.addYears( 1 );    
    distance = today.daysTo(date);
  }

  return distance;
}

QString KickPimRepository::displayName( KPContact* contact )
{
  QString name="";
  if (!contact)
  {
    LogService::logWarn(LogService::CAT_REPOSITORY,"KickPimRepository: DisplayName with null-pointer called!");
    return name;
  }

  QStringMap names = contact->names();

  switch (m_options->displayMode)
  {
    case KickPimOptions::ADDR_FORMATTED:
      name = names["formattedName"];
      break;

    case KickPimOptions::ADDR_SURNAME:
      name = names["familyName"];
      if (!name.isEmpty() && !names["givenName"].isEmpty()) name+= ", "+names["givenName"];
      break;

    case KickPimOptions::ADDR_FORENAME:
      name = names["givenName"];
      if (!name.isEmpty() && !names["familyName"].isEmpty()) name+=" "+names["familyName"];
      break;

    default:
      LogService::logError(LogService::CAT_REPOSITORY, "KickPimRepository: Unknown display mode for names.");

  };
  if (name.isEmpty()) name = names["organizationName"];
  if (name.isEmpty()) {
    name = names["familyName"];
    if (!name.isEmpty() && !names["givenName"].isEmpty()) name += ", "+names["givenName"];
  }
  if (name.isEmpty()) {
    QStringList emailList = contact->emailAddresses();
    if (!emailList.isEmpty())
      name = "<" + emailList.first() + ">";
  }

  return name;
}


void KickPimRepository::clearContacts()
{
  KPContact* contact = m_contacts.first();
  while (contact) {
    delete contact;contact=0; // Bad: events may reference this contact :-(
    contact = m_contacts.next();
  }
  m_contacts.clear();
}

void KickPimRepository::clearEvents()
{
  KPEvent* event = m_events.first();
  while (event) {
    delete event;event=0;
    event = m_events.next();
  }
  m_events.clear();
}

void KickPimRepository::readContacts()
{
  if (LogService::doLogCall) LogService::call("KickPimRepository","readContacts");

  clearContacts();
  m_contactReader->reload();

  KPContact* contact = m_contactReader->first();
  while (contact) {
    m_contacts.append(contact);
    contact = m_contactReader->next();
  }
}

void KickPimRepository::readDistributionLists( )
{
  // TODO: Integrate feature into ContactReader!
  AddressBook* book = StdAddressBook::self();
  m_distLists.clear(); // TODO: Need to remove old distlists from memory?

  DistributionListManager* listMan = new DistributionListManager(book);
  listMan->load();
  QStringList list  = listMan->listNames();
  for( int i=list.size()-1; i>=0; i-- )
  {
     QString name = *list.at(i);
     m_distLists.append(listMan->list(name));
  }

  delete listMan;listMan=0;
}

//
// directories
//


QString& KickPimRepository::dirOfIcons()
{
  return m_dirIcons;
}

QString& KickPimRepository::dirOfInfo()
{
  return m_dirInfo;
}


int KickPimRepository::getWaitingEvents ( KPEventList& )
{
  return 0;
}

int KickPimRepository::getWaitingAnniversaries ( KPEventList& list )
{
  int     count    = 0;
  QString text     = "";
  QString eventText= "";
  
  KPEvent* event = m_events.first();
  KPContactEvent* contactEvent = 0;
  while ( event )
  {
    contactEvent = dynamic_cast<KPContactEvent*>(event);
    if (contactEvent)
    {
      // date and/or time of event
      QString dateTime = "";
      if (contactEvent->date().isValid())
      {
        int distance = KickPIM::rep()->distanceToDate(contactEvent->date(),contactEvent->ignoreYear());
        if (distance > - KickPIM::rep()->options()->remindAnniPast &&
            distance <   KickPIM::rep()->options()->remindAnniFuture )
        {
          count++;
          list.append(event);
        }
      }
    }
    event = m_events.next();
  }
  return count;
}




bool KickPimRepository::addContact()
{
  // Creating the new contact is enough here.
  // It will be auto inserted (argument:true)
  // when the edit dialog exits (contact->changed())

  KPKabContact* contact = new KPKabContact(true);
  return contact->edit();
}

void KickPimRepository::emergencySave() {
  // StdAddressBook::handleCrash();
}


QStringList& KickPimRepository::contactCategories()
{
  // the result list
  static QStringList catList;
  catList.clear();

  QStringList list;
  QString str;
  // get the categories of all contacts
  KPContact* contact = m_contacts.first();
  while (contact) {
    list = contact->categories();
    // insert all new categories to the complete list of categories
    for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
      str = (QString)*it;
      if  ( !catList.contains(str) ) catList += str;
    }
    // next contact
    contact = m_contacts.next();
  }
  catList.sort();
  return catList;
}
