/*
 * kickpimmailmonitor.cpp
 * Copyright (C) 1999-2002 Kurt Granroth <granroth@kde.org>
 *
 * This file contains the implementation of KickPimMailMonitor and
 * associated classes.
 *
 */
#include "kickpimmailmonitor.h"

#include <kmessagebox.h>

#include <sys/types.h>
#ifndef __STRICT_ANSI__
#define __STRICT_ANSI__
#include <sys/socket.h>
#undef __STRICT_ANSI__
#else
#include <sys/socket.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <utime.h>

#include <fcntl.h>
#include <errno.h>

#include <kdebug.h>
#include <krun.h>

#include <qapplication.h>
#include <klocale.h>
#include <qlabel.h>
#include <qstring.h>
#include <qregexp.h>
#include <qdir.h>
#include <qdatetime.h>
#include <ksimpleconfig.h>

#include "kpmailurl.h"
#include "kpmailaccount.h"

#include "../kickpim.h"

#define MAXSTR (1024)

#define MAIL_STATE_FILE "kickpimmail"

#if defined (_HPUX_SOURCE)
extern int h_errno;
#endif
  
static bool real_from(const QString& buffer);
static const char* compare_header(const char* header, const char* field);

KickPimMailMonitor::KickPimMailMonitor(KPMailAccount* account, QObject* eventListener)
    : QObject(),
      m_started(false),
      m_oldCount(-1),
      m_firstRun(false),
      simpleURL(""),
      port(0),
      preauth(false),
      keepalive(false),
      mailState(UnknownState),
      lastSize(0),
      imap(0),
      pop(0),
      nntp(0)
{
    m_eventListener = eventListener;
    if (LogService::doLogConstruct) LogService::construct("KickPimMailMonitor");
    lastRead.setTime_t(0);
    lastModified.setTime_t(0);
    b_new_lastSize     = false;
    b_new_lastRead     = false;
    b_new_lastModified = false;
    b_new_uidlList     = false;

    m_newMailsLabel = 0;
    m_allMailsLabel = 0;

    m_allCount      = account->numberOfEmails();
    m_oldCount      = m_allCount;    
    m_newCount      = 0;

    setData(account);
}

void KickPimMailMonitor::setData( KPMailAccount* accountData ) {
  m_mailAccount = accountData;
  initMailbox();
}

void KickPimMailMonitor::postEvent( MailMonitorEvent* event )
{
  m_mailAccount->setNumberOfEmails( m_allCount );
  QApplication::postEvent( m_eventListener, event );
}


void KickPimMailMonitor::setMailLabels(QLabel* newMails, QLabel* allMails)
{
  //if (m_newMailsLabel) m_newMailsLabel->setText("");
  m_newMailsLabel = newMails;
  //if (m_allMailsLabel) m_allMailsLabel->setText("");
  m_allMailsLabel = allMails;
  updateLabels();
}

KickPimMailMonitor::~KickPimMailMonitor()
{
  if (LogService::doLogConstruct) LogService::destruct("KickPimMailMonitor");
  uidlList.clear();
  delete imap; imap = 0;
  delete pop;  pop  = 0;
  delete nntp; nntp = 0;
  m_newMailsLabel=0;
  m_allMailsLabel=0;
}

void KickPimMailMonitor::readConfig()
{
/*
    KSimpleConfig *config = new KSimpleConfig(MAIL_STATE_FILE);
    config->setDollarExpansion(false);

    QString group;
    group = mailbox + "(" + m_mailAccount->name() + ")";
    config->setGroup(group);

    QStrList list;

    mailState = (KickPimMailState)config->readNumEntry("mailState", UnknownState);
    lastSize = config->readNumEntry("lastSize");
    config->readListEntry("lastRead", list);
    if (list.count()==6)
    {
        lastRead.setDate(QDate(atoi(list.at(0)),atoi(list.at(1)),atoi(list.at(2))));
        lastRead.setTime(QTime(atoi(list.at(3)),atoi(list.at(4)),atoi(list.at(5))));
    }
    config->readListEntry("lastModified", list);
    if (list.count()==6)
    {
      lastModified.setDate(QDate(atoi(list.at(0)),atoi(list.at(1)),atoi(list.at(2))));
      lastModified.setTime(QTime(atoi(list.at(3)),atoi(list.at(4)),atoi(list.at(5))));
    }
    config->readListEntry("uidlList", list);

    char *UIDL;
    uidlList.clear();
    for (UIDL = list.first(); UIDL != 0; UIDL = list.next())
    {
        uidlList.append( new QString(UIDL) );
    }

    m_newCount = config->readNumEntry("m_newCount", 0);
    m_oldCount  = config->readNumEntry("m_oldCount", -1);

    delete config;
*/
}

void KickPimMailMonitor::saveConfig()
{
/*
    // open the config file
    KSimpleConfig *config = new KSimpleConfig(MAIL_STATE_FILE);
    config->setDollarExpansion(false);

    QString group;
    group = mailbox + "(" + m_mailAccount->name() + ")";
    config->setGroup(group);

    QStringList uidlist;
    QString *UIDL;
    for (UIDL = uidlList.first(); UIDL != 0; UIDL = uidlList.next())
    {
        uidlist.append(*UIDL);
    }

    config->writeEntry("mailState", (int)mailState);
    config->writeEntry("lastSize", lastSize);
    config->writeEntry("lastRead",lastRead);
    config->writeEntry("lastModified",lastModified);
    config->writeEntry("uidlList",uidlist);
    config->writeEntry("m_newCount", m_newCount);
    config->writeEntry("m_oldCount", m_oldCount);

    delete config;
*/
}

void KickPimMailMonitor::updateLabels()
{
  // if (LogService::doLogCall) LogService::call("KickPimMailMonitor","updateLabels");

  if (account()->isActive()) {
    if (m_newMailsLabel) {
      int diff = m_newCount;
      QString text = "";
      switch (mailState) {
        case NewMail:
        case NoMail:
        case OldMail:
            text = QString::number( diff );
            if (diff>0) text = "(+"+text+")";
            else        text = "";
            break;
        default:
            text="";
      }
      m_newMailsLabel->setText(text);
      m_newMailsLabel->update();
    }

    if (m_allMailsLabel) {
      QString text="?";
      switch (mailState){
        case NoConn:
            text="!";
            break;
        case NewMail:
        case NoMail:
        case OldMail:
        default:
          text = "-";
          if (m_allCount>=0) text = QString::number( m_allCount );
      }
      m_allMailsLabel->setText(text);
    }
  }
  else if (!account()->isActive()) {
    if (m_allMailsLabel!=0) m_allMailsLabel->setText("");
    if (m_newMailsLabel!=0) m_newMailsLabel->setText("");
    if (m_allMailsLabel!=0) m_allMailsLabel->update();
    if (m_newMailsLabel!=0) m_newMailsLabel->update();
  }
}

void KickPimMailMonitor::onStateChanged()
{
  FUNC_CALL(KickPimMailMonitor::onStateChanged)

  saveConfig();
}

/*void KickPimMailMonitor::start()
{
  int ival = m_mailAccount->pollInterval();
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL, "KickPimMailMonitor: Start polling '"+account()->name()+"' "
                                            "all " + QString::number(ival) + " seconds.");

  readConfig();
  m_started  = true;
  m_firstRun = true;

  resetMailCount();
}
*/

/*
void KickPimMailMonitor::stop()
{
  lastSize   = 0;
  mailState  = UnknownState;
  m_started    = false;
  lastRead.setTime_t(0);
  lastModified.setTime_t(0);
  uidlList.clear();
}
*/

void KickPimMailMonitor::resetMailCount()
{
  if (LogService::doLogCall) LogService::call("KickPimMailMonitor","resetMailCount");
  m_oldCount = m_allCount;
  m_newCount = 0;
  updateLabels();
}



void KickPimMailMonitor::initMailbox()
{
  if (LogService::doLogCall) LogService::call("KickPimMailMonitor","initMailbox");

  delete imap; imap = 0;
  delete pop;  pop  = 0;
  delete nntp; nntp = 0;

  disconnect(this);

  KPMailURL url      = m_mailAccount->url();
  QString   protocol = url.protocol();
  // ** IMAP **
  if (protocol == "imap4") {
    LogService::logInfo( LogService::CAT_MAIL, "Initializing IMAP4");
    if (imap!=0) delete imap;
    imap = new KickPimMailImap();

    mailbox    = url.path().right(url.path().length() - 1);
    port       = (url.port() > 0) ? url.port() : 143;

    preauth    = url.query() == "?preauth" ?  true : false;
    keepalive  = url.ref()   == "keepalive" ? true : false;

    bool async = true;
    // not used yet: async = url.ref()   == "async" ?     true : false;
    imap->setAsync(async);
    simpleURL  = "imap4://" + url.host();
    if (mailbox.length()>0) simpleURL = simpleURL + "/" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkImap()));
  }
  // ** POP3 **
  else if (protocol == "pop3") {
    LogService::logInfo( LogService::CAT_MAIL, "Initializing POP3");
    if (pop!=0) delete pop;
    pop = new KickPimMailPop();

    mailbox    = url.user();
    port       = (url.port() > 0) ? url.port() : 110;

    keepalive  = (url.ref() == "keepalive") ? true : false;

    bool async = true;
    // not used yet: async = (url.ref() == "async")     ? true : false;
    pop->setAsync(async);
    simpleURL  = "pop3://" + url.host() + "/" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkPop()));
  }
  // ** MBOX **
  else if (protocol == "mbox") {
    LogService::logInfo( LogService::CAT_MAIL, "Initializing MBOX");
    mailbox   = url.path();
    simpleURL = "mbox:" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkMbox()));
  }
  // ** FILE **
  else if (protocol == "file") {
  LogService::logInfo( LogService::CAT_MAIL, "Initializing File Mailbox");
    mailbox = url.path();
    simpleURL = "file:" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkLocal()));
  }
  // ** MAILDIR **
  else if (protocol == "maildir") {
    mailbox = url.path();
    simpleURL = "maildir:" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkMaildir()));
  }
  // ** MH **
  else if (protocol == "mh") {
    mailbox = url.path();
    simpleURL = "mh:" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkMHdir()));
  }
  // ** NNTP **
  else if (protocol == "nntp")
  {
    if (nntp!=0) delete nntp;
    nntp = new KickPimMailNntp;

    mailbox  = url.path().right(url.path().length() - 1);
    port     = (url.port() > 0) ? url.port() : 119;

    keepalive = url.ref() == "keepalive" ? true : false;
    bool async = url.ref() == "async" ? true : false;
    nntp->setAsync(async);
    simpleURL = "nntp://" + url.host() + "/" + mailbox;
    connect(this, SIGNAL(signal_checkMail()), SLOT(checkNntp()));
  }
  fetchCommand = m_mailAccount->url().searchPar("fetch");
}

void KickPimMailMonitor::setMailboxIsRead()
{
/*
  lastRead  = QDateTime::currentDateTime();
  if (mailState == NewMail)
  {
    if (b_new_lastSize)     lastSize     = new_lastSize;
    if (b_new_lastRead)     lastRead     = new_lastRead;
    if (b_new_lastModified) lastModified = new_lastModified;
    if (b_new_uidlList)     uidlList     = new_uidlList;

    if (m_allCount!=-1) m_allCount += m_newCount;
    m_newCount = 0;
    b_new_lastSize = false;
    b_new_lastRead = false;
    b_new_lastModified = false;
    b_new_uidlList = false;

    determineState(OldMail);
  }
*/
}

/** Initiates the monitor to check mails (emit signal)
  * if the account is neither paused nor inactivated.
  */
void KickPimMailMonitor::checkMailNow()
{
  if (account()->isActive()) {
    // m_newCount=0;
    emit( signal_checkMail() );
  }
}

void KickPimMailMonitor::checkLocal()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Checking local Mailbox '"+ m_mailAccount->name()+"'");

  // get the information about this local mailbox
  QFileInfo mbox(mailbox);

  // run external fetch client
  if (!fetchCommand.isEmpty()) {
      if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"  Fetch Command = '"+ fetchCommand+"'");
      // TODO: commented due to thread saveness (don't signal to outside listeners)      
      KRun::runCommand( fetchCommand );
  }

  // check if we have new mail
  m_newCount = 0;
  determineState(mbox.size(), mbox.lastRead(), mbox.lastModified());

  m_firstRun = false;
}

void KickPimMailMonitor::checkMbox()
{
  // get the information about this local mailbox  
  QFileInfo mbox(mailbox);
  
  // run external fetch client
  if (!fetchCommand.isEmpty()) {
    KRun::runCommand( fetchCommand );
  }

  unsigned int curSize  = mbox.size();  
  QDateTime curLastMod  = mbox.lastModified();
  QDateTime curLastRead = mbox.lastRead();
  if (LogService::doLogInfo) {
    LogService::logInfo(LogService::CAT_MAIL,"Checking Mbox-Mailbox '"+m_mailAccount->name()+"'");
    LogService::logInfo(LogService::CAT_MAIL,"  - mailbox =  '"+mailbox+"'");
    LogService::logInfo(LogService::CAT_MAIL,"  - fetch   =  '"+fetchCommand+"'");
    LogService::logInfo(LogService::CAT_MAIL,"  - size    =  "+QString::number(curSize)+" ["+QString::number(lastSize)+"]" );
    LogService::logInfo(LogService::CAT_MAIL,"  - modified=  "+curLastMod.toString()   +" ["+lastModified.toString()  +"]" );
    LogService::logInfo(LogService::CAT_MAIL,"  - read    =  "+curLastRead.toString() );    
    
  }      
  
  // see if the state has changed
  if (   ( curLastMod    != lastModified )
      || ( curSize       != lastSize     )
      || ( mailState     == UnknownState )
      || ( m_oldCount    == -1           ) )
  {
    lastModified = curLastMod;
    lastSize     = curSize;

    // ok, the state *has* changed.  see if the number of new 
    // messages has, too.
    cout << "mboxMessages pre:  curCount = " << m_allCount << "   oldCount = " << m_oldCount << "  newCount = " << m_newCount << endl;
    m_allCount = mboxMessages();    
    cout << "mboxMessages post: curCount = " << m_allCount << "   oldCount = " << m_oldCount << "  newCount = " << m_newCount << endl;
    
    // Set access time of the file to what it was. If we don't do
    // this some (all?) MUAs think that the mail has already been
    // read.
    utimbuf buf;    
    QDateTime unixEpoch;
    unixEpoch.setTime_t(0); // 'setTime_t' takes timezone into account
    // do a simple test for those systems with hosed timezones.  if
    // the lastRead and calculated lastRead aren't the same, then
    // something is wrong with the system
    QDateTime calc;
    calc.setTime_t( -curLastRead.secsTo(unixEpoch) );
    if (curLastRead == calc) {
      // QDateTime::secsTo returns a negative value if the argument
      // is before the time the object refers to.
      buf.actime  = -curLastRead.secsTo(unixEpoch);
      buf.modtime = -curLastMod.secsTo(unixEpoch);
      utime( QFile::encodeName(mailbox), &buf );
    }

    // if there are any new messages, consider the state New
    m_newCount = m_allCount - m_oldCount;
    if ( m_newCount > 0 ) {      
      determineState(NewMail);
    }
    else {
      if (m_allCount == 0) determineState(NoMail);
      else                 determineState(OldMail);
    }    
  }
  else if (m_firstRun) {
    KickPimMailState state(mailState);
    mailState = UnknownState;
    determineState(state);
  }

  m_firstRun = false;
}

void KickPimMailMonitor::checkPop()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) "
                                           "Checking Mailbox '"+m_mailAccount->name()+"'");  

  m_firstRun = false;
  
  // try to log on to the POP server
  if ( !pop->active() ) {
    // connect to the server unless it is active already
    QString server = m_mailAccount->url().host();
    QString user   = m_mailAccount->url().user();
    QString passw  = m_mailAccount->url().pass();

    if ( !pop->connectSocket(server, port) ) {
      LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Connection not successful");
      determineState(NoConn);
      return;
    }
    if ( !pop->command( "USER "+user+"\r\n" )) {
      pop->close();
      invalidLogin();
      LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Error with command 'USER "+user+"'");
      return;
    }
    if ( !pop->command("PASS "+passw+"\r\n") ) {
      pop->close();
      invalidLogin();
      LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Error with command 'PASS "+passw+"'");
      return;
    }
  }

  // get the list of mails (try three techniques)
  if ( !pop->command( "UIDL\r\n" ) ) {
    if ( !pop->command( "STAT\r\n" ) ) {
      if ( !pop->command( "LIST\r\n" ) ) {
        // if this still doesn't work, then we close this port
        pop->close();
        LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Error with commands 'UIDL, STAT and LIST'! Closed connection.");
        return;
      }
    }
    else // STAT was successfull
    {
      int count = pop->numberOfMessages();
      LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Command 'UIDL' not successful but 'STAT'.");
      determineState(count);
    }
  }
  else // UIDL was successfull
  {
    m_allCount = uidlList.count();
    m_newCount = m_allCount - m_oldCount;
    if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailMonitor (Pop3) Command 'UIDL' successful: "
                                              +QString::number(m_newCount)+"/"+QString::number(m_allCount)+" mails");
    determineState( pop->getUidlList() );
  }

  if (!keepalive) pop->close();
}

void KickPimMailMonitor::checkImap()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailMonitor (IMAP) "
                                           "Checking Mailbox '"+m_mailAccount->name()+"'");

  m_firstRun = false;

  QString command;
  int seq = 1000;
  bool do_login = false;

  QString server = m_mailAccount->url().host();
  QString user   = m_mailAccount->url().user();
  QString passw  = m_mailAccount->url().pass();

  // run external client (probably to setup SSL)
  if (!fetchCommand.isEmpty()) {
    LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (IMAP) Fetch-Command not successful");
    // TODO: commented due to thread saveness (don't signal to outside listeners)
    // emit(signal_fetchMail(fetchCommand));

    // sleep a bit to allow the connection to take place
    sleep(1);
  }

  // connect to the server
  if (imap->active() == false) {
    if (imap->connectSocket(server, port) == false) {
        LogService::logError(LogService::CAT_MAIL,"KickPimMailMonitor (IMAP) Connect not successful");
        invalidLogin();
        return;
    }
    do_login = true;
  }

  // imap allows spaces in usernames... we need to take care of that
  user = imap->mungeUserPass(user);

  // also asterisks (*) in passwords. maybe it's a good idea
  // to _always_ munge the user and the password.
  passw = imap->mungeUserPass(passw);

  // if we are preauthorized OR we want to keep the session alive, then
  // we don't login.  Otherwise, we do.
  if ((preauth == false) && (do_login == true)) {
      command = QString().setNum(seq) + " LOGIN " + user + " " + passw + "\r\n";
      if (imap->command(command, seq) == false) {
          invalidLogin();
          return;
      }
      seq++;
  }

  // reset the numbers from the last check
  imap->resetNumbers();
  // select the mailbox
  command = QString().setNum(seq) + " EXAMINE " + mailbox + "\r\n";
  if (imap->command(command, seq) == false) return;
  seq++;

  // this gets the number of undeleted messages
  command = QString().setNum(seq) + " SEARCH UNDELETED\r\n";
  if (imap->command(command, seq) == false) return;
  seq++;
  // this gets the number of undeleted _new_ messages
  command = QString().setNum(seq) + " SEARCH UNDELETED UNSEEN\r\n";
  if (imap->command(command, seq) == false) return;
  seq++;

  // lets not logout if we want to keep the session alive
  if (keepalive == false) {
      command = QString().setNum(seq) + " LOGOUT\r\n";
      if (imap->command(command, seq) == false) return;
      imap->close();
  }

  // what state are we in?
  if (imap->numberOfMessages() == 0) {
    m_newCount = 0;
    determineState(NoMail);
  }
  else {    
    m_allCount  = imap->numberOfMessages();    
    m_newCount  = m_allCount - m_oldCount;
    if (m_newCount > 0) determineState(NewMail);
    else if (m_allCount>0) determineState(OldMail);
    else determineState(NoMail);
  }
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailMonitor (IMAP): "
                      +QString::number(m_newCount)+"/"+QString::number(m_allCount)+" mails");
}

void KickPimMailMonitor::checkMaildir()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Checking MailDir-Mailbox '"+m_mailAccount->name()+"'");
  m_firstRun = false;

  // get the information about this local mailbox
  QDir mbox(mailbox);

  // run external fetch client
  if (!fetchCommand.isEmpty()) {
    // TODO: commented due to thread saveness (don't signal to outside listeners)
    // emit(signal_fetchMail(fetchCommand));
  }

  // make sure the mailbox exists
  if (mbox.exists())
  {
    // maildir stores its mail in MAILDIR/new and MAILDIR/cur
    QDir new_mailbox(mailbox + "/new");
    QDir cur_mailbox(mailbox + "/cur");

    // make sure both exist
    if (new_mailbox.exists() && cur_mailbox.exists())
    {
      // check only files
      new_mailbox.setFilter(QDir::Files);
      cur_mailbox.setFilter(QDir::Files);

      // determining "new" (or "unread") mail in maildir folders
      // is a *little* tricky.  all mail in the 'new' folder are
      // new, of course... but so is all mail in the 'cur'
      // folder that doesn't have a ':2,[R|S|T]' after it.
      m_newCount = new_mailbox.count();
      m_allCount = cur_mailbox.count();

      const QFileInfoList *cur_list = cur_mailbox.entryInfoList();
      QFileInfoListIterator it(*cur_list);
      QFileInfo *info;

      static QRegExp suffix(":2,?R?S?T?$");
      while ((info = it.current())) {
        if (info->fileName().findRev(suffix) == -1) {
          m_newCount++;
          m_allCount--;
        }
        ++it;
      }

      // all messages in 'new' are new
      if (m_newCount > 0) {
        determineState(NewMail);
      }
      // failing that, we look for any old ones
      else if (m_allCount > 0) {
        determineState(OldMail);
      }
      // failing that, we have no mail
      else {
        determineState(NoMail);
      }
    }
  }
}

void KickPimMailMonitor::checkNntp()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailMonitor (Nntp) Checking NNTP-Mailbox '"+m_mailAccount->name()+"'");
  m_firstRun = false;

  QString command;
  bool do_login = false;

  QString server = m_mailAccount->url().host();
  QString user   = m_mailAccount->url().user();
  QString passw  = m_mailAccount->url().pass();

  // connect to the server
  if (nntp->active() == false)
  {
      if (nntp->connectSocket(server, port) == false)
      {
          determineState(NoConn);
          return;
      }

      do_login = true;
  }

  // if we are preauthorized OR we want to keep the session alive, then
  // we don't login.  Otherwise, we do.
  if ((preauth == false) && (do_login == true))
  {
      if (user.isEmpty() == false)
      {
          command = "authinfo user " + user + "\r\n";
          if (nntp->command(command) == false)
              return;
      }
      if (passw.isEmpty() == false)
      {
          command = "authinfo pass " + passw + "\r\n";
          if (nntp->command(command) == false)
              return;
      }
  }

  command = "group " + mailbox + "\r\n";
  if (nntp->command(command) == false)
      return;

  // lets not logout if we want to keep the session alive
  if (keepalive == false)
  {
      command = "QUIT\r\n";
      nntp->command(command);
      nntp->close();
  }

  // now, we process the .newsrc file
  QString home(getenv("HOME"));
  QString newsrc_path(home + "/.newsrc");
  QFile newsrc(newsrc_path);
  if (newsrc.open(IO_ReadOnly) == false)
  {
      return;
  }

  char c_buffer[MAXSTR];
  while(newsrc.readLine(c_buffer, MAXSTR) > 0)
  {
      // search for our mailbox name
      QString str_buffer(c_buffer);
      if (str_buffer.left(mailbox.length()) != mailbox)
          continue;

      // we now have our mailbox.  this parsing routine is so
      // ugly, however, that I could almost cry.  it assumes way
      // too much.  the "actual" range MUST be 1-something
      // continuously and our read sequence MUST be sequentially in
      // order
      bool range = false;
      int last = 1;
      m_newCount = 0;
      char *buffer = c_buffer;

      // skip over the mailbox name
      for(; buffer && *buffer != ' '; buffer++);

      // iterate over the sequence until we hit a newline or end of string
      while (buffer && *buffer != '\n' && *buffer != '\0')
      {
          // make sure that this is a digit
          if (!isdigit(*buffer))
          {
              buffer++;
              continue;
          }

          // okay, what digit are we looking at?  atoi() will convert
          // only those digits it recognizes to an it.  this will handily
          // skip spaces, dashes, commas, etc
          char *digit = buffer;
          int current = atoi(digit);

          // if our current digit is greater than is possible, then we
          // should just quit while we're (somewhat) ahead
          if (current > nntp->last())
              break;

          // we treat our sequences different ways if we are in a range
          // or not.  specifically, if we are in the top half of a range,
          // we don't do anything
          if (range == false)
          {
              if (current > last)
                  m_newCount += current - last - 1;
          }
          else
              range = false;

          // set our 'last' one for the next go-round
          last = current;

          // skip over all of these digits
          for(;buffer && isdigit(*buffer); buffer++);

          // is this a range?
          if (*buffer == '-')
              range = true;
      }

      // get the last few new ones
      if (last < nntp->last())
          m_newCount += nntp->last() - last;

      break;
  }
  // with newsgroups, it is either new or non-existant.  it
  // doesn't make sense to count the number of read mails
  if (m_newCount > 0)
      determineState(NewMail);
  else
      determineState(OldMail);
}

/*
 * MH support provided by David Woodhouse <David.Woodhouse@mvhi.com>
 */
void KickPimMailMonitor::checkMHdir()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Checking MHdir-Mailbox '"+m_mailAccount->name()+"'");

  m_firstRun = false;

  // get the information about this local mailbox
  QDir mbox(mailbox);
  char the_buffer[MAXSTR];
  char *buffer = the_buffer;

  // run external fetch client
  if (!fetchCommand.isEmpty()) {
    // TODO: commented due to thread saveness (don't signal to outside listeners)
    // emit(signal_fetchMail(fetchCommand));
  }

  // make sure the mailbox exists
  if (mbox.exists())
  {
    QFile mhseq(mailbox+"/.mh_sequences");
    if (mhseq.open(IO_ReadOnly) == true)
    {
      // Check the .mh_sequences file for 'unseen:'

      buffer[MAXSTR-1]=0;

      while(mhseq.readLine(buffer, MAXSTR-2) > 0)
      {
        if (!strchr(buffer, '\n') && !mhseq.atEnd())
        {
          // read till the end of the line

          int c;
          while((c=mhseq.getch()) >=0 && c !='\n');
        }
        if (!strncmp(buffer, "unseen:", 7))
        {
          // There are unseen messages
          // we will now attempt to count exactly how
          // many new messages there are

          // an unseen sequence looks something like so:
          // unseen: 1, 5-9, 27, 35-41
          bool range = false;
          int last = 0;

          // initialize the number of new messages
          m_newCount = 0;

          // jump to the correct position and iterate through the
          // rest of the buffer
          buffer+=7;
          while(*buffer != '\n' && buffer)
          {
            // is this a digit?  if so, it is the first of possibly
            // several digits
            if (isdigit(*buffer))
            {
              // whether or not this is a range, we are guaranteed
              // of at least *one* new message
              m_newCount++;

              // get a handle to this digit.  atoi() will convert
              // only those digits it recognizes to an int.  so
              // atoi("123garbage") would become 123
              char *digit = buffer;

              // if we are in the second half of a range, we need
              // to compute the number of new messages.
              if (range)
              {
                // remember that we have already counted the
                // two extremes.. hence we need to subtract one.
                m_newCount += atoi(digit) - last - 1;
                range = false;
              }

              // skip over all digits
              for(;buffer && isdigit(*buffer); buffer++);

              // check if we are in a range
              if (*buffer == '-')
              {
                // save the current digit for later computing
                last = atoi(digit);
                range = true;
              }
            }
            else
                buffer++;
          }
          mhseq.close();
          determineState(NewMail);
          return;
        }
      }
      mhseq.close();
    }

    // OK. No new messages listed in .mh_sequences. Check if
    //  there are any old ones.
    //mbox.setFilter(QDir::Files);
    QStringList mails = mbox.entryList(QDir::Files);
    QStringList::Iterator str;

    for (str = mails.begin(); str != mails.end(); str++)
    {
      uint index;
      // Check each file in the directory.
      // If it's a numeric filename, then it's a mail.

      for (index = 0; index < (*str).length(); index++)
      {
        if (!(*str).at(index).isDigit())
            break;
      }
      if (index >= (*str).length())
      {
        // We found a filename which was entirely
        // made up of digits - it's a real mail, so
        // respond accordingly.

        determineState(OldMail);
        return;
      }
    }

    // We haven't found any valid filenames. No Mail.
    determineState(NoMail);
  }
}


/**
 * Takes the complete amount of existing messages and calculate the
 * m_newCount and the State.
 */
void KickPimMailMonitor::determineState(unsigned int count)
{  
  // NoMail
  if (count == 0) {
    if (mailState != NoMail) {
      mailState  = NoMail;
      m_allCount = m_newCount = m_oldCount = 0;      
      postEvent( new MailMonitorEvent(MailMonitorEvent::NO_MAIL) );
      onStateChanged();
    }
    return;
  }
  // new mail
  if (count > m_oldCount) {    
    m_allCount  = count;
    m_newCount  = m_allCount - m_oldCount;
    mailState = NewMail;
    postEvent( new MailMonitorEvent(MailMonitorEvent::NEW_MAIL) );
    onStateChanged();    
    return;
  }

  // if we have *some* mail, but the state is unknown, then we'll consider it's old
  if (mailState == UnknownState)
  {
    mailState = OldMail;
    m_allCount  = count;
    m_newCount  = 0;
    postEvent( new MailMonitorEvent(MailMonitorEvent::OLD_MAIL) );        

    onStateChanged();
    return;
  }
  
  // old mail
  if (count < m_allCount)  {
    if (mailState != OldMail)
    {
      mailState = OldMail;
      m_allCount = count;
      m_newCount = 0;
      postEvent( new MailMonitorEvent(MailMonitorEvent::OLD_MAIL) );
      onStateChanged();
    }
  }

  updateLabels();
}

/**
 * This method tries to calculate the number of new messages from 
 * a UidlList (whateber that is :-)
 */
void KickPimMailMonitor::determineState(KickPimMailUidlList uidl_list)
{
  QString *UIDL;
  unsigned int messages = 0;

  m_allCount = uidl_list.count();
  
  if (m_allCount==0) // no mails
  {
    m_allCount = m_newCount = m_oldCount = 0;
    if (mailState != NoMail)
    {
      lastSize  = 0;      
      mailState = NoMail;
      postEvent( new MailMonitorEvent(MailMonitorEvent::NO_MAIL) );
      //emit(signal_noMail());
      //emit(signal_noMail(simpleURL));
      onStateChanged();
    }
  }
  else
  {
    // if a member of uidl_list is not in the old uidlList then we have new mail
    for (UIDL = uidl_list.first(); UIDL != 0; UIDL = uidl_list.next())
    {
      // If we already have new mail use new_uidlList to see if we have more new messages
      switch (b_new_uidlList)
      {
        case true:  { if (new_uidlList.find(UIDL) == -1) messages++; } break;
        case false: { if (uidlList.find(UIDL)     == -1) messages++; } break;
      }
    }
    // if there are any new messages, then notify..
    if (messages > 0)
    {
      mailState = NewMail;

      postEvent( new MailMonitorEvent(MailMonitorEvent::NEW_MAIL) );
      //emit(signal_newMail());
      //emit(signal_newMail(m_newCount, m_mailAccount->name()));
      onStateChanged();
      // now update m_newCount
      if (b_new_uidlList) 
      { // if we have used new_uidlList for a check
        m_newCount += messages;
      }
      else
      { // if we have used uidlList for a check        
        m_newCount = messages;
      }
      new_uidlList = uidl_list;
      b_new_uidlList = true;
    }
    // this is horrible.  it will reset kickpimmail to OldMail the very next
    // time a pop3 mailbox is checked.  i don't know of a way around
    // this, though :-(
    // MZ: what's wrong with that?
    else if ( (!b_new_uidlList) && mailState != OldMail)
    {
      m_newCount = 0;
      mailState = OldMail;
      postEvent( new MailMonitorEvent(MailMonitorEvent::OLD_MAIL) );
      //emit(signal_oldMail());
      //emit(signal_oldMail(simpleURL));
      onStateChanged();
    }
  }
  //emit(signal_currentStatus(m_newCount, m_mailAccount->name(), mailState));

  updateLabels();
}

/**
 * This method doesnt set any 'count'-Members like m_newCount and m_allCount.
 * This has to  be done before calling the method.
 */
void KickPimMailMonitor::determineState(KickPimMailState state)
{  
  mailStateMessage = "";
  if ( (state == NewMail) && (mailState != NewMail) )
  {    
    mailState = NewMail;
    postEvent( new MailMonitorEvent(MailMonitorEvent::NEW_MAIL) );
    onStateChanged();
  }
  else
  if ((state == NoMail) && (mailState != NoMail))
  {
    mailState = NoMail;
    postEvent( new MailMonitorEvent(MailMonitorEvent::NO_MAIL) );
    onStateChanged();
  }
  else
  if ((state == OldMail) && (mailState != OldMail))
  {
    mailState = OldMail;
    postEvent( new MailMonitorEvent(MailMonitorEvent::OLD_MAIL) );
    onStateChanged();
  }
  else
  if ((state == NoConn) && (mailState != NoConn))
  {
    mailStateMessage = "KickPimMailMonitor: Connection failed to '"+simpleURL+"'";
    mailState = NoConn;
    postEvent( new MailMonitorEvent(MailMonitorEvent::NO_CONNECTION) );
    LogService::logError(LogService::CAT_MAIL,mailStateMessage);
    onStateChanged();
  }
  //emit(signal_currentStatus(m_newCount, m_mailAccount->name(), mailState));

  updateLabels();
}

void KickPimMailMonitor::determineState(unsigned int size, const QDateTime& last_read, const QDateTime& last_modified)
{
  cout << "*** a" << endl;
  // Check for NoMail
  if (size == 0) {
    cout << "*** a1" << endl;
    // Is this a new state?
    if (mailState != NoMail) {
      // Yes, the user has just nuked the entire mailbox
      mailState  = NoMail;
      lastRead   = last_read;
      lastSize   = 0;

      // Let the world know of the new state
      postEvent( new MailMonitorEvent(MailMonitorEvent::NO_MAIL) );
      //emit(signal_noMail());
      //emit(signal_noMail(simpleURL));
      onStateChanged();
    }
  }
  else {
    cout << "*** a2" << endl;
    // There is some mail.  See if it is new or not.  To be new, the
    // mailbox must have been modified after it was last read AND the
    // current size must be greater then it was before.
    if (last_modified>=last_read && size>lastSize) {
      cout << "*** a2a" << endl;
      if (!b_new_lastSize || size>new_lastSize) {
        mailState = NewMail;
        // Let the world know of the new state
        postEvent( new MailMonitorEvent(MailMonitorEvent::NEW_MAIL) );
        //emit(signal_newMail());
        //emit(signal_newMail(1, m_mailAccount->name()));
        onStateChanged();
      }
      new_lastSize = size;
      b_new_lastSize = true;
      new_lastRead = last_read;
      b_new_lastRead = true;
      m_newCount  = 1;
    }
    else {
      cout << "*** a2b" << endl;
      // Finally, check if the state needs to change to OldMail
      if ((mailState != OldMail) && (last_read > lastRead)) {
        mailState = OldMail;
        lastRead  = last_read;
        lastSize  = size;

        // Let the world know of the new state
        postEvent( new MailMonitorEvent(MailMonitorEvent::OLD_MAIL) );
        //emit(signal_oldMail());
        //emit(signal_oldMail(simpleURL));
        onStateChanged();
      }
    }
  }

  // If we get to this point, then the state now is exactly the
  // same as the state when last we checked.  Do nothing at this
  // point.
  //emit(signal_currentStatus(m_newCount, m_mailAccount->name(), mailState));
  cout << "*** end " << endl;
  updateLabels();
}

/**
 * The following function is lifted from unixdrop.cpp in the korn
 * distribution.  It is (C) Sirtaj Singh Kang <taj@kde.org> and is
 * used under the GPL license (and the author's permission).  It has
 * been slightly modified for formatting reasons.
 */
int KickPimMailMonitor::mboxMessages()
{
    QFile mbox(mailbox);
    char buffer[MAXSTR];
    int  count            = 0;
    int  msg_count        = 0;
    bool in_header       = false;
    bool has_content_len = false;
    bool msg_read        = false;
    long content_length  = 0;

    m_oldCount = 0; // TODO: don't reset m_oldCount
    m_allCount = 0;

    if ( !mbox.open(IO_ReadOnly) ) return 0;

    buffer[MAXSTR-1] = 0;

    while (mbox.readLine(buffer, MAXSTR-2) > 0)
    {
      // read a line from the mailbox
      if (!strchr(buffer, '\n') && !mbox.atEnd())
      {
        // read till the end of the line
        int c; while( (c=mbox.getch()) >=0 && c !='\n' ) {;}
      }

      // check if this is the start of a message
      if (!in_header && real_from(buffer))
      {        
        has_content_len = false;
        in_header       = true;
        msg_read        = false;
      }
      else if (in_header)
      {
        // check header fields if we're already in one
        if (compare_header(buffer, "Content-Length"))
        {
          has_content_len = true;
          content_length  = atol(buffer + 15);
        }
        // This should handle those folders that double as IMAP or POP
        // folders.  Possibly PINE uses these always
        if (strcmp(buffer, "Subject: DON'T DELETE THIS MESSAGE -- FOLDER INTERNAL DATA\n") == 0)
        {
          m_oldCount--;
          m_allCount--;
        }
        else
        {
          if (compare_header(buffer, "Status"))
          {
            const char *field = buffer;
            field += 7;
            while (field && (*field== ' ' || *field == '\t')) field++;

            if (*field == 'N' || *field == 'U' || *field == 0x0a) msg_read = false;
            else msg_read = true;
          }
          // Netscape *sometimes* uses X-Mozilla-Status to determine
          // unread vs read mail.  The only pattern I could see for
          // sure, though, was that Read mails m_started with an '8'.
          // I make no guarantees on this...
          else if (compare_header(buffer, "X-Mozilla-Status"))
          {
            const char *field = buffer;
            field += 17;
            while (field && (*field== ' ' || *field == '\t')) { field++; }

            if (*field == '8') msg_read = true;
              else               msg_read = false;
          }
          else if (buffer[0] == '\n' )
          {
            if (has_content_len) mbox.at(mbox.at() + content_length);
            in_header = false;
            m_oldCount++;
            if (!msg_read) count++;
            else           m_allCount++;
          }
        }
      }//in header

      msg_count++;
      
      if(msg_count >= 100 )
      {
        qApp->processEvents();
        msg_count = 0;
      }
    }//while

    mbox.close();
    return count;
}

void KickPimMailMonitor::invalidLogin()
{
  LogService::logWarn(LogService::CAT_MAIL,"KickPimMailMonitor: Invalid Login");
  
  // first, we stop this monitor to be on the safe side  
  determineState(NoConn);
  m_newCount = -1;

  postEvent( new MailMonitorEvent(MailMonitorEvent::INVALID_LOGIN) );  
}

///////////////////////////////////////////////////////////////////////////
// KickPimMailSocket
///////////////////////////////////////////////////////////////////////////
KickPimMailSocket::KickPimMailSocket() : async(false), socketFD(-1), messages(0), newMessages(-1)
{
    FD_ZERO(&socketFDS);

    /*
     * Set the socketTO once and DO NOT use it in any select call as this
     * may alter its value!
     */
    socketTO.tv_sec = SOCKET_TIMEOUT;
    socketTO.tv_usec = 0;
}

KickPimMailSocket::~KickPimMailSocket()
{
    close();
}


int KickPimMailSocket::numberOfMessages()
{
    return messages;
}

int KickPimMailSocket::numberOfNewMessages()
{
    return (newMessages > -1) ? newMessages : 0;
}

void KickPimMailSocket::close()
{

    if (socketFD != -1)
        ::close(socketFD);

    socketFD = -1;
    FD_ZERO(&socketFDS);
}

bool KickPimMailSocket::connectSocket(const QString& hostName, unsigned short int port)
{
  QString host = hostName;
  if (host.length()==0) host=" ";
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailSocket: Connecting to '"+host+"' at port "+QString::number(port));
  sockaddr_in  sin;
  hostent     *hent;
  int addr, n;

  // if we still have a socket, close it
  if (socketFD != -1) close();

  // get the socket
  socketFD = ::socket(AF_INET, SOCK_STREAM, IPPROTO_IP);

  // start setting up the socket info
  memset((char *)&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port   = htons(port);

  // Try to get the address
  if ((addr = inet_addr(host.ascii())) == -1) {
    // Try to get the address by host name
    if ((hent = gethostbyname(host.ascii())) == 0) {
      switch (h_errno) {
        case HOST_NOT_FOUND:
          LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Socket: Host not found: '"+host+"'");
          break;
        case NO_ADDRESS:
          LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Socket: No Address: '"+host+"'");
          break;
        case NO_RECOVERY:
          LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Socket: No recovery: '"+host+"'");
          break;
        case TRY_AGAIN:
          LogService::logWarn(LogService::CAT_MAIL,"KickPimMailSocket: Socket: Try again: '"+host+"'");
          break;
        default:
          LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Socket: Unknown error: "+QString::number(h_errno));
          break;
      }
      close();
      return false;
    }
    memcpy((void *)&sin.sin_addr, *(hent->h_addr_list), hent->h_length);
  }
  else {
    // get the address by IP
    memcpy((void *)&sin.sin_addr, (void *)&addr, sizeof(addr));
  }

  // Set up non-blocking io if requested
  if (async) {
    int flags = fcntl(socketFD, F_GETFL);
    if (flags < 0 || fcntl(socketFD, F_SETFL, flags | O_NONBLOCK) < 0) {
      async = false;
    }
  }

  // the socket is correctly setup.  now connect
  if ((n = ::connect(socketFD, (sockaddr *)&sin, sizeof(sockaddr_in))) == -1 &&
      errno != EINPROGRESS) {
    close();
    LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Connecting the socket failed: ");
    return false;
  }

  // Empty the file descriptor set
  FD_ZERO(&socketFDS);
  FD_SET(socketFD, &socketFDS);

  // For non-blocking io, the connection may need time to finish (n = -1)
  if (n == -1 && async == true) {
    struct timeval tv = socketTO;

    // Wait for the connection to come up
    if (select(socketFD+1, NULL, &socketFDS, NULL, &tv) != 1) {
      errno = ETIMEDOUT;
      LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Time out: '"+host+"'");
      close();
      return false;
    }
    // The connection has finished. Catch any error in a call to readLine()
  }

  // we're connected!  see if the connection is good
  QString line(readLine());
  if (line.isNull() || ((line.find("200") == -1 ) && (line.find("OK") == -1) && (line.find("PREAUTH") == -1)))
  {
    LogService::logError(LogService::CAT_MAIL,"KickPimMailSocket: Connection not good: readLine result = '"+line+"'");
    if (line.isNull())
    close();
    return false;
  }

  // everything is swell
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailSocket: Connection successful");
  return true;
}

bool KickPimMailSocket::active()
{
    return socketFD != -1;
}

bool KickPimMailSocket::isAsync()
{
    return async;
}

void KickPimMailSocket::setAsync(bool on)
{
    int flags = 0;

    async = on;

    if (active()) {
        flags = fcntl(socketFD, F_GETFL);
        switch (async) {
            case false:
                if (flags >= 0)
                    fcntl(socketFD, F_SETFL, flags & ~O_NONBLOCK);
                break;

            case true:
                if (flags < 0 || fcntl(socketFD, F_SETFL, flags | O_NONBLOCK) < 0)
                    async = false;
                break;
        }
    }
}

int KickPimMailSocket::writeLine(const QString& line)
{
    int bytes = 0;

    // Do not try to write to a non active socket. Return error.
    if (!active())
        return -1;

    if ((bytes = ::write(socketFD, line.ascii(), line.length())) <= 0)
        close();

    return bytes;
}

QString KickPimMailSocket::readLine()
{
    QString fault, response;
    char buffer;
    ssize_t bytes = -1;

    if (!async)
        while (((bytes = ::read(socketFD, &buffer, 1)) > 0) && (buffer != '\n'))
            response += buffer;
    else
    {
        while ( (((bytes = ::read(socketFD, &buffer, 1)) > 0) && (buffer != '\n')) ||
            ((bytes < 0) && (errno == EWOULDBLOCK)) )
        {
            if (bytes > 0)
                response += buffer;
            else
            {
                struct timeval tv = socketTO;
                if (select(socketFD+1,  &socketFDS, NULL, NULL, &tv) != 1)
                {
                    errno = ETIMEDOUT;
                    break;
                }
            }
        }
    }

    if (bytes == -1)
    {
        // Close the socket and hope for better luck with a new one
        close();
        return fault;
    }

    return response;
}

///////////////////////////////////////////////////////////////////////////
// KickPimMailImap
///////////////////////////////////////////////////////////////////////////
KickPimMailImap::~KickPimMailImap()
{
    close();
}

bool KickPimMailImap::command(const QString& line, unsigned int seq)
{
    int len, match;
    bool unseen = false;
    QString messagesListString;
    QStringList messagesList;

    if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailImap: Command '"+line/*.left( 11 )*/ + "...' ");

    if (writeLine(line) <= 0) {
        close();
        return false;
    }

    QString ok, bad, no, response;
    ok.sprintf("%d OK", seq);
    bad.sprintf("%d BAD", seq);
    no.sprintf("%d NO", seq);
    while (!(response = readLine()).isNull()) {
      if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailImap:    Response: '"+response+"'");

      // if an error has occurred, we get a null string in return
      if (response.isNull())
          break;

      // if the response is either good or bad, then return
      if (response.find(ok) > -1)
          return true;
      if ((response.find(bad) > -1) || (response.find(no) > -1))
          break;

      QRegExp unseen_re(" UNSEEN");
      if (unseen_re.match(line) > -1)
          unseen = true;

      QRegExp messages_re("SEARCH [0-9 ]*");
      if ((match = messages_re.match(response, 0, &len)) > -1)
      {
          messagesListString = response.mid(match + 7, len - 7);
          messagesList = QStringList::split(' ', messagesListString);
          if (unseen)
              newMessages = messagesList.count();
          else
              messages = messagesList.count();
      }
    }

    close();
    return false;
}

QString KickPimMailImap::mungeUserPass(const QString& old_user)
{
    QString new_user(old_user);

    if (new_user.left(1) != "\"")
        new_user.prepend("\"");
    if (new_user.right(1) != "\"")
        new_user.append("\"");

    return new_user;
}

void KickPimMailImap::resetNumbers()
{
    messages = 0;
    newMessages = 0;
}

///////////////////////////////////////////////////////////////////////////
// KickPimMailPop
///////////////////////////////////////////////////////////////////////////
KickPimMailPop::~KickPimMailPop()
{
    close();
}

void KickPimMailPop::close()
{
    command("QUIT\r\n");
    KickPimMailSocket::close();
}

bool KickPimMailPop::command(const QString& line)
{
  if (writeLine(line) <= 0) return false;

  QString response = readLine();
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"KickPimMailPop: Command '"+line.left( 4 ) + "...' "
                                          +"Response='"+ response.left( response.length()-1) +"'");

  // check if the response was bad.  if so, return now
  if (response.isNull() || response.left(4) == "-ERR") {

    // we used to close the socket here.. but this MAY be
    // because the server didn't understand UIDL.  the server
    // may react better with LIST or STAT so just fail quitely
    // thanks to David Barth (dbarth@videotron.ca)
    return false;
  }

  // if the command was UIDL then build up the newUidlList
  if (line == "UIDL\r\n") {
      uidlList.clear();
      for (response = readLine();
          !response.isNull() && response.left(1) != ".";
          response = readLine())
      {
          uidlList.append(new QString(response.right(response.length() -
                  response.find(" ") - 1)));
      }
  }
  else
  // get all response lines from the LIST command
  // LIST and UIDL are return multilines so we have to loop around
  if (line == "LIST\r\n") {
      for (messages = 0, response = readLine();
          !response.isNull() && response.left(1) != ".";
          messages++, response = readLine());
  }
  else
  if (line == "STAT\r\n") {
      if (!response.isNull())
          sscanf(response.ascii(), "+OK %d", &messages);
  }

  return !response.isNull();
}

KickPimMailUidlList KickPimMailPop::getUidlList() const
{
    return uidlList;
}

///////////////////////////////////////////////////////////////////////////
// KickPimMailNntp
///////////////////////////////////////////////////////////////////////////
KickPimMailNntp::~KickPimMailNntp()
{
    close();
}

bool KickPimMailNntp::command(const QString& line)
{
    int bogus;

    if (writeLine(line) <= 0)
        return false;

    QString response;
    while (!(response = readLine()).isNull())
    {
        // return if the response is bad
        if (response.find("500") > -1)
        {
            close();
            return false;
        }

        // find return codes for tcp, user, pass
        QString code(response.left(3));
        if ((code == "200") || (code == "281") || (code == "381"))
            return true;

        // look for the response to the GROUP command
        // 211 <num> <first> <last> <group>
        if (code == "211")
        {
            sscanf(response.ascii(), "%d %d %d %d",
                    &bogus, &messages, &firstMsg, &lastMsg);
            return true;
        }
    }

    close();
    return false;
}

int KickPimMailNntp::first() const
{
    return firstMsg;
}

int KickPimMailNntp::last() const
{
    return lastMsg;
}

/////////////////////////////////////////////////////////////////////////
/* The following is a (C) Sirtaj Singh Kang <taj@kde.org> */

#define whitespace(c)    (c == ' ' || c == '\t')

#define skip_white(c)     while(c && (*c) && whitespace(*c) ) c++
#define skip_nonwhite(c) while(c && (*c) && !whitespace(*c) ) c++

#define skip_token(buf) skip_nonwhite(buf); if(!*buf) return false; \
    skip_white(buf); if(!*buf) return false;

static const char *month_name[13] = {
    "jan", "feb", "mar", "apr", "may", "jun",
    "jul", "aug", "sep", "oct", "nov", "dec", NULL
};

static const char *day_name[8] = {
    "sun", "mon", "tue", "wed", "thu", "fri", "sat", 0
};

static bool real_from(const QString& orig_buffer)
{
    /*
        A valid from line will be in the following format:

        From <user> <weekday> <month> <day> <hr:min:sec> [TZ1 [TZ2]] <year>
     */

    int day;
    int i;
    int found;

    const char *buffer = (const char*)orig_buffer.ascii();

    /* From */

    if(!buffer || !*buffer)
        return false;

    if (strncmp(buffer, "From ", 5))
        return false;

    buffer += 5;

    skip_white(buffer);

    /* <user> */
    if(*buffer == 0) return false;
    skip_token(buffer);

    /* <weekday> */
    found = 0;
    for (i = 0; day_name[i] != NULL; i++)
        found = found || (qstrnicmp(day_name[i], buffer, 3) == 0);

    if (!found)
        return false;

    skip_token(buffer);

    /* <month> */
    found = 0;
    for (i = 0; month_name[i] != NULL; i++)
        found = found || (qstrnicmp(month_name[i], buffer, 3) == 0);
    if (!found)
        return false;

    skip_token(buffer);

    /* <day> */
    if ( (day = atoi(buffer)) < 0 || day < 1 || day > 31)
        return false;

    return true;
}

static const char* compare_header(const char* header, const char* field)
{
    int len = strlen(field);

    if (qstrnicmp(header, field, len))
        return NULL;

    header += len;

    if( *header != ':' )
        return NULL;

    header++;

    while( *header && ( *header == ' ' || *header == '\t') )
        header++;

    return header;
}
















/***
 ** Mail Monitor Threads
 **
 **
 ***/

KickPimMailMonitorThread::KickPimMailMonitorThread( KickPimMailMonitor* monitor )    
{
  if (monitor!=0) name = monitor->account()->name();
  LogService::construct("KickPimMailMonitorThread '"+name+"'");
  m_monitor    = monitor;
  forceCheck   = false;
  terminate    = false;
  skipChecks   = true;
}

KickPimMailMonitorThread::~KickPimMailMonitorThread( )
{
  m_monitor = 0;
  LogService::destruct("KickPimMailMonitorThread '"+name+"'");
}

KickPimMailMonitor* KickPimMailMonitorThread::monitor()
{
  return m_monitor;
}

void KickPimMailMonitorThread::terminateThread() {
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Trying to terminate this thread ...");
  terminate = true;
  pollIntervalWait.wakeAll();
  terminateWait.wait();
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Success ... terminated.");
}

void KickPimMailMonitorThread::run()
{
  terminate = false;

  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Started.");
  logState();

  int waitTime = 10*1000; // initially wait for 10 seconds
  pollIntervalWait.wait( waitTime );

  forceCheck = false;
  KPMailAccount* account = (m_monitor!=0) ? m_monitor->account() : 0;
  if (account==0) LogService::log(LogService::LVL_FATAL,LogService::CAT_MAIL,"Mail thread without account data started! (Contact the developer!)");

  while (!terminate) {
    // check if the account is active and if we should check for mails
    if ( account->isActive() && !skipChecks ) {
      // check only if autocheck is enabled or if it has been forced
      if ( (account->isAutocheckEnabled() && KickPIM::opt()->mailCheckEnabled) || forceCheck ) {
        if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Checking ...");
        m_monitor->checkMailNow();
      }
    }

    // reset flags
    forceCheck = false;        
    waitTime = 5*1000;        
    if ( account->isAutocheckEnabled() && KickPIM::opt()->mailCheckEnabled ) {
      waitTime = account->pollInterval() * 1000;
    }
    // if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Waiting "+QString::number(waitTime/1000)+" sec. ...");
    if (!terminate) pollIntervalWait.wait( waitTime );
  }

  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Terminated.");
  pollIntervalWait.wait(100); // terminateThread() may need some time to call terminateWait.wait().
  terminateWait.wakeAll();
}

void KickPimMailMonitorThread::setSkipMailchecks( bool on ) {
  skipChecks = on;
  logState();
}


void KickPimMailMonitorThread::logState() {
  QString strAutoCheck = (m_monitor->account()->isAutocheckEnabled()) ? "on" : "off";
  QString strRunState  = (skipChecks)    ? "skip" : "yes";
  QString strActive    = m_monitor->account()->isActive() ? "yes" : "no";
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': active="+strActive+" autocheck="+strAutoCheck+" (checking: "+strRunState+")");
}

void KickPimMailMonitorThread::checkMailNow()
{
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_MAIL,"Mail Monitor Thread '"+name+"': Check Mail Now (waking up sleeping thread).");
  forceCheck = true;
  pollIntervalWait.wakeAll();
}

