/*
 * This file is a part of KleanSweep.
 *
 * Copyright (C) 2005 Pawel Stolowski <pawel.stolowski@wp.pl>
 *
 * KleanSweep is free software; you can redestribute it and/or modify it
 * under terms of GNU General Public License by Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY. See GPL for more details.
 */

#include "fileviewitem.h"
#include "filelistview.h"
#include "sweepfileinfo.h"
#include <kglobal.h>
#include <klocale.h>
#include <qpainter.h>

//
// width/height of score meter boxes
const int FileViewItem::ScoreW = 6;
const int FileViewItem::ScoreH = 4;
const int FileViewItem::ScoreTotalW = 9*6;

FileViewItem::FileViewItem(FileListView *parent, SweepFileInfo *f): QCheckListItem(parent, QString::null, CheckBox), finfo(f)
{
	setColumns(parent->getColumns());
}

FileViewItem::FileViewItem(FileViewItem *parent, SweepFileInfo *f): QCheckListItem(parent, QString::null, CheckBox), finfo(f)
{
	if (FileListView *list = dynamic_cast<FileListView*>(listView()))
		setColumns(list->getColumns());
}

FileViewItem::~FileViewItem()
{
}

void FileViewItem::setColumns(int cols)
{
	if (finfo)
	{
		int i = 0;
		setPixmap(i, finfo->pixmap(16));
		if (cols & FileName)
			setText(i++, finfo->name());
		if (cols & FileDir)
			setText(i++, finfo->url().directory());
		if (cols & Score)
			setText(i++, QString::number(finfo->getScore()));
		if (cols & Target)
			setText(i++, finfo->getTarget());
		if (cols & Size)
			setText(i++, KGlobal::locale()->formatNumber(finfo->size(), 0));
		if (cols & ModTime)
			setText(i++, finfo->timeString(KIO::UDS_MODIFICATION_TIME));
		if (cols & AccessTime)
			setText(i++, finfo->timeString(KIO::UDS_ACCESS_TIME));
		if (cols & Permissions)
			setText(i++, finfo->permissionsString());
		if (cols & Owner)
			setText(i++, finfo->user());
		if (cols & Group)
			setText(i++, finfo->group());
		setOn(finfo->isMarkedForRemoval());
	}
}

SweepFileInfo* FileViewItem::fileItem() const
{
	return finfo;
}

void FileViewItem::stateChange(bool f)
{
	if (finfo && finfo->isMarkedForRemoval() != f)
	{
		finfo->markForRemoval(f);
		if (FileListView *list = dynamic_cast<FileListView*>(listView()))
			list->stateChanged(finfo, f);
	}
}

int FileViewItem::compare(QListViewItem *i, int col, bool ascending) const
{
	if (FileListView *list = dynamic_cast<FileListView*>(listView()))
	{
		const int t = list->getColumnType(col);
		if (t == Size)
		{
			if (FileViewItem *item = dynamic_cast<FileViewItem*>(i))
			{
				return finfo->size() - item->finfo->size();
			}
		} else if (t == Score)
		{
			if (FileViewItem *item = dynamic_cast<FileViewItem*>(i))
			{
				return finfo->getScore() - item->finfo->getScore();
			}
		}
		return QListViewItem::compare(i, col, ascending);
	}
}

int FileViewItem::width(const QFontMetrics &fm, const QListView *lv, int c) const
{
	FileListView *list = dynamic_cast<FileListView*>(const_cast<QListView *>(lv)); //ugly...
	if (list && list->getColumnType(c) == Score)
	{
		return FileViewItem::ScoreTotalW + list->itemMargin() + 1;
	}
	return QCheckListItem::width(fm, lv, c);
}

void FileViewItem::paintCell(QPainter *p, const QColorGroup &cg, int col, int width, int align)
{
	if (FileListView *list = dynamic_cast<FileListView*>(listView()))
	{
		if (list->getColumnType(col) == Score)
		{
			const int len = finfo->getScore() * ScoreW;
			const int y = (height()/2) - ScoreH;
			const int margin = list->itemMargin();
			const QBrush brush = cg.brush(QColorGroup::Highlight);
			p->eraseRect(0, 0, width, height());
			for (int i=0; i<len; i+=ScoreW)
				p->fillRect(margin + i, y, ScoreW-1, ScoreH, brush);
			return;
		}
		if (finfo->getScore() == 0) //special handling for files with score = 0
		{
			if (list->getColumnType(col) == FileName) //make file name italic
			{
				QFont font = p->font();
				font.setItalic(true);
				p->setFont(font);
			}
			QListViewItem::paintCell(p, cg, col, width, align);
			return;
		}
	}
	QCheckListItem::paintCell(p, cg, col, width, align);
}

void FileViewItem::paintFocus(QPainter *p, const QColorGroup &cg, const QRect &r)
{
	if (finfo->getScore() == 0) //special handling for files with score = 0
		QListViewItem::paintFocus(p, cg, r);
	else
		QCheckListItem::paintFocus(p, cg, r);
}

