/***************************************************************************
 *
 * knetworkmanager-storage.h - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Stefan Bogner      <sbogner@suse.de>, <bochi@kmobiletools.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *         Helmut Schaa       <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#ifndef _KNETWORKMANAGERSTORAGE_H_
#define _KNETWORKMANAGERSTORAGE_H_

#include <kwallet.h>
#include <qmap.h>
#include <qobject.h>
#include "knetworkmanager-network.h"

class KConfigGroup;

class CredentialsRequest : public QObject
{
	Q_OBJECT
	public:
		CredentialsRequest(const QString&);
		~CredentialsRequest();
		void loadCredentials();

	public slots:
		void slotWalletOpened(bool);
		void slotCancelRequest();

	signals:
		void credentialsLoaded(QString, QMap<QString, QString>, bool);

	private:
		QString _id;
		bool    _canceled;
		static KWallet::Wallet * _wallet;
};

class KNetworkManagerStorage : public QObject
{
	Q_OBJECT

	public:
		// TODO add parent or static deleter.
		KNetworkManagerStorage ();
		~KNetworkManagerStorage ();
		/** 
		 * KNetworkManagerStorage is a singleton.  Use this method to access it.
		 */
		static KNetworkManagerStorage *getInstance();
		/**
		 * Get a list of all the VPN Connections that we know
		 */
		QStringList vpnConnectionGroups() const;
		/**
		 * Create new VPN connection Group
		 */
		QString vpnConnectionNewGroup();
		/**
		 * Get a list of all the ESSIDs that we know
		 */
		QStringList networks() const;
		/**
		 * Get the details for a specific network.  This method is not const because it can open the KDE wallet
		 * @return a Network struct with the following fields set.  
		 * Other fields should be treated as unset.
		 */
		Network* networkProperties( const QString & essid, const QString & hwaddr = QString::null, bool * hwAddrMatched = 0 );
		/**
		 * Retrieve the credentials (key or passphrase) given network, from secure storage (asynchonous)
		 */
		CredentialsRequest* credentialsAsync ( const QString& );
		/**
		 * Retrieve the credentials (key or passphrase) given network, from secure storage
		 */
		QMap< QString, QString>  credentials ( const QString& );
		/**
		 * Returns whether the credentials are store in the wallet
		 */
		bool hasCredentialsStored ( const QString& );
		/**
		 * Store simple key to secure storage
		 */
		bool storeKey ( const QString&, const QString& );
		/**
		 * Store credentials to secure storage
		 */
		bool storeCredentials ( const QString&, const QMap< QString, QString>& );
		/**
		 * Save the details for the given network
		 */
		void storeNetwork( const Network*, bool updateTimeStamp = false );
		/**
		 * Remove a stored network
		 */
		void removeNetwork( const Network * );
		/**
		 * Persist whether wireless should be enabled/disabled at startup
		 */
		void persistWireless (bool);
		/**
		 * Check whether wireless should be enabled/disabled at startup
		 */
		bool getWireless (void);
		/**
		 * Persist whether offline mode should be enabled/disabled at startup
		 */
		void persistOfflineMode (bool);
		/**
		 * Check whether offline mode should be enabled/disabled at startup
		 */
		bool getOfflineMode (void);
		/**
		 * Returns whether kwallet is used to store passwords or the config file
		 */
		bool getStoreKeysUnencrypted (void);
		/**
		 * Sets whether kwallet is used to store passwords or the config file
		 */
		bool setStoreKeysUnencrypted (bool);

	public slots:
		/**
		 * Update stored network data with changes
		 */
		void updateNetwork( Network *, bool );
		/**
		 * Notification when a CredentialRequest gets destoryed
		 */
		void slotCredentialsRequestDestroyed ( QObject* );
		/**
		 * Cancel all pending credential requests
		 */
		void slotCancelAllCredentialRequests ( );

	private:
		/** 
		 * look up the group name for a given Network
		 * @param net The network we are looking for.
		 * @param hwAddr if supplied, try to find a network that contains this hardware address.
		 * @param fuzzyMatched this bool is set if we only found a group by matching on ESSID
		 * @return the KConfig group name if found, an empty string if not.
		 */
		QString lookupNetworkGroupName( const QString & essid, const QString & hwAddr = QString::null, bool * hwAddrMatched = 0 ) const;
		QString lookupVPNConnectionGroupName( const QString & name) const;
		static KNetworkManagerStorage *m_instance;
		KWallet::Wallet *m_wallet;
		uint m_walletRefCount;
		QValueList<CredentialsRequest*> _requests;

	private slots:
		void slotWalletClosed ();
};

#endif
