/***************************************************************************
 *   Copyright (C) 2006 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "launcher.h"

#include <qcursor.h>
#include <qfile.h>
#include <qstringlist.h>
#include <qtextstream.h>

#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kprocess.h>

#include "constants.h"
#include "ksmoothdock.h"
#include "showdesktop.h"

/// PUBLIC ///

/** 
 * Constructor
 */

Launcher::Launcher() {}

/** 
 * Constructor
 * @param desc description of the dock item
 * @param iconName the name of the icon that represents the dock item
 * @param command the command associated with this launcher
 */

Launcher::Launcher(QString desc, QString iconName, QString command)
    : m_command(command), m_iconName(iconName)
{
    m_description = desc;
}

/** 
 * Constructor
 * @param fileName the link file
 */

Launcher::Launcher(QString fileName) {
    QString line;
    QFile in(fileName);
    in.open(IO_ReadOnly);

    if (in.readLine(line, MAX_LINE_LENGTH) != -1) {
        if (line.contains("[Desktop Entry]", FALSE) > 0) {
            while (in.readLine(line, MAX_LINE_LENGTH) != -1) {
                if (line.startsWith("Exec=")) {
                    m_command = line.mid(5).stripWhiteSpace();
                } else if (line.startsWith("Icon=")) {
                    m_iconName = line.mid(5).stripWhiteSpace();
                } else if (line.startsWith("Name=")) {
                    m_description = line.mid(5).stripWhiteSpace();
                }
            }
        }
    }

    in.close();
}

/** 
 * Constructor
 * @param desc description of the dock item
 * @param desktop the desktop the dock item is on
 * @param iconName the name of the icon that represents the dock item
 * @param baseSize the original size of the icon to load
 * @param minSize the minimum height of the dock item's icon
 * @param maxSize the maximum height of the dock item's icon
 * @param orientation orienation of the dock
 * @param command the command associated with this launcher
 */

Launcher::Launcher(KSmoothDock* parent, int itemId, QString desc, int desktop, QString iconName, int baseSize, int minSize, int maxSize, Qt::Orientation orientation, QString command)
 : IconBasedDockItem(parent, itemId, desc, desktop, iconName, baseSize, minSize, maxSize, orientation), m_command(command), m_iconName(iconName)
{
}

/** 
 * Constructor
 * @param fileName the link file
 * @param baseSize the original size of the icon to load
 * @param minSize the minimum size of the dock item's icon
 * @param maxSize the maximum size of the dock item's icon
 * @param orientation orienation of the dock
 */

Launcher::Launcher(KSmoothDock* parent, int itemId, QString fileName, int baseSize, int minSize, int maxSize, Qt::Orientation orientation) {
    QString line;
    QFile in(fileName);
    in.open(IO_ReadOnly);

    if (in.readLine(line, MAX_LINE_LENGTH) != -1) {
        if (line.contains("[Desktop Entry]", FALSE) > 0) {
            while (in.readLine(line, MAX_LINE_LENGTH) != -1) {
                if (line.startsWith("Exec=")) {
                    m_command = line.mid(5).stripWhiteSpace();
                } else if (line.startsWith("Icon=")) {
                    m_iconName = line.mid(5).stripWhiteSpace();
                } else if (line.startsWith("Name=")) {
                    m_description = line.mid(5).stripWhiteSpace();
                }
            }
        }
    }

    in.close();

    m_parent = parent;
    m_itemId = itemId;
    m_minSize = minSize;
    m_maxSize = maxSize;
    m_orientation = orientation;
    m_desktop = ON_ALL_DESKTOPS;    

    KIconLoader* iLoader = KGlobal::iconLoader();

    QPixmap pix = iLoader->loadIcon(m_iconName, KIcon::NoGroup, baseSize);
    setIcon(pix);
}

/**
 * Get the command
 */

QString Launcher::getCommand() {
    return m_command;
}

/**
 * Get the icon name
 */

QString Launcher::getIconName() {
    return m_iconName;
}

/**
 * Set new command
 * @param command the new command
 */

void Launcher::setCommand(QString command) {
    m_command = command;
}

/**
 * Set new icon
 * @param iconName the name of the new icon
 */

void Launcher::setIconName(QString iconName) {
    m_iconName = iconName;
    KIconLoader* iLoader = KGlobal::iconLoader();

    QPixmap pix = iLoader->loadIcon(m_iconName, KIcon::NoGroup, KIcon::SizeEnormous);
    setIcon(pix);
}

/** 
 * Mouse pressed event handler
 */

void Launcher::mousePressEvent(QMouseEvent* e) {
    if (e->button() == Qt::LeftButton) { // if left-click, run the application
        if (m_command.compare("SHOW_DESKTOP") == 0) {
            ShowDesktop::the()->toggle();
        } else {
    
            // remove leading and trailing single and double quotes
        
            int len = m_command.length();
            if ((m_command.at(0) == '\'' && m_command.at(len - 1) == '\'') || (m_command.at(0) == '\"' && m_command.at(len - 1) == '\"')) {
                m_command = m_command.mid(1, len - 2);
            }
        
            // run the command
        
            KProcess proc;
            QStringList list;
            list = QStringList::split(" ", m_command);
            for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
                proc << (*it);
            }
        
            if(!proc.start(KProcess::DontCare))
                KMessageBox::information(0, i18n("Could not run the application. Please check the command and try again."));
        }
    } else { // if right-click, display the dock's popup menu
        m_parent->showPopupMenu(QCursor::pos());
    }
}

/**
 * Draw itself into the offscreen buffer with mouse clicked acknowledgement
 */

void Launcher::drawAcknowledgement(QPixmap& buffer, int x, int y, int size) {
    if (size >= m_minSize && size <= m_maxSize) {
        KPixmap pix(getIcon(size));
        KPixmapEffect::fade(pix, 0.5, QColor("#FFFFFF"));
        bitBlt(&buffer, x, y, &pix);
    }
}

/** 
 * Save the launcher info into a Link-to-Application file
 */

void Launcher::saveToFile(QString fileName) {
    QFile out(fileName);
    if (out.open(IO_WriteOnly)) {
        QTextStream out_s(&out);
        out_s << "[Desktop Entry]" << endl;
        out_s << "Comment=" << endl;
        out_s << "Comment[en_GB]=" << endl;
    
        out_s << "Encoding=UTF-8" << endl;
        out_s << "Exec=" << m_command << endl;
        out_s << "GenericName=" << endl;
        out_s << "GenericName[en_GB]=" << endl;
        out_s << "Icon=" << m_iconName << endl;
        out_s << "MimeType=" << endl;
        out_s << "Name=" << m_description << endl;
        out_s << "Name[en_GB]=" << endl;
        out_s << "Path=" << endl;
        out_s << "ServiceTypes=" << endl;
        out_s << "SwallowExec=" << endl;
        out_s << "SwallowTitle=" << endl;
        out_s << "Terminal=false" << endl;
        out_s << "TerminalOptions=" << endl;
        out_s << "Type=Application" << endl;
        out_s << "X-KDE-SubstituteUID=false" << endl;
        out_s << "X-KDE-Username=" << endl;
    
        out.close();
    }
}

