/***************************************************************************
 *   Copyright (C) 2006 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "normalzoomstate.h"

#include <iostream>
#include <unistd.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qpainter.h>

#include <kpixmapeffect.h>
#include <kwin.h>

#include "ksmoothdock.h"
#include "launcher.h"
#include "showdesktop.h"

/// PUBLIC ///

/**
 * Constructors
 */

NormalZoomState::NormalZoomState(KSmoothDock* dock)
 : DockState(dock) {
    m_w = 0;
    m_h = 0;
    m_timer = new QTimer(this);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(updateZoom()));
    m_animationNumSteps = 10;
    if (m_dock->m_isAutohide) {
        m_isHidden = true;
    } else {
        m_isHidden = false;
    }
    m_maskOn = false;
}

/**
 * Dock created event
 */

void NormalZoomState::dockCreatedEvent() {
    if (m_dock->m_isAutohide && m_isHidden)
        m_dock->setDockBelow();
}

/**
 * Update the layout
 */

void NormalZoomState::updateLayout(bool reset) {
    m_lastFocusedItemIndex = -1;
    m_clickedLauncherIndex = -1;
    m_itemsSize.clear();
    m_itemsSizeIncrement.clear();
    for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
        m_itemsSize.push_back(m_dock->m_smallIconSize);
        m_itemsSizeIncrement.push_back(0);
    }
    m_firstIndexToUpdate = 0;
    m_lastIndexToUpdate = m_dock->m_items.size() - 1;

    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    int old_w = m_w;
    int old_h = m_h;
    m_w = m_h = 0;
    if (m_dock->m_orientation == Qt::Horizontal) {
        // calculate w and h

        for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
            m_w += m_dock->m_items[i]->getMaxWidth();
        }
        m_h = m_dock->m_bigIconSize;
    } else { // Vertical
        // calculate w and h
        
        for (unsigned int i = 0; i < m_dock->m_items.size(); i++)
            m_h += m_dock->m_items[i]->getMaxHeight();
        m_w = m_dock->m_bigIconSize;
    }

    if (m_dock->m_orientation == Qt::Horizontal && m_h != old_h) {
        m_buffer.reset(new QPixmap(dw, m_h));
        m_background.reset(new KPixmap(*m_buffer));
        
        // update background
        updateBackgroundImage();
    } else if (m_dock->m_orientation == Qt::Vertical && m_w != old_w) {
        m_buffer.reset(new QPixmap(m_w, dh));
        m_background.reset(new KPixmap(*m_buffer));        

        // update background
        updateBackgroundImage();
    }

    int x = (dw - m_w) / 2;
    int y = (dh - m_h) / 2;    

    // set strut of the dock and move it

    switch(m_dock->m_position) {
    case LEFT:
        if (m_dock->m_isAutohide) {
            KWin::setExtendedStrut(m_dock->winId(), 1, 0, dh, 0, 0, 0, 0, 0, 0, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(0, 0, dw, dh);
        } else {
            KWin::setExtendedStrut(m_dock->winId(), m_w, 0, dh, 0, 0, 0, 0, 0, 0, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(m_w, 0, dw - m_w, dh);
        }        
        m_dock->move(0, y);
        break;
    case RIGHT:
        if (m_dock->m_isAutohide) {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 1, 0, dh, 0, 0, 0, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(0, 0, dw, dh);
        } else {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, m_w, 0, dh, 0, 0, 0, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(0, 0, dw - m_w, dh);
        }        
        m_dock->move(dw - m_w, y);
        break;
    case TOP:
        if (m_dock->m_isAutohide) {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, 1, 0, dw, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(0, 0, dw, dh);
        } else {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, m_h, 0, dw, 0, 0, 0);
            KSmoothDock::setDesktopIconsArea(0, m_h, dw, dh - m_h);
        }
        
        m_dock->move(x, 0);
        break;
    case BOTTOM:
        if (m_dock->m_isAutohide) {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, dw);
            KSmoothDock::setDesktopIconsArea(0, 0, dw, dh);
        } else {
            KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, 0, 0, 0, m_h, 0, dw);
            KSmoothDock::setDesktopIconsArea(0, 0, dw, dh - m_h);
        }        
        m_dock->move(x, dh - m_h);
        break;
    default:
        break;
    }

    // resize the dock
    m_dock->resize(m_w, m_h);

    if (m_dock->m_isAutohide && m_isHidden) {
        setMaskOn(true);
    } else {
        setMaskOn(false);
    }
    m_dock->repaint();
}

/**
 * Update the background
 */

void NormalZoomState::updateBackground() {
    updateBackgroundImage();
    m_dock->repaint();
}

/**
 * Paint event handler    
 */

void NormalZoomState::paintEvent(QPaintEvent* e) {
    int x = 0;
    int y = 0;
    int w = 0;
    int h = 0;
    unsigned int i = 0;

    if (m_dock->m_isAutohide && m_isHidden) {
        if (!m_maskOn) {
            setMaskOn(true);
        }

    } else {
        if (m_maskOn) {
            setMaskOn(false);
        }

        if (m_dock->m_orientation == Qt::Horizontal) {
            // draw the background
            for (i = 0; i < m_firstIndexToUpdate; i++)
                x += m_dock->m_items[i]->getMaxWidth();
            for (i = m_firstIndexToUpdate; i <= m_lastIndexToUpdate; i++)
                w += m_dock->m_items[i]->getMaxWidth();
            w += 3; // fix the drawing bug
    
            bitBlt(m_buffer.get(), x, 0, m_background.get(), x, 0, w, m_h);
    
            // draw the items
            int x_item = x;
            for (i = m_firstIndexToUpdate; i <= m_lastIndexToUpdate; i++) {
                DockItem* item = m_dock->m_items[i];
                int dx = (item->getMaxWidth() - item->getWidth(m_itemsSize[i])) / 2;
                int dy = (item->getMaxHeight() - item->getHeight(m_itemsSize[i])) / 2;
    
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, x_item + dx, dy, m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, x_item + dx, dy, m_itemsSize[i]);
                }
    
                x_item += item->getMaxWidth();
            }
    
            // update the dock
            bitBlt(m_dock, x, 0, m_buffer.get(), x, 0, w, m_h);
        } else { // Vertical
            // draw the background
            for (i = 0; i < m_firstIndexToUpdate; i++)
                y += m_dock->m_items[i]->getMaxHeight();
            for (i = m_firstIndexToUpdate; i <= m_lastIndexToUpdate; i++)
                h += m_dock->m_items[i]->getMaxHeight();
            h += 3; // fix the drawing bug
    
            bitBlt(m_buffer.get(), 0, y, m_background.get(), 0, y, m_w, h);
    
            // draw the items
            int y_item = y;
            for (i = m_firstIndexToUpdate; i <= m_lastIndexToUpdate; i++) {
                DockItem* item = m_dock->m_items[i];
                int dx = (item->getMaxWidth() - item->getWidth(m_itemsSize[i])) / 2;
                int dy = (item->getMaxHeight() - item->getHeight(m_itemsSize[i])) / 2;
    
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, dx, y_item + dy, m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, dx, y_item + dy, m_itemsSize[i]);
                }
    
                y_item += item->getMaxHeight();
            }
    
            // update the dock
            bitBlt(m_dock, 0, y, m_buffer.get(), 0, y, m_w, h);
        }
    }    
}

/**
 * Mouse pressed event handler
 */

void NormalZoomState::mousePressEvent(QMouseEvent* e) {
    // find the m_dock item's index
    int i = findItemIndex(e->x(), e->y());

    // call the item's mouse event handler
    m_dock->m_items[i]->mousePressEvent(e);

    // launcher acknowledgement
    if (e->button() == Qt::LeftButton) {
        Launcher l;
        if (typeid(*m_dock->m_items[i]) == typeid(l)) {
            m_clickedLauncherIndex = i;
            QTimer::singleShot(500, this, SLOT(stopLauncherAcknowledgement()));
            m_dock->repaint();

            Launcher* launcher = dynamic_cast<Launcher*>(m_dock->m_items[i]);
            if (launcher->getCommand().compare("SHOW_DESKTOP") == 0) { // Show Desktop button
                updateAfterShowingDesktop();
            }
        }
    }
}

/**
 * Mouse moved event handler
 */

void NormalZoomState::mouseMoveEvent(QMouseEvent* e) {
    // find the m_dock item's index
    int i = findItemIndex(e->x(), e->y());

    if (m_lastFocusedItemIndex != i) {
        // show tooltip
        if (m_dock->m_showTooltip) {
            m_dock->m_tooltip.setText(m_dock->m_items[i]->getDescription());

            int x = m_dock->x();
            int y  = m_dock->y();
            if (m_dock->m_orientation == Qt::Horizontal) {
                for (int j = 0; j < i; j++) {
                    x += m_dock->m_items[j]->getMaxWidth();
                }
                x = x + m_dock->m_items[i]->getMaxWidth() / 2 - m_dock->m_tooltip.width() / 2;

                if (m_dock->m_position == TOP)
                    y = y + m_dock->height() + TOOLTIP_SPACE;
                else // BOTTOM
                    y = y - m_dock->m_tooltip.height() - TOOLTIP_SPACE;
            } else { // Vertical
                for (int j = 0; j < i; j++) {
                    y += m_dock->m_items[j]->getMaxHeight();
                }
                y = y + m_dock->m_items[i]->getMaxHeight() / 2 - m_dock->m_tooltip.height() / 2;

                if (m_dock->m_position == LEFT)
                    x = x + m_dock->width() + TOOLTIP_SPACE;
                else // RIGHT
                    x = x - m_dock->m_tooltip.width() - TOOLTIP_SPACE;
            }

            m_dock->m_tooltip.move(x, y);
            if (!m_dock->m_tooltip.isShown())
                m_dock->m_tooltip.show();
        }
        
        // zoom items
        m_itemsSizeIncrement[i] = 1;
        if (m_lastFocusedItemIndex >= 0 && m_lastFocusedItemIndex < (int) m_dock->m_items.size())
            m_itemsSizeIncrement[m_lastFocusedItemIndex] = -1;
        if (!m_timer->isActive())
            m_timer->start(32 - m_dock->m_NzZoomSpeed);

        m_lastFocusedItemIndex = i;
    }
}

/**
 * Enter event handler
 */

void NormalZoomState::enterEvent(QEvent* e) {
    if (m_dock->m_isAutohide && m_isHidden) {
        m_isHidden = false;
        updateBackgroundImage();
        m_dock->setDockAbove();
        updateLayout();
    }
}

/**
 * Leave event handler
 */

void NormalZoomState::leaveEvent(QEvent* e) {
    if (m_dock->m_tooltip.isShown())
            m_dock->m_tooltip.hide();

    if (m_dock->m_isAutohide && !m_isHidden) {
        m_isHidden = true;
        updateBackgroundImage();
        m_dock->setDockBelow();
        updateLayout();
    } else {
        m_itemsSizeIncrement[m_lastFocusedItemIndex] = -1;
        m_lastFocusedItemIndex = -1;
        if (!m_timer->isActive())
            m_timer->start((32 - m_dock->m_NzZoomSpeed));
    }
}

/**
 * Update a specific item
 * @TODO: fix and use the effient method (at the moment it does not work properly when the dock is vertical)
 */

void NormalZoomState::updateItem(int itemIndex) { 
    /*
    int x = 0;
    int y = 0;
    DockItem* item = m_dock->m_items[itemIndex];
    int w = item->getWidth(m_itemsSize[itemIndex]);
    int h = item->getHeight(m_itemsSize[itemIndex]);    
    int dx = (item->getMaxWidth() - w) / 2;
    int dy = (item->getMaxHeight() - h) / 2;

    for (int i = 0; i < itemIndex; i++) {
        if (m_dock->m_orientation == Qt::Horizontal) {
            x += m_dock->m_items[i]->getMaxWidth();
        } else { // Vertical
            y += m_dock->m_items[i]->getMaxHeight();
        }
    }
    x += dx;
    y += dy;

    bitBlt(m_buffer.get(), x, y, m_background.get(), x, y, w, h);
    item->draw(*m_buffer, x, y, m_itemsSize[itemIndex]);

    bitBlt(m_dock, x, y, m_buffer.get(), x, y, w, h); (
    */
    
    // for now use this inefficient method
    m_dock->repaint();
}

/**
 * Update the dock when the active window has been changed
 */

void NormalZoomState::activeWindowChanged(WId id) {
    if (!m_dock->m_isAutohide)
        return;

    if ( (id == m_dock->winId()) || (id == 0) || ShowDesktop::the()->desktopShowing() )
        return;

    if (!m_isHidden) { // zoom mode
        KWin::WindowInfo info(id, 0, 0);
        QRect windowFrameGeometry = info.frameGeometry();
    
        QRect rect = m_dock->frameGeometry();
        NET::WindowType type = info.windowType(0xffff);
        NET::MappingState mapping_state = info.mappingState();
        if ((mapping_state == NET::Visible) && (type == NET::Normal || type == NET::Dialog || type == NET::Unknown || type == NET::Override) && rect.intersects(windowFrameGeometry)) {
            m_dock->setDockBelow();
            QRect intersection = rect & windowFrameGeometry;
            pauseExecution();
            
            QPixmap intersection_pix = QPixmap::grabWindow(qt_xrootwin(), intersection.x(), intersection.y(), intersection.width(), intersection.height());    
    
            QPoint pos = m_dock->mapFromGlobal(intersection.topLeft());

            switch(m_dock->m_position) {
            case TOP:
                bitBlt(m_background.get(), pos.x(), pos.y() - 1, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case BOTTOM:
                bitBlt(m_background.get(), pos.x(), pos.y(), &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case LEFT:
                bitBlt(m_background.get(), pos.x() - 1, pos.y(), &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case RIGHT:
                bitBlt(m_background.get(), pos.x(), pos.y(), &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            default:
                break;
            }

            // fade the background
            if (m_dock->m_dockOpacity > 0)
                KPixmapEffect::fade(*m_background, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
        
            // draw borders
            QPainter p(m_background.get());
            if (m_dock->m_showBorders) {
                p.setPen(m_dock->m_borderColor);
        
                switch(m_dock->m_position) {
                case LEFT:
                    p.drawLine(0, 0, m_w - 1, 0);
                    p.drawLine(m_w - 1, 0, m_w - 1, m_h - 1);
                    p.drawLine(m_w - 1, m_h - 1, 0, m_h - 1);
                    break;
                case RIGHT:
                    p.drawLine(0, 0, m_w - 1, 0);
                    p.drawLine(0, 0, 0, m_h - 1);
                    p.drawLine(m_w - 1, m_h - 1, 0, m_h - 1);
                    break;
                case TOP:
                    p.drawLine(0, m_h - 1, m_w - 1, m_h - 1);
                    p.drawLine(0, 0, 0, m_h - 1);
                    p.drawLine(m_w - 1, m_h - 1, m_w - 1, 0);
                    break;
                case BOTTOM:
                    p.drawLine(0, 0, m_w - 1, 0);
                    p.drawLine(0, 0, 0, m_h - 1);
                    p.drawLine(m_w - 1, 0, m_w - 1, m_h - 1);
                    break;
                default:
                    break;
                }
            }
                        
            m_dock->setDockAbove();
            m_dock->repaint();
        }
    }
}


/**
 * Update autohide mode
 */

void NormalZoomState::updateAutohideMode() {
    if (m_dock->m_isAutohide && m_isHidden)
        m_dock->setDockBelow();
    updateBackgroundImage();
    updateLayout();
}

/**
 * Zoom timer event handler
 */

void NormalZoomState::updateZoom() {
    bool keep_timer = FALSE;

    for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
        if (m_itemsSizeIncrement[i] != 0) {
            m_itemsSize[i] += m_itemsSizeIncrement[i] * (m_dock->m_bigIconSize - m_dock->m_smallIconSize) / m_animationNumSteps;
    
            if (m_itemsSize[i] < m_dock->m_smallIconSize)
                m_itemsSize[i] = m_dock->m_smallIconSize;
            if (m_itemsSize[i] > m_dock->m_bigIconSize)
                m_itemsSize[i] = m_dock->m_bigIconSize;

            if (m_itemsSize[i] == m_dock->m_smallIconSize || m_itemsSize[i] == m_dock->m_bigIconSize) {
                m_itemsSizeIncrement[i] = 0;
            }
            if (m_firstIndexToUpdate > i)
                m_firstIndexToUpdate = i;
            if (m_lastIndexToUpdate < i)
                m_lastIndexToUpdate = i;
        }
    }
    
    m_dock->repaint();
    
    for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
        if (m_itemsSizeIncrement[i] != 0) {
            m_firstIndexToUpdate = i;
            keep_timer = TRUE;
            break;
        }
    }

    for (int i = (int) m_dock->m_items.size() - 1; i >= 0; i--) {
        if (m_itemsSizeIncrement[i] != 0) {
            m_lastIndexToUpdate = i;
            break;
        }
    }
    
    if (!keep_timer) {
        m_timer->stop();
        m_firstIndexToUpdate = 0;
        m_lastIndexToUpdate = m_dock->m_items.size() - 1;
    }
}

/**
 * Stop launcher acknowledgement
 */

void NormalZoomState::stopLauncherAcknowledgement() {
    m_clickedLauncherIndex = -1;
    m_dock->repaint();
}

/// PRIVATE ///

/**
 * Find the index of the item with mouse focus given mouse's x and y position
 */

int NormalZoomState::findItemIndex(int x, int y) {
    unsigned int i = 0;
    
    if (m_dock->m_orientation == Qt::Horizontal) {
        int w = 0;
        while (i < m_dock->m_items.size() && x > w) {
            w += m_dock->m_items[i]->getMaxWidth();
            i++;
        }
        if (i >= 1)
            i--;
    } else { // Vertical
        int h = 0;
        while (i < m_dock->m_items.size() && y > h) {
            h += m_dock->m_items[i]->getMaxHeight();
            i++;
        }
        if (i >= 1)
            i--;
    }

    return i;
}

/**
 * Update the background image
 */

void NormalZoomState::updateBackgroundImage() {
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
    int x = 0;
    int y = 0;

    switch(m_dock->m_position) {
    case LEFT:
        y = (dh - m_h) / 2;
        break;
    case RIGHT:
        x = dw - m_w;
        y = (dh - m_h) / 2;
        break;
    case TOP:
        x = (dw - m_w) / 2;
        break;
    case BOTTOM:
        x = (dw - m_w) / 2;
        y = dh - m_h;
        break;
    default:
        break;
    }

    if (m_dock->m_isAutohide) {
        if (m_isHidden || ShowDesktop::the()->desktopShowing()) {
            // copy from the wallpaper
            bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), x, y, m_w, m_h);
        } else {
            m_dock->setDockBelow();
            pauseExecution();
            *m_background = QPixmap::grabWindow(qt_xrootwin(), x, y, m_w, m_h);
            m_dock->setDockAbove();
        }
    } else {
        // copy from the wallpaper
        bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), x, y, m_w, m_h);
    }

    if (!m_dock->m_isAutohide || !m_isHidden) {
        // fade the background
        if (m_dock->m_dockOpacity > 0)
            KPixmapEffect::fade(*m_background, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
    
        // draw borders
        QPainter p(m_background.get());
        if (m_dock->m_showBorders) {
            p.setPen(m_dock->m_borderColor);
    
            switch(m_dock->m_position) {
            case LEFT:
                p.drawLine(0, 0, m_w - 1, 0);
                p.drawLine(m_w - 1, 0, m_w - 1, m_h - 1);
                p.drawLine(m_w - 1, m_h - 1, 0, m_h - 1);
                break;
            case RIGHT:
                p.drawLine(0, 0, m_w - 1, 0);
                p.drawLine(0, 0, 0, m_h - 1);
                p.drawLine(m_w - 1, m_h - 1, 0, m_h - 1);
                break;
            case TOP:
                p.drawLine(0, m_h - 1, m_w - 1, m_h - 1);
                p.drawLine(0, 0, 0, m_h - 1);
                p.drawLine(m_w - 1, m_h - 1, m_w - 1, 0);
                break;
            case BOTTOM:
                p.drawLine(0, 0, m_w - 1, 0);
                p.drawLine(0, 0, 0, m_h - 1);
                p.drawLine(m_w - 1, 0, m_w - 1, m_h - 1);
                break;
            default:
                break;
            }
        }
    }
}

/**
 * Update after showing/un-showing the desktop
 */

void NormalZoomState::updateAfterShowingDesktop() {
    if (m_dock->m_isAutohide) {
        bool desktopShowing = ShowDesktop::the()->desktopShowing();
        if (desktopShowing) {
            updateBackgroundImage();
        } else {
            m_dock->setDockBelow();
            
            pauseExecution();
                
            updateBackgroundImage();
    
            m_dock->setDockAbove();
        }
        m_dock->repaint();
    }
}

/**
 * Pause execution
 */

void NormalZoomState::pauseExecution() {
    usleep(100000); // pause for 100 miliseconds
}

/**
 * Set mask on/off
 */

void NormalZoomState::setMaskOn(bool maskOn) {
    if (maskOn) {
        QBitmap mask(m_dock->size());
        QPainter pm;
        pm.begin(&mask);
        pm.fillRect(m_dock->rect(), Qt::color0);

        int rx = 0;
        int ry = 0;

        switch(m_dock->m_position) {
        case LEFT:
            rx = 0; // for the sake of clarity
            ry = 0;
            pm.fillRect(QRect(rx, ry, 1, m_h), Qt::color1);
            break;
        case RIGHT:
            rx = m_w - 1;
            ry = 0;
            pm.fillRect(QRect(rx, ry, 1, m_h), Qt::color1);
            break;
        case TOP:
            rx = 0;
            ry = 0;            
            pm.fillRect(QRect(rx, ry, m_w, 1), Qt::color1);
            break;
        case BOTTOM:
            rx = 0;
            ry = m_h - 1;            
            pm.fillRect(QRect(rx, ry, m_w, 1), Qt::color1);
            break;
        default:
            break;
        }

        pm.end();
        m_dock->setMask(mask);
        m_maskOn = true;
    } else {
        // for some reason, clearMask does not work anymore
        //m_dock->clearMask();

        QBitmap mask(m_dock->size());
        QPainter pm;
        pm.begin(&mask);
        pm.fillRect(m_dock->rect(), Qt::color1);
        pm.end();
        m_dock->setMask(mask);
        m_maskOn = false;
    }
}

#include "normalzoomstate.moc"
