/***************************************************************************
 *   Copyright (C) 2006 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "paraboliczoomstate.h"

#include <iostream>
#include <unistd.h>

#include <qapplication.h>
#include <qbitmap.h>
#include <qcursor.h>
#include <qpainter.h>

#include <kpixmapeffect.h>
#include <kwin.h>

#include "desktopselector.h"
#include "ksmoothdock.h"
#include "launcher.h"
#include "showdesktop.h"

/// PUBLIC ///

/**
 * Constructors
 */

ParabolicZoomState::ParabolicZoomState(KSmoothDock* dock)
 : DockState(dock) {
    m_w = 0;
    m_h = 0;
    m_timer = new QTimer(this);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(updateZoom()));
    m_animationNumSteps = 20;
    // m_animationStepDelay = 32 - m_dock->m_PzZoomSpeed;
    m_isMinimised = true;
    m_maskOn = false;
    m_justEntered = false;
    m_justLeft = false;
    m_inAnimation = false;
}

/**
 * Dock created event
 */

void ParabolicZoomState::dockCreatedEvent() {
    m_dock->setDockBelow();
}

/**
 * Update the layout
 */

void ParabolicZoomState::updateLayout(bool reset) {
    m_clickedLauncherIndex = -1;
    
    m_itemsSize.clear();
    m_itemsStartSize.clear();
    m_itemsEndSize.clear();
    m_itemsCenterMin.clear();
    m_itemsCenter.clear();
    m_itemsStartCenter.clear();
    m_itemsEndCenter.clear();
    for (unsigned int i = 0; i < m_dock->m_items.size() + 1; i++) {
        m_itemsSize.push_back(m_dock->m_smallIconSize);
        m_itemsStartSize.push_back(m_dock->m_smallIconSize);
        m_itemsEndSize.push_back(m_dock->m_smallIconSize);

        m_itemsCenterMin.push_back(0);
        m_itemsCenter.push_back(0);
        m_itemsStartCenter.push_back(0);
        m_itemsEndCenter.push_back(0);
    }

    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    int old_min_w = m_minW;
    int old_min_h = m_minH;
    int old_w = m_w;
    int old_h = m_h;

    m_itemSpacing = m_dock->m_smallIconSize / 2;
    m_functionMaxX = (int) (2.5 * (m_dock->m_smallIconSize + m_itemSpacing));

    m_minW = m_minH = 0;
    if (m_dock->m_orientation == Qt::Horizontal) {
        // calculate w and h

        for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
            int item_min_w = m_dock->m_items[i]->getMinWidth();
            m_minW += item_min_w;
        }
        m_minW += m_dock->m_items.size() * m_itemSpacing;
        if (m_dock->m_isAutohide) {
            m_minH = 1;
        } else {
            m_minH = m_dock->m_smallIconSize + m_itemSpacing;
        }

        int distance = m_dock->m_smallIconSize + m_itemSpacing;

        if (m_dock->m_items.size() >= 5) {
            m_w = function(0) + 2*function(distance) + 2*function(2*distance) - 5*m_dock->m_smallIconSize + m_minW;
        } else if (m_dock->m_items.size() == 4) {
            m_w = function(0) + 2*function(distance) + function(2*distance) - 4*m_dock->m_smallIconSize + m_minW;
        } else if (m_dock->m_items.size() == 3) {
            m_w = function(0) + 2*function(distance) - 3*m_dock->m_smallIconSize + m_minW;
        } else if (m_dock->m_items.size() == 2) {
            m_w = function(0) + function(distance) - 2*m_dock->m_smallIconSize + m_minW;
        } else if (m_dock->m_items.size() == 1) {
            m_w = function(0) - m_dock->m_smallIconSize + m_minW;
        }

        m_h = m_dock->m_bigIconSize + m_itemSpacing;

        // calculate items center

        m_itemsCenterMin[0] = m_itemSpacing / 2 + (m_w - m_minW) / 2 + m_dock->m_items[0]->getMinWidth() / 2;
        for (unsigned int i = 1; i < m_dock->m_items.size(); i++) {
            m_itemsCenterMin[i] = m_itemsCenterMin[i-1] + m_dock->m_items[i-1]->getMinWidth() / 2 + m_itemSpacing + m_dock->m_items[i]->getMinWidth() / 2;
        }
    } else { // Vertical
        // calculate w and h

        for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
            int item_min_h = m_dock->m_items[i]->getMinHeight();
            m_minH += item_min_h;
        }
        m_minH += m_dock->m_items.size() * m_itemSpacing;
        if (m_dock->m_isAutohide) {
            m_minW = 1;
        } else {
            m_minW = m_dock->m_smallIconSize + m_itemSpacing;
        }

        int distance = m_dock->m_smallIconSize + m_itemSpacing;

        if (m_dock->m_items.size() >= 5) {
            m_h = function(0) + 2*function(distance) + 2*function(2*distance) - 5*m_dock->m_smallIconSize + m_minH;
        } else if (m_dock->m_items.size() == 4) {
            m_h = function(0) + 2*function(distance) + function(2*distance) - 4*m_dock->m_smallIconSize + m_minH;
        } else if (m_dock->m_items.size() == 3) {
            m_h = function(0) + 2*function(distance) - 3*m_dock->m_smallIconSize + m_minH;
        } else if (m_dock->m_items.size() == 2) {
            m_h = function(0) + function(distance) - 2*m_dock->m_smallIconSize + m_minH;
        } else if (m_dock->m_items.size() == 1) {
            m_h = function(0) - m_dock->m_smallIconSize + m_minH;
        }

        m_w = m_dock->m_bigIconSize + m_itemSpacing;

        // calculate items center

        m_itemsCenterMin[0] = m_itemSpacing / 2 + (m_h - m_minH) / 2 + m_dock->m_items[0]->getMinHeight() / 2;
        for (unsigned int i = 1; i < m_dock->m_items.size(); i++) {
            m_itemsCenterMin[i] = m_itemsCenterMin[i-1] + m_dock->m_items[i-1]->getMinHeight() / 2 + m_itemSpacing + m_dock->m_items[i]->getMinHeight() / 2;
        }
    }

    for (unsigned int i = 0; i < m_dock->m_items.size(); i++) {
        m_itemsCenter[i] = m_itemsCenterMin[i];
        m_itemsStartCenter[i] = m_itemsCenterMin[i];
        m_itemsEndCenter[i] = m_itemsCenterMin[i];
    }

    // resize pixmaps

    if (m_dock->m_orientation == Qt::Horizontal && (m_minH != old_min_h || m_h != old_h)) {
        m_buffer.reset(new QPixmap(dw, m_h));
        QPixmap pix(dw, m_minH);
        m_background.reset(new KPixmap(pix));
        m_originalBackground.reset(new QPixmap(dw, m_h));
        if (m_dock->m_isAutohide) {
            m_sideBorder.reset(new QPixmap(1, m_dock->m_smallIconSize + m_itemSpacing));
        } else {
            m_sideBorder.reset(new QPixmap(1, m_minH));
        }
        
        // update background
        updateBackgroundImage();
    } else if (m_dock->m_orientation == Qt::Vertical && (m_minW != old_min_w || m_w != old_w)) {
        m_buffer.reset(new QPixmap(m_w, dh));
        QPixmap pix(m_minW, dh);
        m_background.reset(new KPixmap(pix));
        m_originalBackground.reset(new QPixmap(m_w, dh));
        if (m_dock->m_isAutohide) {
            m_sideBorder.reset(new QPixmap(m_dock->m_smallIconSize + m_itemSpacing, 1));
        } else {
            m_sideBorder.reset(new QPixmap(m_minW, 1));
        }
        
        // update background
        updateBackgroundImage();
    }

    int x = (dw - m_w) / 2;
    int y = (dh - m_h) / 2;    

    // set strut of the dock and move it

    switch(m_dock->m_position) {
    case LEFT:
        m_dock->move(0, y);
        KWin::setExtendedStrut(m_dock->winId(), m_minW, 0, dh, 0, 0, 0, 0, 0, 0, 0, 0, 0);
        KSmoothDock::setDesktopIconsArea(m_minW, 0, dw - m_minW, dh);
        break;
    case RIGHT:
        m_dock->move(dw - m_w, y);
        KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, m_minW, 0, dh, 0, 0, 0, 0, 0, 0);
        KSmoothDock::setDesktopIconsArea(0, 0, dw - m_minW, dh);
        break;
    case TOP:
        m_dock->move(x, 0);
        KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, m_minH, 0, dw, 0, 0, 0);
        KSmoothDock::setDesktopIconsArea(0, m_minH, dw, dh - m_minH);
        break;
    case BOTTOM:
        m_dock->move(x, dh - m_h);
        KWin::setExtendedStrut(m_dock->winId(), 0, 0, 0, 0, 0, 0, 0, 0, 0, m_minH, 0, dw);
        KSmoothDock::setDesktopIconsArea(0, 0, dw, dh - m_minH);
        break;
    default:
        break;
    }

    if (!reset) {
        if (!m_isMinimised) { // zoom case

            // the dock has not been resized yet, so we have to do mouse position's translation ourselves

            int rx = 0;
            int ry = 0;

            switch(m_dock->m_position) {
            case LEFT:
                rx = 0; // for the sake of clarity
                ry = y;
                break;
            case RIGHT:
                rx = dw - m_w;
                ry = y;
                break;
            case TOP:
                rx = x;
                ry = 0;
                break;
            case BOTTOM:
                rx = x;
                ry = dh - m_h;
                break;
            default:
                break;
            }

            QRect rect(rx, ry, m_w, m_h);
            QPoint mouse_pos = QCursor::pos();
            if (rect.contains(mouse_pos)) {                
                mouse_pos.setX(mouse_pos.x() - rx);
                mouse_pos.setY(mouse_pos.y() - ry);
                setItemsPosition(mouse_pos.x(), mouse_pos.y());
            } else {
                leaveEvent(NULL);
            }            
        } else {        
            m_dock->setDockBelow();
        }
    } else {
        m_isMinimised = true;
        m_dock->setDockBelow();
    }

    // resize the dock
    m_dock->resize(m_w, m_h);

    // reset the mask
    if (m_isMinimised) {
        setMaskOn(true);
    } else {
        setMaskOn(false);
    }

    m_dock->repaint();    
}

/**
 * Update the background
 */

void ParabolicZoomState::updateBackground() {
    updateBackgroundImage();
    m_dock->repaint();
}

/**
 * Paint event handler    
 */

void ParabolicZoomState::paintEvent(QPaintEvent* e) {
    unsigned int i = 0;
    unsigned int no_items = m_dock->m_items.size();

    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    if (m_dock->m_position == TOP) {
        if (m_isMinimised) { // Minimised case
            // draw the background
            int dx = (m_w - m_minW) / 2;
            int dy = m_h - m_minH;
            bitBlt(m_buffer.get(), dx, 0, m_background.get(), (dw - m_minW) / 2, 0, m_minW, m_minH);
            bitBlt(m_buffer.get(), 0, m_minH, m_originalBackground.get(), (dw - m_w) / 2, m_minH, m_w, dy);
            bitBlt(m_buffer.get(), 0, 0, m_originalBackground.get(), (dw - m_w) / 2, 0, dx, m_minH);
            bitBlt(m_buffer.get(), m_w - dx, 0, m_originalBackground.get(), (dw + m_minW) / 2, 0, dx, m_minH);

            if (m_dock->m_showBorders) {
                bitBlt(m_buffer.get(), dx, 0, m_sideBorder.get(), 0, 0, 1, m_minH);
                bitBlt(m_buffer.get(), dx + m_minW - 1, 0, m_sideBorder.get(), 0, 0, 1, m_minH);
            }

            // draw the items
            for (i = 0; i < no_items; i++) {
                DockItem* item = m_dock->m_items[i];
                int x_item = m_itemsCenterMin[i] - item->getMinWidth() / 2;
                item->draw(*m_buffer, x_item, m_itemSpacing/2, m_dock->m_smallIconSize);
            }
        } else { // Zoomed case

            // draw the background
            int dy = m_h - m_minH;
            bitBlt(m_buffer.get(), 0, 0, m_background.get(), (dw - m_w) / 2, 0, m_w, m_minH);
            bitBlt(m_buffer.get(), 0, m_minH, &m_snapshot, (dw - m_w) / 2, 0, m_w, dy);

            if (m_dock->m_showBorders) {
                if (m_dock->m_isAutohide) {
                    int minH = m_dock->m_smallIconSize + m_itemSpacing;
                    bitBlt(m_buffer.get(), 0, 0, m_sideBorder.get(), 0, 0, 1, minH);
                    bitBlt(m_buffer.get(), m_w - 1, 0, m_sideBorder.get(), 0, 0, 1, minH);
                } else {
                    bitBlt(m_buffer.get(), 0, 0, m_sideBorder.get(), 0, 0, 1, m_minH);
                    bitBlt(m_buffer.get(), m_w - 1, 0, m_sideBorder.get(), 0, 0, 1, m_minH);
                }
            }

            // draw the items
            for (i = 0; i <= no_items - 1; i++) {
                DockItem* item = m_dock->m_items[i];
                int x_item = m_itemsCenter[i] - item->getWidth(m_itemsSize[i]) / 2;
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, x_item, m_itemSpacing/2 , m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, x_item, m_itemSpacing/2, m_itemsSize[i]);
                }
            }
        }
    } else if (m_dock->m_position == BOTTOM) {
        if (m_isMinimised) { // Minimised case
            // draw the background
            int dx = (m_w - m_minW) / 2;
            int dy = m_h - m_minH;
            bitBlt(m_buffer.get(), dx, dy, m_background.get(), (dw - m_minW) / 2, 0, m_minW, m_minH);
            bitBlt(m_buffer.get(), 0, 0, m_originalBackground.get(), (dw - m_w) / 2, 0, m_w, dy);
            bitBlt(m_buffer.get(), 0, dy, m_originalBackground.get(), (dw - m_w) / 2, dy, dx, m_minH);
            bitBlt(m_buffer.get(), m_w - dx, dy, m_originalBackground.get(), (dw + m_minW) / 2, dy, dx, m_minH);

            if (m_dock->m_showBorders) {
                bitBlt(m_buffer.get(), dx, dy, m_sideBorder.get(), 0, 0, 1, m_minH);
                bitBlt(m_buffer.get(), dx + m_minW - 1, dy, m_sideBorder.get(), 0, 0, 1, m_minH);
            }

            // draw the items
            for (i = 0; i < no_items; i++) {
                DockItem* item = m_dock->m_items[i];
                int x_item = m_itemsCenterMin[i] - item->getMinWidth() / 2;
                int dy = m_h - m_itemSpacing / 2 - item->getMinHeight();
                item->draw(*m_buffer, x_item, dy, m_dock->m_smallIconSize);
            }
        } else { // Zoomed case

            // draw the background
            int dy = m_h - m_minH;
            bitBlt(m_buffer.get(), 0, dy, m_background.get(), (dw - m_w) / 2, 0, m_w, m_minH);
            bitBlt(m_buffer.get(), 0, 0, &m_snapshot, (dw - m_w) / 2, 0, m_w, dy);

            if (m_dock->m_showBorders) {
                if (m_dock->m_isAutohide) {
                    int minH = m_dock->m_smallIconSize + m_itemSpacing;
                    bitBlt(m_buffer.get(), 0, m_h - minH, m_sideBorder.get(), 0, 0, 1, minH);
                    bitBlt(m_buffer.get(), m_w - 1, m_h - minH, m_sideBorder.get(), 0, 0, 1, minH);
                } else {
                    bitBlt(m_buffer.get(), 0, dy, m_sideBorder.get(), 0, 0, 1, m_minH);
                    bitBlt(m_buffer.get(), m_w - 1, dy, m_sideBorder.get(), 0, 0, 1, m_minH);
                }
            }

            // draw the items
            for (i = 0; i <= no_items - 1; i++) {
                DockItem* item = m_dock->m_items[i];
                int x_item = m_itemsCenter[i] - item->getWidth(m_itemsSize[i]) / 2;
                int dy = m_h - m_itemSpacing / 2 - item->getHeight(m_itemsSize[i]);
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, x_item, dy, m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, x_item, dy, m_itemsSize[i]);
                }
            }
        }
    } else if (m_dock->m_position == LEFT) {
        if (m_isMinimised) { // Minimised case
            // draw the background
            int dy = (m_h - m_minH) / 2;
            int dx = m_w - m_minW;
            bitBlt(m_buffer.get(), 0, dy, m_background.get(), 0, (dh - m_minH) / 2, m_minW, m_minH);
            bitBlt(m_buffer.get(), m_minW, 0, m_originalBackground.get(), m_minW, (dh - m_h) / 2, dx, m_h);
            bitBlt(m_buffer.get(), 0, 0, m_originalBackground.get(), 0, (dh - m_h) / 2, m_minW, dy);
            bitBlt(m_buffer.get(), 0, m_h - dy, m_originalBackground.get(), 0, (dh + m_minH) / 2, m_minW, dy);

            if (m_dock->m_showBorders) {
                bitBlt(m_buffer.get(), 0, dy, m_sideBorder.get(), 0, 0, m_minW, 1);
                bitBlt(m_buffer.get(), 0, dy + m_minH - 1, m_sideBorder.get(), 0, 0, m_minW, 1);
            }

            // draw the items
            for (i = 0; i < no_items; i++) {
                DockItem* item = m_dock->m_items[i];
                int y_item = m_itemsCenterMin[i] - item->getMinHeight() / 2;
                item->draw(*m_buffer, m_itemSpacing / 2, y_item, m_dock->m_smallIconSize);
            }
        } else { // Zoomed case

            // draw the background
            int dx = m_w - m_minW;
            bitBlt(m_buffer.get(), 0, 0, m_background.get(), 0, (dh - m_h) / 2, m_minW, m_h);
            bitBlt(m_buffer.get(), m_minW, 0, &m_snapshot, 0, (dh - m_h) / 2, dx, m_h);

            if (m_dock->m_showBorders) {
                if (m_dock->m_isAutohide) {
                    int minW = m_dock->m_smallIconSize + m_itemSpacing;
                    bitBlt(m_buffer.get(), 0, 0, m_sideBorder.get(), 0, 0, minW, 1);
                    bitBlt(m_buffer.get(), 0, m_h - 1, m_sideBorder.get(), 0, 0, minW, 1);
                } else {
                    bitBlt(m_buffer.get(), 0, 0, m_sideBorder.get(), 0, 0, m_minW, 1);
                    bitBlt(m_buffer.get(), 0, m_h - 1, m_sideBorder.get(), 0, 0, m_minW, 1);
                }
            }

            // draw the items
            for (i = 0; i <= no_items - 1; i++) {
                DockItem* item = m_dock->m_items[i];
                int y_item = m_itemsCenter[i] - item->getHeight(m_itemsSize[i]) / 2;
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, m_itemSpacing / 2, y_item, m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, m_itemSpacing / 2, y_item, m_itemsSize[i]);
                }
            }
        }
    } else if (m_dock->m_position == RIGHT) {
        if (m_isMinimised) { // Minimised case
            // draw the background
            int dy = (m_h - m_minH) / 2;
            int dx = m_w - m_minW;
            bitBlt(m_buffer.get(), dx, dy, m_background.get(), 0, (dh - m_minH) / 2, m_minW, m_minH);
            bitBlt(m_buffer.get(), 0, 0, m_originalBackground.get(), 0, (dh - m_h) / 2, dx, m_h);
            bitBlt(m_buffer.get(), dx, 0, m_originalBackground.get(), dx, (dh - m_h) / 2, m_minW, dy);
            bitBlt(m_buffer.get(), dx, m_h - dy, m_originalBackground.get(), dx, (dh + m_minH) / 2, m_minW, dy);

            if (m_dock->m_showBorders) {
                bitBlt(m_buffer.get(), dx, dy, m_sideBorder.get(), 0, 0, m_minW, 1);
                bitBlt(m_buffer.get(), dx, dy + m_minH - 1, m_sideBorder.get(), 0, 0, m_minW, 1);
            }

            // draw the items
            for (i = 0; i < no_items; i++) {
                DockItem* item = m_dock->m_items[i];
                int y_item = m_itemsCenterMin[i] - item->getMinHeight() / 2;
                int dx = m_w - m_itemSpacing / 2 - item->getMinWidth();
                item->draw(*m_buffer, dx, y_item, m_dock->m_smallIconSize);
            }
        } else { // Zoomed case

            // draw the background
            int dx = m_w - m_minW;
            bitBlt(m_buffer.get(), dx, 0, m_background.get(), 0, (dh - m_h) / 2, m_minW, m_h);
            bitBlt(m_buffer.get(), 0, 0, &m_snapshot, 0, (dh - m_h) / 2, dx, m_h);

            if (m_dock->m_showBorders) {
                if (m_dock->m_isAutohide) {                    
                    int minW = m_dock->m_smallIconSize + m_itemSpacing;
                    bitBlt(m_buffer.get(), m_w - minW, 0, m_sideBorder.get(), 0, 0, minW, 1);
                    bitBlt(m_buffer.get(), m_w - minW, m_h - 1, m_sideBorder.get(), 0, 0, minW, 1);
                } else {
                    bitBlt(m_buffer.get(), dx, 0, m_sideBorder.get(), 0, 0, m_minW, 1);
                    bitBlt(m_buffer.get(), dx, m_h - 1, m_sideBorder.get(), 0, 0, m_minW, 1);
                }
            }

            // draw the items
            for (i = 0; i <= no_items - 1; i++) {
                DockItem* item = m_dock->m_items[i];
                int y_item = m_itemsCenter[i] - item->getHeight(m_itemsSize[i]) / 2;
                int dx = m_w - m_itemSpacing / 2 - item->getWidth(m_itemsSize[i]);
                if ((int)i == m_clickedLauncherIndex) { // launcher clicked acknowlegement
                    Launcher* l = dynamic_cast<Launcher*>(item);
                    l->drawAcknowledgement(*m_buffer, dx, y_item, m_itemsSize[i]);
                } else {
                    item->draw(*m_buffer, dx, y_item, m_itemsSize[i]);
                }
            }
        }
    }
    
    if (m_isMinimised) {
        if (!m_maskOn) { // set mask
            setMaskOn(true);
        }
    } else {
        if (m_maskOn) { // clear mask
            setMaskOn(false);
        }
    }

    bitBlt(m_dock, 0, 0, m_buffer.get(), 0, 0, (m_w < dw) ? m_w : dw, (m_h < dh) ? m_h : dh);
}

/**
 * Mouse pressed event handler
 */

void ParabolicZoomState::mousePressEvent(QMouseEvent* e) {    
    if (m_inAnimation)
        return;

    // find the m_dock item's index
    int i = findItemIndex(e->x(), e->y());
    if (i < 0 || i >= m_dock->m_items.size())
        return;

    if (e->button() == Qt::LeftButton) {
        m_dock->m_items[i]->mousePressEvent(e);

        // launcher acknowledgement

        Launcher l;
        if (typeid(*m_dock->m_items[i]) == typeid(l)) {
            m_clickedLauncherIndex = i;
            QTimer::singleShot(500, this, SLOT(stopLauncherAcknowledgement()));
            m_dock->repaint();

            Launcher* launcher = dynamic_cast<Launcher*>(m_dock->m_items[i]);
            if (launcher->getCommand().compare("SHOW_DESKTOP") == 0) { // Show Desktop button
                updateAfterShowingDesktop();
            }
        }        
    } else if (e->button() == Qt::RightButton) {
        m_dock->m_items[i]->mousePressEvent(e);
    }
}

/**
 * Mouse moved event handler
 */

void ParabolicZoomState::mouseMoveEvent(QMouseEvent* e) {
    if (m_inAnimation)
        return;

    if (!m_isMinimised || enableZoom(e->x(), e->y())) {
        // zoom items
    
        setItemsPosition(e->x(), e->y());
    
        // show tooltip
        
        if (m_dock->m_showTooltip) {
            // find the m_dock item's index
            int i = findItemIndex(e->x(), e->y());
            if (i < 0 || i >= m_dock->m_items.size()) {
                m_dock->m_tooltip.hide();
            } else {
                showTooltip(i);
            }
        }
    }
}

/**
 * Enter event handler
 */

void ParabolicZoomState::enterEvent(QEvent* e) {
    QPoint pos = QCursor::pos();
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
    int x = 0;
    int y = 0;
    
    switch(m_dock->m_position) {
    case TOP:
        x = (dw - m_minW) / 2;
        break;
    case BOTTOM:
        x = (dw - m_minW) / 2;
        y = dh - m_minH;
        break;
    case LEFT:
        y = (dh - m_minH) / 2;
        break;
    case RIGHT:
        x = dw - m_minW;
        y = (dh - m_minH) / 2;
        break;
    default:
        break;
    }

    QRect rect(x, y, m_minW, m_minH);
    if (!rect.contains(pos)) // not a real enter event
        return;
    
    m_justEntered = true;
}

/**
 * Leave event handler
 */

void ParabolicZoomState::leaveEvent(QEvent* e) {
    QPoint pos = QCursor::pos();
    if (m_dock->frameGeometry().contains(pos)) // we have not really left the dock yet
        return;

    m_justLeft = true;
    m_inAnimation = true;
    for (int i = 0; i < m_dock->m_items.size(); i++) {
        m_itemsStartSize[i] = m_itemsSize[i];
        m_itemsEndSize[i] = m_dock->m_smallIconSize;
        m_itemsStartCenter[i] = m_itemsCenter[i];
        m_itemsEndCenter[i] = m_itemsCenterMin[i];
    }
    if (!m_timer->isActive()) {
        m_animationCurrentStep = 0;
        m_timer->start(32 - m_dock->m_PzZoomSpeed);
    }
}

/**
 * Update a specific item
* @TODO: fix and use the effient method (at the moment it does not work properly when the dock is vertical or in zoomed mode)
 */

void ParabolicZoomState::updateItem(int itemIndex) {
    /*
    int x = 0;
    int y = 0;
    DockItem* item = m_dock->m_items[itemIndex];
    int w = item->getWidth(m_itemsSize[itemIndex]);
    int h = item->getHeight(m_itemsSize[itemIndex]);    
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    if (m_dock->m_position == TOP) {
        y = m_itemSpacing/2;
        if (m_isMinimised) { // Minimised case
            x = m_itemsCenterMin[itemIndex] - item->getMinWidth() / 2;
            bitBlt(m_buffer.get(), x, y, m_background.get(), x + (dw - m_w) / 2, y, w, h);    
        } else { // Zoomed case
            x = m_itemsCenter[itemIndex] - item->getWidth(m_itemsSize[itemIndex]) / 2;
        }
    } else if (m_dock->m_position == BOTTOM) {
        if (m_isMinimised) { // Minimised case
            x = m_itemsCenterMin[itemIndex] - item->getMinWidth() / 2;
            y = m_h - m_itemSpacing / 2 - item->getMinHeight();
            bitBlt(m_buffer.get(), x, y, m_background.get(), x + (dw - m_w) / 2, y - m_h + m_minH, w, h);             
        } else { // Zoomed case
            x = m_itemsCenter[itemIndex] - w / 2;
            y = m_h - m_itemSpacing / 2 - h;
        }
    } else if (m_dock->m_position == LEFT) {
        x = m_itemSpacing / 2;
        if (m_isMinimised) { // Minimised case
            y = m_itemsCenterMin[itemIndex] - item->getMinHeight() / 2;
            bitBlt(m_buffer.get(), x, y, m_background.get(), x, y + (dh - m_h) / 2, w, h);    
        } else { // Zoomed case
            y = m_itemsCenter[itemIndex] - item->getHeight(m_itemsSize[itemIndex]) / 2;
        }
    } else if (m_dock->m_position == RIGHT) {
        if (m_isMinimised) { // Minimised case
            y = m_itemsCenterMin[itemIndex] - item->getMinHeight() / 2;
            x = m_w - m_itemSpacing / 2 - item->getMinWidth();
            bitBlt(m_buffer.get(), x, y, m_background.get(), x - m_w + m_minW, y + (dh - m_h) / 2, w, h);  
            std::cout << (x - m_w + m_minW) << " " << (y + (dh - m_h) / 2) << " " << w << " " << h << " " << dh << std::endl;
        } else { // Zoomed case
            y = m_itemsCenter[itemIndex] - item->getHeight(m_itemsSize[itemIndex]) / 2;
            x = m_w - m_itemSpacing / 2 - item->getWidth(m_itemsSize[itemIndex]);
        }
    }    
    item->draw(*m_buffer, x, y, m_itemsSize[itemIndex]);
    bitBlt(m_dock, x, y, m_buffer.get(), x, y, w, h);
    */

    // for now use this inefficient method
    m_dock->repaint();
}

/**
 * Update the dock when the active window has been changed
 */

void ParabolicZoomState::activeWindowChanged(WId id) { 
    if ( (id == m_dock->winId()) || (id == 0) || ShowDesktop::the()->desktopShowing() )
        return;

    if (!m_isMinimised) { // zoom mode
        KWin::WindowInfo info(id, 0, 0);
        QRect windowFrameGeometry = info.frameGeometry();
        int x = (QApplication::desktop()->width() - m_w) / 2;
        int y = (QApplication::desktop()->height() - m_h) / 2;
    
        QRect rect = m_dock->frameGeometry();
        NET::WindowType type = info.windowType(0xffff);
        NET::MappingState mapping_state = info.mappingState();
        if ((mapping_state == NET::Visible) && (type == NET::Normal || type == NET::Dialog || type == NET::Unknown || type == NET::Override) && rect.intersects(windowFrameGeometry)) {
            m_dock->setDockBelow();
            QRect intersection = rect & windowFrameGeometry;
            pauseExecution();
            
            QPixmap intersection_pix = QPixmap::grabWindow(qt_xrootwin(), intersection.x(), intersection.y(), intersection.width(), intersection.height());    
    
            QPoint pos = m_dock->mapFromGlobal(intersection.topLeft());

            switch(m_dock->m_position) {
            case TOP:
                bitBlt(&m_snapshot, pos.x() + x, pos.y() - m_minH, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case BOTTOM:
                bitBlt(&m_snapshot, pos.x() + x, pos.y(), &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case LEFT:
                bitBlt(&m_snapshot, pos.x() - m_minW, pos.y() + y, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            case RIGHT:
                bitBlt(&m_snapshot, pos.x(), pos.y() + y, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                break;
            default:
                break;
            }

            if (m_dock->m_isAutohide) {
                fadeSnapshot();
            }
                        
            m_dock->setDockAbove();
            m_dock->repaint();
        }
    }
}

/**
 * Update autohide mode
 */

void ParabolicZoomState::updateAutohideMode() {
    updateLayout();
}

/**
 * Zoom timer event handler
 */

void ParabolicZoomState::updateZoom() {
    int i;

    for (i = 0; i < (int) m_dock->m_items.size(); i++) {
        m_itemsSize[i] = m_itemsStartSize[i] + (m_itemsEndSize[i] - m_itemsStartSize[i]) * m_animationCurrentStep / m_animationNumSteps;
        m_itemsCenter[i] = m_itemsStartCenter[i] + (m_itemsEndCenter[i] - m_itemsStartCenter[i]) * m_animationCurrentStep / m_animationNumSteps;
    }
    m_dock->repaint();

    m_animationCurrentStep++;
    if (m_animationCurrentStep == 1 && !m_isMinimised)
        m_dock->setDockAbove();

    if (m_animationCurrentStep >= m_animationNumSteps) {
        m_timer->stop();
        m_inAnimation = false;

        if (m_justLeft) {
            if (!m_isMinimised) {
                m_isMinimised = true;
            }
        
            if (m_dock->m_tooltip.isShown())
                m_dock->m_tooltip.hide();
            m_justLeft = false;

            m_dock->repaint();
            m_dock->setDockBelow();
        }
    }
}

/**
 * Stop launcher acknowledgement
 */

void ParabolicZoomState::stopLauncherAcknowledgement() {
    m_clickedLauncherIndex = -1;
    m_dock->repaint();
}

/// PRIVATE ///

/**
 * Update the background image
 */

void ParabolicZoomState::updateBackgroundImage() {
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    // copy from the wallpaper

    switch(m_dock->m_position) {
    case LEFT:
        bitBlt(m_originalBackground.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, 0, m_w, dh);
        bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, 0, m_minW, dh);
        break;
    case RIGHT:
        bitBlt(m_originalBackground.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), dw - m_w, 0, m_w, dh);
        bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), dw - m_minW, 0, m_minW, dh);
        break;
    case TOP:
        bitBlt(m_originalBackground.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, 0, dw, m_h);
        bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, 0, dw, m_minH);
        break;
    case BOTTOM:
        bitBlt(m_originalBackground.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, dh - m_h, dw, m_h);
        bitBlt(m_background.get(), 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, dh - m_minH, dw, m_minH);
        break;
    default:
        break;
    }

    // fade the background
    if (m_dock->m_dockOpacity > 0) {
        KPixmapEffect::fade(*m_background, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);        
    }

    // draw borders
    if (m_dock->m_showBorders) {
        QPainter p(m_background.get());
        QPainter p1(m_sideBorder.get());

        p.setPen(m_dock->m_borderColor);
        p1.fillRect(m_sideBorder->rect(), m_dock->m_borderColor);

        switch(m_dock->m_position) {
        case LEFT:            
            p.drawLine(m_minW - 1, 0, m_minW - 1, dh - 1);
            break;
        case RIGHT:            
            p.drawLine(0, 0, 0, dh - 1);
            break;
        case TOP:            
            p.drawLine(0, m_minH - 1, dw - 1, m_minH - 1);
            break;
        case BOTTOM:            
            p.drawLine(0, 0, dw - 1, 0);
            break;
        default:
            break;
        }
    }

    // update the snapshot

    if (!m_isMinimised) {
        updateSnapshotFromWallpaper();    
        m_dock->setDockBelow();
        pauseExecution();
        updateSnapshotFromDesktop();
        m_dock->setDockAbove();
    }
}

/**
 * Update after showing/un-showing the desktop
 */

void ParabolicZoomState::updateAfterShowingDesktop() {
    bool desktopShowing = ShowDesktop::the()->desktopShowing();
    if (desktopShowing) {
        updateSnapshotFromWallpaper();        
    } else {
        m_dock->setDockBelow();
        
        pauseExecution();

        // update the snapshot from the desktop
        updateSnapshotFromDesktop();

        m_dock->setDockAbove();
    }
    m_dock->repaint();
}

/**
 * Update the snapshot, copying from wallpaper image
 */

void ParabolicZoomState::updateSnapshotFromWallpaper() {
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    switch(m_dock->m_position) {
    case TOP:
        bitBlt(&m_snapshot, 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, m_minH, dw, m_h - m_minH);
        break;
    case BOTTOM:
        bitBlt(&m_snapshot, 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), 0, dh - m_h, dw, m_h - m_minH);
        break;
    case LEFT:
        bitBlt(&m_snapshot, 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), m_minW, 0, m_w - m_minW, dh);
        break;
    case RIGHT:
        bitBlt(&m_snapshot, 0, 0, &m_dock->m_wallpaperManager->getWallpaper(KWin::currentDesktop()), dw - m_w, 0, m_w - m_minW, dh);
        break;
    default:
        break;
    }
    if (m_dock->m_isAutohide) {
        fadeSnapshot();
    }  
}

/**
 * Update the snapshot, copying from desktop
 */

void ParabolicZoomState::updateSnapshotFromDesktop() {
    int x = (QApplication::desktop()->width() - m_w) / 2;
    int y = (QApplication::desktop()->height() - m_h) / 2;

    QRect rect = m_dock->frameGeometry();
    
    for (QValueList<WId>::ConstIterator it = m_dock->m_windowManager.stackingOrder().begin(); it != m_dock->m_windowManager.stackingOrder().end(); it++) {
        if (m_dock->m_windowManager.hasWId(*it)) {
            KWin::WindowInfo info(*it, 0, 0);                
        
            if (info.onAllDesktops() || info.desktop() == KWin::currentDesktop()) {
                QRect windowFrameGeometry = info.frameGeometry();
                NET::WindowType type = info.windowType(0xffff);
                NET::MappingState mapping_state = info.mappingState();
                if ((mapping_state == NET::Visible) && (type == NET::Normal || type == NET::Dialog || type == NET::Unknown || type == NET::Override) && rect.intersects(windowFrameGeometry)) {
                    QRect intersection = rect & windowFrameGeometry;
            
                    QPixmap intersection_pix = QPixmap::grabWindow(qt_xrootwin(), intersection.x(), intersection.y(), intersection.width(), intersection.height());    
            
                    QPoint pos = m_dock->mapFromGlobal(intersection.topLeft());
        
                    switch(m_dock->m_position) {
                    case TOP:
                        bitBlt(&m_snapshot, pos.x() + x, pos.y() - m_minH, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                        break;
                    case BOTTOM:
                        bitBlt(&m_snapshot, pos.x() + x, pos.y(), &intersection_pix, 0, 0, intersection.width(), intersection.height());
                        break;
                    case LEFT:
                        bitBlt(&m_snapshot, pos.x() - m_minW, pos.y() + y, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                        break;
                    case RIGHT:
                        bitBlt(&m_snapshot, pos.x(), pos.y() + y, &intersection_pix, 0, 0, intersection.width(), intersection.height());
                        break;
                    default:
                        break;
                    }
                }                
            }
        }
    }

    if (m_dock->m_isAutohide && !m_isMinimised) {
        fadeSnapshot();
    }
}

/**
 * Find the index of the item with mouse focus given the mouse's x and y position
 */

int ParabolicZoomState::findItemIndex(int x, int y) {
    unsigned int i = 0;
    unsigned int no_items = m_dock->m_items.size();

    if (m_isMinimised) {
        if (m_dock->m_orientation == Qt::Horizontal) {
            int dx = (m_w - m_minW)/2;
            if ((x < dx) || (x > m_w - dx))
                return -1;
            if ((m_dock->m_position == TOP && y > m_minH) || (m_dock->m_position == BOTTOM && y < m_h - m_minH))
                return -1;
        } else { // Vertical
            int dy = (m_h - m_minH)/2;
            if ((y < dy) || (y > m_h - dy))
                return -1;
            if ((m_dock->m_position == LEFT && x > m_minW) || (m_dock->m_position == RIGHT && x < m_w - m_minW))
                return -1;
        }
    }

    
    if (m_dock->m_orientation == Qt::Horizontal) {
        int w = 0;
        while (i < no_items && x > w) {
            w = m_itemsCenter[i] - m_dock->m_items[i]->getWidth(m_itemsSize[i]) / 2;
            i++;
        }
        if (i == no_items && x > w)
            i = no_items - 1;
        else
            i -= 2;
    } else { // Vertical
        int h = 0;
        while (i < m_dock->m_items.size() && y > h) {
            h = m_itemsCenter[i] - m_dock->m_items[i]->getHeight(m_itemsSize[i]) / 2;
            i++;
        }
        if (i == no_items && y > h)
            i = no_items - 1;
        else
            i -= 2;
    }

    return i;
}

/**
 * Set isMinimised property given the mouse's x and y position
 */

bool ParabolicZoomState::enableZoom(int x, int y) {
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();

    int start_x = (m_w - m_minW + m_itemSpacing) / 2;
    int start_y = (m_h - m_minH + m_itemSpacing) / 2;

    switch(m_dock->m_position) {
    case LEFT:
        if (y >= start_y && y < start_y + m_minH - m_itemSpacing) {
            m_snapshot = QPixmap::grabWindow(qt_xrootwin(), m_minW, 0, m_w - m_minW, dh);
            if (m_dock->m_isAutohide) {
                fadeSnapshot();
            }
            m_isMinimised = false;
        }
        break;
    case RIGHT:
        if (y >= start_y && y < start_y + m_minH - m_itemSpacing) {
            m_snapshot = QPixmap::grabWindow(qt_xrootwin(), dw - m_w, 0, m_w - m_minW, dh);
            if (m_dock->m_isAutohide) {
                fadeSnapshot();
            }
            m_isMinimised = false;
        }
        break;
    case TOP:
        if (x >= start_x && x < start_x + m_minW - m_itemSpacing) {
            m_snapshot = QPixmap::grabWindow(qt_xrootwin(), 0, m_minH, dw, m_h - m_minH);
            if (m_dock->m_isAutohide) {
                fadeSnapshot();
            }
            m_isMinimised = false;
        }
        break;
    case BOTTOM:
        if (x >= start_x && x < start_x + m_minW - m_itemSpacing) {
            m_snapshot = QPixmap::grabWindow(qt_xrootwin(), 0, dh - m_h, dw, m_h - m_minH);
            if (m_dock->m_isAutohide) {
                fadeSnapshot();
            }
            m_isMinimised = false;
        }
        break;
    default:
        break;
    }

    return !m_isMinimised;
}

/**
 * Set items' position (and size) given the mouse's x and y position
 * size is a function of position calculated by the parabolic function
 */

void ParabolicZoomState::setItemsPosition(int x, int y) {
    int i = 0;
    unsigned int no_items = m_dock->m_items.size();
    int first_unchanged_index_right = no_items;

    if (m_dock->m_orientation == Qt::Horizontal) {

        // determine the size of the icons

        for (i = 0; i < no_items; i++) {
            int delta_x = abs(m_itemsCenterMin[i] - x);
            if (delta_x < m_functionMaxX) {
                m_itemsSize[i] = function(delta_x);
            } else {
                m_itemsSize[i] = m_dock->m_smallIconSize;
                if (m_itemsCenterMin[i] > x && first_unchanged_index_right == no_items)
                    first_unchanged_index_right = i;
            }
        }

        // determine the position of the icons

        m_itemsCenter[0] = m_itemSpacing / 2 + m_dock->m_items[0]->getWidth(m_itemsSize[0])/2;
        for (i = 1; i < no_items; i++) {
            m_itemsCenter[i] = m_itemsCenter[i-1] + (m_dock->m_items[i]->getWidth(m_itemsSize[i]) + m_dock->m_items[i-1]->getWidth(m_itemsSize[i-1]))/2 + m_itemSpacing;
        }

        if (first_unchanged_index_right < no_items) {
            m_itemsCenter[no_items - 1] = m_w - m_dock->m_items[no_items - 1]->getWidth(m_dock->m_smallIconSize)/2 - m_itemSpacing/2;
            for (i = no_items - 2; i >= first_unchanged_index_right; i--) {
                m_itemsCenter[i] = m_itemsCenter[i+1] - (m_dock->m_items[i]->getWidth(m_itemsSize[i]) + m_dock->m_items[i+1]->getWidth(m_itemsSize[i+1]))/2 - m_itemSpacing;
            }
        }

        if (first_unchanged_index_right <= 5 ) {
             for (i = first_unchanged_index_right - 1; i >= 0; i--) {
                m_itemsCenter[i] = m_itemsCenter[i+1] - (m_dock->m_items[i]->getWidth(m_itemsSize[i]) + m_dock->m_items[i+1]->getWidth(m_itemsSize[i+1]))/2 - m_itemSpacing;
            }
        }
    } else { // Vertical

        // determine the size of the icons

        for (i = 0; i < no_items; i++) {
            int delta_y = abs(m_itemsCenterMin[i] - y);
            if (delta_y < m_functionMaxX) {
                m_itemsSize[i] = function(delta_y);
            } else {
                m_itemsSize[i] = m_dock->m_smallIconSize;
                if (m_itemsCenterMin[i] > y && first_unchanged_index_right == no_items)
                    first_unchanged_index_right = i;
            }
        }

        // determine the position of the icons

        m_itemsCenter[0] = m_itemSpacing / 2 + m_dock->m_items[0]->getHeight(m_itemsSize[0])/2;
        for (i = 1; i < no_items; i++) {
            m_itemsCenter[i] = m_itemsCenter[i-1] + (m_dock->m_items[i]->getHeight(m_itemsSize[i]) + m_dock->m_items[i-1]->getHeight(m_itemsSize[i-1]))/2 + m_itemSpacing;
        }

        if (first_unchanged_index_right < no_items) {
            m_itemsCenter[no_items - 1] = m_h - m_dock->m_items[no_items - 1]->getHeight(m_dock->m_smallIconSize)/2 - m_itemSpacing/2;
            for (i = no_items - 2; i >= first_unchanged_index_right; i--) {
                m_itemsCenter[i] = m_itemsCenter[i+1] - (m_dock->m_items[i]->getHeight(m_itemsSize[i]) + m_dock->m_items[i+1]->getHeight(m_itemsSize[i+1]))/2 - m_itemSpacing;
            }
        }

        if (first_unchanged_index_right <= 5 ) {
             for (i = first_unchanged_index_right - 1; i >= 0; i--) {
                m_itemsCenter[i] = m_itemsCenter[i+1] - (m_dock->m_items[i]->getHeight(m_itemsSize[i]) + m_dock->m_items[i+1]->getHeight(m_itemsSize[i+1]))/2 - m_itemSpacing;
            }
        }
    }

    if (m_justEntered) {
        m_justEntered = false;
        m_inAnimation = true;
        for (i = 0; i < no_items; i++) {
            m_itemsStartSize[i] = m_dock->m_smallIconSize;
            m_itemsEndSize[i] = m_itemsSize[i];
            m_itemsStartCenter[i] = m_itemsCenterMin[i];
            m_itemsEndCenter[i] = m_itemsCenter[i];
        }
        if (!m_timer->isActive()) {
            m_animationCurrentStep = 0;
            m_timer->start(32 - m_dock->m_PzZoomSpeed);
        }
    } else if (!m_inAnimation) {
        m_dock->repaint();
    }
}

/**
 * Show tooltip at item index
 */

void ParabolicZoomState::showTooltip(int index) {
    m_dock->m_tooltip.setText(m_dock->m_items[index]->getDescription());

    int x = m_dock->x();
    int y  = m_dock->y();
    if (m_dock->m_orientation == Qt::Horizontal) {
        x += m_itemSpacing / 2;
        for (int i = 0; i < index; i++) {
            x += m_dock->m_items[i]->getWidth(m_itemsSize[i]) + m_itemSpacing;
        }
        x = x + m_dock->m_items[index]->getWidth(m_itemsSize[index]) / 2 - m_dock->m_tooltip.width() / 2;

        if (m_dock->m_position == TOP)
            y = y + m_dock->height() + TOOLTIP_SPACE;
        else // BOTTOM
            y = y - m_dock->m_tooltip.height() - TOOLTIP_SPACE;
    } else { // Vertical
        y += m_itemSpacing / 2;
        for (int i = 0; i < index; i++) {
            y += m_dock->m_items[i]->getHeight(m_itemsSize[i]) + m_itemSpacing;
        }
        y = y + m_dock->m_items[index]->getHeight(m_itemsSize[index]) / 2 - m_dock->m_tooltip.height() / 2;

        if (m_dock->m_position == LEFT)
            x = x + m_dock->width() + TOOLTIP_SPACE;
        else // RIGHT
            x = x - m_dock->m_tooltip.width() - TOOLTIP_SPACE;
    }

    m_dock->m_tooltip.move(x, y);
    if (!m_dock->m_tooltip.isShown())
        m_dock->m_tooltip.show();
}

/**
 * Fade and draw a border on the snapshot (in autohide mode)
 */

void ParabolicZoomState::fadeSnapshot() {
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
    int minW = m_dock->m_smallIconSize + m_itemSpacing;
    int minH = m_dock->m_smallIconSize + m_itemSpacing;
    KPixmap pix;

    switch(m_dock->m_position) {
    case LEFT:
        pix = QPixmap(minW, m_h);
        bitBlt(&pix, 0, 0, &m_snapshot, 0, (dh - m_h) / 2, minW, m_h);
        KPixmapEffect::fade(pix, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
        bitBlt(&m_snapshot, 0, (dh - m_h) / 2, &pix, 0, 0, minW, m_h);
        break;
    case RIGHT:
        pix = QPixmap(minW, m_h);
        bitBlt(&pix, 0, 0, &m_snapshot, m_w - minW, (dh - m_h) / 2, minW, m_h);
        KPixmapEffect::fade(pix, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
        bitBlt(&m_snapshot, m_w - minW, (dh - m_h) / 2, &pix, 0, 0, minW, m_h);
        break;
    case TOP:
        pix = QPixmap(m_w, minH);
        bitBlt(&pix, 0, 0, &m_snapshot, (dw - m_w) / 2, 0, m_w, minH);
        KPixmapEffect::fade(pix, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
        bitBlt(&m_snapshot, (dw - m_w) / 2, 0, &pix, 0, 0, m_w, minH);
        break;
    case BOTTOM:
        pix = QPixmap(m_w, minH);
        bitBlt(&pix, 0, 0, &m_snapshot, (dw - m_w) / 2, m_h - minH, m_w, minH);
        KPixmapEffect::fade(pix, ((float)m_dock->m_dockOpacity)*0.01, m_dock->m_backgroundColor);
        bitBlt(&m_snapshot, (dw - m_w) / 2, m_h - minH, &pix, 0, 0, m_w, minH);
        break;
    default:
        break;
    }

    // draw borders
    if (m_dock->m_showBorders) {       
        QPainter p(&m_snapshot);

        p.setPen(m_dock->m_borderColor);

        switch(m_dock->m_position) {
        case LEFT:
            p.drawLine(minW - 1, 0, minW - 1, dh - 1);            
            break;
        case RIGHT:
            p.drawLine(m_w - minW, 0, m_w - minW, dh - 1);
            break;
        case TOP:
            p.drawLine(0, minH - 1, dw - 1, minH - 1);
            break;
        case BOTTOM:
            p.drawLine(0, m_h - minH, dw - 1, m_h - minH);
            break;
        default:
            break;
        }
    }
}

/**
 * Pause execution
 */

void ParabolicZoomState::pauseExecution() {
    usleep(100000); // pause for 100 miliseconds    
}

/**
 * Set mask on/off
 */

void ParabolicZoomState::setMaskOn(bool maskOn) {
    if (maskOn) { // set mask
        QBitmap mask(m_dock->size());
        QPainter pm;
        pm.begin(&mask);
        pm.fillRect(m_dock->rect(), Qt::color0);
        
        int rx = 0;
        int ry = 0;

        switch(m_dock->m_position) {
        case LEFT:
            rx = 0; // for the sake of clarity
            ry = (m_h - m_minH) / 2;
            break;
        case RIGHT:
            rx = m_w - m_minW;
            ry = (m_h - m_minH) / 2;
            break;
        case TOP:
            rx = (m_w - m_minW) / 2;
            ry = 0;
            break;
        case BOTTOM:
            rx = (m_w - m_minW) / 2;
            ry = m_h - m_minH;
            break;
        default:
            break;
        }

        QRect rect(rx, ry, m_minW, m_minH);
        pm.fillRect(rect, Qt::color1);
        pm.end();
        m_dock->setMask(mask);
        m_maskOn = true;
    } else {

        // for some reason, clearMask does not work anymore
        //m_dock->clearMask();

        QBitmap mask(m_dock->size());
        QPainter pm;
        pm.begin(&mask);
        pm.fillRect(m_dock->rect(), Qt::color1);
        pm.end();
        m_dock->setMask(mask);
        m_maskOn = false;
    }
}

/**
 * Parabolic function
 * @param x the distant from the center of the icon to the mouse position
 * @return the icon size
 */

int ParabolicZoomState::function(int x) {
    if ( (x < -m_functionMaxX) || (x > m_functionMaxX) )
        return m_dock->m_smallIconSize;
    else
        return (m_dock->m_bigIconSize - (x * x * (m_dock->m_bigIconSize - m_dock->m_smallIconSize)) / (m_functionMaxX * m_functionMaxX));
}

#include "paraboliczoomstate.moc"

