/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <math.h>

#include <qapp.h>
#include <qthread.h>
#include <qevent.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <qpopupmenu.h>

#include <klistview.h>
#include <klocale.h>
#include <kaccel.h>

#include "UserEvents.h"
#include "Process.h"
#include "StrTok.h"
#include "KYumConf.h"
#include "KYumView.h"
#include "KYumPreferences.h"
#include "DlgUserPrompt.h"
#include "DlgFindPackages.h"
#include "DlgStopAction.h"
#include "DlgCleanCache.h"
#include "DlgChangeLog.h"
#include "DlgDependencies.h"
#include "DlgRemovePkgs.h"
#include "DlgRun.h"
#include "ActionsView.h"


enum
  {
    c_idxPackage,
    c_idxVersion,
//    c_idxInstVer,
    c_idxArch,
    c_idxRepoTag,
    c_idxSize,
    c_idxSummary
  };

static struct
  {
    const char  * m_pTitle;
    int           m_cx;
  }
  g_columns[] =
    {
      { /*i18n*/("Package"),       -1 },
      { /*i18n*/("Version"),       -1 },
//      { /*i18n*/("Inst. Version"), -1 },
      { /*i18n*/("Architecture"),  -1 },
      { /*i18n*/("Repository"),    -1 },
      { /*i18n*/("Size"),          -1 },
      { /*i18n*/("Summary"),       -1 }
    };


#define ArraySize(a)  (sizeof(a) / sizeof((a)[0]))


class PkgListView;

class PkgListItem : public QCheckListItem
  {
    private:
      PackageInfo::Ptr  m_ptrPkgInfo;


    protected:
      virtual void      stateChange (bool bStatus);


    public:
                        PkgListItem (PkgListView *     pParent,
                                     PackageInfo::Ptr  ptrPkgInfo);

      virtual         ~ PkgListItem ();

      virtual void      paintCell   (QPainter *          pPainter,
                                     const QColorGroup & cg,
                                     int                 column,
                                     int                 width,
                                     int                 alignment);

      PackageInfo::Ptr  getPkgInfo  ()       { return m_ptrPkgInfo; }

      const
      PackageInfo::Ptr  getPkgInfo  () const { return m_ptrPkgInfo; }

      void              setPkgInfo  (PackageInfo::Ptr ptrInfo);
  };


class PkgListView : public KListView
  {
    private:
      typedef std::map<QString, PkgListItem *> ItemMap;

    private:
      ItemMap       m_itemMap;

    public:
                    PkgListView(QWidget * pParent, const char * pName = 0)
                      : KListView(pParent, pName)
                    {
                    }

      virtual     ~ PkgListView()
                    {
                    }

      void          addItemToMap(QString strName, PkgListItem * pItem)
                    {
                        m_itemMap[strName] = pItem;
                    }

      void          removeItemFromMap(QString strName)
                    {
                        m_itemMap.erase(m_itemMap.find(strName));
                    }

      const
      PkgListItem * getItemWithKey(const QString & strName) const;

      PkgListItem * getItemWithKey(const QString & strName)
                    {
                        return (PkgListItem *)
                               ((const PkgListView *) this)->getItemWithKey(strName);
                    }
};


class ListTitle : public QWidget
  {
    private:
      QLabel      * m_pTitle;

    public:
                    ListTitle (QWidget * pParent, const char * pName);
      virtual     ~ ListTitle () {}

      void          setTitle  (QString strTitle) { m_pTitle->setText(strTitle); }
  };


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ListTitle::ListTitle(QWidget * pParent, const char * pName)
  : QWidget(pParent, pName)
{
    QHBoxLayout * pLayout = new QHBoxLayout(this, 5, 5);

    m_pTitle = new QLabel(this);
    m_pTitle->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

    QFont titleFont(m_pTitle->font());
    titleFont.setPointSize(14);
    titleFont.setBold(true);
    m_pTitle->setFont(titleFont);

    pLayout->addWidget(m_pTitle);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

const PkgListItem * PkgListView::getItemWithKey(const QString & strKey) const
{
    ItemMap::const_iterator it = m_itemMap.find(strKey);
    return it == m_itemMap.end() ? 0 : it->second;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

PkgListItem::PkgListItem(PkgListView * pParent, PackageInfo::Ptr ptrInfo)

  : QCheckListItem(pParent, ptrInfo->m_strName, QCheckListItem::CheckBox),
    m_ptrPkgInfo(ptrInfo)

{
    pParent->addItemToMap(ptrInfo->m_strName, this);
    setPkgInfo(ptrInfo);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

PkgListItem::~PkgListItem()
{
    PkgListView * pList = dynamic_cast<PkgListView *>(parent());

    if ( pList )
        pList->removeItemFromMap(m_ptrPkgInfo->m_strName);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void PkgListItem::setPkgInfo(PackageInfo::Ptr ptrInfo)
{
    if ( ptrInfo.isValid() )
    {
        setText(c_idxVersion, ptrInfo->m_strVersion);
//        setText(c_idxInstVer, ptrInfo->m_strInstalledVer);
        setText(c_idxArch,    ptrInfo->m_strArch);
        setText(c_idxRepoTag, ptrInfo->m_strRepoTag);
        setText(c_idxSummary, QString::fromUtf8(ptrInfo->m_strSummary));
        setText(c_idxSize,    ptrInfo->m_strSize);

        setOn(ptrInfo->m_bChecked);

        m_ptrPkgInfo = ptrInfo;
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void PkgListItem::stateChange(bool /*bStatus*/)
{
    //
    // If we do this here, ActionsView::slotListItemSelected() won't work
    // properly since we cannot determine if the package was selected
    // before (i.e. if the total count of selected packages must be
    // increased or decreased).
    //
    //m_ptrPkgInfo->m_bChecked = isOn();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void PkgListItem::paintCell(QPainter *          pPainter,
                            const QColorGroup & clrGroup,
                            int                 column,
                            int                 width,
                            int                 alignment)
{
    QColorGroup cg(clrGroup);
    QColor      clr = cg.text();

    if ( isOn() )
        cg.setColor(QColorGroup::Text, Qt::blue);

    QCheckListItem::paintCell(pPainter, cg, column, width, alignment);
    cg.setColor(QColorGroup::Text, clr);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ActionsView::ActionsView(QWidget * pParent)

  : QWidget           (pParent),
    m_pProcess        (0),
    m_currentAction   (c_NoAction),
    m_currentList     (c_NoList),
    m_pCurrentItem    (0),
    m_numCheckedItems (0),
    m_totalPkgSize    (0.0)

{
    m_strStdErrBuf = "*** ";

    QHBoxLayout * pTopLayout = new QHBoxLayout(this, 0, 6, "TopLayout");

    //
    // Left button panel
    //
    QWidget     * pVButtonPanel = new QWidget(this, "VButtonPanel");
    QVBoxLayout * pVLayout      = new QVBoxLayout(pVButtonPanel, 6);

    m_pButtonListAvail = new QPushButton(pVButtonPanel, "buttonListAvail");
    m_pButtonListAvail->setText(i18n("List &Available"));

    m_pButtonListUpdates = new QPushButton(pVButtonPanel, "buttonListUpdates");
    m_pButtonListUpdates->setText(i18n("List &Updates"));

    m_pButtonListInstalled = new QPushButton(pVButtonPanel, "buttonListInstalled");
    m_pButtonListInstalled->setText(i18n("&List Installed"));

    m_pButtonSearch = new QPushButton(pVButtonPanel, "buttonSearch");
    m_pButtonSearch->setText(i18n("&Find..."));

    m_pCheckDescriptions = new QCheckBox(pVButtonPanel, "checkDescriptions");
    m_pCheckDescriptions->setText(i18n("&Descriptions"));
    m_pCheckDescriptions->setChecked(true);

    m_pButtonRemove = new QPushButton(pVButtonPanel, "buttonRemove");
    m_pButtonRemove->setText(i18n("&Remove Selected"));

    m_pButtonInstall = new QPushButton(pVButtonPanel, "buttonInstall");
    m_pButtonInstall->setText(i18n("&Install Selected"));

    m_pButtonUpdate = new QPushButton(pVButtonPanel, "buttonUpdate");
    m_pButtonUpdate->setText(i18n("Update S&ystem"));

    m_pCheckObsoletes = new QCheckBox(pVButtonPanel, "checkObsoletes");
    m_pCheckObsoletes->setText(i18n("&Obsoletes"));

    m_pCheckCache = new QCheckBox(pVButtonPanel, "checkCache");
    m_pCheckCache->setText(i18n("Use &Cache"));

    pVLayout->addWidget(m_pButtonListAvail);
    pVLayout->addWidget(m_pButtonListUpdates);
    pVLayout->addWidget(m_pButtonListInstalled);
    pVLayout->addWidget(m_pButtonSearch);
    pVLayout->addWidget(m_pCheckDescriptions);

    QSpacerItem * pVSpacer = new QSpacerItem(60, 20, QSizePolicy::Minimum, QSizePolicy::Expanding);
    pVLayout->addItem(pVSpacer);

    pVLayout->addWidget(m_pButtonRemove);
    pVLayout->addWidget(m_pButtonInstall);
    pVLayout->addWidget(m_pButtonUpdate);
    pVLayout->addWidget(m_pCheckObsoletes);
    pVLayout->addWidget(m_pCheckCache);

    pTopLayout->addWidget(pVButtonPanel);

    //
    // List view and buttons below it...
    //
    QWidget     * pRightView    = new QWidget(this, "RightView");
    QVBoxLayout * pRightLayout  = new QVBoxLayout(pRightView, 0, 6, "RightLayout");
    QWidget     * pHButtonPanel = new QWidget(pRightView, "HButtonPanel");
    QHBoxLayout * pHLayout      = new QHBoxLayout(pHButtonPanel, 2);


    m_pButtonSelectAll = new QPushButton(pHButtonPanel, "buttonSelectAll");
    m_pButtonSelectAll->setText(i18n("&Select All"));

    m_pButtonDeselectAll = new QPushButton(pHButtonPanel, "buttonDeselectAll");
    m_pButtonDeselectAll->setText(i18n("&Deselect All"));

    //
    //
    //
    QString       strSampleInfo    = i18n("Selected : 99999/99999 packages, 99999.99 MB");
    QWidget     * pSelInfoPanel    = new QWidget(pHButtonPanel, "SelInfoPanel");
    QHBoxLayout * pInfoPanelLayout = new QHBoxLayout(pSelInfoPanel, 0, 6);

    pSelInfoPanel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Maximum);

    m_pSelectionInfo = new QLabel(pSelInfoPanel, "SelectionInfo");
    m_pSelectionInfo->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Maximum);

    QFontMetrics fm(m_pSelectionInfo->font());
    m_pSelectionInfo->setMinimumWidth(fm.width(strSampleInfo));

    pInfoPanelLayout->addSpacing(25);
    pInfoPanelLayout->addWidget(m_pSelectionInfo);
    pInfoPanelLayout->addSpacing(25);

    m_pListView = 0;  // To avoid crash in updateSelectionInfo()
    updateSelectionInfo();


    m_pLabelFindInList = new QLabel(pHButtonPanel, "labelFindInList" );
    m_pLabelFindInList->setText(i18n("Search List:"));
    m_pEditFindInList = new QLineEdit(pHButtonPanel, "editFindInList");
    m_pEditFindInList->setMinimumWidth(100);

    m_pButtonFindPrevInList  = new QPushButton(pHButtonPanel, "buttonFindPrevInList");
    m_pButtonFindPrevInList->setText("<<");

    m_pButtonFindNextInList  = new QPushButton(pHButtonPanel, "buttonFindNextInList");
    m_pButtonFindNextInList->setText(">>");

    m_pButtonFindNextInList->setEnabled(false);
    m_pButtonFindPrevInList->setEnabled(false);

/*
    QSpacerItem * pHSpacer = new QSpacerItem(60, 0, QSizePolicy::Expanding, QSizePolicy::Maximum);
*/
    pHLayout->addWidget(m_pButtonSelectAll);
    pHLayout->addWidget(m_pButtonDeselectAll);
  //  pHLayout->addItem(pHSpacer);
    pHLayout->addWidget(pSelInfoPanel);
    pHLayout->addWidget(m_pLabelFindInList);
    pHLayout->addWidget(m_pEditFindInList);
    pHLayout->addWidget(m_pButtonFindPrevInList);
    pHLayout->addWidget(m_pButtonFindNextInList);

    m_pListTitle = new ListTitle(pRightView, "m_pListTitle");
    m_pListView  = new PkgListView(pRightView, "m_pListView");

    pRightLayout->addWidget(m_pListTitle);
    pRightLayout->addWidget(m_pListView, 1);
    pRightLayout->addWidget(pHButtonPanel);

    pTopLayout->addWidget(pRightView);

    m_pListView->setAllColumnsShowFocus(true);
    m_pListView->setSelectionMode(QListView::Single);

    for ( int idx=0; idx < (int) ArraySize(g_columns); idx++ )
        m_pListView->addColumn(g_columns[idx].m_pTitle, g_columns[idx].m_cx);

    m_pMutex = new QMutex;

    KAccel * pAccel = new KAccel(this);

    pAccel->insert(KStdAccel::FindNext, this, SLOT(slotFindNextInList()));
    pAccel->insert(KStdAccel::FindPrev, this, SLOT(slotFindPrevInList()));

    enableActions(true);

    m_pProcess = new Process(this);

    connect(m_pButtonListAvail,     SIGNAL(clicked()), this, SLOT(listAvailable()));
    connect(m_pButtonListUpdates,   SIGNAL(clicked()), this, SLOT(listUpdates()));
    connect(m_pButtonListInstalled, SIGNAL(clicked()), this, SLOT(listInstalled()));
    connect(m_pButtonSearch,        SIGNAL(clicked()), this, SLOT(slotSearch()));

    connect(m_pButtonSelectAll,     SIGNAL(clicked()), this, SLOT(slotSelectAll()));
    connect(m_pButtonDeselectAll,   SIGNAL(clicked()), this, SLOT(slotDeselectAll()));

    connect(m_pButtonRemove,        SIGNAL(clicked()), this, SLOT(slotRemove()));
    connect(m_pButtonInstall,       SIGNAL(clicked()), this, SLOT(slotInstall()));
    connect(m_pButtonUpdate,        SIGNAL(clicked()), this, SLOT(slotUpdate()));

    connect(m_pButtonFindPrevInList, SIGNAL(clicked()),
            this,                   SLOT(slotFindPrevInList()));

    connect(m_pButtonFindNextInList, SIGNAL(clicked()),
            this,                   SLOT(slotFindNextInList()));

    connect(m_pEditFindInList,      SIGNAL(textChanged(const QString &)),
            this,                   SLOT(slotFindInList(const QString &)));

    connect(m_pListView,            SIGNAL(currentChanged(QListViewItem *)),
            this,                   SLOT(slotListItemSelected(QListViewItem *)));

    connect(m_pListView,  SIGNAL(selectionChanged(QListViewItem *)),
            this,         SLOT(slotListItemSelected(QListViewItem *)));

    connect(m_pListView,  SIGNAL(clicked(QListViewItem *)),
            this,         SLOT(slotListItemSelected(QListViewItem *)));

    connect(m_pListView,
            SIGNAL(contextMenuRequested(QListViewItem *, const QPoint &, int)),
            this,
            SLOT(slotListContextMenu(QListViewItem *, const QPoint &, int)));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

ActionsView::~ActionsView()
{
    delete m_pMutex;
    delete m_pProcess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::updateSelectionInfo()
{
    QString strInfo;

    strInfo.sprintf(i18n("Selected: %d/%d packages, %.2f MB"),
                    m_numCheckedItems,
                    m_pListView ? m_pListView->childCount() : 0,
                    m_totalPkgSize);

    m_pSelectionInfo->setText(strInfo);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::clearPkgList()
{
    m_pListView->clear();
    m_packageMap.clear();
    m_pkgListMap.clear();

    m_numCheckedItems = 0;
    m_totalPkgSize    = 0.0;

    updateSelectionInfo();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::enableActions(bool bEnable)
{
    m_bActionsEnabled = bEnable;

    if ( bEnable && 0 < m_numCheckedItems )
    {
        m_pButtonRemove->setEnabled(c_Installed    == m_currentList ||
                                    c_SearchResult == m_currentList);

        m_pButtonInstall->setEnabled(m_currentList != c_Installed);
    }
    else
    {
        m_pButtonRemove->setEnabled(false);
        m_pButtonInstall->setEnabled(false);
    }

    m_pButtonListAvail->setEnabled(bEnable);
    m_pButtonListUpdates->setEnabled(bEnable);
    m_pButtonListInstalled->setEnabled(bEnable);
    m_pButtonSearch->setEnabled(bEnable);
    m_pButtonSelectAll->setEnabled(bEnable);
    m_pButtonDeselectAll->setEnabled(bEnable);

    m_pButtonUpdate->setEnabled(bEnable);

    m_pCheckObsoletes->setEnabled(bEnable);
    m_pCheckCache->setEnabled(bEnable);
    m_pCheckDescriptions->setEnabled(bEnable);

    m_pEditFindInList->setEnabled(0 < m_pListView->childCount());

    bool bSearchEnabled = !m_pEditFindInList->text().isEmpty();

    m_pButtonFindNextInList->setEnabled(bSearchEnabled);
    m_pButtonFindPrevInList->setEnabled(bSearchEnabled);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::listAvailable()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    if ( m_pCheckDescriptions->isChecked() )
    {
        m_pProcess->addArgument("info");
        m_currentAction = c_InfoAvail;
    }
    else
    {
        m_pProcess->addArgument("list");
        m_currentAction = c_ListAvail;
    }

    m_pProcess->addArgument("available");

    clearPkgList();


    if ( startAction() )
        m_currentList = c_Available;
    else
    {
        m_currentAction = c_NoAction;
        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

PackageInfo::Ptr ActionsView::getPackageFromMap(const QString & strKey) const
{
    PackageInfo::Ptr            ptrPackage;
    PackageMap::const_iterator  it = m_packageMap.find(strKey);

    if ( it != m_packageMap.end() )
        ptrPackage = it->second;

    return ptrPackage;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

PackageInfo::Ptr ActionsView::getPackageFromMap(const PackageInfo & package) const
{
    return getPackageFromMap(package.getMapKey());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

PackageInfo::Ptr ActionsView::getPackageFromMap(const QListViewItem * pItem) const
{
    PackageInfo::Ptr            ptrPackage;
    PkgListMap::const_iterator  it = m_pkgListMap.find(pItem);

    if ( it != m_pkgListMap.end() )
        ptrPackage = it->second;

    return ptrPackage;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::putPackageIntoMap(PackageInfo::Ptr ptrPackage, QListViewItem * pItem)
{
    m_packageMap[ptrPackage->getMapKey()] = ptrPackage;
    m_pkgListMap[pItem] = ptrPackage;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

double ActionsView::getPkgSize(const QListViewItem * pItem) const
{
    PackageInfo::Ptr ptrPackage = getPackageFromMap(pItem);

    return ptrPackage.isValid() ? ptrPackage->getPkgSize() : 0.0;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::listUpdates()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    if ( m_pCheckDescriptions->isChecked() )
    {
        m_pProcess->addArgument("info");
        m_currentAction = c_InfoUpdates;
    }
    else
    {
        m_pProcess->addArgument("list");
        m_currentAction = c_ListUpdates;
    }

    m_pProcess->addArgument("updates");

    clearPkgList();

    if ( startAction() )
        m_currentList = c_Updates;
    else
    {
        m_currentAction = c_NoAction;
        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::listInstalled()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    if ( m_pCheckDescriptions->isChecked() )
    {
        m_pProcess->addArgument("info");
        m_currentAction = c_InfoInstalled;
    }
    else
    {
        m_pProcess->addArgument("list");
        m_currentAction = c_ListInstalled;
    }

    m_pProcess->addArgument("installed");

    clearPkgList();

    if ( startAction() )
        m_currentList = c_Installed;
    else
    {
        m_currentAction = c_NoAction;
        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotSearch()
{
    DlgFindPackages dlg(this);

    if ( QDialog::Accepted == dlg.exec() )
    {
        m_pProcess->clearArguments();
        m_pProcess->addArgument("yum");
        m_pProcess->addArgument("-y");

        if ( m_pCheckCache->isChecked() )
            m_pProcess->addArgument("-C");

        switch ( dlg.getMode() )
        {
          case DlgFindPackages::c_Search:
                      if ( dlg.getInstall() )
                      {
                          m_currentAction = c_Install;
                          m_pProcess->addArgument("install");
                      }
                      else
                      {
                          if ( m_pCheckDescriptions->isChecked() )
                          {
                              m_currentAction = c_Info;
                              m_pProcess->addArgument("info");
                          }
                          else
                          {
                              m_currentAction = c_Search;
                              m_pProcess->addArgument("search");
                          }
                      }
                      break;

          case DlgFindPackages::c_Provides:
                      if ( dlg.getInstall() )
                          m_currentAction = c_FindAndInstall;
                      else
                          m_currentAction = c_Search;

                      m_pProcess->addArgument("whatprovides");
                      break;
        }


        QStringList                 patterns;
        QStringList::ConstIterator  it;

        dlg.getPatterns(patterns);

        for ( it = patterns.begin(); it != patterns.end(); ++it )
        {
            m_pProcess->addArgument(*it);
        }

        clearPkgList();

        if ( startAction() )
            m_currentList = c_SearchResult;
        else
        {
            m_currentAction = c_NoAction;
            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::switchAffectedRepos()
{
//
// Disabled because other repositories might provide needed dependencies!
//
#if 0
    //
    // First go through the package list and find out which repositories
    // we actually need....
    //
    QListViewItemIterator   itList(m_pListView);
    std::map<QString, bool> neededMap;

    while ( itList.current() )
    {
        QCheckListItem *  pItem   = dynamic_cast<QCheckListItem *>(itList.current());
        QString           strRepo = pItem->text(c_idxRepoTag);

        if ( pItem->isOn() )
            neededMap[strRepo] = true;

        ++itList;
    }

    //
    // Then walk through the list of repository files and through all
    // repositories within them to disable those that we do not need...
    //
    const KYumConf &                yumConf   = KYumView::getKYumView()->getYumConf();
    const RepoFile::List &          repoFiles = yumConf.getRepoFiles();
    RepoFile::List::const_iterator  itFiles;
    QString                         strArg;

    for ( itFiles = repoFiles.begin(); itFiles != repoFiles.end(); itFiles++ )
    {
        RepoFile::Ptr               ptrFile  = *itFiles;
        const Repo::List &          repoList = ptrFile->getRepos();
        Repo::List::const_iterator  itRepos;

        for ( itRepos = repoList.begin(); itRepos != repoList.end(); itRepos++ )
        {
            Repo::Ptr ptrRepo = *itRepos;

            if ( neededMap.find(ptrRepo->getTag()) == neededMap.end() )
            {
                if ( ptrRepo->isEnabled() )
                {
                    strArg  = "--disablerepo=";
                    strArg += ptrRepo->getTag();

                    m_pProcess->addArgument(strArg);
                }
            }
        }
    }
#endif
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotLocalUpdate()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotLocalInstall()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotInstall()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    switchAffectedRepos();

    if ( c_Updates == m_currentList )
    {
        if ( m_pCheckObsoletes->isChecked() )
            m_pProcess->addArgument("upgrade");
        else
            m_pProcess->addArgument("update");
    }
    else
        m_pProcess->addArgument("install");


    //
    // Find out which list items are checked and mark them for
    // installation...
    //
    QListViewItemIterator it(m_pListView);
    int                   numSelected = 0;

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        if ( pItem->isOn() )
        {
            m_pProcess->addArgument(pItem->text(c_idxPackage));
            numSelected++;

            PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

            if ( ptrPkgInfo )
                ptrPkgInfo->m_bInstalling = true;
        }

        ++it;
    }

    if ( 0 < numSelected )
    {
        m_currentAction = c_Install;

        if ( !startAction() )
        {
            m_currentAction = c_NoAction;
            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
        }
    }
    else
    {
        m_currentAction = c_NoAction;
        QMessageBox::information(this, "KYum", i18n("The selection is empty."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotUpdate()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");

    if ( m_pCheckCache->isChecked() )
        m_pProcess->addArgument("-C");

    if ( m_pCheckObsoletes->isChecked() )
    {
        m_pProcess->addArgument("upgrade");
    }
    else
        m_pProcess->addArgument("update");


    m_currentAction = c_Install;

    if ( !startAction() )
    {
        m_currentAction = c_NoAction;
        QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotRemove()
{
    m_pProcess->clearArguments();
    m_pProcess->addArgument("yum");
    m_pProcess->addArgument("-y");
    m_pProcess->addArgument("remove");

    QListViewItemIterator it(m_pListView);
    int                   numSelected = 0;
    QStringList           pkgList;

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        if ( pItem->isOn() )
        {
            PackageInfo::Ptr ptrPkg = getPackageFromMap(pItem);

            m_pProcess->addArgument(ptrPkg->getNameAndVersion());
            pkgList.push_back(ptrPkg->m_strName);
            numSelected++;
        }

        ++it;
    }

    bool bContinue = false;

    if ( 0 < numSelected )
    {
        DlgRemovePkgs dlg(pkgList, this);

        //
        // Never forget: The process must be started before exec(). No
        //               need to call dlg.setArgs(), because this has
        //               already been done it the constructor of DlgRemovePkgs.
        //
        dlg.startProcess();

        if ( dlg.exec() == DlgRemovePkgs::Accepted )
        {
            m_currentAction = c_Remove;
            bContinue       = true;
        }
    }
    else
        QMessageBox::warning(this, "KYum", i18n("The selection is empty."));


    if ( !bContinue || !startAction() )
    {
        m_currentAction = c_NoAction;

        if ( bContinue )
            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::cleanCache()
{
    DlgCleanCache dlg(this);

    if ( DlgCleanCache::Accepted == dlg.exec() )
    {
        int which = dlg.cleanWhich();

        m_pProcess->clearArguments();
        m_pProcess->addArgument("yum");
        m_pProcess->addArgument("-y");
        m_pProcess->addArgument("clean");

        if ( which == DlgCleanCache::c_cleanAll )
        {
            m_pProcess->addArgument("all");
        }
        else
        {
            if ( which & DlgCleanCache::c_cleanHeaders )
                m_pProcess->addArgument("headers");

            if ( which & DlgCleanCache::c_cleanPackages )
                m_pProcess->addArgument("packages");

            if ( which & DlgCleanCache::c_cleanMetadata )
                m_pProcess->addArgument("metadata");

            if ( which & DlgCleanCache::c_cleanCache )
                m_pProcess->addArgument("cache");

            if ( which & DlgCleanCache::c_cleanDBCache )
                m_pProcess->addArgument("dbcache");
        }

        m_currentAction = c_Clean;

        if ( !startAction() )
        {
            m_currentAction = c_NoAction;
            QMessageBox::warning(this, "KYum", i18n("Could not start yum."));
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotReadyStdout()
{
    while ( m_pProcess->canReadLineStdout() )
    {
        QString strLine = m_strStdOutBuf + m_pProcess->readLineStdout();

        m_strYumOutput += strLine;
        m_strStdOutBuf  = "";

        //
        // Are we going to display any list at all?
        //
        if ( 0 != (m_currentAction & c_fl_AllLists) || c_Search == m_currentAction )
        {
            if ( c_Search != m_currentAction && m_pCheckDescriptions->isChecked() )
            {
                addYumInfo(strLine);
            }
            else
                addYumLineInfo(strLine);
        }
        else
            emit yumInfo(strLine);
    }

    m_strStdOutBuf += QString(m_pProcess->readStdout());

/*
    if ( c_Remove == m_currentAction
            && !KYumPreferences::getAssumeYesOnRemove()
            && m_strStdOutBuf.startsWith("Is this ok [") )
    {
        enum { c_id_Yes, c_id_No };
        const char *  responses[] = { 0, 0, 0 };

        responses[c_id_Yes] = i18n("Yes");
        responses[c_id_No]  = i18n("No");

        DlgUserPrompt dlg(m_strYumOutput, responses, c_id_Yes, this);

        m_pProcess->writeToStdin(c_id_Yes == dlg.exec() ? "y\n" : "n\n");
    }
*/
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotReadyStderr()
{
    //QMutexLocker locker(m_pMutex);

#if 0
    while ( m_pProcess->canReadLineStderr() )
    {
        QString strLine("*** ");

        strLine += m_pProcess->readLineStderr();

        addYumError(strLine);
    }
#else

  #if (QT_VERSION < 0x040000)
    QString strErr = m_pProcess->readStderr();
  #else
    QString strErr = m_pProcess->readAllStandardError();
  #endif
    int     pos1   = 0,
            pos2   = 0;

    while ( pos1 < (int) strErr.length() && 0 <= (pos2 = (int) strErr.find('\n', pos1)) )
    {
        m_strStdErrBuf += strErr.mid(pos1, pos2 - pos1);
        addYumError(m_strStdErrBuf);

        m_strStdErrBuf = "*** ";
        pos1 = pos2 + 1;
    }

    if ( pos1 < (int) strErr.length() )
        m_strStdErrBuf += strErr.mid(pos1);
#endif
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotSelectAll()
{
    QListViewItemIterator it(m_pListView);

    m_totalPkgSize = 0.0;

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        pItem->setOn(true);

        PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

        if ( ptrPkgInfo.isValid() )
            ptrPkgInfo->m_bChecked = true;

        m_totalPkgSize += getPkgSize(pItem);
        ++it;
    }

    m_numCheckedItems = m_pListView->childCount();

    updateSelectionInfo();

    if ( m_bActionsEnabled )
        enableActions(true);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotDeselectAll()
{
    QListViewItemIterator it(m_pListView);

    while ( it.current() )
    {
        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

        PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

        if ( ptrPkgInfo.isValid() )
            ptrPkgInfo->m_bChecked = false;

        pItem->setOn(false);
        ++it;
    }

    m_numCheckedItems = 0;
    m_totalPkgSize    = 0.0;

    updateSelectionInfo();

    if ( m_bActionsEnabled )
        enableActions(true);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotLaunchFinished()
{
    //QMutexLocker locker(m_pMutex);

    m_strYumOutput = "";

    emit actionStarted(m_pProcess);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotProcessExited()
{
    if ( m_ptrCurPackage.isValid() )
    {
        addPackageToList(m_ptrCurPackage);
        m_ptrCurPackage = 0;
    }

    if ( c_FindAndInstall == m_currentAction && 0 < m_pListView->childCount() )
    {
        slotSelectAll();
        slotInstall();
    }
    else
    {
        //
        // If desired, remove the processed items from the list.
        //
        if ( (c_Install == m_currentAction || c_Remove == m_currentAction)
              && m_pProcess->normalExit()
              && 0 == m_pProcess->exitStatus()
              && KYumPreferences::getRemoveFromListAfterInstall() )
        {
            QListViewItemIterator it(m_pListView);

            while ( it.current() )
            {
                QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

                ++it;

                PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

                if ( ptrPkgInfo && ptrPkgInfo->m_bInstalling )
                {
                    if ( pItem->isOn() )
                    {
                        m_numCheckedItems--;
                        m_totalPkgSize -= getPkgSize(pItem);
                    }

                    delete pItem;

                    // We should actually delete the map entry, too, but we don't
                    // care because the entry won't hurt and will be deleted anyway
                    // when a new list is generated...
                }
            }

            updateSelectionInfo();
        }
        else
        {
            if ( c_Install == m_currentAction || c_Remove == m_currentAction )
            {
                QListViewItemIterator it(m_pListView);

                while ( it.current() )
                {
                    QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

                    ++it;

                    PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

                    if ( ptrPkgInfo && ptrPkgInfo->m_bInstalling )
                    {
                        ptrPkgInfo->m_bInstalling = false;
                    }
                }
            }

            updateSelectionInfo();
        }


        emit actionTerminated(m_pProcess);
        m_currentAction = c_NoAction;

        if ( 0 == m_pListView->childCount() )
            m_currentList = c_NoList;

        switch ( m_currentList )
        {
          case c_Updates:   m_pListTitle->setTitle("Packages Available for Update");
                            break;

          case c_Available: m_pListTitle->setTitle("Packages Available for Install");
                            break;

          case c_Installed: m_pListTitle->setTitle("Installed Packages");
                            break;

          case c_SearchResult:
                            m_pListTitle->setTitle("Result of Package Search");
                            break;

          default:          m_pListTitle->setTitle("");
                            break;
        }

        delete m_pProcess;
        m_pProcess = new Process(this);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::addYumError(QString strLine)
{
    emit yumError(strLine);
}


/***************************************************************************/
/**
 *  Creates a list view item for the specified package and stores the
 *  corresponding package information in the package map.
 *
 ***************************************************************************/

void ActionsView::addPackageToList(PackageInfo::Ptr ptrNewPackage)
{
    //
    // Do we already have this package in our map?
    //
    PackageInfo::Ptr ptrOldPackage = getPackageFromMap(ptrNewPackage->getMapKey());

    if ( ptrOldPackage.isValid() )
    {
        bool  bReplace  = ptrOldPackage->m_strArch == ptrNewPackage->m_strArch &&
                          ptrOldPackage->m_strVersion < ptrNewPackage->m_strVersion,
              bUpdate   = !bReplace && ptrNewPackage->m_strRepoTag == "installed";

        //
        // If there already is a list entry for this package, we will replace
        // it only if the new one refers to a later version. However, the new
        // package might be an installed one, so let's first check whether we
        // should update the "installed version" information for the package
        // in the list...
        //
        if ( bReplace || bUpdate )
        {
            PkgListItem * pItem = m_pListView->getItemWithKey(ptrOldPackage->m_strName);

            if ( pItem )
            {
                if ( bReplace )
                {
                    if ( ptrOldPackage->m_strRepoTag == "installed" )
                    {
                        ptrNewPackage->m_strInstalledVer = ptrOldPackage->m_strVersion;
                    }
                }
                else if ( bUpdate )
                {
                    ptrOldPackage->m_strInstalledVer = ptrNewPackage->m_strVersion;
                    ptrNewPackage = ptrOldPackage;
                }

                pItem->setPkgInfo(ptrNewPackage);
            }
        }
    }
    else
    {
        QCheckListItem * pNewItem = new PkgListItem(m_pListView, ptrNewPackage);

        if ( pNewItem )
            putPackageIntoMap(ptrNewPackage, pNewItem);
    }
}


/***************************************************************************/
/**
 *  Parses one output line returned by yum. This is meant for those of
 *  yum's commands that return multiple lines of information per package
 *  (i.e. the various versions of the 'info' command).
 *
 ***************************************************************************/

void ActionsView::addYumInfo(QString strLine)
{
    /*
      Name   : wxGTK-xrc
      Arch   : i386
      Version: 2.4.2
      Release: 7
      Size   : 465 k
      Repo   : installed
      Summary: The XML-based resource system for the wxWidgets library

      Description:
      ...
    */
    bool    bMaybeDescription = false;
    int     pos               = strLine.find(':');

    //
    // Try to determine the type of information...
    //
    if ( 0 < pos && pos < 12 )
    {
        QString strTag = strLine.left(pos).stripWhiteSpace().lower(),
                strValue;

        if ( ++pos < (int) strLine.length() )
            strValue = strLine.mid(pos).stripWhiteSpace();
        else
            strValue = "";

        if ( strTag == "name" )
        {
            //
            // Finish off current package...
            //
            if ( m_ptrCurPackage.isValid() )
                addPackageToList(m_ptrCurPackage);

            //
            // Start new package...
            //
            m_ptrCurPackage            = new PackageInfo;
            m_ptrCurPackage->m_strName = strValue;
            m_curInfoTag               = c_InfoTag_Name;
        }
        else if ( m_ptrCurPackage.isValid() && c_InfoTag_Description != m_curInfoTag )
        {
            if ( strTag == "arch" )
            {
                m_ptrCurPackage->m_strArch = strValue;
                m_curInfoTag               = c_InfoTag_Arch;
            }
            else if ( strTag == "version" )
            {
                m_ptrCurPackage->m_strVersion = strValue;
                m_curInfoTag                  = c_InfoTag_Version;
            }
            else if ( strTag == "release" )
            {
                m_ptrCurPackage->m_strVersion += '-';
                m_ptrCurPackage->m_strVersion += strValue;
                m_curInfoTag                   = c_InfoTag_Release;
            }
            else if ( strTag == "repo" )
            {
                m_ptrCurPackage->m_strRepoTag = strValue;
                m_curInfoTag                  = c_InfoTag_Repo;
            }
            else if ( strTag == "summary" )
            {
                m_ptrCurPackage->m_strSummary = strValue;
                m_curInfoTag                  = c_InfoTag_Summary;
            }
            else if ( strTag == "description" )
            {
                m_curInfoTag = c_InfoTag_Description;
                // Description starts in next line
            }
            else if ( strTag == "size" )
            {
                m_ptrCurPackage->m_strSize = strValue;
                m_curInfoTag               = c_InfoTag_Size;
            }
            else
                bMaybeDescription = true;
        }
        else
            bMaybeDescription = true;
    }
    else
        bMaybeDescription = true;

    //
    // If we could not find a known tag, we might have another line of
    // description...
    //
    if ( bMaybeDescription )
    {
        if ( c_InfoTag_Description == m_curInfoTag )
        {
            //
            // A new paragraph?
            //
            if ( strLine.isEmpty() )
            {
                if ( !m_ptrCurPackage->m_strDescription.isEmpty() )
                    m_ptrCurPackage->m_strDescription += "</p>";

                m_ptrCurPackage->m_strDescription += "<p>";
            }
            else
            {
                if ( m_ptrCurPackage->m_strDescription.isEmpty() )
                    m_ptrCurPackage->m_strDescription = "<p>";
                else
                    m_ptrCurPackage->m_strDescription += ' ';

                m_ptrCurPackage->m_strDescription += strLine;
            }
        }
        else if ( c_InfoTag_Summary == m_curInfoTag )
        {
            if ( !strLine.isEmpty() )
            {
                if ( !m_ptrCurPackage->m_strSummary.isEmpty() )
                    m_ptrCurPackage->m_strSummary += ' ';

                m_ptrCurPackage->m_strSummary += strLine;
            }
        }
    }

    if ( m_ptrCurPackage.isValid() )
    {
     //   emit yumPkgInfo(strLine);
    }
    else
        emit yumInfo(strLine);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::addYumLineInfo(QString strLine)
{
    //
    // Only add this line to the QListView if we are actually listing packages...
    //
    if ( c_Install != m_currentAction )
    {
        //
        // Format: <package.arch> <version> <repo>
        //
        PackageInfo::Ptr ptrNewPackage = new PackageInfo;

        if ( ptrNewPackage->getFromLine(strLine) )
        {
            PackageInfo::Ptr  ptrOldPackage;

            emit yumPkgInfo(strLine);

            //
            // Only mask out duplicate entries (i.e. different versions of the
            // same package), if we are not displaying the list of packages that
            // are currently installed on our system...
            //
            if ( c_ListInstalled != m_currentAction )
                 ptrOldPackage = getPackageFromMap(*ptrNewPackage.getPtr());

            //
            // Do we already have this package in our list? Then we will not
            // add a second item, but rather overwrite the existing one, if
            // the current one is newer.
            //
            if ( ptrOldPackage.isValid() )
            {
                bool bOverwrite = ptrOldPackage->m_strVersion < ptrNewPackage->m_strVersion;

                if ( !bOverwrite )
                {
                    //
                    // We want to show that we have the latest version installed - if we have.
                    //
                    if ( ptrOldPackage->m_strVersion == ptrNewPackage->m_strVersion )
                        bOverwrite = ptrNewPackage->m_strRepoTag == "installed";
                }


                if ( ptrOldPackage->m_strVersion < ptrNewPackage->m_strVersion )
                {
                    QListViewItemIterator it(m_pListView);

                    while ( it.current() )
                    {
                        QCheckListItem * pItem = dynamic_cast<QCheckListItem *>(it.current());

                        if ( pItem->text(c_idxPackage) == ptrOldPackage->m_strName &&
                             pItem->text(c_idxArch)    == ptrOldPackage->m_strArch )
                        {
                            pItem->setText(c_idxVersion,  ptrNewPackage->m_strVersion);
                            pItem->setText(c_idxRepoTag,  ptrNewPackage->m_strRepoTag);
                            pItem->setText(c_idxSummary,  ptrNewPackage->m_strSummary);

                            //
                            // This will overwrite the old map entry since both have the
                            // same key...
                            //
                            putPackageIntoMap(ptrNewPackage, pItem);
                            break;
                        }

                        ++it;
                    }

                }
            }
            //
            // The package was not yet in our list, or we want all packages to be
            // displayed...
            //
            else
                addPackageToList(ptrNewPackage);
        }
        else
            emit yumInfo(strLine);
    }
    else
        emit yumInfo(strLine);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::showChangeLog(PackageInfo::Ptr ptrPkgInfo)
{
    DlgChangeLog(this, ptrPkgInfo).exec();
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::showDependencies(PackageInfo::Ptr ptrPkgInfo)
{
#if 0
    DlgDependencies(this, ptrPkgInfo).exec();
#else
    DlgRun      dlg(this);
    QStringList args;

    args.push_back("yum");
    args.push_back("deplist");
    args.push_back(ptrPkgInfo->getNameAndVersion());

    dlg.setArgs(args);
    dlg.startProcess();
    dlg.exec();
#endif
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::stopAction()
{
    if ( c_NoAction != m_currentAction )
    {
        DlgStopAction dlg(this);
        QProcess::PID pid = m_pProcess->processIdentifier();

        if ( DlgStopAction::Accepted == dlg.exec() )
        {
            QMutexLocker locker(m_pMutex);

            //
            // The process may have terminated already
            //
            if ( m_pProcess && m_pProcess->processIdentifier() == pid )
            {
                if ( dlg.graceful() )
                {
                    emit yumError(i18n("Sending terminate signal to process..."));
                    m_pProcess->tryTerminate();
                }
                else
                {
                    emit yumError(i18n("Killing process..."));
                    m_pProcess->kill();
                }
            }
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool ActionsView::startAction()
{
    bool bSuccess = false;
    //QMutexLocker locker(m_pMutex);

    if ( !m_pProcess->isRunning() )
    {
        // This is now done in KYum::slotActionStarted()
        //enableActions(false);

        m_ptrCurPackage = 0;
        m_curInfoTag    = c_InfoTag_None;
        m_pCurrentItem  = 0;

        m_pListTitle->setTitle("");

        connect(m_pProcess, SIGNAL(launchFinished()),  this, SLOT(slotLaunchFinished()));
        connect(m_pProcess, SIGNAL(processExited()),   this, SLOT(slotProcessExited()));
        connect(m_pProcess, SIGNAL(readyReadStdout()), this, SLOT(slotReadyStdout()));
        connect(m_pProcess, SIGNAL(readyReadStderr()), this, SLOT(slotReadyStderr()));

        if ( m_pProcess->start() )
        {
            emit actionStarted(m_pProcess);
            bSuccess = true;
        }
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::findPrevInList(QListViewItemIterator & it)
{
//    QMutexLocker  locker(m_pMutex);
    QString       strText = m_pEditFindInList->text();

    for ( bool bFound=false; !bFound && it.current(); --it )
    {
        PackageInfo::Ptr ptrPackage = getPackageFromMap(it.current());

        if ( ptrPackage.isValid() )
        {
            if ( ptrPackage->m_strName.contains(strText) ||
                 ptrPackage->m_strSummary.contains(strText) ||
                 ptrPackage->m_strDescription.contains(strText) ||
                 ptrPackage->m_strArch.contains(strText) ||
                 ptrPackage->m_strVersion.contains(strText) ||
                 ptrPackage->m_strRepoTag.contains(strText) )
            {
                bFound = true;

                m_pListView->ensureItemVisible(it.current());
                m_pListView->setCurrentItem(it.current());
                slotListItemSelected(it.current());
            }
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::findNextInList(QListViewItemIterator & it)
{
//    QMutexLocker  locker(m_pMutex);
    QString       strText = m_pEditFindInList->text();

    for ( bool bFound=false; !bFound && it.current(); it++ )
    {
        PackageInfo::Ptr ptrPackage = getPackageFromMap(it.current());

        if ( ptrPackage.isValid() )
        {
            if ( ptrPackage->m_strName.contains(strText) ||
                 ptrPackage->m_strSummary.contains(strText) ||
                 ptrPackage->m_strDescription.contains(strText) ||
                 ptrPackage->m_strArch.contains(strText) ||
                 ptrPackage->m_strVersion.contains(strText) ||
                 ptrPackage->m_strRepoTag.contains(strText) )
            {
                bFound = true;

                m_pListView->ensureItemVisible(it.current());
                m_pListView->setCurrentItem(it.current());
                slotListItemSelected(it.current());
            }
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotFindPrevInList()
{
    if ( m_pCurrentItem )
    {
        QListViewItemIterator itPrev(m_pCurrentItem);
        findPrevInList(--itPrev);
    }
    else
    {
        QListViewItemIterator itLast(m_pListView);
        findNextInList(itLast);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotFindNextInList()
{
    if ( m_pCurrentItem )
    {
        QListViewItemIterator itNext(m_pCurrentItem);

        itNext++;
        findNextInList(itNext);
    }
    else
    {
        QListViewItemIterator itFirst(m_pListView);
        findNextInList(itFirst);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotFindInList(const QString & strText)
{
    if ( strText.isEmpty() )
    {
        m_pButtonFindPrevInList->setEnabled(false);
        m_pButtonFindNextInList->setEnabled(false);
    }
    else
    {
        m_pButtonFindPrevInList->setEnabled(true);
        m_pButtonFindNextInList->setEnabled(true);

        QListViewItemIterator itFirst(m_pListView);
        findNextInList(itFirst);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotListItemSelected(QListViewItem * pItem)
{
    if ( pItem )
    {
        PackageInfo::Ptr ptrInfo = getPackageFromMap(pItem);

        //
        // Create an empty package info to erase the current info view content,
        // if we could not find a valid package info.
        //
        if ( !ptrInfo.isValid() )
            ptrInfo = new PackageInfo;

        if ( m_pCurrentItem != pItem )
        {
            m_pCurrentItem = pItem;

            emit showPackageInfo(*ptrInfo.getPtr());
        }

        QCheckListItem * pCheckItem = dynamic_cast<QCheckListItem *>(pItem);

        if ( pCheckItem ) // this should NEVER fail!
        {
#if 0
            //
            // Don't allow selecting/deselecting items while installing, updating
            // or removing installed packages...
            //

            if ( m_currentAction & (c_fl_Install | c_fl_Remove) )
            {
                pCheckItem->setOn(ptrInfo->m_bChecked);
            }
            //
            // Else update number and size of selected packages accordingly...
            //
            else
#endif
            if ( pCheckItem->isOn() )
            {
                if ( !ptrInfo->m_bChecked )
                {
                    m_numCheckedItems++;
                    m_totalPkgSize += ptrInfo->getPkgSize();

                    ptrInfo->m_bChecked = true;
                }
            }
            else
            {
                if ( ptrInfo->m_bChecked )
                {
                    m_numCheckedItems--;
                    m_totalPkgSize -= ptrInfo->getPkgSize();

                    ptrInfo->m_bChecked = false;
                }
            }

            updateSelectionInfo();
        }

        //
        // Enable/disable buttons appropriately
        //
        if ( m_bActionsEnabled )
            enableActions(true);
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void ActionsView::slotListContextMenu(QListViewItem * pItem,
                                      const QPoint &  pos,
                                      int             /*column*/)
{
    if ( c_NoAction == m_currentAction )
    {
        QPopupMenu * pPopup = new QPopupMenu(this);

        enum
          {
            c_id_ChangeLog,
            c_id_Deps,
            c_id_InstalledVer,
            c_id_InstalledVerExt,
            c_id_RemoveFromList
          };


        //
        // The following menu item are only valid if the mouse is over a
        // repository item.
        //
        if ( pItem )
        {
            PackageInfo::Ptr ptrPkgInfo = getPackageFromMap(pItem);

            pPopup->insertItem(i18n("View &ChangeLog"), c_id_ChangeLog);
            pPopup->insertItem(i18n("Show &Dependencies"), c_id_Deps);
            pPopup->insertItem(i18n("&Remove From List"), c_id_RemoveFromList);

            pPopup->setItemEnabled(c_id_ChangeLog, false);

            if ( !(m_currentList & c_fl_Installed) )
            {
                pPopup->insertItem(i18n("&Query Installed Version (short)"),
                                   c_id_InstalledVer);

                pPopup->insertItem(i18n("&Query Installed Version (extended)"),
                                   c_id_InstalledVerExt);
            }


            switch ( pPopup->exec(pos) )
            {
              case c_id_ChangeLog:
                                showChangeLog(ptrPkgInfo);
                                break;

              case c_id_Deps:   showDependencies(ptrPkgInfo);
                                break;

              case c_id_InstalledVer:
                                {
                                    DlgRun      dlg(this);
                                    QStringList args;

                                    args.push_back("rpm");
                                    args.push_back("-q");
                                    args.push_back(ptrPkgInfo->m_strName);

                                    dlg.setArgs(args);
                                    dlg.startProcess();
                                    dlg.exec();
                                }
                                break;

              case c_id_InstalledVerExt:
                                {
                                    DlgRun      dlg(this);
                                    QStringList args;

                                    args.push_back("rpm");
                                    args.push_back("-qi");
                                    args.push_back(ptrPkgInfo->m_strName);

                                    dlg.setArgs(args);
                                    dlg.startProcess();
                                    dlg.exec();
                                }
                                break;

              case c_id_RemoveFromList:
                                {
                                    QCheckListItem * pCheckItem = dynamic_cast<QCheckListItem *>(pItem);

                                    if ( ptrPkgInfo->m_bChecked )
                                    {
                                        m_numCheckedItems--;
                                        m_totalPkgSize -= ptrPkgInfo->getPkgSize();
                                    }

                                    delete pCheckItem;
                                    updateSelectionInfo();
                                }
                                break;
            }
        }
    }
}


#include "ActionsView.moc"
