/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <qapp.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qprocess.h>
#include <qsplitter.h>

#include <klocale.h>
#include <klistview.h>
#include <ktextedit.h>

#include "StrTok.h"
#include "Process.h"
#include "KYumPreferences.h"
#include "UserEvents.h"
#include "DlgDependencies.h"


class DepListView : public KListView
  {
    public:
                    DepListView(QWidget * pParent, const char * pName = "DepListView")
                      : KListView(pParent, pName)
                    {
                    }

      virtual     ~ DepListView()
                    {
                    }
  };


class DepListItem : public QListViewItem
  {
    private:
      PackageInfo::List m_providers;

    public:
                    DepListItem(DepListView * pListView, QString strDep);

      void          addProvider(QString strPkg, QString strArch, QString strVersion);
  };


enum
  {
    c_idxPackage,
    c_idxProviders
  };

enum
  {
    c_idxDependency,
    c_idxProvider
  };

static struct
  {
    const char  * m_pTitle;
    int           m_cx;
  }
  g_pkgListColumns[] =
    {
      { /*i18n*/("Package"),         -1 },
      { /*i18n*/("Known Providers"), -1 }
    },
  g_depListColumns[] =
    {
      { /*i18n*/("Dependency"),      -1 },
      { /*i18n*/("Provider"),        -1 }
    };

#define ArraySize(a)  (sizeof(a) / sizeof((a)[0]))

/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DepListItem::DepListItem(DepListView * pListView, QString strDep)

  : QListViewItem(pListView, strDep)

{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DepListItem::addProvider(QString strPkg, QString strArch, QString strVersion)
{
    m_providers.push_back(new PackageInfo(strPkg, strArch, strVersion));

    setText(c_idxProviders, QString::number(m_providers.size(), 10));
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgDependencies::DlgDependencies(QWidget * pParent, PackageInfo::Ptr ptrPkgInfo)

  : ProcessDlg(pParent, "DlgDependencies"),
    m_ptrPkgInfo(ptrPkgInfo),
    m_pCurDepItem(0)

{
    QVBoxLayout * pTopLayout = new QVBoxLayout(this, 5, 5);
    pTopLayout->setAutoAdd(true);

    resize(600, 800);

    //
    // Vertical splitter between list view and output view
    //
    QSplitter * pVSplitter = new QSplitter(this);
    pVSplitter->setOrientation(QSplitter::Vertical);


    QWidget   * pListWidget = new QWidget(pVSplitter, "ListWidget");
    QWidget   * pBottomView = new QWidget(pVSplitter, "BottomView");

    QSplitter * pHSplitter  = new QSplitter(pListWidget);

    m_pPkgListView = new DepListView(pHSplitter, "PkgListView");
    m_pPkgListView->setAllColumnsShowFocus(true);
    m_pPkgListView->setSelectionMode(QListView::Single);

    for ( int idx=0; idx < (int) ArraySize(g_pkgListColumns); idx++ )
    {
        m_pDepListView->addColumn(g_pkgListColumns[idx].m_pTitle,
                                  g_pkgListColumns[idx].m_cx);
    }

    m_pDepListView = new DepListView(pHSplitter, "DepListView");
    m_pDepListView->setAllColumnsShowFocus(true);
    m_pDepListView->setSelectionMode(QListView::Single);

    for ( int idx=0; idx < (int) ArraySize(g_depListColumns); idx++ )
    {
        m_pDepListView->addColumn(g_depListColumns[idx].m_pTitle,
                                  g_depListColumns[idx].m_cx);
    }


    QVBoxLayout * pBottomViewLayout = new QVBoxLayout(pBottomView, 0, 5);
    pBottomViewLayout->setAutoAdd(true);

    m_pEdOutput = new KTextEdit(pBottomView, "OutputView");

    QWidget     * pButtonWidget   = new QWidget(pBottomView, "ButtonWidget");
    QHBoxLayout * pButtonsLayout  = new QHBoxLayout(pButtonWidget, 0, 6, "ButtonsLayout");

/*
    m_pButtonHelp = new QPushButton(pButtonWidget, "m_pButtonHelp");
    m_pButtonHelp->setAutoDefault(TRUE);
    m_pButtonHelp->setText(i18n("Help"));
    m_pButtonsLayout->addWidget(m_pButtonHelp);
*/
    QSpacerItem * pSpacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
    pButtonsLayout->addItem(pSpacer);

/*
    m_pButtonOk = new QPushButton(pButtonWidget, "m_pButtonOk");
    m_pButtonOk->setAutoDefault(TRUE);
    m_pButtonOk->setText(i18n("Ok"));
    pButtonsLayout->addWidget(m_pButtonOk);
*/

    m_pButtonCancel = new QPushButton(pButtonWidget, "m_pButtonCancel");
    m_pButtonCancel->setAutoDefault(TRUE);
    m_pButtonCancel->setText(i18n("E&xit"));
    pButtonsLayout->addWidget(m_pButtonCancel);

    connect(m_pButtonCancel, SIGNAL(clicked()), this, SLOT(reject()));

    QStringList args;

    args.push_back("yum");
    args.push_back("deplist");
    args.push_back(ptrPkgInfo->getNameAndVersion());

    setArgs(args);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

DlgDependencies::~DlgDependencies()
{
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::slotProcessStarted()
{
    m_pEdOutput->setColor(KYumPreferences::c_clrInfo);

    QString strMsg;

    strMsg.sprintf(i18n("Process %d (%s) has been started.\n"
                        "Please be patient, this may be a lengthy operation.\n\n"),
                        m_pProcess->getCmdName().ascii(),
                        m_pProcess->processIdentifier());

    m_pEdOutput->append(strMsg);

    m_pDepListView->setUpdatesEnabled(FALSE);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::slotProcessExited()
{
    QString strMsg;

    m_pDepListView->setUpdatesEnabled(TRUE);

    if ( m_pProcess->normalExit() )
    {
        m_pEdOutput->setColor(KYumPreferences::c_clrInfo);

        strMsg.sprintf(i18n("Process %d exited normally with code %d"),
                       m_pProcess->processIdentifier(),
                       m_pProcess->exitStatus());
    }
    else
    {
        m_pEdOutput->setColor(KYumPreferences::c_clrError);

        strMsg.sprintf(i18n("Process %d exited abnormally with code %d"),
                       m_pProcess->processIdentifier(),
                       m_pProcess->exitStatus());
    }

    m_pEdOutput->append(strMsg);
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::processStdout(QString strLine)
{
    m_pEdOutput->setColor(KYumPreferences::c_clrInfo);
    m_pEdOutput->append(strLine);

    if ( !strLine.isEmpty() && ' ' == strLine[0] )
    {
        StrTok  strTok(strLine);
        QString strKey;

        strTok.nextToken(':', strKey);

        if ( strKey == "dependency" )
        {
            strTok.nextToken('\n', strKey);
            m_pCurDepItem = new DepListItem(dynamic_cast<DepListView *>(m_pDepListView),
                                            strKey);
        }
        else if ( m_pCurDepItem && strKey == "provider" )
        {
            QString strPkg,
                    strArch,
                    strVersion;

            strTok.nextToken('.', strPkg);
            strTok.nextToken(' ', strArch);
            strTok.nextToken('\n', strVersion);

            dynamic_cast<DepListItem *>(m_pCurDepItem)->addProvider(strPkg,
                                                                    strArch,
                                                                    strVersion);
        }
    }
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::slotReadyStdout()
{
    while ( m_pProcess->canReadLineStdout() )
        processStdout(m_pProcess->readLineStdout());
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::slotReadyStderr()
{
    m_pEdOutput->setColor(KYumPreferences::c_clrError);

#if 1
    m_pEdOutput->append(m_pProcess->readStderr());
#else
    while ( m_pProcess->canReadLineStderr() )
        m_pEdOutput->append(m_pProcess->readLineStderr());
#endif
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

void DlgDependencies::threadMain()
{
    m_pButtonCancel->setText(i18n("&Cancel"));
    m_pEdOutput->setText("");

    ProcessDlg::threadMain();
}

#include "DlgDependencies.moc"
