/***************************************************************************
 *
 *   KYum - a KDE GUI for yum
 *
 *   Copyright (C) 2005 by Steffen Offermann
 *   steffen_ac@yahoo.com
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <stdlib.h>
#include <unistd.h>

#include <qfile.h>
#include <qprocess.h>

#include <kstandarddirs.h>

#include "StrTok.h"
#include "KYumConf.h"
#include "KYumView.h"
#include "System.h"


class SystemInfo : public QObject
  {
    private:
      bool      m_bValid;
      int       m_releaseVer;
      QString   m_strBaseArch,
                m_strArch,
                m_strError;

    public:
                SystemInfo();

      bool      isValid       () const { return m_bValid; }
      int       getReleaseVer () const { return m_releaseVer; }
      QString   getBaseArch   () const { return m_strBaseArch; }
      QString   getArch       () const { return m_strArch; }
      QString   getErrorMsg   () const { return m_strError; }
  };

static QString  g_strBaseArch,
                g_strArch;
static int      g_releaseVer;
static bool     g_bInitialized = false;


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

SystemInfo::SystemInfo()
{
    QProcess  process(this);
    QString   strScript = KStandardDirs::kde_default("data");

    if ( !strScript.endsWith("/") )
        strScript += '/';

    strScript += "kyum/kyum_sysinfo.py";

    if ( !strScript.startsWith("/") )
    {
        if ( !strScript.startsWith("usr") )
            strScript.prepend("/usr/");
        else
            strScript.prepend("/");
    }

    process.addArgument("python");
    process.addArgument(strScript);
    process.addArgument(KYumView::getKYumView()->getYumConf().getDistroVerPkg());

    m_bValid = false;

    if ( process.launch("") )
    {
        QString strStdOut;

        while ( process.isRunning() )
        {
            strStdOut  += QString(process.readStdout());
            m_strError += QString(process.readStderr());

            ::usleep(1000);
        }

        if ( !strStdOut.isEmpty() )
        {
            StrTok strTok(strStdOut);

            m_bValid = strTok.nextToken(';', m_strBaseArch) &&
                       strTok.nextToken(';', m_strArch);

            if ( m_bValid )
            {
                QString strVer;

                strTok.nextToken(';', strVer);
                m_bValid = 0 != (m_releaseVer = strVer.stripWhiteSpace().toInt());

                m_strBaseArch = m_strBaseArch.stripWhiteSpace();
                m_strArch     = m_strArch.stripWhiteSpace();
            }
        }
    }

}

/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

bool System::getSystemInfo()
{
    SystemInfo  info;
    bool        bSuccess = info.isValid();

    if ( bSuccess )
    {
        g_strBaseArch   = info.getBaseArch();
        g_strArch       = info.getArch();
        g_releaseVer    = info.getReleaseVer();
        g_bInitialized  = true;
    }

    return bSuccess;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString System::getBaseArch()
{
    if ( !g_bInitialized )
        getSystemInfo();

    return g_strBaseArch;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString System::getArch()
{
    if ( !g_bInitialized )
        getSystemInfo();

    return g_strArch;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

int System::getReleaseVer()
{
    if ( !g_bInitialized )
        getSystemInfo();

    return g_releaseVer;
}


/***************************************************************************/
/**
 *
 *
 ***************************************************************************/

QString System::expandString(QString str)
{
    if ( !g_bInitialized )
        getSystemInfo();

    if ( g_bInitialized )
    {
        str.replace("$releasever", QString::number(getReleaseVer(), 10));
        str.replace("$arch",       getArch());
        str.replace("$basearch",   getBaseArch());
    }

    return str;
}
