
/*
 * LaTeX Symbols Selector
 *
 * Copyright (C) 2006 Tomasz Mąka <pasp@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "gui.h"
#include "send.h"
#include "prefs.h"
#include "config.h"
#include "i18n.h"

/* include data */
#include "graphics.h"
#include "symbols-arrows.h"
#include "symbols-arrows-table.h"
#include "symbols-greek_letters.h"
#include "symbols-greek_letters-table.h"
#include "symbols-operators.h"
#include "symbols-operators-table.h"
#include "symbols-relations.h"
#include "symbols-relations-table.h"
#include "symbols-delimiters.h"
#include "symbols-delimiters-table.h"
#include "symbols-accents.h"
#include "symbols-accents-table.h"
#include "symbols-misc_math.h"
#include "symbols-misc_math-table.h"
#include "symbols-text_mode.h"
#include "symbols-text_mode-table.h"

#include "ams_symbols-accents.h"
#include "ams_symbols-accents-table.h"
#include "ams_symbols-arrows.h"
#include "ams_symbols-arrows-table.h"
#include "ams_symbols-misc_math.h"
#include "ams_symbols-misc_math-table.h"
#include "ams_symbols-delimiters.h"
#include "ams_symbols-delimiters-table.h"
#include "ams_symbols-relations.h"
#include "ams_symbols-relations-table.h"
#include "ams_symbols-operators.h"
#include "ams_symbols-operators-table.h"


#define NUMBER_OF_SETS  8

gint symbolset_len[NUMBER_OF_SETS] = {
        LSYMBOLS_accents,
        LSYMBOLS_arrows,
        LSYMBOLS_delimiters,
        LSYMBOLS_greek_letters,
        LSYMBOLS_misc_math,
        LSYMBOLS_operators,
        LSYMBOLS_relations,
        LSYMBOLS_text_mode
};

const guint8 **symbolset_icon[NUMBER_OF_SETS] = {
        symbols_accents_data,
        symbols_arrows_data,
        symbols_delimiters_data,
        symbols_greek_letters_data,
        symbols_misc_math_data,
        symbols_operators_data,
        symbols_relations_data,
        symbols_text_mode_data
};

const gchar **symbolset_name[NUMBER_OF_SETS] = {
        symbols_accents_name,
        symbols_arrows_name,
        symbols_delimiters_name,
        symbols_greek_letters_name,
        symbols_misc_math_name,
        symbols_operators_name,
        symbols_relations_name,
        symbols_text_mode_name
};

const gchar *group_name[NUMBER_OF_SETS] = {
        N_("Accents"),
        N_("Arrows"),
        N_("Delimiters"),
        N_("Greek Letters"),
        N_("Miscellaneous"),
        N_("Operators"),
        N_("Relations"),
        N_("Text Mode")
};

#define AMS_NUMBER_OF_SETS  6

gint ams_symbolset_len[AMS_NUMBER_OF_SETS] = {
        AMS_LSYMBOLS_accents,
        AMS_LSYMBOLS_arrows,
        AMS_LSYMBOLS_delimiters,
        AMS_LSYMBOLS_misc_math,
        AMS_LSYMBOLS_operators,
        AMS_LSYMBOLS_relations
};

const guint8 **ams_symbolset_icon[AMS_NUMBER_OF_SETS] = {
        ams_symbols_accents_data,
        ams_symbols_arrows_data,
        ams_symbols_delimiters_data,
        ams_symbols_misc_math_data,
        ams_symbols_operators_data,
        ams_symbols_relations_data
};

const gchar **ams_symbolset_name[AMS_NUMBER_OF_SETS] = {
        ams_symbols_accents_name,
        ams_symbols_arrows_name,
        ams_symbols_delimiters_name,
        ams_symbols_misc_math_name,
        ams_symbols_operators_name,
        ams_symbols_relations_name
};

const gchar *ams_group_name[AMS_NUMBER_OF_SETS] = {
        N_("AMS Accents"),
        N_("AMS Arrows"),
        N_("AMS Delimiters"),
        N_("AMS Miscellaneous"),
        N_("AMS Operators"),
        N_("AMS Relations")
};


GtkWidget           *main_window;
GtkWidget           *icon_view;
GtkWidget           *popup_menu;
GtkWidget           *statusbar;
GtkListStore        *store;
GtkWidget           *list;
GtkListStore        *list_store = NULL;
GtkUIManager        *uim_widget = NULL;
GtkTreeSelection    *list_selection;

gint                context_id;

/*------------------------------------------------------------------------------*/

void
show_status (gchar *symbol_name, gint status) {

    gchar message[BUFFER_SIZE], tmp[BUFFER_SIZE];


    if (status & SELECTED) {
        sprintf(message, _("[ %s ] symbol selected"), symbol_name);
    }

    if ((status & SENT_TO_GVIM) && (status & SENT_TO_CLIPBOARD)) {

        sprintf(tmp, _(" and sent to gVIM and clipboard"));
        strcat(message, tmp);

    } else if (status & SENT_TO_GVIM) {

        sprintf(tmp, _(" and sent to gVIM"));
        strcat(message, tmp);

    } else if (status & SENT_TO_CLIPBOARD) {

        sprintf(tmp, _(" and sent to clipboard"));
        strcat(message, tmp);
    }

    gtk_statusbar_pop (GTK_STATUSBAR(statusbar), context_id);
    gtk_statusbar_push (GTK_STATUSBAR(statusbar), context_id, message);

}

/*------------------------------------------------------------------------------*/

void
group_selected_cb (GtkTreeSelection *selection, gpointer data) {

    GtkTreeIter iter;
    GtkTreeModel *model;
    gint i;
    gchar message[BUFFER_SIZE];

    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

        gtk_tree_model_get (model, &iter, 1, &i, -1);

        config.current_group = i;

        if (i >= NUMBER_OF_SETS) {
            sprintf(message, _("[ %s ] group selected (%d symbols)"), ams_group_name[i-NUMBER_OF_SETS], ams_symbolset_len[i-NUMBER_OF_SETS]);
        } else {
            sprintf(message, _("[ %s ] group selected (%d symbols)"), group_name[i], symbolset_len[i]);
        }

        gtk_statusbar_pop (GTK_STATUSBAR(statusbar), context_id);
        gtk_statusbar_push (GTK_STATUSBAR(statusbar), context_id, message);

        display_symbols_list(config.current_group);

    }

}

/*------------------------------------------------------------------------------*/

void
show_statusbar_option_desc(gchar *message, gint state) {

    gchar buffer[BUFFER_SIZE];

    if (state) {
        sprintf(buffer, _("'%s' option activated"), message);
    } else {
        sprintf(buffer, _("'%s' option deactivated"), message);
    }

    gtk_statusbar_pop (GTK_STATUSBAR(statusbar), context_id);
    gtk_statusbar_push (GTK_STATUSBAR(statusbar), context_id, buffer);
}

void
options_eams_cb (GtkAction *action) {

    config.enable_ams_symbols =  gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));

    if (!config.enable_ams_symbols) {

        if (config.current_group >= NUMBER_OF_SETS) {

            config.current_group = NUMBER_OF_SETS - 1;
        }

    }

    display_names_list ();
    display_symbols_list(config.current_group);
    set_cursor_on_group(config.current_group);

    show_statusbar_option_desc(_("Enable AMS symbols"), config.enable_ams_symbols);
}

void
options_dsn_cb (GtkAction *action) {

    config.display_symbol_name =  gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));

    if(config.display_symbol_name) {

        gtk_icon_view_set_text_column (GTK_ICON_VIEW (icon_view), COL_NAME);
        gtk_icon_view_set_item_width (GTK_ICON_VIEW (icon_view), 120);

    } else {

        gtk_icon_view_set_text_column (GTK_ICON_VIEW (icon_view), -1);
        gtk_icon_view_set_item_width (GTK_ICON_VIEW (icon_view), -1);
    }

    show_statusbar_option_desc(_("Display symbol name"), config.display_symbol_name);
}

void
options_sc_cb (GtkAction *action) {

    config.single_click =  gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));

    show_statusbar_option_desc(_("Activate action on single click"), config.single_click);
}

void
options_stg_cb (GtkAction *action) {

    config.send_to_gvim =  gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));

    show_statusbar_option_desc(_("Send symbol to gVIM"), config.send_to_gvim);
}

void
options_stc_cb (GtkAction *action) {

    config.send_to_clipboard =  gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action));

    show_statusbar_option_desc(_("Send symbol to clipboard"), config.send_to_clipboard);
}

/*------------------------------------------------------------------------------*/

void
display_about_cb (void) {

GdkPixbuf *logo;

const gchar *authors[] = {
    "\n"
    "Tomasz Mąka <pasp@users.sourceforge.net>",
    NULL
};
const gchar *artists[] = {
    "\n"
    "Maja Kocoń <ironya@ll.pl>\n\thttp://ironya.ll.pl",
    NULL
};

const gchar license[] = {
  "\nThis program is free software; you can redistribute it and/or modify "
  "it under the terms of the GNU General Public License as published by "
  "the Free Software Foundation; either version 2 of the License, or "
  "(at your option) any later version."
  "\n\n"
  "This program is distributed in the hope that it will be useful, "
  "but WITHOUT ANY WARRANTY; without even the implied warranty of "
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
  "GNU General Public License for more details."
  "\n\n"
  "You should have received a copy of the GNU General Public License "
  "along with this program; if not, write to the Free Software "
  "Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA."
};

    logo = gdk_pixbuf_new_from_inline(-1, lss_logo, FALSE, NULL);

    gtk_show_about_dialog(GTK_WINDOW(main_window), 
                          "name", "LaTeX Symbols Selector",
                          "version", VERSION,
                          "website", "http://clay.ll.pl/lss",
                          "copyright", "Copyright © 2006 Tomasz Mąka",
                          "authors", authors,
                          "artists", artists,
                          "license", license,
                          "wrap-license", TRUE,
                          "logo", logo,
                          NULL);

    g_object_unref(logo);

}

/*------------------------------------------------------------------------------*/

void
gui_window_closed_cb ( GtkWidget *widget, gpointer data ) {

    gtk_window_get_size (GTK_WINDOW(main_window),
                        &config.window_size_x, &config.window_size_y);

    gdk_window_get_root_origin (main_window->window,
                        &config.window_x, &config.window_y);

    gtk_main_quit ();
}

/*------------------------------------------------------------------------------*/

void
display_names_list(void) {

    gint i;
    GtkTreeIter iter;

    /* first clear the store */
    gtk_list_store_clear (list_store);

    for(i = 0; i < NUMBER_OF_SETS; i++) {

        gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(list_store), &iter, NULL, i);
        gtk_list_store_append(list_store, &iter);
        gtk_list_store_set(list_store, &iter, 0, gettext(group_name[i]), 1, i, -1);

    }

    if (config.enable_ams_symbols) {

        for(i = 0; i < AMS_NUMBER_OF_SETS; i++) {

            gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(list_store), &iter, NULL, NUMBER_OF_SETS+i);
            gtk_list_store_append(list_store, &iter);
            gtk_list_store_set(list_store, &iter, 0, gettext(ams_group_name[i]), 1, NUMBER_OF_SETS+i, -1);

        }

    }
}

void
display_symbols_list(gint setnumber) {

    gint i;
    GtkTreeIter iter;
    GdkPixbuf *icon;

    /* clear the store first */
    gtk_list_store_clear (store);

    /* add items */

    if (setnumber < NUMBER_OF_SETS) {

        for(i = 0; i < symbolset_len[setnumber]; i++) {

            gtk_list_store_append (store, &iter);

            icon = gdk_pixbuf_new_from_inline(-1, symbolset_icon[setnumber][i], FALSE, NULL);
            gtk_list_store_set (store, &iter, COL_PIXBUF, icon, COL_NAME, symbolset_name[setnumber][i], -1);

            g_object_unref(icon);

        }

    } else {

        setnumber -= NUMBER_OF_SETS;

        for(i = 0; i < ams_symbolset_len[setnumber]; i++) {

            gtk_list_store_append (store, &iter);

            icon = gdk_pixbuf_new_from_inline(-1, ams_symbolset_icon[setnumber][i], FALSE, NULL);
            gtk_list_store_set (store, &iter, COL_PIXBUF, icon, COL_NAME, ams_symbolset_name[setnumber][i], -1);

            g_object_unref(icon);

        }

    }
}

/*------------------------------------------------------------------------------*/
/* double click callback */

static void
icon_activated_dc_cb   (GtkIconView *icon_view,
                        GtkTreePath *tree_path,
                        gpointer     user_data)
{
GtkListStore *store;
gchar *icon_name;
GtkTreeIter iter;
gint status;

    if (!config.single_click) {

        store = GTK_LIST_STORE (user_data);

        gtk_tree_model_get_iter (GTK_TREE_MODEL (store), &iter, tree_path);
        gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                          COL_NAME, &icon_name,
                          -1);

        status = SELECTED;

        if (config.send_to_gvim) {
            if (send_symbol_to_gvim(icon_name) == TRUE)
                status |= SENT_TO_GVIM;
        }
        if (config.send_to_clipboard) {
            send_symbol_to_clipboard(icon_name);
            status |= SENT_TO_CLIPBOARD;
        }

        show_status (icon_name, status);

        g_free (icon_name);
    }

}


/* single click callback */
void
icon_activated_sc_cb   (GtkIconView *iconview,
                        gpointer user_data)
{
GtkTreePath *tree_path;
GtkTreeIter iter;
gchar *icon_name;
GList *items_list, *current;
gint status;

    if (config.single_click) {

        store = GTK_LIST_STORE (user_data);
        items_list = gtk_icon_view_get_selected_items (iconview);

        current = g_list_nth (items_list, 0);

        if(current != NULL) {

            tree_path = current->data;

            gtk_tree_model_get_iter (GTK_TREE_MODEL (store), &iter, tree_path);
            gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                COL_NAME, &icon_name,
                                -1);
            status = SELECTED;

            if (config.send_to_gvim) {
                if (send_symbol_to_gvim(icon_name) == TRUE)
                    status |= SENT_TO_GVIM;
            }
            if (config.send_to_clipboard) {
                send_symbol_to_clipboard(icon_name);
                status |= SENT_TO_CLIPBOARD;
            }

            show_status (icon_name, status);

            g_free (icon_name);

        }

        g_list_foreach (items_list, (GFunc) gtk_tree_path_free, NULL);
        g_list_free (items_list);

    }
}

/*------------------------------------------------------------------------------*/

gboolean
mouse_button_press_cb( GtkWidget *widget,
                       GdkEventButton *event,
                       gpointer user_data)
{

    if (event->type == GDK_BUTTON_PRESS && event->button == 3) {

        GdkEventButton *bevent = (GdkEventButton *) event; 

        gtk_menu_popup (GTK_MENU (popup_menu), NULL, NULL, NULL, NULL,
                        bevent->button, bevent->time);
        return TRUE;
    }

    return FALSE;
}

/*------------------------------------------------------------------------------*/
/* set cursor on last selected group */

void
set_cursor_on_group(gint group) {

    gint i;
    GtkTreeIter iter;
    GtkTreePath *visible_path;

    if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &iter)) {

        do {
            gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter, 1, &i, -1);

            if (i == group) {

                visible_path = gtk_tree_model_get_path (GTK_TREE_MODEL(list_store), &iter);
                gtk_tree_view_set_cursor (GTK_TREE_VIEW (list), visible_path, NULL, TRUE);
                gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (list), visible_path, NULL, TRUE, 0.5, 0.0);
                gtk_tree_path_free(visible_path);
                break;
            }

        } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &iter));

    }
}

gint key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data)
{

    switch (event->keyval) {

        case GDK_Escape:
            gui_window_closed_cb(NULL, NULL);
            return TRUE;

        case GDK_F1:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F1 */
                set_cursor_on_group(GROUP_AMS_ACCENTS);
            } else {
                set_cursor_on_group(GROUP_ACCENTS);
            }
            return TRUE;
        case GDK_F2:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F2 */
                set_cursor_on_group(GROUP_AMS_ARROWS);
            } else {
                set_cursor_on_group(GROUP_ARROWS);
            }
            return TRUE;
        case GDK_F3:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F3 */
                set_cursor_on_group(GROUP_AMS_DELIMITERS);
            } else {
                set_cursor_on_group(GROUP_DELIMITERS);
            }
            return TRUE;
        case GDK_F4:
            set_cursor_on_group(GROUP_GREEK_LETTERS);
            return TRUE;
        case GDK_F5:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F5 */
                set_cursor_on_group(GROUP_AMS_MISCELLANEOUS);
            } else {
                set_cursor_on_group(GROUP_MISCELLANEOUS);
            }
            return TRUE;
        case GDK_F6:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F6 */
                set_cursor_on_group(GROUP_AMS_OPERATORS);
            } else {
                set_cursor_on_group(GROUP_OPERATORS);
            }
            return TRUE;
        case GDK_F7:
            if (event->state & GDK_SHIFT_MASK) {  /* SHIFT + F7 */
                set_cursor_on_group(GROUP_AMS_RELATIONS);
            } else {
                set_cursor_on_group(GROUP_RELATIONS);
            }
            return TRUE;
        case GDK_F8:
            set_cursor_on_group(GROUP_TEXT_MODE);
            return TRUE;
    }

    return FALSE;
}

/*------------------------------------------------------------------------------*/

void gui_create_window(void) {

GtkWidget           *hbox;
GtkWidget           *vbox1;
GtkWidget           *vbox2;
GtkWidget           *scrolledwindow_icons;
GtkWidget           *scrolledwindow_list;
GtkCellRenderer     *renderer;
GtkTreeViewColumn   *column;
GtkCellRenderer     *renderer_n;
GtkTreeViewColumn   *column_n;
GtkActionGroup      *action_group = NULL;
GdkPixbuf           *icon;
GtkAction           *a_action;


GtkActionEntry entries[] = {
  { "about",    GTK_STOCK_ABOUT,    NULL, NULL, NULL, G_CALLBACK (display_about_cb) },
  { "quit",     GTK_STOCK_QUIT,     NULL, NULL, NULL, G_CALLBACK (gui_window_closed_cb) }
};

guint n_entries = G_N_ELEMENTS (entries);

GtkToggleActionEntry toggle_entries[] = {
  { "send_to_gvim",        NULL, _("Send symbol to GVIM"),             NULL, NULL, G_CALLBACK (options_stg_cb), FALSE },
  { "send_to_clipboard",   NULL, _("Send symbol to clipboard"),        NULL, NULL, G_CALLBACK (options_stc_cb), FALSE },
  { "single_click",        NULL, _("Activate action on single click"), NULL, NULL, G_CALLBACK (options_sc_cb),  FALSE },
  { "display_symbol_name", NULL, _("Display symbol name"),             NULL, NULL, G_CALLBACK (options_dsn_cb), FALSE },
  { "enable_ams_symbols",  NULL, _("Enable AMS symbols"),              NULL, NULL, G_CALLBACK (options_eams_cb), FALSE }
};

static guint n_toggle_entries = G_N_ELEMENTS (toggle_entries);

const gchar *ui_info =
"  <popup name='popup_menu'>\n"
"      <menuitem action='send_to_clipboard' />\n"
"      <menuitem action='send_to_gvim' />\n"
"      <separator />\n"
"      <menuitem action='single_click' />\n"
"      <menuitem action='enable_ams_symbols' />\n"
"      <menuitem action='display_symbol_name' />\n"
"      <separator />\n"
"      <menuitem action='about' />\n"
"      <separator />\n"
"      <menuitem action='quit' />\n"
"  </popup>\n";


    main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (main_window), "LaTeX Symbols Selector " VERSION);

    icon = gdk_pixbuf_new_from_inline(-1, lss_icon, FALSE, NULL);
    gtk_window_set_icon (GTK_WINDOW(main_window), icon);
    g_object_unref(icon);

    gtk_window_move (GTK_WINDOW (main_window), config.window_x, config.window_y);
    gtk_window_set_default_size (GTK_WINDOW(main_window), config.window_size_x, config.window_size_y);
    gtk_window_set_resizable (GTK_WINDOW (main_window), TRUE);

    g_signal_connect (G_OBJECT (main_window), "delete_event",
                      G_CALLBACK(gui_window_closed_cb), NULL);
    g_signal_connect (G_OBJECT (main_window), "key_press_event",
                        G_CALLBACK (key_press_cb), NULL);

    action_group = gtk_action_group_new ("lss_actions");
    gtk_action_group_add_actions (action_group, entries, n_entries, NULL);
    gtk_action_group_add_toggle_actions (action_group, toggle_entries, n_toggle_entries, NULL);
    gtk_action_group_set_sensitive(action_group, TRUE);

    uim_widget = gtk_ui_manager_new ();

    gtk_ui_manager_insert_action_group (uim_widget, action_group, 0);

    gtk_window_add_accel_group (GTK_WINDOW (main_window), 
                                gtk_ui_manager_get_accel_group (uim_widget));

    if (!gtk_ui_manager_add_ui_from_string (uim_widget, ui_info, -1, NULL)) {
        g_message ("building menus failed");
    }

    gtk_ui_manager_ensure_update (uim_widget);

    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox2);
    gtk_container_add (GTK_CONTAINER (main_window), vbox2);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);

    /* groups list */

    scrolledwindow_list = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow_list);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow_list), 6);
    gtk_box_pack_start (GTK_BOX (hbox), scrolledwindow_list, FALSE, TRUE, 0);
    gtk_widget_set_size_request (GTK_WIDGET(scrolledwindow_list), 170, -1);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow_list), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow_list), GTK_SHADOW_IN);

    list_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
    list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store));
    g_signal_connect_swapped (G_OBJECT (list), "button_press_event", G_CALLBACK (mouse_button_press_cb), NULL);
    gtk_widget_show (list);
    list_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (list));
    g_signal_connect(G_OBJECT(list_selection), "changed", G_CALLBACK(group_selected_cb), NULL);
    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("Symbol groups", renderer, "text", 0, NULL);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column),
                                    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column), FALSE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

    renderer_n = gtk_cell_renderer_text_new();
    column_n = gtk_tree_view_column_new_with_attributes("N", renderer_n, "text", 0, NULL);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column_n),
                                    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_resizable(GTK_TREE_VIEW_COLUMN(column_n), FALSE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(list), column_n);
    gtk_tree_view_column_set_visible(GTK_TREE_VIEW_COLUMN(column_n), FALSE);

    gtk_container_add (GTK_CONTAINER (scrolledwindow_list), list);

    /* symbols list */

    vbox1 = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox1);
    gtk_box_pack_start (GTK_BOX (hbox), vbox1, TRUE, TRUE, 0);

    scrolledwindow_icons = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow_icons);
    gtk_container_set_border_width (GTK_CONTAINER (scrolledwindow_icons), 6);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow_icons), GTK_SHADOW_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow_icons), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start (GTK_BOX (vbox1), scrolledwindow_icons, TRUE, TRUE, 0);

    store = gtk_list_store_new (NUM_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING);

    icon_view = gtk_icon_view_new_with_model (GTK_TREE_MODEL (store));
    g_signal_connect_swapped (G_OBJECT (icon_view), "button_press_event", G_CALLBACK (mouse_button_press_cb), NULL);
    gtk_widget_show(icon_view);
    gtk_icon_view_set_pixbuf_column (GTK_ICON_VIEW (icon_view), COL_PIXBUF);

    g_signal_connect (icon_view, "selection_changed",
                            G_CALLBACK (icon_activated_sc_cb), store);
    g_signal_connect (icon_view, "item_activated",
                            G_CALLBACK (icon_activated_dc_cb), store);

    gtk_container_add (GTK_CONTAINER (scrolledwindow_icons), icon_view);
    g_object_unref (store);

    /* statusbar */

    statusbar = gtk_statusbar_new ();
    gtk_widget_show (statusbar);

    context_id = gtk_statusbar_get_context_id( GTK_STATUSBAR(statusbar), "lss");

    gtk_box_pack_start (GTK_BOX (vbox2), statusbar, FALSE, TRUE, 0);

    /* popup menu */

    popup_menu = gtk_ui_manager_get_widget (uim_widget, "/popup_menu");

    /* restore state of toggle items */

    a_action = gtk_ui_manager_get_action(uim_widget, "/popup_menu/send_to_gvim");
    if (config.send_to_gvim)
        gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(a_action), TRUE);
    a_action = gtk_ui_manager_get_action(uim_widget, "/popup_menu/send_to_clipboard");
    if (config.send_to_clipboard)
        gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(a_action), TRUE);
    a_action = gtk_ui_manager_get_action(uim_widget, "/popup_menu/single_click");
    if (config.single_click)
        gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(a_action), TRUE);
    a_action = gtk_ui_manager_get_action(uim_widget, "/popup_menu/display_symbol_name");
    if (config.display_symbol_name)
        gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(a_action), TRUE);
    a_action = gtk_ui_manager_get_action(uim_widget, "/popup_menu/enable_ams_symbols");
    if (config.enable_ams_symbols)
        gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(a_action), TRUE);

    display_names_list();

    set_cursor_on_group(config.current_group);

    gtk_widget_show(main_window);

    gtk_statusbar_push (GTK_STATUSBAR(statusbar), context_id, _("Please select symbol..."));

}

/*------------------------------------------------------------------------------*/

