/*
 * btnode.h - NILFS BT-Node prototypes and definitions
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * btnode.h,v 1.45 2007-07-12 11:22:31 kihara Exp
 *
 * Written by Seiji Kihara <kihara@osrg.net>
 */

#ifndef _NILFS_BTNODE_H
#define _NILFS_BTNODE_H

#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/buffer_head.h>
#include <linux/spinlock.h>
#include <linux/radix-tree.h>

#include "nilfs_types.h"
#include "kern_feature.h"
#include "debug.h"

#define	NILFS_PAGECACHE_TAG_PDIRTY	PAGECACHE_TAG_WRITEBACK

/* nilfs_dat_translate(&dat_inode, vbn, &pbn) */
typedef int nilfs_dat_translate_t(struct inode *, nilfs_sector_t, sector_t *);

struct nilfs_btnode_cache_ops {
	nilfs_dat_translate_t *translate;
};

struct nilfs_btnode_cache {
	struct radix_tree_root page_tree;
	rwlock_t tree_lock;
	struct nilfs_btnode_cache_ops ops;
};

struct nilfs_btnode_chkey_ctxt {
	nilfs_sector_t oldkey;
	nilfs_sector_t newkey;
	struct buffer_head *bh;
	struct buffer_head *newbh;
};

static inline void
nilfs_btnode_cache_init_once(struct nilfs_btnode_cache *btnc)
{
	INIT_RADIX_TREE(&btnc->page_tree, GFP_ATOMIC);
	rwlock_init(&btnc->tree_lock);
	btnc->ops.translate = NULL;
	page_debug(3, "translate set to NULL.\n");
}

static inline void
nilfs_btnode_cache_init(struct nilfs_btnode_cache *btnc,
			nilfs_dat_translate_t *translate)
{
	btnc->ops.translate = translate;
	page_debug(3, "translate=%p\n", translate);
}

int
nilfs_btnode_page_delete_cache(struct page *page);

int
__nilfs_btnode_get(struct nilfs_btnode_cache *, nilfs_sector_t, sector_t,
		   struct buffer_head **, int);

static inline int
nilfs_btnode_get(struct nilfs_btnode_cache *btnc, nilfs_sector_t bn,
		 struct buffer_head **result)
{
	return __nilfs_btnode_get(btnc, bn, 0, result, 1 /* exist */);
}

static inline int
nilfs_btnode_get_new(struct nilfs_btnode_cache *btnc, nilfs_sector_t bn,
		     struct buffer_head **result)
{
	return __nilfs_btnode_get(btnc, bn, 0, result, 0 /* new */);
}

static inline int
nilfs_btnode_get_pb(struct nilfs_btnode_cache *btnc, nilfs_sector_t bn,
		    sector_t pbn, struct buffer_head **result)
{
	return __nilfs_btnode_get(btnc, bn, pbn, result, 1 /* exist */);
}

int nilfs_btnode_delete_bh(struct buffer_head *);

int __nilfs_btnode_delete(struct buffer_head *, int);

static inline int
nilfs_btnode_delete_nolock(struct buffer_head *bh)
{
	return __nilfs_btnode_delete(bh, 0);
}

static inline int
nilfs_btnode_delete(struct buffer_head *bh)
{
	return __nilfs_btnode_delete(bh, 1);
}

int nilfs_btnode_delete_all(struct nilfs_btnode_cache *);

/*
 * void nilfs_btnode_read_lock(struct nilfs_btnode_cache *, unsigned long);
 * void nilfs_btnode_read_unlock(struct nilfs_btnode_cache *, unsigned long);
 * void nilfs_btnode_write_lock(struct nilfs_btnode_cache *, unsigned long);
 * void nilfs_btnode_write_unlock(struct nilfs_btnode_cache *, unsigned long);
 *
 * Note: caller must declare unsigned long variable, and must not touch it.
 */
#define nilfs_btnode_read_lock(c, f)	\
	read_lock_irqsave(&(c)->tree_lock, (f))
#define nilfs_btnode_read_unlock(c, f) \
	read_unlock_irqrestore(&(c)->tree_lock, (f))
#define nilfs_btnode_write_lock(c, f)	\
	write_lock_irqsave(&(c)->tree_lock, (f))
#define nilfs_btnode_write_unlock(c, f) \
	write_unlock_irqrestore(&(c)->tree_lock, (f))

static inline int
nilfs_btnode_page_referenced(struct page *p, int ref)
{
	return page_count(p) - ref >
		!!p->mapping + page_has_buffers(p) + PageLRU(p);
}

static inline unsigned int
nilfs_btnode_gang_lookup_nolock(struct nilfs_btnode_cache *btnc,
				struct page **pages,
				unsigned long index, int size)
{
	unsigned long ret, i;

	ret = radix_tree_gang_lookup(&btnc->page_tree, (void **)pages,
				     index, size);
	/* getting page references to guard against shrinker. */
	for (i = 0; i < ret; i++)
		page_cache_get(pages[i]);
	return ret;
}

static inline unsigned int
nilfs_btnode_gang_lookup(struct nilfs_btnode_cache *btnc, struct page **pages,
			 unsigned long index, int size)
{
	unsigned long res, flags;

	read_lock_irqsave(&btnc->tree_lock, flags);
	res = nilfs_btnode_gang_lookup_nolock(btnc, pages, index, size);
	read_unlock_irqrestore(&btnc->tree_lock, flags);
	return res;
}

static inline unsigned int
nilfs_btnode_gang_lookup_tag_nolock(struct nilfs_btnode_cache *btnc,
				    struct page **pages, unsigned long index,
				    int size, int tag)
{
	unsigned long ret, i;

	ret = radix_tree_gang_lookup_tag(&btnc->page_tree, (void **)pages,
					  index, size, tag);
	/* getting page references, to fit with standard VFS routines */
	for (i = 0; i < ret; i++)
		page_cache_get(pages[i]);
	return ret;
}

static inline unsigned int
nilfs_btnode_gang_lookup_tag(struct nilfs_btnode_cache *btnc,
			     struct page **pages, unsigned long index,
			     int size, int tag)
{
	unsigned long res, flags;

	read_lock_irqsave(&btnc->tree_lock, flags);
	res = nilfs_btnode_gang_lookup_tag_nolock(btnc, pages, index, size,
						  tag);
	read_unlock_irqrestore(&btnc->tree_lock, flags);
	return res;
}

void __nilfs_btnode_mark_dirty(struct buffer_head *, int);

static inline void
nilfs_btnode_mark_dirty(struct buffer_head *bh)
{
	__nilfs_btnode_mark_dirty(bh, PAGECACHE_TAG_DIRTY);
}

static inline void
nilfs_btnode_mark_prepare_dirty(struct buffer_head *bh)
{
	__nilfs_btnode_mark_dirty(bh, NILFS_PAGECACHE_TAG_PDIRTY);
}

void nilfs_btnode_page_clear_dirty(struct page *, int);
void nilfs_btnode_clear_dirties(struct buffer_head *);

static inline void
nilfs_btnode_clear_dirty(struct buffer_head *bh)
{
	nilfs_btnode_clear_dirties(bh);
}

static inline void
nilfs_btnode_clear_prepare_dirty(struct buffer_head *bh)
{
	nilfs_btnode_clear_dirties(bh);
}

int nilfs_btnode_prepare_change_key(struct nilfs_btnode_cache *,
				    struct nilfs_btnode_chkey_ctxt *);
void nilfs_btnode_commit_change_key(struct nilfs_btnode_cache *,
				    struct nilfs_btnode_chkey_ctxt *);
void nilfs_btnode_abort_change_key(struct nilfs_btnode_cache *,
				   struct nilfs_btnode_chkey_ctxt *);
void nilfs_btnode_do_copy_dirty_pages(struct nilfs_btnode_cache *,
				      struct nilfs_btnode_cache *, int);

static inline void
nilfs_btnode_copy_dirty_pages(struct nilfs_btnode_cache *src,
			      struct nilfs_btnode_cache *dst)
{
	nilfs_btnode_do_copy_dirty_pages(src, dst, PAGECACHE_TAG_DIRTY);
	nilfs_btnode_do_copy_dirty_pages(src, dst, NILFS_PAGECACHE_TAG_PDIRTY);
}

void nilfs_btnode_commit_gcdat_cache(struct nilfs_btnode_cache *,
				     struct nilfs_btnode_cache *);
void nilfs_btnode_clear_gcdat_cache(struct nilfs_btnode_cache *);
#endif	/* _NILFS_BTNODE_H */

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
