/*
 * debug.c - NILFS debug code and Proc-fs handling code.
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * debug.c,v 1.37 2007-06-11 09:39:09 kihara Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>,
 *            Ryusuke Konishi <ryusuke@osrg.net>
 */

#include <linux/buffer_head.h>
#include <linux/mpage.h>
#include <linux/writeback.h>
#include <linux/blkdev.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include <linux/parser.h>
#include "nilfs.h"
#include "sufile.h"

static int proc_calc_metrics(char *page, char **start, off_t off,
			     int count, int *eof, int len);

/*
 * Counters
 */
void nilfs_init_counters(void)
{
}

/*
 * debug info
 */
struct nilfs_debug_info nilfs_debug_info;

DEFINE_SPINLOCK(debug_print_lock);
DEFINE_SPINLOCK(debug_info_lock);

enum {
	Opt_quiet, Opt_verbose, Opt_verbose2, Opt_verbose3,
        /* Opt_quiet ~ Opt_verbose3 must be successive. */
	Opt_err
};

#define MAX_VLEVEL  3   /* Maximum level of Opt_verbose */

static match_table_t opt_tokens = {
	{Opt_verbose, "v"},
	{Opt_verbose2, "vv"},
	{Opt_verbose3, "vvv"},
	{Opt_quiet, "n"},
	{Opt_err, NULL}
};


static match_table_t class_tokens = {
	{NILFS_VERBOSE_FS, "fs"},
	{NILFS_VERBOSE_SEGMENT, "segment"},
	{NILFS_VERBOSE_SEGINFO, "seginfo"},
	{NILFS_VERBOSE_RECOVERY, "recovery"},
	{NILFS_VERBOSE_INODE, "inode"},
	{NILFS_VERBOSE_MDT, "mdt"},
	{NILFS_VERBOSE_BMAP, "bmap"},
	{NILFS_VERBOSE_DAT, "dat"},
	{NILFS_VERBOSE_PAGE, "page"},
	{-1, NULL},
};

static inline const char *
find_token(int token, match_table_t tokens)
{
	struct match_token *pt;

	for (pt = tokens; pt->pattern != NULL; pt++)
		if (pt->token == token)
			return pt->pattern;
	return NULL;
}

void nilfs_fill_debug_info(int level)
{
	int i;

	for (i = 0; i < NILFS_VERBOSE_LIMIT; i++)
		nilfs_debug_info.verbose[i] = level;
}

static int nilfs_read_debug_option(char *page, char **start, off_t off,
				   int count, int *eof, void *data)
{
	int len = 0;
	int flag;

	spin_lock(&debug_info_lock);
	
	for (flag = 0; flag < NILFS_VERBOSE_LIMIT; flag ++) {
		const char *vopt, *p;
		int level = min(MAX_VLEVEL,
				(int)nilfs_debug_info.verbose[flag]);

		if (level >= 0) {
			vopt = find_token(Opt_quiet + level, opt_tokens);
			BUG_ON(vopt == NULL);

			p = find_token(flag, class_tokens);
			if (!p) break;
				
			if (len > 0)
				page[len++] = ' ';
			len += sprintf(page + len, "-%s %s", vopt, p);
		}
	}

	spin_unlock(&debug_info_lock);
	page[len++] = '\n';

	return proc_calc_metrics(page, start, off, count, eof, len);
}

static int
nilfs_parse_verbose_option(char **dp, char *vopt, substring_t args[],
			   int level)
{
	char *p = "";
	int flag;

	while ((p = strsep(dp, " \t\n")) != NULL) {
		if (!*p) continue;

		if (strcmp(p, "all") == 0) {
			nilfs_fill_debug_info(level);
			return 0;
		}
		flag = match_token(p, class_tokens, args);
		if (flag < 0) break;

		nilfs_debug_info.verbose[flag] = (char)level;
		return 0;
	}
	printk(KERN_ERR
	       "NILFS: Unrecognized verbose option \"-%s %s\"\n",
	       vopt, p);
	return -EINVAL;
}

static int nilfs_parse_debug_option(char *data)
{
	char *p;
	substring_t args[MAX_OPT_ARGS];
	int err;

	while ((p = strsep(&data, " \t\n")) != NULL) {
		int token, level = -1;

		if (!*p) continue;
		else if (*p != '-') goto bad_option;

		token = match_token(++p, opt_tokens, args);
		switch(token) {
		case Opt_verbose:  level = 1; break;
		case Opt_verbose2: level = 2; break;
		case Opt_verbose3: level = 3; break;
		case Opt_quiet: level = 0; break;
		default:
			goto bad_option;
		}
		if (level >= 0) {
			err = nilfs_parse_verbose_option(&data, p, args, level);
			if (err < 0)
				return err;
		}
	}
	return 0;

 bad_option:
	printk(KERN_ERR "NILFS: Unrecognized debug option \"%s\"\n", p);
	return -EINVAL;
}

static int
nilfs_write_debug_option(struct file *file, const char __user *buffer,
			 unsigned long count, void *data)
{
	char *tmp;
	int ret = -EFAULT;

	tmp = kmalloc(count + 1, GFP_KERNEL);
	if (unlikely(!tmp))
		return -ENOMEM;

	if (copy_from_user(tmp, buffer, count))
		goto out;

	tmp[count] = '\0';

	spin_lock(&debug_info_lock);

	ret = nilfs_parse_debug_option(tmp);
	if (!ret)
		ret = count;

	spin_unlock(&debug_info_lock);
 out:
	kfree(tmp);
	return ret;
}

/*
 * VINODE
 */
#define nbar(n) ((n)++?"|":"")
#define MSIZ 256
void nilfs_vinode_debug(const char *fname, int line, char *m, struct inode *inode)
{
	int n = 0, len = 0;
	char b[MSIZ];
	struct nilfs_inode_info *ii;

	if (inode == NULL) {
		printk(KERN_DEBUG "VINODE %s: inode=NULL %s at %d\n", m, fname, line);
		return;
	}
	ii = container_of(inode, struct nilfs_inode_info, vfs_inode);

	len += snprintf(b + len, MSIZ - len, "VINODE %p %s: current %p ino=%lu nlink=%u count=%u mode=0%o mapping=%p i_bh=%p",
	       inode, m, current,
	       inode->i_ino, inode->i_nlink, atomic_read(&inode->i_count), inode->i_mode,
	       inode->i_mapping, ii->i_bh);

	len += snprintf(b + len, MSIZ - len, " %s(%d) i_state=", fname, line);
	if (inode->i_state & I_DIRTY_SYNC) {len += snprintf(b + len, MSIZ - len, "DIRTY_SYNC"); n++;}
	if (inode->i_state & I_DIRTY_DATASYNC) {len += snprintf(b + len, MSIZ - len, "%sDIRTY_DATASYNC",nbar(n));}
	if (inode->i_state & I_DIRTY_PAGES) {len += snprintf(b + len, MSIZ - len, "%sDIRTY_PAGES",nbar(n));}
	if (inode->i_state & I_LOCK) {len += snprintf(b + len, MSIZ - len, "%sLOCK",nbar(n));}
	if (inode->i_state & I_FREEING) {len += snprintf(b + len, MSIZ - len, "%sFREEING",nbar(n));}
	if (inode->i_state & I_CLEAR) {len += snprintf(b + len, MSIZ - len, "%sCLEAR",nbar(n));}
	if (inode->i_state & I_NEW) {len += snprintf(b + len, MSIZ - len, "%sNEW",nbar(n));}
#ifdef I_WILL_FREE
	if (inode->i_state & I_WILL_FREE) {len += snprintf(b + len, MSIZ - len, "%sWILL_FREE",nbar(n));}
#endif

	n=0;
	if (ii->i_state)
		len += snprintf(b + len, MSIZ - len, " vi_state=");
	if (test_bit(NILFS_I_NEW, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "NEW"); n++;}
	if (test_bit(NILFS_I_DIRTY, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sDIRTY",nbar(n));}
	if (test_bit(NILFS_I_QUEUED, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sQUEUED",nbar(n));}
	if (test_bit(NILFS_I_BUSY, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sBUSY",nbar(n));}
	if (test_bit(NILFS_I_COLLECTED, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sCOLLECTED",nbar(n));}
	if (test_bit(NILFS_I_UPDATED, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sSTATE_UPDATED",nbar(n));}
	if (test_bit(NILFS_I_INODE_DIRTY, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sINODE_DIRTY",nbar(n));}
	if (test_bit(NILFS_I_BMAP, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sBMAP",nbar(n));}
	if (test_bit(NILFS_I_GCINODE, &ii->i_state)) {len += snprintf(b+len, MSIZ-len, "%sGC",nbar(n));}

	printk(KERN_DEBUG "%s\n", b);
	BH_DEBUG(ii->i_bh, "");
}

/*
 * BH_DEBUG
 */
void
nilfs_bh_debug(const char *fname, int line, char *m, struct buffer_head *bh)
{
	int n = 0, len = 0;
	char b[MSIZ];
	if (bh == NULL) {
		printk(KERN_DEBUG "BH %s: bh=NULL %s at %d\n", m, fname, line);
		return;
	}
	len += snprintf(b + len, MSIZ - len, "BH %p %s: page=%p cnt=%d blk#=%llu",
	       bh, m, bh->b_page, atomic_read(&bh->b_count), (unsigned long long)bh->b_blocknr);
	if (bh->b_page)
		len += snprintf(b + len, MSIZ - len, " pagecnt=%d", page_count(bh->b_page));
	len += snprintf(b + len, MSIZ - len, " %s(%d) state=", fname, line);
	if (buffer_uptodate(bh)) {len += snprintf(b + len, MSIZ - len, "Uptodate"); n++;}
	if (buffer_dirty(bh)) {len += snprintf(b + len, MSIZ - len, "%sDirty",nbar(n));}
	if (buffer_locked(bh)) {len += snprintf(b + len, MSIZ - len, "%sLocked",nbar(n));}
	if (buffer_req(bh)) {len += snprintf(b + len, MSIZ - len, "%sReq",nbar(n));}
	if (buffer_mapped(bh)) {len += snprintf(b + len, MSIZ - len, "%sMapped",nbar(n));}
	if (buffer_new(bh)) {len += snprintf(b + len, MSIZ - len, "%sNew",nbar(n));}
	if (buffer_async_read(bh)) {len += snprintf(b + len, MSIZ - len, "%sARead",nbar(n));}
	if (buffer_async_write(bh)) {len += snprintf(b + len, MSIZ - len, "%sAWrite",nbar(n));}
	if (buffer_delay(bh)) {len += snprintf(b + len, MSIZ - len, "%sDelay",nbar(n));}
	if (buffer_boundary(bh)) {len += snprintf(b + len, MSIZ - len, "%sBoundary",nbar(n));}
	if (buffer_write_io_error(bh)) {len += snprintf(b + len, MSIZ - len, "%sWriteIOErr",nbar(n));}
	if (buffer_ordered(bh)) {len += snprintf(b + len, MSIZ - len, "%sOrdered",nbar(n));}
	if (buffer_eopnotsupp(bh)) {len += snprintf(b + len, MSIZ - len, "%sENOTSUPP",nbar(n));}
	/* nilfs private */
	if (buffer_prepare_dirty(bh)) {len += snprintf(b + len, MSIZ - len, "%sPrepare_Dirty",nbar(n));}
	if (buffer_nilfs_freeze(bh)) {len += snprintf(b + len, MSIZ - len, "%sFreeze",nbar(n));}
	if (buffer_nilfs_allocated(bh)) {len += snprintf(b + len, MSIZ - len, "%sAllocated",nbar(n));}
	if (buffer_nilfs_node(bh)) {len += snprintf(b + len, MSIZ - len, "%sNode",nbar(n));}
	if (buffer_nilfs_volatile(bh)) {len += snprintf(b + len, MSIZ - len, "%sVolatile",nbar(n));}

	printk(KERN_DEBUG "%s\n", b);
}

/*
 * Seginfo
 */
void nilfs_print_seginfo(struct nilfs_segment_info *seginfo)
{
	int n, len;
	char b[MSIZ];
	unsigned int flags;

	if (nilfs_debug_info.verbose[NILFS_VERBOSE_SEGINFO] <= 1)
		return;

	n = 0;  len = 0;
	b[0] = '\0';
	flags = seginfo->sum.flags;

	if (!flags) {
		len += snprintf(b + len, MSIZ - len, "<none>");  n++;
	} else {
		if (flags & NILFS_SS_LOGBGN) {
			len += snprintf(b + len, MSIZ - len, "LOGBGN");  n++;
		}
		if (flags & NILFS_SS_LOGEND)
			len += snprintf(b + len, MSIZ - len, "%sLOGEND", nbar(n));
		if (flags & NILFS_SS_SR)
			len += snprintf(b + len, MSIZ - len, "%sSR", nbar(n));
		if (flags & NILFS_SS_SYNDT)
			len += snprintf(b + len, MSIZ - len, "%sSYNDT", nbar(n));
		if (flags & NILFS_SS_GC)
			len += snprintf(b + len, MSIZ - len, "%sGC", nbar(n));
	}

	printk(KERN_DEBUG "========= NILFS SEGMENT INFORMATION ========\n");
	printk(KERN_DEBUG "full segment: segnum=%lu, start=%llu, end=%llu\n", seginfo->segnum,
	       (unsigned long long)seginfo->fseg_start, (unsigned long long)seginfo->fseg_end);
	printk(KERN_DEBUG "partial segment: start=%llu, rest=%u\n",
	       (unsigned long long)seginfo->pseg_start, seginfo->rest_blocks);
	printk(KERN_DEBUG "------------------ SUMMARY -----------------\n");
	printk(KERN_DEBUG "nfinfo     = %lu (number of files)\n", seginfo->sum.nfinfo);
	printk(KERN_DEBUG "nblocks    = %lu (number of blocks)\n", seginfo->sum.nblocks);
	printk(KERN_DEBUG "sumbytes   = %lu (size of summary in bytes)\n", seginfo->sum.sumbytes);
	printk(KERN_DEBUG "nsumblk    = %lu (number of summary blocks)\n", seginfo->sum.nsumblk);
	printk(KERN_DEBUG "flags      = %s\n", b);
	printk(KERN_DEBUG "============================================\n");
}

void nilfs_print_segment_list(const char *name, struct list_head *head, struct the_nilfs *nilfs)
{
	struct nilfs_segment_list_head *slh;

	printk(KERN_DEBUG "segment list: %s\n", name);
	list_for_each_entry(slh, head, list) {
		struct nilfs_segment_usage segusage;
		char b[MSIZ];
		int n = 0, len = 0;
		unsigned flags = slh->flags;
		int err;

		b[0] = '\0';

		len += snprintf(b + len, MSIZ - len, "slh-flags=");  n++;
		if (!flags) {
			len += snprintf(b + len, MSIZ - len, "<none>");  n++;
		} else {
			if (flags & NILFS_SLH_COMMIT) {
				len += snprintf(b + len, MSIZ - len, "COMMIT");  n++;
			}
			if (flags & NILFS_SLH_FREED)
				len += snprintf(b + len, MSIZ - len, "%sFREED", nbar(n));
		}
		err = nilfs_sufile_get_segment_usages(nilfs->ns_sufile, slh->segnum, &segusage, 1);
		if (likely(!err)) {
			len += snprintf(b + len, MSIZ - len, " su-flags=");  n++;
			if (nilfs_segment_usage_active(&segusage)) {
				len += snprintf(b + len, MSIZ - len, "ACTIVE");  n++;
			}
			if (nilfs_segment_usage_dirty(&segusage))
				len += snprintf(b + len, MSIZ - len, "%sDIRTY", nbar(n));
			if (nilfs_segment_usage_error(&segusage))
				len += snprintf(b + len, MSIZ - len, "%sERROR", nbar(n));
			if (nilfs_segment_usage_volatile_active(&segusage))
				len += snprintf(b + len, MSIZ - len, "%sVACTIVE", nbar(n));
		}
		printk(KERN_DEBUG "SLH(segnum=%lu) %s bh_su=%p raw_su=%p\n",
		       slh->segnum, b, slh->bh_su, slh->raw_su);
	}
}
#undef MSIZ
#undef nbar

#define MSIZ 512
void nilfs_print_finfo(sector_t blocknr, ino_t ino,
		       unsigned long nblocks, unsigned long ndatablk)
{
	unsigned long nnodeblk = nblocks - ndatablk;
	sector_t node_start = blocknr + ndatablk;
	char b[MSIZ];
	int len;

	if (nilfs_debug_info.verbose[NILFS_VERBOSE_SEGINFO] < 3)
		return;

	len = 0;
	b[0] = '\0';

	if (ndatablk)
		len += snprintf(b + len, MSIZ - len, " data[%llu,%llu]",
				(unsigned long long)blocknr, (unsigned long long)node_start - 1);
	else
		len += snprintf(b + len, MSIZ - len, " data[<none>]");

	if (nnodeblk)
		len += snprintf(b + len, MSIZ - len, " node[%llu,%llu]",
				(unsigned long long)node_start,
				(unsigned long long)(node_start + nnodeblk));
	else
		len += snprintf(b + len, MSIZ - len, " node[<none>]");

	printk(KERN_DEBUG "FINFO(ino=%lu)%s\n", ino, b);
}

void nilfs_print_binfo(sector_t blocknr, union nilfs_binfo *binfo,
		       print_binfo_proc_t proc)
{
	char b[MSIZ];

	if (nilfs_debug_info.verbose[NILFS_VERBOSE_SEGINFO] < 3)
		return;

	if (proc) {
		(*proc)(b, MSIZ, binfo);
		printk(KERN_DEBUG "BINFO(blocknr=%llu): %s\n", (unsigned long long)blocknr, b);
	}
}
#undef MSIZ

/*
 * Proc-fs entries
 */
struct proc_dir_entry *nilfs_proc_root;

static int proc_calc_metrics(char *page, char **start, off_t off,
			     int count, int *eof, int len)
{
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}

static int
nilfs_print_page_counters(char *page, char **start, off_t off,
			  int count, int *eof, void *data)
{
	int len;
	unsigned long counters[2];

	nilfs_read_page_counters(counters);
	len = sprintf(page, "nr_active: %lu\n", counters[0]);
	len += sprintf(page + len, "nr_inactive: %lu\n", counters[1]);
	return proc_calc_metrics(page, start, off, count, eof, len);
}

int nilfs_init_proc_entries(void)
{
	struct proc_dir_entry *entry;

	nilfs_proc_root = proc_mkdir("nilfs2", proc_root_fs);
	if (!nilfs_proc_root) {
		printk(KERN_WARNING "NILFS: cannot create proc root\n");
		return 0; /* We don't abort when failed to make 
			     proc entries */
	}
	nilfs_proc_root->owner = THIS_MODULE;

	/* /proc entries */
	entry = create_proc_entry("debug_option", 
				  S_IFREG | S_IRUGO | S_IWUSR, 
				  nilfs_proc_root);
	if (entry) {
		entry->read_proc = nilfs_read_debug_option;
		entry->write_proc = nilfs_write_debug_option;
	}

	entry = create_proc_entry("page", 0, nilfs_proc_root);
	if (entry)
		entry->read_proc = nilfs_print_page_counters;

	return 0;
}

void nilfs_remove_proc_entries(void)
{
	remove_proc_entry("page", nilfs_proc_root);
	remove_proc_entry("debug_option", nilfs_proc_root);
	remove_proc_entry("nilfs2", proc_root_fs);
}

/*
 * For inode and page debug
 */
int nilfs_releasepage(struct page *page, gfp_t gfp_mask)
{
	int ret;
	int verbose = (nilfs_debug_info.verbose[NILFS_VERBOSE_PAGE] > 1);

	if (!verbose) {
		struct address_space *mapping = page->mapping;
		struct inode *inode;

		if (mapping) {
			inode = mapping->host;
			if (!inode)
				inode = container_of(page->mapping, struct inode, i_data);
			if (inode->i_sb && !(inode->i_sb->s_flags & MS_ACTIVE))
				verbose = 1;
		}
	}

	if (!PagePrivate(page)) {
		page_debug(1, "Page %p has no buf.\n", page);
		return 0;
	}
	BUG_ON(buffer_nilfs_allocated(page_buffers(page)));

	ret = try_to_free_buffers(page);
	if (verbose && !ret) {
		page_debug(2, "warning: try_to_free_buffers() failed (page=%p, count=%d)\n",
			   page, page_count(page));
		if (page_has_buffers(page)) {
			struct buffer_head *head, *bh;
			head = bh = page_buffers(page);
			do {
				BH_DEBUG(bh, "");
			} while ((bh = bh->b_this_page) != head);
		}
	}
	if (ret && page->mapping && page->mapping->host) {
		if (page_count(page) > 2 + !PageLRU(page))
			page_debug(1, "warning: too many page_count=%d (page=%p)\n",
				   page_count(page), page);
	}
	return ret;
}

#if NEED_SYNC_PAGE_RETVAL
int
#else
void
#endif
nilfs_sync_page(struct page *page)
{
	page_debug(3, "called (page=%p)\n", page);
#if NEED_SYNC_PAGE_RETVAL
	return 0;
#endif
}

#if NEED_INVALIDATEPAGE_RETVAL
#define __ESC(f)   return (f)
int
#else
#define __ESC(f)   do { (f); return; } while(1)
void
#endif /* NEED_INVALIDATEPAGE_RETVAL */
nilfs_invalidatepage(struct page *page, unsigned long offset)
{
	struct buffer_head *bh = NULL;

	if (PagePrivate(page)) {
		bh = page_buffers(page);
		BUG_ON(buffer_nilfs_allocated(bh));
	}
	__ESC(block_invalidatepage(page, offset));
}
#undef __ESC

void
nilfs_put_inode(struct inode *inode)
{
	VINODE_DEBUG(inode, "");
}

/*
 * Radix-tree checker
 */
#define S_N_PAGEVEC  16
void
nilfs_check_radix_tree(const char *fname, struct address_space *mapping,
		       int blocksize_bits)
{
	struct page *pages[S_N_PAGEVEC];
	unsigned int nr_page;
	pgoff_t index = 0;
	int nr_found = 0;
	int i;

 repeat:
	READ_LOCK_IRQ(&mapping->tree_lock);
	nr_page = radix_tree_gang_lookup(&mapping->page_tree,
					 (void **)pages,
					 index,
					 S_N_PAGEVEC);
	for (i = 0; i < nr_page; i++)
		page_cache_get(pages[i]);
	READ_UNLOCK_IRQ(&mapping->tree_lock);

	if (nr_page == 0) {
		if (nr_found)
			printk(KERN_WARNING "%s: found %d leaking pages\n",
			       fname, nr_found);
		return;
	}
	index = pages[nr_page - 1]->index + 1;

	for(i = 0; i < nr_page; i++) {
		struct buffer_head *bh, *head;
		unsigned long fbn;

		fbn = pages[i]->index >> (PAGE_SHIFT - blocksize_bits);
		printk(KERN_WARNING 
		       "found leaking page(addr=%p, index=%lu)\n",
		       pages[i], fbn);
		nr_found++;
		if (!page_has_buffers(pages[i]))
			goto skip_page;
		bh = head = page_buffers(pages[i]);
		do {
			BH_DEBUG(bh, "leaking block");
			bh = bh->b_this_page;
			fbn++;
		} while (bh != head);
	skip_page:
		page_cache_release(pages[i]);
	}
	goto repeat;
}

void nilfs_print_bmap_direct_pointers(struct inode *inode, struct nilfs_inode *raw_inode)
{
	int i;
	__le64 *direct_ptr;

	if (nilfs_debug_info.verbose[NILFS_VERBOSE_INODE] < 3)
		return;
	printk(KERN_DEBUG "NILFS inode (ino=%lu)\n", inode->i_ino);
	for (i = 0, direct_ptr = raw_inode->i_bmap;
	     i < NILFS_INODE_BMAP_SIZE; i++, direct_ptr++)
		printk(KERN_DEBUG "  i_bmap[%d]=%llu\n",
		       i, (unsigned long long)le64_to_cpu(*direct_ptr));
}


/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
