/*
 * mdt.h - NILFS meta data file (provisional) prototype and definitions
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * mdt.h,v 1.35 2007-07-23 11:25:58 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 */

#ifndef _NILFS_MDT_H
#define _NILFS_MDT_H

#include <linux/buffer_head.h>
#include <linux/blockgroup_lock.h>
#include "nilfs.h"
#include "nilfs_types.h"
#include "kern_feature.h"


struct nilfs_mdt_info {
	struct the_nilfs       *mi_nilfs;	/* Back pointer to the_nilfs struct */
	struct rw_semaphore	mi_sem;		/* Reader/Writer semaphore for contents */
	struct blockgroup_lock *mi_bgl;		/* Per-blockgroup locking */
	unsigned		mi_entry_size;	/* Size of an entry */
	unsigned long		mi_entries_per_block; /* Number of entries in a block */
	unsigned long		mi_blocks_per_group;  /* Number of blocks in a group */
	unsigned long		mi_groups_count; /* Number of groups */
};

static inline struct nilfs_mdt_info *NILFS_MDT(struct inode *inode)
{
#if NEED_INODE_GENERIC_IP
	return inode->u.generic_ip;
#else
	return inode->i_private;
#endif
}

static inline struct the_nilfs *NILFS_I_NILFS(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;

	return sb ? NILFS_SB(sb)->s_nilfs : NILFS_MDT(inode)->mi_nilfs;
}

typedef void (nilfs_mdt_init_block_t)(struct inode *, struct buffer_head *, void *);


/* Default GFP flags using highmem */
#define NILFS_MDT_GFP      (__GFP_WAIT | __GFP_IO | __GFP_HIGHMEM)

extern int nilfs_mdt_create_block(struct inode *, nilfs_blkoff_t, struct buffer_head **, nilfs_mdt_init_block_t *);
extern int nilfs_mdt_read_block(struct inode *, nilfs_blkoff_t, struct buffer_head **);
extern int nilfs_mdt_get_block(struct inode *, nilfs_blkoff_t, int, nilfs_mdt_init_block_t *, struct buffer_head **);
extern int nilfs_mdt_delete_block(struct inode *, nilfs_blkoff_t);
extern int nilfs_mdt_forget_block(struct inode *, nilfs_blkoff_t);
extern int nilfs_mdt_truncate_blocks(struct inode *, nilfs_blkoff_t);
extern int nilfs_mdt_mark_block_dirty(struct inode *, nilfs_blkoff_t);

extern struct inode *nilfs_mdt_new(struct the_nilfs *, struct super_block *, ino_t, gfp_t);
extern struct inode *nilfs_mdt_new_common(struct the_nilfs *, struct super_block *, ino_t, gfp_t);
extern void nilfs_mdt_clear(struct inode *);
extern struct inode *nilfs_mdt_new_with_blockgroup(struct the_nilfs *, struct super_block *, ino_t, gfp_t, unsigned, unsigned long);

#if NEED_OLD_MARK_BUFFER_DIRTY
void FASTCALL(nilfs_mdt_mark_buffer_dirty(struct buffer_head *bh));
#else
#define nilfs_mdt_mark_buffer_dirty(bh)		mark_buffer_dirty(bh)
#endif

static inline void nilfs_mdt_mark_dirty(struct inode *inode)
{
	mdt_debug(3, "called (ino=%lu)\n", inode->i_ino);
	set_bit(NILFS_I_DIRTY, &NILFS_I(inode)->i_state);
}

static inline void nilfs_mdt_clear_dirty(struct inode *inode)
{
	clear_bit(NILFS_I_DIRTY, &NILFS_I(inode)->i_state);
}

static inline int nilfs_mdt_test_dirty(struct inode *inode)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);

	if (nilfs_bmap_test_and_clear_dirty(ii->i_bmap)) {
		set_bit(NILFS_I_DIRTY, &ii->i_state);
		return 1;
	}
	return test_bit(NILFS_I_DIRTY, &ii->i_state);
}

static inline void nilfs_mdt_destroy(struct inode *inode)
{
	extern void nilfs_destroy_inode(struct inode *);

	mdt_debug(2, "called (ino=%lu)\n", inode->i_ino);
	if (NILFS_MDT(inode)->mi_bgl)
		kfree(NILFS_MDT(inode)->mi_bgl);
	kfree(NILFS_MDT(inode));
	nilfs_destroy_inode(inode);
	mdt_debug(2, "done\n");
}

static inline nilfs_cno_t nilfs_mdt_cno(struct inode *inode)
{
	return NILFS_MDT(inode)->mi_nilfs->ns_cno;
}

#define nilfs_mdt_bgl_lock(inode, block_group) \
	(&NILFS_MDT(inode)->mi_bgl->locks[(block_group) & (NR_BG_LOCKS-1)].lock)


static inline int
nilfs_mdt_read_inode_direct(struct inode *inode, struct buffer_head *bh, unsigned n)
{
	extern int nilfs_dat_translate(struct inode *, nilfs_sector_t, sector_t *);
	struct nilfs_inode *raw_inode = (struct nilfs_inode *)(bh->b_data + n);
	int err = nilfs_read_inode_common(inode, raw_inode, 
					  inode->i_ino == NILFS_DAT_INO ?
					  NULL : nilfs_dat_translate);
	/* Other field initializations */
	return err;
}

static inline void 
nilfs_mdt_write_inode_direct(struct inode *inode, struct buffer_head *bh, unsigned n)
{
	nilfs_write_inode_common(inode, (struct nilfs_inode *)(bh->b_data + n), 1);
}

#endif /* _NILFS_MDT_H */

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
