/*
 * page.c - buffer/page management for NILFS
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * page.c,v 1.76 2007-06-11 05:18:42 kihara Exp
 *
 * Modified for NILFS by Amagai Yoshiji <amagai@osrg.net>,
 *                       Ryusuke Konishi <ryusuke@osrg.net>,
 *			 Seiji Kihara <kihara@osrg.net>.
 */
/*
 *  linux/fs/buffer.c
 *
 *  Copyright (C) 1991, 1992, 2002  Linus Torvalds
 */

#include <linux/pagemap.h>
#include <linux/writeback.h>
#include <linux/swap.h>
#include <linux/bitops.h>
#include <linux/page-flags.h>
#include <linux/list.h>
#include "nilfs.h"
#include "page.h"
#include "btnode.h"

struct buffer_head *nilfs_bread_slow(struct buffer_head *bh)
{
	lock_buffer(bh);
	if (buffer_uptodate(bh)) {
		unlock_buffer(bh);
		return bh;
	} else {
#ifdef CONFIG_NILFS_DEBUG
		char b[BDEVNAME_SIZE];
#endif

		get_bh(bh);
		page_debug(3, "try to read block (dev=%s, dbn=%llu)\n",
			   bdevname(bh->b_bdev, b),
			   (unsigned long long)bh->b_blocknr);
		bh->b_end_io = end_buffer_read_sync;
		submit_bh(READ, bh);
		wait_on_buffer(bh);
		if (buffer_uptodate(bh))
			return bh;
	}
	brelse(bh);
	return NULL;
	/*
	 * __bread_slow() releases buffer count when it fails to read.
	 * The caller must not release buffer_head if NULL is returned.
	 * Note that an increment by get_bh() in __bread_slow() is consumed by
	 * the BIO submission.
	 */
}

/* end copied functions from fs/buffer.c */

struct buffer_head *
nilfs_get_page_block(struct page *page, nilfs_blkoff_t block, pgoff_t index,
		     int blkbits)

{
	nilfs_blkoff_t first_block;
	struct buffer_head *bh;

	BUG_ON(!PageLocked(page));

	if (!page_has_buffers(page)) {
		struct buffer_head *head;

		mdt_debug(3, "page has no buffer heads. allocating.. (page=%p)\n", page);
		head = alloc_page_buffers(page, 1 << blkbits, 1);
		if (!head)
			return NULL;
		nilfs_link_buffers(page, head);
#if 0
		bh = head;
		do {
			bh->b_blocknr = 0;
			bh = bh->b_this_page;
		} while (bh != head);
#endif
#ifdef NILFS_BH_DEBUG
		BH_DEBUG(bh, "linked");
#endif
	}

	first_block = (nilfs_blkoff_t)index << (PAGE_CACHE_SHIFT - blkbits);
	bh = nilfs_page_get_nth_block(page, block - first_block);
	BUG_ON(!bh);

	touch_buffer(bh);
	wait_on_buffer(bh);
	return bh;
}

/*
 * low-level nilfs pages, page functions
 */
static struct nilfs_pages {
	spinlock_t lru_lock;
	struct list_head active;
	struct list_head inactive;
	unsigned long nr_active;
	unsigned long nr_inactive;
} nilfs_pages;

/*
 * XXX per-cpu pagevecs may be able to reduce the overhead of list handlings
 *
 * static DEFINE_PER_CPU(struct pagevec, nilfs_lru_active) = { 0, };
 * static DEFINE_PER_CPU(struct pagevec, nilfs_lru_inactive) = { 0, };
 */

void nilfs_pages_init(void)
{
	INIT_LIST_HEAD(&nilfs_pages.active);
	INIT_LIST_HEAD(&nilfs_pages.inactive);
	spin_lock_init(&nilfs_pages.lru_lock);
	nilfs_pages.nr_active = 0;
	nilfs_pages.nr_inactive = 0;
}

#ifdef CONFIG_NILFS_DEBUG
void nilfs_read_page_counters(unsigned long *counters)
{
	spin_lock_irq(&nilfs_pages.lru_lock);
	counters[0] = nilfs_pages.nr_active;
	counters[1] = nilfs_pages.nr_inactive;
	spin_unlock_irq(&nilfs_pages.lru_lock);
}
#endif

static /* inline */ void
nilfs_free_page(struct page *page)
{
	BUG_ON(!PageLocked(page));
	/* ref page for __free_page */
	if (unlikely(page_count(page) != 1)) {
		page_debug(1, "*** PAGE COUNT WRONG (%d,%p) ***\n",
			   page_count(page), page);
		BUG();
	}
#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(page->mapping != NULL);
	BUG_ON(PageLRU(page));
	BUG_ON(PagePrivate(page));
	BUG_ON(PageActive(page));
	BUG_ON(PageWriteback(page));
	BUG_ON(PageDirty(page));
	BUG_ON(!list_empty(&page->lru) && (page->lru.next != LIST_POISON1) &&
	       (page->lru.prev != LIST_POISON2));
#endif
	unlock_page(page);
	__free_page(page);
}

static int
nilfs_pages_free_lru(int active, int nr, int *nf, int destroy)
{
	struct list_head *lru_list;
	spinlock_t *lock = &nilfs_pages.lru_lock;
	int n_freed = 0, n_scanned = 0;
	unsigned long flags, *nr_lru;
	struct page *page, *next;
	struct buffer_head *bh, *head;
	LIST_HEAD(l_victim);

	lru_list = active ? &nilfs_pages.active : &nilfs_pages.inactive;
	nr_lru = active ? &nilfs_pages.nr_active : &nilfs_pages.nr_inactive;
	spin_lock_irqsave(lock, flags);
	if ((*nr_lru < nr) || (nr == 0))
		nr = *nr_lru;
	list_for_each_entry_safe(page, next, lru_list, lru) {
		if (nr == 0)
			break;
		nr--;
		n_scanned++;
		page_cache_get(page);
		if (TestSetPageLocked(page)) {
			page_debug(2, "Skipped(Locked) %p\n", page);
			goto keep_in_list;
		}
		page_debug(3, "locked page %p.\n", page);
		/* prevent deletion by others */
		if (!nilfs_btnode_page_referenced(page, 1)) {
			list_del(&page->lru);
			list_add(&page->lru, &l_victim);
			continue;
		}
		unlock_page(page);
	keep_in_list:
		page_cache_release(page);
	}
	spin_unlock_irqrestore(lock, flags);
	list_for_each_entry_safe(page, next, &l_victim, lru) {
		if (PageDirty(page) || PageWriteback(page)) {
			page_debug(2, "Skipped(Dirty|WB) %p\n", page);
			goto keep_locked;
		}
		BUG_ON(!PageLRU(page));
		BUG_ON(!PageLocked(page));
		if (page_has_buffers(page)) {
			bh = head = page_buffers(page);
			if (buffer_nilfs_node(head)) {
				/* remove from cache and clear nilfs node */
				if (nilfs_btnode_page_delete_cache(page)) {
					page_debug(2, "Skipped(BufBusy) %p\n",
						   page);
					goto keep_locked;
				}
				do {
					if (buffer_mapped(bh)) {
						get_bh(bh);
						nilfs_btnode_delete_bh(bh);
					}
					clear_buffer_nilfs_node(bh);
					bh = bh->b_this_page;
				} while (bh != head);
			}
		}
		__ClearPageLRU(page);
		list_del_init(&page->lru);
		n_freed++;
		page_cache_release(page);
		ClearPageActive(page);
		ClearPageReferenced(page);
		if (page_has_buffers(page)) {
			page_debug(3, "freeing page %p with buffers.\n", page);
			wait_on_page_writeback(page);
			nilfs_free_buffer_page(page);
		} else {
			page_debug(1, "freeing page %p without no bufs\n",
				   page);
			nilfs_free_page(page);
		}
		continue;
	keep_locked:
		unlock_page(page);
		/* page_debug(3, "unlocked page %p.\n", page); */
		page_cache_release(page);
	}
	spin_lock_irqsave(lock, flags);
	list_splice(&l_victim, lru_list);
	*nr_lru -= n_freed;
	spin_unlock_irqrestore(lock, flags);
	*nf = n_freed;
	return n_scanned;
}

static void
nilfs_pages_forget_lru(int active)
{
	struct list_head *lru_list;
	spinlock_t *lock = &nilfs_pages.lru_lock;
	unsigned long flags;
	struct page *page, *next;
	struct buffer_head *bh, *head;

	lru_list = active ? &nilfs_pages.active : &nilfs_pages.inactive;
	spin_lock_irqsave(lock, flags);
	list_for_each_entry_safe(page, next, lru_list, lru) {
		if (TestSetPageLocked(page)) {
			page_debug(1, "Locked %p, wait\n", page);
			lock_page(page);
		}
		wait_on_page_writeback(page);	/* wait writer */
		if (page_has_buffers(page) && nilfs_page_buffers_busy(page)) {
			head = bh = page_buffers(page);
			do {
				get_bh(bh);
				if (buffer_locked(bh)) {
					page_debug(1, "BH %p locked, wait.\n",
						   bh);
					wait_on_buffer(bh);
				}
				if (buffer_dirty(bh)) {
					page_debug(1, "BH %p dirty, clear\n",
						   bh);
					clear_buffer_dirty(bh);
				}
				while (atomic_read(&bh->b_count) > 1) {
					page_debug(1, "BH %p ref %d\n", bh,
						   atomic_read(&bh->b_count));
					brelse(bh);
				}
				brelse(bh);
			} while ((bh = bh->b_this_page) != head);
		}
		if (PageDirty(page)) {
			page_debug(1, "Page %p dirty, force clear\n", page);
			ClearPageDirty(page);
		}
		/*
		 * Note: radix-tree's lock should be hold when checking page
		 * count, but this function must be called only when exiting
		 * the file system and when nobody refer it.
		 */
		while (nilfs_btnode_page_referenced(page, 1)) {
			page_debug(1, "ref %d:%d%d for %p\n",
				   page_count(page), !!page->mapping,
				   page_has_buffers(page), page);
			page_cache_release(page);
		}
	}
	spin_unlock_irqrestore(lock, flags);
}

void nilfs_pages_destroy(void)
{
	int nf;

	page_debug(3, "freeing inactive lru\n");
	nilfs_pages_free_lru(0 /* inactive */, 0, &nf, 1 /* destroy */);
	if (nilfs_pages.nr_inactive) {
		nilfs_pages_forget_lru(0);
		nilfs_pages_free_lru(0, 0, &nf, 1);
		if (nilfs_pages.nr_inactive)
			page_debug(1,
				   "*** %ld pages remain on inactive list ***",
				   nilfs_pages.nr_inactive);

	}
	page_debug(3, "freeing active lru\n");
	nilfs_pages_free_lru(1 /* active */, 0, &nf, 1 /* destroy */);
	if (nilfs_pages.nr_active) {
		nilfs_pages_forget_lru(1);
		nilfs_pages_free_lru(1, 0, &nf, 1);
		if (nilfs_pages.nr_active)
			page_debug(1,
				   "*** %ld pages remain on active list ***",
				   nilfs_pages.nr_active);
	}
	page_debug(3, "finished.\n");
}

#ifdef NILFS_SHRINKER_ENABLE
int nilfs_pages_shrink(int nr, GFP_T gfp_mask)
{
	int ns, nfi = 0, nfa = 0;

	if (nr) {
		page_debug(2, "called with %d, a:i=%lu:%lu.\n", nr,
			   nilfs_pages.nr_active, nilfs_pages.nr_inactive);

		ns = nilfs_pages_free_lru(0 /* inactive*/, nr, &nfi,
					  0 /* normal */);
		nr -= ns;
		if (nr > 0)
			ns += nilfs_pages_free_lru(1 /* active */, nr, &nfa,
						   0 /* normal */);
		/*
		 * XXX: must move pages from active to inactive for right PFRA.
		 */
		/* nilfs_pages_refill_inactive(nr + ns); */

		page_debug(2, "%d scanned, %d+%d freed, return %d(%lu+%lu).\n",
			   ns, nfa, nfi,
			   (int) (nilfs_pages.nr_active +
				  nilfs_pages.nr_inactive),
			   nilfs_pages.nr_active, nilfs_pages.nr_inactive);
	}
	return nilfs_pages.nr_active + nilfs_pages.nr_inactive;
}
#endif

static inline void
__nilfs_page_add_to_inactive(struct page *page)
{
#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(!list_empty(&page->lru) && (page->lru.next != LIST_POISON1) &&
	       (page->lru.prev != LIST_POISON2));
#endif
	BUG_ON(PageLRU(page));
	SetPageLRU(page);
	list_add_tail(&page->lru, &nilfs_pages.inactive);
	page_cache_get(page);	/* LRU */
	nilfs_pages.nr_inactive++;
}

void nilfs_page_add_to_inactive(struct page *page)
{
	unsigned long flags;
	spin_lock_irqsave(&nilfs_pages.lru_lock, flags);
	__nilfs_page_add_to_inactive(page);
	spin_unlock_irqrestore(&nilfs_pages.lru_lock, flags);
}

static inline void
__nilfs_page_delete_from_inactive(struct page *page)
{
#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(!PageLRU(page));
	BUG_ON((page->lru.next == LIST_POISON1) ||
	       (page->lru.prev == LIST_POISON2));
#endif
	__ClearPageLRU(page);
	list_del_init(&page->lru);
	page_cache_release(page); /* LRU */
	nilfs_pages.nr_inactive--;
}

static inline void
__nilfs_page_add_to_active(struct page *page)
{
#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(!list_empty(&page->lru) && (page->lru.next != LIST_POISON1) &&
	       (page->lru.prev != LIST_POISON2));
	BUG_ON(PageLRU(page));
#endif
	list_add_tail(&page->lru, &nilfs_pages.active);
	page_cache_get(page);	/* LRU */
	nilfs_pages.nr_active++;
	SetPageActive(page);
	SetPageLRU(page);
}

void nilfs_page_add_to_active(struct page *page)
{
	unsigned long flags;
	spin_lock_irqsave(&nilfs_pages.lru_lock, flags);
	__nilfs_page_add_to_active(page);
	spin_unlock_irqrestore(&nilfs_pages.lru_lock, flags);
}

static inline void
__nilfs_page_delete_from_active(struct page *page)
{
#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(!PageLRU(page));
	BUG_ON((page->lru.next == LIST_POISON1) ||
	       (page->lru.prev == LIST_POISON2));
#endif
	__ClearPageLRU(page);
	list_del_init(&page->lru);
	__ClearPageActive(page);
	page_cache_release(page); /* LRU */
	nilfs_pages.nr_active--;
}

static inline void
__nilfs_page_delete_from_lru(struct page *page)
{
	if (PageActive(page))
		__nilfs_page_delete_from_active(page);
	else
		__nilfs_page_delete_from_inactive(page);
}

void nilfs_page_delete_from_lru(struct page *page)
{
	unsigned long flags;

	spin_lock_irqsave(&nilfs_pages.lru_lock, flags);
	__nilfs_page_delete_from_lru(page);
	spin_unlock_irqrestore(&nilfs_pages.lru_lock, flags);
}

/* borrow from mm/swap.c::mark_page_accessed() and activate_page() */
void nilfs_page_mark_accessed(struct page *page)
{
	unsigned long flags;

	if (!PageLRU(page) || (PageActive(page) && PageReferenced(page)))
		return;
	lock_page(page);
	if (!PageReferenced(page)) {
		SetPageReferenced(page);
		goto out_unlock;
	}
	spin_lock_irqsave(&nilfs_pages.lru_lock, flags);
	list_move_tail(&page->lru, &nilfs_pages.active);
	nilfs_pages.nr_inactive--;
	nilfs_pages.nr_active++;
	/* inc_page_state(pgactivate); */
	SetPageActive(page);
	spin_unlock_irqrestore(&nilfs_pages.lru_lock, flags);
	ClearPageReferenced(page);
out_unlock:
	unlock_page(page);
	return;
}

/**
 * nilfs_alloc_buffer_page - allocate a private page with buffer heads
 *
 * Return Value: On success, a pointer to the allocated page is returned.
 * On error, NULL is returned.
 */
struct page *
nilfs_alloc_buffer_page(struct block_device *bdev, int size)
{
	struct buffer_head *bufs, *bh;
	struct page *page;

	page = alloc_page(GFP_NOFS); /* page_count of the returned page is 1 */
	if (unlikely(!page))
		return NULL;

#ifdef CONFIG_NILFS_DEBUG
	BUG_ON(PageLRU(page));
	BUG_ON(!list_empty(&page->lru) && (page->lru.next != LIST_POISON1) &&
	       (page->lru.prev != LIST_POISON2));
	BUG_ON(page->mapping);
#endif

	lock_page(page);
	bufs = alloc_page_buffers(page, size, 0);
	if (unlikely(!bufs)) {
		nilfs_free_page(page);
		return NULL;
	}
	nilfs_link_buffers(page, bufs);
	bh = bufs;
	do {
		set_buffer_nilfs_allocated(bh);
		bh->b_bdev = bdev;	/* for a compatibility reason */
		bh = bh->b_this_page;
	} while (bh != bufs);

	return page;
}

/**
 * nilfs_free_buffer_page - try to free a private buffer page.
 * page: page to be freed
 *
 * This function should be called when the reference count of
 * each buffer becomes zero.
 * The page is freed only when it has no active buffers.
 * In that case, buffer heads attached to the page are destroyed.
 *
 * Note: the page should be locked and page's ref hold by caller.
 */
int nilfs_free_buffer_page(struct page *page)
{
	int err = -1;

	BUG_ON(!PageLocked(page));
	if (page->mapping) {
		struct buffer_head *head;
		page_debug(3, "Page %p has mapping %p\n", page, page->mapping);
		if (page_has_buffers(page)) {
			head = page_buffers(page);
			if (buffer_nilfs_node(head)) {
				if (nilfs_btnode_page_delete_cache(page)) {
					page_debug(3, "Cannot del from cache\n");
					return err;
				}
			} else
				page_debug(3, "Page is not nilfs page.\n");
		} else
			page_debug(3, "Page has mapping but no bufs.\n");
	}
	if (page_has_buffers(page) && !try_to_free_buffers(page)) {
		struct buffer_head *head, *bh;

		page_debug(3, "try_to_free_buffers failed (page=%p)\n", page);
		head = bh = page_buffers(page);
		do {
			BH_DEBUG(bh, "");
		} while ((bh = bh->b_this_page) != head);
		return err;
	}
	nilfs_free_page(page);
	return 0;
}

/**
 * nilfs_copy_page_buffers - make a clone copy of page with buffers
 * @bh: buffer head of start block
 * @nblock: number of blocks to be copied
 *
 * On success, returns buffers whose page is locked.
 */
struct buffer_head *nilfs_copy_page_buffers(struct buffer_head *bh, int nblocks)
{
	struct page *page, *orig_page = bh->b_page;
	struct buffer_head *bufs;
	void *kaddr;

	BUG_ON(nblocks <= 0);
	page = nilfs_alloc_buffer_page(bh->b_bdev, bh->b_size);
	if (unlikely(!page))
		return NULL;

	bufs = page_buffers(page);
	kaddr = kmap_atomic(orig_page, KM_USER0);
	memcpy(bufs->b_data, kaddr + bh_offset(bh), bh->b_size * nblocks);
	kunmap_atomic(kaddr, KM_USER0);

	nilfs_page_add_to_active(page);

	return bufs;
}

/**
 * nilfs_page_buffers_clean - check if a page has dirty buffers or not.
 * @page: page to be checked
 *
 * nilfs_page_buffers_clean() returns zero if the page has dirty buffers.
 * Otherwise, it returns non-zero value.
 * For the btnode page, the prepare-dirty state flag for written buffers
 * cleared here, and return values are:
 *	00(b): page state unchanged (remains dirty or prepare-dirty)
 *	01(b): page state will be changed from dirty to clean
 *	10(b): page state will be changed from prepare-dirty to dirty
 *	11(b): page state will be changed from prepare-dirty to clean
 */
int nilfs_page_buffers_clean(struct page *page)
{
	struct buffer_head *bh, *head;
	int d = 0, pd = 0, pc = 0;

	BUG_ON(!page_has_buffers(page));

	bh = head = page_buffers(page);
	do {
		int _d = buffer_dirty(bh), _p = buffer_prepare_dirty(bh);
		d |= _d;
		pd |= _p & _d;
		if (_p & !_d) {
			/* Note: buffer's prepare_dirty bit is cleared here */
			clear_buffer_prepare_dirty(bh);
			pc = 1;
		}
		bh = bh->b_this_page;
	} while (bh != head);
	return (!pd & pc) << NILFS_PAGECACHE_TAG_PDIRTY |
		!d << PAGECACHE_TAG_DIRTY;
}

#if !HAVE_CLEAR_PAGE_DIRTY
int __nilfs_clear_page_dirty(struct page *page)
{
	struct address_space *mapping = page->mapping;
	unsigned long flags;

	if (mapping) {
		write_lock_irqsave(&mapping->tree_lock, flags);
		if (test_bit(PG_dirty, &page->flags)) {
			radix_tree_tag_clear(&mapping->page_tree,
					     page_index(page),
					     PAGECACHE_TAG_DIRTY);
			write_unlock_irqrestore(&mapping->tree_lock, flags);
			return clear_page_dirty_for_io(page);
		}
		write_unlock_irqrestore(&mapping->tree_lock, flags);
		return 0;
	}
	return TestClearPageDirty(page);
}
#endif

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
