/*
 * segbuf.c - NILFS segment buffer
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * segbuf.c,v 1.5 2007-07-19 07:50:49 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */

#include <linux/buffer_head.h>
#include <linux/writeback.h>
#include <linux/backing-dev.h>
#include "page.h"
#include "segbuf.h"


#define NILFS_BYTES_SECTOR(b)      ((((b) - 1) >> NILFS_SECTOR_BITS) + 1)


int nilfs_segbuf_init(struct nilfs_segment_buffer *segbuf,
		      unsigned max_blocks)
{
	unsigned long bytes;

	segbuf->chunk.nbuffers = max_blocks;
	bytes = sizeof(struct buffer_head *) * max_blocks;
	segbuf->chunk.bhs = kmalloc(bytes, GFP_NOFS);
	if (!unlikely(segbuf->chunk.bhs))
		return -ENOMEM;
	memset(segbuf->chunk.bhs, 0, bytes);
	segbuf->pbh = segbuf->chunk.bhs;
	return 0;
}

void nilfs_segbuf_destroy(struct nilfs_segment_buffer *segbuf)
{
	if (segbuf->chunk.bhs) {
		kfree(segbuf->chunk.bhs);
		segbuf->chunk.bhs = NULL;
	}
}

void nilfs_segbuf_seek(struct nilfs_segment_buffer *segbuf,
		       int from, int index)
{
	if (from == NILFS_SEGBUF_SEEK_FROM_HEAD)
		segbuf->pbh = segbuf->chunk.bhs + index;
	else if (from == NILFS_SEGBUF_SEEK_FROM_TAIL)
		segbuf->pbh = segbuf->chunk.bhs +
			segbuf->chunk.nbuffers - (index + 1);
}

/*
 * BIO operations
 */
static int
nilfs_segbuf_end_bio_write(struct bio *bio, unsigned int bytes_done,
			   int err)
{
	const int uptodate = test_bit(BIO_UPTODATE, &bio->bi_flags);
	struct nilfs_segbuf_write_info *wi;
	unsigned int blk_done;

	if (bio->bi_size)
		return 1;

	wi = bio->bi_private;

	if (err == -EOPNOTSUPP) {
		set_bit(BIO_EOPNOTSUPP, &bio->bi_flags);
		bio_put(bio);
		return 0; /* to be detected by submit_seg_bio() */
	}

	blk_done = bytes_done >> wi->sb->s_blocksize_bits;

	if (!uptodate)
		atomic_inc(&wi->err);

	seg_debug(3, "blocks written by this bio = %u, "
		  "residual blocks of this segment = %u\n",
		  blk_done, atomic_read(&wi->bio_blk_cnt));
	if (atomic_sub_and_test(blk_done, &wi->bio_blk_cnt) && wi->end_io)
		wi->end_io(wi->private, wi->segbuf, uptodate);

	bio_put(bio);
	complete(&wi->bio_event);
	return 0;
}

int nilfs_segbuf_write(struct nilfs_segbuf_write_info *wi, int mode)
{
	struct bio *bio = wi->bio;
	int err;

	if (bdi_write_congested(wi->bdi)) {
		seg_debug(3, "waiting for a segment\n");
		wait_for_completion(&wi->bio_event);
		wi->nbio--;
		if (unlikely(atomic_read(&wi->err))) {
			seg_debug(2, "detected io-error\n");
			bio_put(bio);
			err = -EIO;
			goto failed;
		}
	}

	bio->bi_end_io = nilfs_segbuf_end_bio_write;
	bio->bi_private = wi;
	seg_debug(3, "submitting bio (start_sector=%llu, size=%u, "
		  "vcnt=%hu, barrier=%d)\n",
		  (unsigned long long)bio->bi_sector,
		  bio->bi_size, bio->bi_vcnt,
		  (mode & (1 << BIO_RW_BARRIER)) != 0);

	bio_get(bio);
	submit_bio(mode, bio);
	if (bio_flagged(bio, BIO_EOPNOTSUPP)) {
		seg_debug(2, "aborted bio submission\n");
		bio_put(bio);
		err = -EOPNOTSUPP;
		goto failed;
	}
	wi->nbio++;
	bio_put(bio);

	wi->bio = NULL;
	wi->rest_blocks -= wi->end - wi->start;
	wi->nr_vecs = min(wi->max_pages, wi->rest_blocks);
	wi->start = wi->end;
	return 0;

 failed:
	wi->bio = NULL;
	return err;
}

/** 
 * alloc_seg_bio - allocate a bio for writing segment.
 * @sb: super block
 * @start: beginning disk block number of this BIO.
 * @nr_vecs: request size of page vector.
 *
 * alloc_seg_bio() allocates a new BIO structure and initialize it.
 *
 * Return Value: On success, pointer to the struct bio is returned.
 * On error, NULL is returned.
 */
static struct bio *
alloc_seg_bio(struct super_block *sb, sector_t start, int nr_vecs)
{
	struct bio *bio;

	bio = bio_alloc(GFP_NOIO, nr_vecs);
	if (bio == NULL && (current->flags & PF_MEMALLOC)) {
		while (!bio && (nr_vecs >>= 1))
			bio = bio_alloc(GFP_NOIO, nr_vecs);
	}
	if (likely(bio)) {
		bio->bi_bdev = sb->s_bdev;
		bio->bi_sector = (sector_t)start << (sb->s_blocksize_bits - NILFS_SECTOR_BITS);
		seg_debug(3, "allocated bio (max_vecs=%d)\n", bio->bi_max_vecs);
	}
	return bio;
}

/** 
 * seg_bio_add_bh - appends a buffer head to specified bio.
 * @bio: bio 
 * @bh: buffer head to be appended to the bio.
 *
 * seg_bio_add_bh() adds a buffer head @bh to a page vector of @bio.
 * It also gets the reference of the buffer and locks it.
 * When returning an error, it will do nothing to the buffer.
 *
 * Return Value: On success, 0 is returned. If the page vector is full,
 * 1 is returned.
 */
static int
seg_bio_add_bh(struct bio *bio, struct buffer_head *bh)
{
	unsigned int offset = bh_offset(bh);
	unsigned int len;
	request_queue_t *q = bdev_get_queue(bio->bi_bdev);

	BUG_ON(bio_flagged(bio, BIO_CLONED));
	/* Cloned bio must not modify io-vectors.
	   This is a check against future changes */

	if (NILFS_BYTES_SECTOR(bio->bi_size + bh->b_size) > q->max_sectors)
		return 1; /* FULL */

	if (bio->bi_vcnt > 0) {
		struct bio_vec *bvec = &bio->bi_io_vec[bio->bi_vcnt - 1];

		if (bvec->bv_page == bh->b_page &&
		    bvec->bv_offset + bvec->bv_len == offset) {
			bvec->bv_len += bh->b_size;
			goto out;
		}
	}
	len = bio_add_page(bio, bh->b_page, bh->b_size, offset);
	if (len == 0)
		return 1;
 out:
	return 0;
}

void nilfs_segbuf_prepare_write(struct nilfs_segment_buffer *segbuf,
				struct nilfs_segbuf_write_info *wi,
				sector_t start_blocknr, int nr_blocks)
{
	wi->bio = NULL;
	wi->segbuf = segbuf;
	wi->rest_blocks = nr_blocks;
	wi->max_pages = bio_get_nr_vecs(wi->sb->s_bdev);
	wi->nr_vecs = min(wi->max_pages, wi->rest_blocks);
	wi->start = wi->end = 0;
	wi->nbio = 0;
	wi->blocknr = start_blocknr;

	atomic_set(&wi->err, 0);
	atomic_set(&wi->bio_blk_cnt, nr_blocks);
	init_completion(&wi->bio_event);
}

int nilfs_segbuf_commit_bh(struct nilfs_segment_buffer *segbuf,
			   struct nilfs_segbuf_write_info *wi,
			   int mode)
{
	int res;

	BUG_ON(wi->nr_vecs <= 0);
 retry_add:
	if (!wi->bio) {
		wi->bio = alloc_seg_bio(wi->sb,
					wi->blocknr + wi->end,
					wi->nr_vecs);
		if (unlikely(!wi->bio)) {
			seg_debug(2, "failed to allocated bio\n");
			return -ENOMEM;
		}
	}
	res = seg_bio_add_bh(wi->bio, *segbuf->pbh);
	if (unlikely(res < 0)) {
		bio_put(wi->bio);
	} else if (res > 0) { /* bio is FULL */
		int err = nilfs_segbuf_write(wi, mode); /* never submit current bh */

		if (unlikely(err))
			return err;
		goto retry_add;
	} else /* res == 0 */
		wi->end++;

	return res;
}


/**
 * nilfs_segbuf_wait - wait for completion of requested BIOs
 * @wi: nilfs_segbuf_write_info
 *
 * Return Value: On Success, 0 is returned. On Error, one of the following
 * negative error code is returned.
 *
 * %-EIO - I/O error
 */
int nilfs_segbuf_wait(struct nilfs_segbuf_write_info *wi)
{
	int err = 0;

	seg_debug(3, "called nbio=%d\n", wi->nbio);
	if (!wi->nbio)
		return 0;

	do {
		wait_for_completion(&wi->bio_event);
	} while (--wi->nbio > 0);

	seg_debug(3, "wait completed\n");
	if (unlikely(atomic_read(&wi->err) > 0)) {
		printk(KERN_ERR "NILFS: IO error writing segment\n");
		err = -EIO;
	}
	return err;
}

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
