/*
 * segbuf.h - NILFS Segment buffer prototypes and definitions
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * segbuf.h,v 1.5 2007-06-17 05:30:18 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */
#ifndef _NILFS_SEGBUF_H
#define _NILFS_SEGBUF_H

#include <linux/fs.h>
#include <linux/buffer_head.h>
#include <linux/bio.h>
#include <linux/completion.h>

/*
 * Linux VFS sector size (used when submitting BIO)
 */
#define NILFS_SECTOR_BITS	9
#define NILFS_SECTOR_SIZE	(1 << SECTOR_BITS)

struct nilfs_buffer_chunk {
	struct buffer_head	**bhs;    /* buffer head array */
	unsigned		nbuffers; /* Number of buffer heads per
					     nilfs_buffer_chunk */
};

struct nilfs_segment_buffer {
	struct nilfs_buffer_chunk chunk;
	struct buffer_head	**pbh;    /* Current pointer */
};

struct nilfs_segment_buffer_ptr {
	struct nilfs_buffer_chunk *pchunk;
	struct buffer_head	**pbh;
};
typedef struct nilfs_segment_buffer_ptr nilfs_segbuf_ptr_t;

#define NILFS_SEGBUF_NULL	{NULL, NULL}
#define NILFS_SEGBUF_IS_NULL(ptr)   (!(ptr)->pchunk)
#define NILFS_SEGBUF_SIZE(segbuf)   ((segbuf)->chunk.nbuffers)
#define NILFS_SEGBUF_PTR_BH(ptr)    (*(ptr)->pbh)
#define NILFS_SEGBUF_PTR_INDEX(ptr) ((ptr)->pbh - (ptr)->pchunk->bhs)

static inline struct nilfs_segment_buffer *NILFS_SEGBUF(nilfs_segbuf_ptr_t *ptr)
{
	return container_of(ptr->pchunk, struct nilfs_segment_buffer, chunk);
}

typedef void (nilfs_segbuf_end_io_t)(void *, struct nilfs_segment_buffer *, int);

struct nilfs_segbuf_write_info {
	struct bio 		*bio;
	struct nilfs_segment_buffer *segbuf;
	int 			start, end; /* The region to be submitted */
	int			rest_blocks;
	int			max_pages;
	int			nr_vecs;
	sector_t		blocknr;

	int			nbio;
	atomic_t		err;
	atomic_t		bio_blk_cnt;
	struct completion	bio_event; /* completion event of segment write */

	/*
	 * The following fields must be set explicitly
	 */
	struct super_block      *sb;
	struct backing_dev_info *bdi; /* backing dev info */
	nilfs_segbuf_end_io_t	*end_io;
	void			*private;
};

enum {
	NILFS_SEGBUF_SEEK_NOP = 0,
	NILFS_SEGBUF_SEEK_FROM_HEAD,
	NILFS_SEGBUF_SEEK_FROM_TAIL,
};

/*
 * Operations for segment buffer
 */
extern int nilfs_segbuf_init(struct nilfs_segment_buffer *, unsigned);
extern void nilfs_segbuf_destroy(struct nilfs_segment_buffer *);

extern void nilfs_segbuf_seek(struct nilfs_segment_buffer *, int, int);

static inline struct buffer_head *
nilfs_segbuf_read_bh(struct nilfs_segment_buffer *segbuf)
{
	BUG_ON(segbuf->pbh < segbuf->chunk.bhs ||
	       segbuf->pbh >= segbuf->chunk.bhs + segbuf->chunk.nbuffers);
	return *segbuf->pbh;
}

static inline void
nilfs_segbuf_set_bh(struct nilfs_segment_buffer *segbuf, struct buffer_head *bh)
{
	BUG_ON(segbuf->pbh < segbuf->chunk.bhs ||
	       segbuf->pbh >= segbuf->chunk.bhs + segbuf->chunk.nbuffers);
	if (unlikely(*segbuf->pbh))
		__brelse(*segbuf->pbh);
	*segbuf->pbh = bh;
	get_bh(bh);
}

static inline void nilfs_segbuf_release_bh(struct nilfs_segment_buffer *segbuf)
{
	BUG_ON(segbuf->pbh < segbuf->chunk.bhs ||
	       segbuf->pbh >= segbuf->chunk.bhs + segbuf->chunk.nbuffers);
	BUG_ON(!*segbuf->pbh);
	__brelse(*segbuf->pbh);
	*segbuf->pbh = NULL;
}

static inline void nilfs_segbuf_move_next(struct nilfs_segment_buffer *segbuf)
{
	++(segbuf->pbh);
}

static inline void nilfs_segbuf_move_prev(struct nilfs_segment_buffer *segbuf)
{
	--(segbuf->pbh);
}

static inline struct buffer_head *
nilfs_segbuf_copy_page_buffers(nilfs_segbuf_ptr_t *from, int nblocks)
{
	extern struct buffer_head *nilfs_copy_page_buffers(struct buffer_head *, int);
	struct buffer_head *bufs;

	bufs = nilfs_copy_page_buffers(*from->pbh, nblocks);
	if (unlikely(!bufs))
		return NULL;
	return bufs;
}

static inline void
nilfs_segbuf_read_ptr(struct nilfs_segment_buffer *segbuf, nilfs_segbuf_ptr_t *ptr)
{
	ptr->pchunk = &segbuf->chunk;
	ptr->pbh = segbuf->pbh;
}

static inline void
nilfs_segbuf_set_ptr(struct nilfs_segment_buffer *segbuf, nilfs_segbuf_ptr_t *ptr)
{
	BUG_ON(&segbuf->chunk != ptr->pchunk);
	segbuf->pbh = ptr->pbh;
}


extern void nilfs_segbuf_prepare_write(struct nilfs_segment_buffer *,
				       struct nilfs_segbuf_write_info *,
				       sector_t, int);
extern int nilfs_segbuf_commit_bh(struct nilfs_segment_buffer *,
				  struct nilfs_segbuf_write_info *, int);
extern int nilfs_segbuf_write(struct nilfs_segbuf_write_info *, int);
extern int nilfs_segbuf_wait(struct nilfs_segbuf_write_info *);


#endif /* _NILFS_SEGBUF_H */

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
