/*
 * segment.h - NILFS Segment constructor prototypes and definitions
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * segment.h,v 1.50 2007-06-17 05:30:18 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */
#ifndef _NILFS_SEGMENT_H
#define _NILFS_SEGMENT_H

#include <linux/fs.h>
#include <linux/buffer_head.h>
#include "nilfs_fs.h"
#include "sb.h"
#include "segbuf.h"

/*
 * Recovery info
 */
struct nilfs_recovery_info {

#define NILFS_RECOVERY_SR_UPDATED	 1	/* The super root was updated */
#define NILFS_RECOVERY_ROLLFORWARD_DONE	 2	/* Rollforward was carried out */

	int			ri_need_recovery;
	sector_t		ri_super_root;	/* Block number of the last super root */
	nilfs_cno_t		ri_cno;		/* Number of the last checkpoint */

	sector_t		ri_lsegs_start;	/* region for roll-forwarding */
	sector_t		ri_lsegs_end;
	u64			ri_lsegs_start_seq; /* sequence value of the segment at lsegs_start */
	struct list_head	ri_used_segments;
	sector_t		ri_pseg_start;	/* Block number of the last partial segment */
	u64			ri_seq;		/* Sequence number on the last partial segment */
	nilfs_segnum_t		ri_segnum;	/* Segment number on the last partial segment */
	nilfs_segnum_t		ri_nextnum;	/* Next segment number on the last partial segment */
};

/*
 * Transaction info
 */
struct nilfs_transaction_info {

#define NILFS_TI_MAGIC		0xd9e392fb

	u32			ti_magic;
	void		       *ti_save;	/*
						 * Backup of journal_info field of
						 * task_struct. This should never
						 * used. If this will happen, some
						 * other filesystem has a bug.
						 */
#define NILFS_TI_DYNAMIC_ALLOC	0x0001
#define NILFS_TI_SYNC		0x0002		/* Force to construct segment
						   at the end of transaction. */
#define NILFS_TI_GC		0x0004		/* GC context */
#define NILFS_TI_COMMIT		0x0008		/* Change happened or not */

	unsigned short		ti_flags;
	unsigned short		ti_count;
	struct list_head	ti_garbage;	/* List of inode to be put when releasing
						   semaphore */
};

/*
 * vacancy_check flags for nilfs_transaction_begin()
 */
#define NILFS_TI_NOCHECK		0
#define NILFS_TI_BLOCK_ON_LOW_CAPACITY	0x01	/* Wait for GC when disk capacity is low */
#define NILFS_TI_BLOCK_ON_DISK_FULL	0x02	/* Block on a disk full condition (not supported) */

/*
 * On-memory segment summary
 */
struct nilfs_segsum_info {
	unsigned int		flags;
	unsigned long		nfinfo;		/* Number of file information structures */
	unsigned long		nblocks;	/* Number of blocks included in the partial segment */
	unsigned long		nsumblk;	/* Number of summary blocks */
	unsigned long		sumbytes;	/* Byte count of segment summary */
	unsigned long		nfileblk;	/* Total number of file blocks */
	u64			seg_seq;	/* Segment sequence number */
	time_t			ctime;		/* Creation time */
	sector_t		next;		/* Blocknr of the next full segment */
};

#define NILFS_SEG_HAS_SR(sum)    ((sum)->flags & NILFS_SS_SR)
#define NILFS_SEG_LOGBGN(sum)    ((sum)->flags & NILFS_SS_LOGBGN)
#define NILFS_SEG_LOGEND(sum)    ((sum)->flags & NILFS_SS_LOGEND)
#define NILFS_SEG_SIMPLEX(sum)   (((sum)->flags & (NILFS_SS_LOGBGN | NILFS_SS_LOGEND)) == (NILFS_SS_LOGBGN | NILFS_SS_LOGEND))
#define NILFS_SEG_DSYNC(sum)     ((sum)->flags & NILFS_SS_SYNDT)

struct nilfs_segment_list_head {
	nilfs_segnum_t		segnum;

#define NILFS_SLH_COMMIT	0x0001	/* not to be cancelled also if constrution failed */
#define NILFS_SLH_FREED		0x0002	/* Segment was freed and must be reallocated if
					   construction fails */

	unsigned		flags;
	struct list_head	list;
	struct buffer_head     *bh_su;
	struct nilfs_segment_usage *raw_su;
};

/*
 * Segment information
 */
struct nilfs_segment_info {
	struct nilfs_segsum_info sum;
	struct list_head	list;		/* List of nilfs_segment_info */
	unsigned		rest_blocks;	/* Residual blocks */
	unsigned 		submit_blocks;
	unsigned		npayload;	/* Current index on buffer head array */
	unsigned		nheader;	/* Reverse index (for summary blocks) */
	sector_t		pseg_start;	/* Disk block number of partial segment */
	struct nilfs_segment_buffer  segbuf;

	/* Full segment information */
	nilfs_segnum_t		segnum;
	nilfs_segnum_t		nextnum;
	sector_t		fseg_start, fseg_end;
	int			io_error;
};

/*
 * Context of collection
 */
struct nilfs_collection_stage {
	char			main;		/* Collection stage */
	char			sub;		/* Collection stage in a file */
	unsigned short		done;
	unsigned short		started;
	struct nilfs_inode_info *dirty_file_ptr; /* Pointer on dirty_files list, or
						    inode of a target file for data only segment */
	struct nilfs_inode_info *gc_inode_ptr;
};

/*
 * Segment constructor information
 */
struct nilfs_sc_info {
	struct super_block     *sc_super;	/* back pointer to super_block struct */

	unsigned long		sc_nblk_inc;	/* block count of current generation */

	struct list_head	sc_dirty_files;	/* list of files to be written */
	struct list_head	sc_gc_inodes;	/* list of GC inodes having blocks to be written */
	struct list_head	sc_used_active_segments; /* Used segments to be deactivated */
	struct list_head	sc_cleaning_segments; /* Segments to be freed */

	/* Segment buffers */
	struct nilfs_segment_info sc_segments;
	nilfs_segbuf_ptr_t	sc_super_root;	/* pointer to the super root block */

	struct nilfs_collection_stage sc_stage;
	struct nilfs_segment_info *sc_curseg;	/* Current (partial) segment */

	unsigned long		sc_blk_cnt;	/* data or node block count of current file */
	unsigned long		sc_datablk_cnt;	/* data block count */
	unsigned long		sc_finfo_offset;  /* current finfo offset in segment summary */
	unsigned long		sc_binfo_offset;  /* current binfo offset in segment summary */
	unsigned long		sc_nblk_this_inc;
	time_t			sc_seg_ctime;

	unsigned long		sc_flags;	/* Internal flags defined below */

	/*
	 * Pointer to an inode of the sketch.
	 * This pointer is kept only while it contains data.
	 * We protect it with a semaphore of the segment constructor.
	 */
	struct inode	       *sc_sketch_inode;

	spinlock_t		sc_state_lock;

#define NILFS_SEGCTOR_QUIT		0x0001	/* segctord is being destroyed */
#define NILFS_SEGCTOR_INIT		0x0002	/* segctord is being started */
#define NILFS_SEGCTOR_COMMIT		0x0004	/* at least one transaction has been committed */
#define NILFS_SEGCTOR_FLUSH_DATA	0x0010
#define NILFS_SEGCTOR_FLUSH_IFILE	0x0020
#define NILFS_SEGCTOR_FLUSH  \
	  (NILFS_SEGCTOR_FLUSH_DATA | NILFS_SEGCTOR_FLUSH_IFILE)

	unsigned long		sc_state;	/* segctord state flags */

	wait_queue_head_t	sc_wait_request;/* client request queue */
	wait_queue_head_t	sc_wait_daemon;	/* daemon wait queue */
	wait_queue_head_t	sc_wait_task;	/* start/end wait queue */

	__u32			sc_seq_request;	/* request counter to thread */
	__u32			sc_seq_done;	/* completion counter of thread */

	int			sc_sync;	/* Request of explicit sync operation */
	unsigned long		sc_interval;	/* Timeout value of segment construction */
	unsigned long		sc_mjcp_freq;	/* Frequency of creating checkpoints. */
	unsigned long		sc_lseg_stime;	/* The time when the latest logical segment
						   started to construct (in 1/HZ seconds) */
	struct timer_list      *sc_timer;

	unsigned long		sc_nr_dirty;	/* Number of dirty data buffers (in block) */
	unsigned long		sc_watermark;	/* Watermark of dirty buffers */

	struct task_struct     *sc_task;	/* current constructor thread */
};

/* sc_flags */
enum {
	NILFS_SC_DIRTY,		/* One or more dirty meta-data blocks exist */
	NILFS_SC_UNCLOSED,	/* Logical segment is not closed */
	NILFS_SC_SUPER_ROOT,	/* The latest segment has a super root */
	NILFS_SC_GC_COPY,	/* Copying GC blocks */
};

/*
 * Constant parameters
 */
#define NILFS_SC_CLEANUP_RETRY		3	/* Retry count of construction when
						   destroying the segment constructor. */

/*
 * Default values of timeout, in seconds.
 */
#define NILFS_SC_DEFAULT_TIMEOUT	5	/* Timeout value of buffered dirty blocks.
						   This timeout triggers construction of a
						   logical segment having a super root. */
#define NILFS_SC_DEFAULT_SR_FREQ	30	/* Maximum frequency of super root creation */
#define NILFS_SC_DEFAULT_SB_FREQ	30	/* Minimum interval of periodical update of
						   superblock (reserved) */

/*
 * The default threshold amount of data, in block counts.
 */
#define NILFS_SC_DEFAULT_WATERMARK	3600


/* segment.c */
extern int nilfs_init_transaction_cache(void);
extern void nilfs_destroy_transaction_cache(void);
extern int nilfs_transaction_begin(struct super_block *, struct nilfs_transaction_info *, int);
extern int nilfs_transaction_end(struct super_block *, int);

extern int nilfs_set_file_dirty(struct nilfs_sb_info *, struct inode *);
extern int nilfs_commit_dirty_file(struct inode *);
extern void nilfs_dirty_inode(struct inode *);

extern int nilfs_construct_segment(struct super_block *);
extern int nilfs_construct_dsync_segment(struct super_block *, struct inode *);
extern void nilfs_flush_segment(struct nilfs_sb_info *, ino_t);
extern int nilfs_clean_segments(struct super_block *, unsigned long);

extern void nilfs_segctor_add_dirty(struct nilfs_sc_info *, unsigned);
extern int nilfs_segctor_add_segments_to_be_freed(struct nilfs_sc_info *, nilfs_segnum_t *, size_t);
extern void nilfs_segctor_clear_segments_to_be_freed(struct nilfs_sc_info *);

extern int nilfs_attach_segment_constructor(struct nilfs_sb_info *, struct nilfs_recovery_info *);
extern void nilfs_detach_segment_constructor(struct nilfs_sb_info *);

/* recovery.c */
extern int nilfs_read_super_root_block(struct super_block *, sector_t, struct buffer_head **, int);
extern void nilfs_dispose_segment_list(struct list_head *);
extern int nilfs_search_super_root(struct the_nilfs *, struct nilfs_sb_info *, struct nilfs_recovery_info *);
extern int nilfs_recover_logical_segments(struct the_nilfs *, struct nilfs_sb_info *, struct nilfs_recovery_info *);


static inline struct nilfs_sc_info *NILFS_SC(struct nilfs_sb_info *sbi)
{
	return sbi->s_sc_info;
}

static inline void nilfs_init_recovery_info(struct nilfs_recovery_info *ri)
{
	memset(ri, 0, sizeof(*ri));
	INIT_LIST_HEAD(&ri->ri_used_segments);
}

static inline void nilfs_clear_recovery_info(struct nilfs_recovery_info *ri)
{
	nilfs_dispose_segment_list(&ri->ri_used_segments);
}


static inline void nilfs_set_transaction_flag(unsigned int flag)
{
	struct nilfs_transaction_info *ti = current->journal_info;

	BUG_ON(!ti);
	ti->ti_flags |= flag;
}

#if 0
static inline void nilfs_clear_transaction_flag(unsigned int flag)
{
	struct nilfs_transaction_info *ti = current->journal_info;

	BUG_ON(!ti);
	ti->ti_flags &= ~flag;
}
#endif

static inline int nilfs_test_transaction_flag(unsigned int flag)
{
	struct nilfs_transaction_info *ti = current->journal_info;

	if (ti == NULL || ti->ti_magic != NILFS_TI_MAGIC)
		return 0;
	return !!(ti->ti_flags & flag);
}

#endif /* _NILFS_SEGMENT_H */

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
