/*
 * the_nilfs.h - the_nilfs shared structure.
 *
 * Copyright (C) 2005-2007 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * the_nilfs.h,v 1.22 2007-06-17 05:28:45 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */

#ifndef _THE_NILFS_H
#define _THE_NILFS_H

#include "nilfs_types.h"

/* the_nilfs struct */
enum {
	THE_NILFS_INIT = 0,     /* Information from super_block is set */
	THE_NILFS_LOADED,       /* Roll-back/roll-forward has done and
				   the latest checkpoint was loaded */
	THE_NILFS_LOCKING_WRITES,/* Locking normal write operations */
	THE_NILFS_DISK_FULL,	/* Cause -ENOSPC failure on writes */
	THE_NILFS_DISCONTINUED,	/* Segment chain by the 'next' pointer has broken */
	THE_NILFS_COND_WRITE_LOCKED,	/* Condition to wake up cleanerd */
	THE_NILFS_COND_NONGC_WRITE,	/* Condition to wake up cleanerd */
};

struct the_nilfs {
	unsigned long		ns_flags;
        atomic_t		ns_count;

	struct block_device    *ns_bdev;
	struct backing_dev_info *ns_bdi;	/* backing dev info */
	struct nilfs_sb_info   *ns_writer;	/* back pointer to the writable nilfs_sb_info
						   (protected by ns_writer_sem) */
	struct rw_semaphore	ns_sem;		/* Semaphore for shared states */
	struct semaphore	ns_writer_sem;	/* Semaphore for writer attach/detach */
	atomic_t		ns_writer_sem_count; /* Count referrers of writer */

        /* 
	 * used for
	 * - loading the latest checkpoint exclusively.
	 * - allocating a new full segment.
	 * - protecting s_dirt in the struct super_block (see nilfs_write_super)
	 *   and the following fields.
	 */
	struct buffer_head     *ns_sbh;		/* Buffer containing the super block */
	struct nilfs_super_block *ns_sbp;	/* Pointer to the disk super block
						   in the buffer */
	struct list_head	ns_used_segments;  /* Full segments in volatile active state */
	unsigned		ns_mount_state;
	struct list_head	ns_supers;	/* List of nilfs supers */

	/*
	 * Following fields are dedicated to a writable FS-instance.
	 * Except for the period seeking checkpoint, code outside the segment constructor
	 * must lock a segment semaphore with transaction_begin()/transaction_end(),
	 * when accessing these fields.
	 * The writable FS-instance is sole during a lifetime of the_nilfs.
	 */
	u64			ns_seg_seq;	/* Segment sequence counter */
	nilfs_segnum_t		ns_segnum;	/* Full segment index last used */
	nilfs_segnum_t		ns_nextnum;	/* Full segment index to be used next */
	unsigned long		ns_pseg_offset;	/* Offset of next partial segment
						   in the current full segment */
	nilfs_cno_t		ns_cno;		/* Next checkpoint number */
	time_t			ns_ctime;	/* Write time of the last segment */
	time_t			ns_nongc_ctime; /* Write time of the last segment not for
						   cleaner operation */

	/*
	 * The following fields hold information on the latest partial segment written to disk
	 * with a super root;  ns_last_segment_lock protects them.
	 */
	spinlock_t		ns_last_segment_lock;
	sector_t		ns_last_pseg;	/* Start block number of the latest segment */
	u64			ns_last_seq;	/* Sequence value of the latest segment */
	nilfs_cno_t		ns_last_cno;	/* Checkpoint number of the latest segment */
	unsigned long		ns_free_segments_count;

	/* Segment constructor semaphore */
	struct rw_semaphore	ns_segctor_sem;

	/*
	 * Following fields are lock free except for the period before the_nilfs
	 * is initialized.
	 */
	struct inode	       *ns_dat;		/* DAT file inode */
	struct inode	       *ns_cpfile;	/* checkpoint file inode */
	struct inode	       *ns_sufile;	/* segusage file inode */
	struct inode	       *ns_gc_dat;	/* DAT file inode for GC */

	/* GC inode list and hash table head */
	spinlock_t		ns_gc_inode_lock; /* lock list/hlist operations */
	struct list_head	ns_gc_inodes;	/* keep live blocks */
	struct hlist_head      *ns_gc_inodes_h;	/* keep live blocks in hash list */

	/* cleanerd */
	wait_queue_head_t	ns_cleanerd_wq;

	/* Disk layout information (static) */
	unsigned int		ns_blocksize_bits;
	unsigned long		ns_nsegments;	/* Number of segments in filesystem */
	unsigned long		ns_blocks_per_segment; /* Number of blocks per segment */
	unsigned long		ns_r_segments_percentage;/* Reserved segments percentage */
	unsigned long		ns_first_data_block;
	int			ns_inode_size;
	int			ns_first_ino;
	u32			ns_crc_seed;	/* Seed value of CRC32 calculation */
};

#define NILFS_GCINODE_HASH_BITS		8	/* allocate hash list head at alloc_nilfs() */
#define NILFS_GCINODE_HASH_SIZE		(1<<NILFS_GCINODE_HASH_BITS)


#define THE_NILFS_FNS(bit, name)					\
static inline void set_nilfs_##name(struct the_nilfs *nilfs)		\
{									\
	set_bit(THE_NILFS_##bit, &(nilfs)->ns_flags);			\
}									\
static inline void clear_nilfs_##name(struct the_nilfs *nilfs)		\
{									\
	clear_bit(THE_NILFS_##bit, &(nilfs)->ns_flags);			\
}									\
static inline int nilfs_##name(struct the_nilfs *nilfs)			\
{									\
	return test_bit(THE_NILFS_##bit, &(nilfs)->ns_flags);		\
}

THE_NILFS_FNS(INIT, init)
THE_NILFS_FNS(LOADED, loaded)
THE_NILFS_FNS(LOCKING_WRITES, write_locked)
THE_NILFS_FNS(DISK_FULL, disk_full)
THE_NILFS_FNS(DISCONTINUED, discontinued)
THE_NILFS_FNS(COND_WRITE_LOCKED, cond_write_locked)
THE_NILFS_FNS(COND_NONGC_WRITE, cond_nongc_write)

extern void nilfs_set_last_segment(struct the_nilfs *, sector_t, u64, nilfs_cno_t);
extern struct the_nilfs *alloc_nilfs(struct block_device *);
extern void put_nilfs(struct the_nilfs *);
extern int init_nilfs(struct the_nilfs *, struct nilfs_sb_info *, char *);
extern int load_nilfs(struct the_nilfs *, struct nilfs_sb_info *);
extern int nilfs_count_free_blocks(struct the_nilfs *, sector_t *);
extern void nilfs_dispose_used_segments(struct the_nilfs *);
extern int nilfs_wait_bit_interruptible(void *);
extern int nilfs_wait_bit_uninterruptible(void *);
extern void FASTCALL(nilfs_lock_write(struct the_nilfs *));
extern void FASTCALL(nilfs_unlock_write(struct the_nilfs *));
extern int nilfs_checkpoint_is_mounted(struct the_nilfs *, nilfs_cno_t, int);


static inline void get_nilfs(struct the_nilfs *nilfs)
{
	/* Caller must have at least one reference of the_nilfs. */
	atomic_inc(&nilfs->ns_count);
}

static inline struct nilfs_sb_info *nilfs_get_writer(struct the_nilfs *nilfs)
{
	if (atomic_inc_and_test(&nilfs->ns_writer_sem_count))
		down(&nilfs->ns_writer_sem);
	return nilfs->ns_writer;
}

static inline void nilfs_put_writer(struct the_nilfs *nilfs)
{
	if (atomic_add_negative(-1, &nilfs->ns_writer_sem_count))
		up(&nilfs->ns_writer_sem);
}

static inline void
nilfs_attach_writer(struct the_nilfs *nilfs, struct nilfs_sb_info *sbi)
{
	down(&nilfs->ns_writer_sem);
	nilfs->ns_writer = sbi;
	up(&nilfs->ns_writer_sem);
}

static inline void
nilfs_detach_writer(struct the_nilfs *nilfs, struct nilfs_sb_info *sbi)
{
	down(&nilfs->ns_writer_sem);
	if (sbi == nilfs->ns_writer)
		nilfs->ns_writer = NULL;
	up(&nilfs->ns_writer_sem);
}

static inline void nilfs_wait_on_write(struct the_nilfs *nilfs)
{
	wait_on_bit(&(nilfs)->ns_flags, THE_NILFS_LOCKING_WRITES,
		    nilfs_wait_bit_uninterruptible, TASK_UNINTERRUPTIBLE);
}

static inline int nilfs_wait_on_write_interruptible(struct the_nilfs *nilfs)
{
	return wait_on_bit(&(nilfs)->ns_flags, THE_NILFS_LOCKING_WRITES,
			   nilfs_wait_bit_interruptible, TASK_INTERRUPTIBLE);
}

static inline void
nilfs_get_segment_range(struct the_nilfs *nilfs, nilfs_segnum_t segnum,
			sector_t *seg_start, sector_t *seg_end)
{
	*seg_start = (sector_t)nilfs->ns_blocks_per_segment * segnum;
	*seg_end = *seg_start + nilfs->ns_blocks_per_segment - 1;
	if (segnum == 0)
		*seg_start = nilfs->ns_first_data_block;
}

static inline sector_t
nilfs_get_segment_start_blocknr(struct the_nilfs *nilfs, nilfs_segnum_t segnum)
			
{
	return (segnum == 0) ? nilfs->ns_first_data_block :
		(sector_t)nilfs->ns_blocks_per_segment * segnum;
}

static inline nilfs_segnum_t
nilfs_get_segnum_of_block(struct the_nilfs *nilfs, sector_t blocknr)
{
	sector_t segnum = blocknr;

	sector_div(segnum, nilfs->ns_blocks_per_segment);
	return segnum;
}

static inline void
nilfs_terminate_segment(struct the_nilfs *nilfs, sector_t seg_start, sector_t seg_end)
{
	nilfs->ns_pseg_offset = seg_end - seg_start + 1;
}

static inline nilfs_cno_t nilfs_last_cno(struct the_nilfs *nilfs)
{
	nilfs_cno_t cno;

	spin_lock(&nilfs->ns_last_segment_lock);
	cno = nilfs->ns_last_cno;
	spin_unlock(&nilfs->ns_last_segment_lock);
	return cno;
}


#endif /* _THE_NILFS_H */

/* Local Variables:		*/
/* eval: (c-set-style "linux")	*/
/* End:				*/
