#include "configutils.h"

#include <qapplication.h>
#include <qprocess.h>
#include <qobject.h>
#include <qfile.h>
#include <qdir.h>
#include <qregexp.h>
#ifdef Q_WS_WIN
#include <windows.h>
#endif
#ifdef Q_OS_UNIX
#include <stdlib.h>
#endif

QString *qtDir = 0;
int processes = 0;
int errors = 0;
int warnings = 0;
bool autoShutdown = TRUE;

static QStringList static_messages;

void message( const QString &str )
{
    static_messages.append( str );
}

QStringList messages()
{
    return static_messages;
}

class BlockingProcess : public QProcess
{
    Q_OBJECT

public:
    BlockingProcess()
    {
	connect(this, SIGNAL(readyReadStdout()), this, SLOT(readOut()));
	connect(this, SIGNAL(readyReadStderr()), this, SLOT(readErr()));
	connect(this, SIGNAL(processExited()), this, SLOT(exited()));
	outUsed = errUsed = 0;
    }

public slots:
    void readOut()
    {
 	QByteArray rout = readStdout();
	if (outUsed + rout.size() > out.size())
	    out.resize(outUsed + rout.size());
	memcpy(out.data() + outUsed, rout, rout.size());
	outUsed += rout.size();
    }

    void readErr()
    {
 	QByteArray rerr = readStderr();
	if (errUsed + rerr.size() > err.size())
	    err.resize(errUsed + rerr.size());
	memcpy(err.data() + errUsed, rerr, rerr.size());
	errUsed += rerr.size();
    }

    void exited()
    {
	qApp->exit_loop();
    }

public:
    QByteArray out;
    QByteArray err;
    int outUsed;
    int errUsed;
};


static bool execute( const QStringList &args )
{
    BlockingProcess bp;
    bp.setArguments(args);
    if (!bp.start())
	return FALSE;
    qApp->enter_loop();
    return !bp.exitStatus() && bp.normalExit();
}

static void runQMake(const QString &d, const QStringList &configs, const QString &prefix,
                     const QString &target)
{
    QDir dir(d);
    QString runDir = dir.absPath();
    dir.cdUp();
    QString oldDir = dir.absPath();
    QDir::setCurrent(runDir);
    // make the top level Makefile
    QStringList args;
    args.append( "qmake" );
    if ( !configs.isEmpty() )
	args.append( "CONFIG+=" + configs.join( " " ) );
    if ( !prefix.isEmpty() )
	args.append( "QSA_INSTALL_PREFIX=" + prefix );
    if (!target.isNull()) {
        args.append("-o");
        args.append(target);
    }
    if( !execute( args ) )
	warnings++;
    QDir::setCurrent(oldDir);
}


void runQMake( const QStringList &configs, const QString &prefix )
{
    runQMake(".", configs, prefix, "Makefile.qsa");
}

void mkDir( const QString &dir )
{
    QDir current;
    current.mkdir( dir );
}


void copy( const QString &source, const QString &dest )
{
    QString s = QDir::convertSeparators( source );
    QString d = QDir::convertSeparators( dest );
#ifdef Q_OS_UNIX
    system( "cp " + QFile::encodeName( s ) + " " + QFile::encodeName( d ) );
    system( "chmod +w " + QFile::encodeName( d ) );
#else
    QT_WA(
    {
	if ( !CopyFileW( (TCHAR*) s.ucs2(), (TCHAR*) d.ucs2(), FALSE ) ) {
	    message( "Failed to copy file: " + s );
	    errors++;
	}
	if ( !SetFileAttributesW( (TCHAR*) d.ucs2(), FILE_ATTRIBUTE_NORMAL ) ) {
	    message( "Failed to set file attributes to normal" );
	    errors++;
	}
    }, {
	if ( !CopyFileA( QFile::encodeName( s ), QFile::encodeName( d ), FALSE ) ) {
	    message( "Failed to copy file: " + s );
	    errors++;
	}
	if ( !SetFileAttributesA( QFile::encodeName( d ), FILE_ATTRIBUTE_NORMAL ) ) {
	    message( "Failed to set file attributes to normal" );
	    errors++;
	}
    } );

#endif
}


void symLink( const QString &source, const QString &dest )
{
#ifdef Q_OS_UNIX
    QString s = QDir::convertSeparators( source );
    QFileInfo info( s );
    s = info.absFilePath();
    QString d = QDir::convertSeparators( dest );
    system( "rm -f " + QFile::encodeName( d ) );
    system( "ln -s " + QFile::encodeName( s ) + " " + QFile::encodeName( d ) );
#else
    copy( source, dest );
#endif
}

bool writeQSConfig( bool buildIde )
{
    QFile file( "src/qsa/qsconfig.h" );
    if( !file.open( IO_WriteOnly ) ) {
	message( "Failed to open 'src/qsa/qsconfig.h' for writing." );
	return FALSE;
    }
    QTextStream txt( &file );
    txt << "// This file is autogenerated by QSA configure, do not modify it!\n"
	<< "#ifndef QS_CONFIG_H\n"
	<< "#define QS_CONFIG_H\n"
	<< "\n";
    if( !buildIde )
	txt << "#define QSA_NO_IDE\n";
    txt << "\n"
	<< "#endif\n";
    return TRUE;
}

void rmDirRecursive( const QDir &dir )
{
    const QFileInfoList* list = dir.entryInfoList( QDir::All | QDir::System | QDir::Hidden );
    if ( list ) {
	QFileInfoListIterator it( *list );
	QFileInfo* fi;

	while( ( fi = it.current() ) ) {
	    if( ( fi->fileName() != "." ) && ( fi->fileName() != ".." ) ){
		if( fi->isDir() )
		    rmDirRecursive( QDir(fi->absFilePath()) );
		else
		    QFile::remove( fi->absFilePath() );
	    }
	    ++it;
	}
    }
    // Remove this dir as well
    dir.rmdir( dir.absPath() );
}


#include "configutils.moc"

