#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "getHome.h"
#include "ppaSetBase.h"

// We don't actually are pnm2ppa.c, but we need to tell defaults.h that we are
// in order to let defaults to be set.
#define __PNM2PPA_C__
#include "defaults.h"
#undef __PNM2PPA_C__

#define PNM2PPA_CONF "/etc/pnm2ppa.conf"
#define MAXLINELEN 240

void defaultSettingsForPrinter(ppaSet_t *settings, int i){
  if ( (i<0) || (i>2) )
    return;
  // Some of them are set to 0, because the command line
  // are ADDED to the default ones.
  //  settings->xOffset = gPrinterDefaults[i].x_offset;
  //  settings->yOffset = gPrinterDefaults[i].y_offset;
  settings->xOffset = 0;
  settings->yOffset = 0;
  settings->topMargin = gPrinterDefaults[i].top_margin;
  settings->bottomMargin = gPrinterDefaults[i].bottom_margin;
  settings->leftMargin = gPrinterDefaults[i].left_margin;
  settings->rightMargin = gPrinterDefaults[i].right_margin;
  //  settings->bwShearing = gPrinterDefaults[i].r2l_bw_offset;
  //  settings->colorShearing = gPrinterDefaults[i].r2l_col_offset;
  settings->bwShearing = 0;
  settings->colorShearing = 0;
  //  settings->xOffsetColor = gPrinterDefaults[i].ColBwOffsX;
  //  settings->yOffsetColor = gPrinterDefaults[i].ColBwOffsY;
  settings->xOffsetColor = 0;
  settings->yOffsetColor = 0;
}

int getSystemDefaults(ppaSet_t *settings){
  char line[MAXLINELEN];
  char variable[MAXLINELEN];
  char value[MAXLINELEN];
  FILE *confFile;
  int i=0;
  char commented = 0;

  /* We first fill hard-coded defaults: 720 series and its associated values */
  settings->type = TYPE_HP720;
  defaultSettingsForPrinter(settings, 0);
  /* Let's add a few hard-coded defaults */			    
  settings->color = COLOR_COLOR;
  settings->dither = DITHER_FS;
  settings->mode = MODE_NORMAL;
  settings->paper = PAPER_A4;
  settings->biUni = BIUNI_BI;
  settings->blackness = 3;
  settings->gammaMode = GAMMA_DEFAULT;
  settings->gammaEnhR = 1.0;
  settings->gammaEnhG = 1.0;
  settings->gammaEnhB = 1.0;
  settings->gammaIndexR = 0;
  settings->gammaIndexG = 0;
  settings->gammaIndexB = 0;
  strncpy(settings->gammaFile,"/etc/pnm2ppa.gamma",GAMMA_FILE_PATH_LEN);

  /* Now, lets parse the system configuration file */
  if ( (confFile = fopen(PNM2PPA_CONF, "r")) == NULL){
    perror("ppaSetBase");
    return -1;
  }

  while (1) { // We will use "break"
    if (fgets(line, MAXLINELEN, confFile) == NULL){
      /* This is an EOF, so exit*/
      break;
    }
    /* Let's remove comments, if available */
    commented = 0;
    for(i=0; i<MAXLINELEN; i++){
      if (commented == 1)
	line[i] = 0;
      else if (line[i] == '#'){
	line[i] = 0;
	commented = 1;
      }
      else if (line[i] == '\n'){
	line[i] = 0;
	commented = 1; // Fine... it's not a comment, but...
      }
    }
    /* If the resulting is a blank line, let's do nothing */
    if (line[0] == 0)
      continue;

    /* Finally, lets look for VARIABLE VALUE pairs */
    if ( (sscanf(line, "%s %s", variable, value)) != 2){
      fprintf(stderr, "ppaSetBase: couldn't match a pair in %s... I will try to continue\n", line);
    }
    if (!strncmp(variable,"version",7)) {
      if (!strncmp(value, "ERROR", 5)) {
	fprintf(stderr, "ppaSetBase: you administrator must select a printer version in /etc/pnm2ppa.conf\n");
	return -1;
      } else if(!strncmp(value, "710", 3)) {
	settings -> type = TYPE_HP710;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "712", 3)) {
	settings -> type = TYPE_HP712;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "720", 3)) {
	settings -> type = TYPE_HP720;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "722", 3)) {
	settings -> type = TYPE_HP722;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "820", 3)) {
	settings -> type = TYPE_HP820;
	defaultSettingsForPrinter(settings, 1);
      } else if(!strncmp(value, "1000", 4)) {
	settings -> type = TYPE_HP1000;
	defaultSettingsForPrinter(settings, 2);
      } else {
	fprintf(stderr, "ppaSetBase: unrecognized value %s for variable %s... I will try to continue\n", value, variable);
      }
    } else if (!strncmp(variable,"silent",6)) {
      // Ignore it, it's used in pnm2ppa
    } else if (!strncmp(variable,"verbose",7)) {
      // Ignore it, it's used in pnm2ppa
    } else if (!strncmp(variable,"topmargin",9)) {
      settings -> topMargin = atoi(value);
    } else if (!strncmp(variable,"bottommargin",12)) {
      settings -> bottomMargin = atoi(value);
    } else if (!strncmp(variable,"leftmargin",10)) {
      settings -> leftMargin = atoi(value);
    } else if (!strncmp(variable,"rightmargin",11)) {
      settings -> rightMargin = atoi(value);
    } else if (!strncmp(variable,"xoffset",7)) {
      settings -> xOffset = atoi(value);
    } else if (!strncmp(variable,"yoffset",7)) {
      settings -> yOffset = atoi(value);
    } else if (!strncmp(variable,"ColOffsX",8)) {
      settings -> xOffsetColor = atoi(value);
    } else if (!strncmp(variable,"ColOffsY",8)) {
      settings -> yOffsetColor = atoi(value);
    } else if (!strncmp(variable,"colorshear",10)) {
      settings -> colorShearing = atoi(value);
    } else if (!strncmp(variable,"blackshear",10)) {
      settings -> bwShearing = atoi(value);
    } else if (!strncmp(variable,"blackness",9)) {
      settings -> blackness = atoi(value);
    } else if (!strncmp(variable,"gammaR",6)) {
      settings -> gammaMode = GAMMA_ENH;
      settings -> gammaEnhR = atof(value);
    } else if (!strncmp(variable,"gammaG",6)) {
      settings -> gammaMode = GAMMA_ENH;
      settings -> gammaEnhG = atof(value);
    } else if (!strncmp(variable,"gammaB",6)) {
      settings -> gammaMode = GAMMA_ENH;
      settings -> gammaEnhB = atof(value);
    } else if (!strncmp(variable,"RedGammaIdx",11)) {
      settings -> gammaMode = GAMMA_INDEX;
      settings -> gammaIndexR = atof(value);
    } else if (!strncmp(variable,"GreenGammaIdx",13)) {
      settings -> gammaMode = GAMMA_INDEX;
      settings -> gammaIndexG = atof(value);
    } else if (!strncmp(variable,"BlueGammaIdx",12)) {
      settings -> gammaMode = GAMMA_INDEX;
      settings -> gammaIndexB = atof(value);
    } else if (!strncmp(variable,"papersize",9)) {
      if (!strncmp(value, "letter", 6)){
	settings -> paper = PAPER_LETTER;
      } else if (!strncmp(value, "legal", 5)){
	settings -> paper = PAPER_LEGAL;
      } else if (!strncmp(value, "a4", 2)){
	settings -> paper = PAPER_A4;
      } else if (!strncmp(value, "a3", 2)){
	// This value is not documented in /etc/pnm2ppa.conf, but in pnm2ppa man page
	settings -> paper = PAPER_A3;
      } else {
	fprintf(stderr, "ppaSetBase: unrecognized value %s for variable %s... I will try to continue\n", value, variable);
      }
    } else if (!strncmp(variable,"unimode",7)) {
      if (!strncmp(value, "1", 1)){
	settings -> biUni = BIUNI_UNI;
      }
    } else if (!strncmp(variable,"black_ink",9)) {
      // Debug only, forget it.
    } else if (!strncmp(variable,"color_ink",9)) {
      // Debug only, forget it.
    } else if (!strncmp(variable,"cyan_ink",8)) {
      // Debug only, forget it.
    } else if (!strncmp(variable,"magenta_ink",11)) {
      // Debug only, forget it.
    } else if (!strncmp(variable,"yellow_ink",10)) {
      // Debug only, forget it.
    } else {
      fprintf(stderr, "ppaSetBase: unrecognized variable %s... I will try to continue\n", variable);
    }
  }
  return 0;
}

void settings2RC_conf(FILE *output, ppaSet_t settings){
  fprintf(output,"#File generated by ppaSetBase (ppaSet)\n");
  fprintf(output,"#Do not edit unless you know what you are doing\n");

  if (settings.type == TYPE_HP710){
    fprintf(output, "version 710\n");
  } else if (settings.type == TYPE_HP712){
    fprintf(output, "version 712\n");
  } else if (settings.type == TYPE_HP720){
    fprintf(output, "version 720\n");
  } else if (settings.type == TYPE_HP722){
    fprintf(output, "version 722\n");
  } else if (settings.type == TYPE_HP820){
    fprintf(output, "version 820\n");
  } else if (settings.type == TYPE_HP1000){
    fprintf(output, "version 1000\n");
  }
  fprintf(output, "blackshear %i\n", settings.bwShearing);
  fprintf(output, "colorshear %i\n", settings.colorShearing);
  fprintf(output, "ColOffsX %i\n", settings.xOffsetColor);
  fprintf(output, "ColOffsY %i\n", settings.yOffsetColor);

  if (settings.gammaMode == GAMMA_DEFAULT){
  } else if (settings.gammaMode == GAMMA_NONE){
  } else if (settings.gammaMode == GAMMA_ENH){
    fprintf(output, "GammaR %f\n", settings.gammaEnhR);
    fprintf(output, "GammaG %f\n", settings.gammaEnhG);
    fprintf(output, "GammaB %f\n", settings.gammaEnhB);
  } else if (settings.gammaMode == GAMMA_INDEX){
    fprintf(output, "RedGammaIdx %i\n", settings.gammaIndexR);
    fprintf(output, "GreenGammaIdx %i\n", settings.gammaIndexG);
    fprintf(output, "BlueGammaIdx %i\n", settings.gammaIndexB);
  } else if (settings.gammaMode == GAMMA_FILE){
  }
}

void settings2RC(FILE *output, ppaSet_t settings){
  fprintf(output,"#File generated by ppaSetBase (ppaSet)\n");
  fprintf(output,"#Do not edit unless you know what you are doing\n");

  if (settings.type == TYPE_HP710){
    fprintf(output, "TYPE=710\n");
  } else if (settings.type == TYPE_HP712){
    fprintf(output, "TYPE=712\n");
  } else if (settings.type == TYPE_HP720){
    fprintf(output, "TYPE=720\n");
  } else if (settings.type == TYPE_HP722){
    fprintf(output, "TYPE=722\n");
  } else if (settings.type == TYPE_HP820){
    fprintf(output, "TYPE=820\n");
  } else if (settings.type == TYPE_HP1000){
    fprintf(output, "TYPE=1000\n");
  }
  if (settings.color == COLOR_COLOR ) {
    fprintf(output, "COLOR=ppmraw\n");
  } else if (settings.color == COLOR_GREYSCALE ) {
    fprintf(output, "COLOR=pgmraw\n");
  } else if (settings.color == COLOR_BLACKWHITE ) {
    fprintf(output, "COLOR=pbmraw\n");
  }
  if (settings.dither == DITHER_FS){
    fprintf(output, "DITHER=\n");
  } else if (settings.dither == DITHER_FAST){
    fprintf(output, "DITHER=--fd\n");
  }
  if (settings.mode == MODE_ECO){
    fprintf(output, "MODE=--eco\n");
  } else if (settings.mode == MODE_NORMAL){
    fprintf(output, "MODE=\n");
  } else if (settings.mode == MODE_PHOTO){
    fprintf(output, "MODE=-p\n");
  }
  if (settings.paper == PAPER_A4){
    fprintf(output, "PAPER=a4\n");
  } else if (settings.paper == PAPER_A3){
    fprintf(output, "PAPER=a3\n");
  } else if (settings.paper == PAPER_LEGAL){
    fprintf(output, "PAPER=legal\n");
  } else if (settings.paper == PAPER_LETTER){
    fprintf(output, "PAPER=letter\n");
  }
  fprintf(output, "XOFFSET=%i\n", settings.xOffset);
  fprintf(output, "YOFFSET=%i\n", settings.yOffset);
  fprintf(output, "TOPMARGIN=%i\n", settings.topMargin);
  fprintf(output, "BOTTOMMARGIN=%i\n", settings.bottomMargin);
  fprintf(output, "LEFTMARGIN=%i\n", settings.leftMargin);
  fprintf(output, "RIGHTMARGIN=%i\n", settings.rightMargin);
  if (settings.biUni == BIUNI_BI){
    fprintf(output, "BIUNI=--bi\n");
  } else if (settings.biUni == BIUNI_UNI){
    fprintf(output, "BIUNI=--uni\n");
  }
  fprintf(output, "BLACKNESS=%i\n", settings.blackness);
  if (settings.gammaMode == GAMMA_DEFAULT){
    fprintf(output, "GAMMA=\n");
  } else if (settings.gammaMode == GAMMA_NONE){
    fprintf(output, "GAMMA=--noGamma\n");
  } else if (settings.gammaMode == GAMMA_ENH){
    fprintf(output, "GAMMA=\n");    
  } else if (settings.gammaMode == GAMMA_INDEX){
    fprintf(output, "GAMMA=\n");
  } else if (settings.gammaMode == GAMMA_FILE){
    fprintf(output, "GAMMA=-F %s\n", settings.gammaFile);
  } else if (settings.gammaMode == GAMMA_CAL){
    fprintf(output, "GAMMA=\"--nogamma -g\"\n");
  }
}

int createPpaSetRC(ppaSet_t settings){
  FILE *rcFile;
  FILE *rcConfFile;
  char *home;
  char rcString[MAXPATHLEN];
  char rcConfString[MAXPATHLEN];

  if ( (home = getenv("HOME")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't get HOME enviroment variable\n");
    return -1;
  }
  strncpy(rcString, home, MAXPATHLEN);
  strcat(rcString, "/.ppaSetRC");
  strncpy(rcConfString, home, MAXPATHLEN);
  strcat(rcConfString, "/.ppaSetRC.conf");

  if ( (rcFile = fopen(rcString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcString);
    return -1;
  }
  if ( (rcConfFile = fopen(rcConfString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcConfString);
    return -1;
  }
  settings2RC(rcFile, settings);
  fclose(rcFile);
  settings2RC_conf(rcConfFile, settings);
  fclose(rcConfFile);
  return 0;
}


int createPpaSet1RC(ppaSet_t settings){
  FILE *rcFile;
  FILE *rcConfFile;
  char *home;
  char rcString[MAXPATHLEN];
  char rcConfString[MAXPATHLEN];

  if ( (home = getenv("HOME")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't get HOME enviroment variable\n");
    return -1;
  }
  strncpy(rcString, home, MAXPATHLEN);
  strcat(rcString, "/.ppaSet1RC");
  strncpy(rcConfString, home, MAXPATHLEN);
  strcat(rcConfString, "/.ppaSet1RC.conf");

  if ( (rcFile = fopen(rcString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcString);
    return -1;
  }
  if ( (rcConfFile = fopen(rcConfString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcConfString);
    return -1;
  }
  settings2RC(rcFile, settings);
  fclose(rcFile);
  settings2RC_conf(rcConfFile, settings);
  fclose(rcConfFile);
  return 0;
}

int createPpaSetCalRC(ppaSet_t settings){
  FILE *rcFile;
  FILE *rcConfFile;
  char *home;
  char rcString[MAXPATHLEN];
  char rcConfString[MAXPATHLEN];

  if ( (home = getenv("HOME")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't get HOME enviroment variable\n");
    return -1;
  }
  strncpy(rcString, home, MAXPATHLEN);
  strcat(rcString, "/.ppaSetCalRC");
  strncpy(rcConfString, home, MAXPATHLEN);
  strcat(rcConfString, "/.ppaSetCalRC.conf");

  if ( (rcFile = fopen(rcString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcString);
    return -1;
  }
  if ( (rcConfFile = fopen(rcConfString, "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open %s for writing\n", rcConfString);
    return -1;
  }
  settings2RC(rcFile, settings);
  fclose(rcFile);
  settings2RC_conf(rcConfFile, settings);
  fclose(rcConfFile);
  return 0;
}

int getPpaSetRC(ppaSet_t *settings){ 
  FILE *rcFile;
  FILE *rcConfFile;
  char *home;
  char rcString[MAXPATHLEN];
  char rcConfString[MAXPATHLEN];
  char line[MAXLINELEN];
  char variable[MAXLINELEN];
  char value[MAXLINELEN];
  int i=0;
  char commented = 0;

  /* Firtly, let's set the system defaults, so a missgiven value can be overriden
     by them */
  if (getSystemDefaults(settings)==-1){
    return -1;
  }

  if ( (home = getenv("HOME")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't get HOME enviroment variable\n");
    return -1;
  }
  strncpy(rcString, home, MAXPATHLEN);
  strcat(rcString, "/.ppaSetRC");
  strncpy(rcConfString, home, MAXPATHLEN);
  strcat(rcConfString, "/.ppaSetRC.conf");

  if ( (rcFile = fopen(rcString, "r")) == NULL ){
    // We can't read it, so return 0 to let our client to use system defaults.
    return 1;
  }
  if ( (rcConfFile = fopen(rcConfString, "r")) == NULL ){
    // We can't read it, so return 0 to let our client to use system defaults.
    return 1;
  }

  /*************************/
  /* Let's parse .ppaSetRC */
  /*************************/
  while (1) { // We will use "break"
    if (fgets(line, MAXLINELEN, rcFile) == NULL){
      /* This is an EOF, so exit*/
      break;
    }
    /* Let's remove comments, if available */
    commented = 0;
    for(i=0; i<MAXLINELEN; i++){
      if (commented == 1)
	line[i] = 0;
      else if (line[i] == '#'){
	line[i] = 0;
	commented = 1;
      }
      else if (line[i] == '\n'){
	line[i] = 0;
	commented = 1; // Fine... it's not a comment, but...
      } else if (line[i] == '=') {
	// We'll replace '=' with ' ' in order to use sscanf later
	line[i] = ' ';
      }
    }
    /* If the resulting is a blank line, let's do nothing */
    if (line[0] == 0)
      continue;
    //    printf("%s\n", line);

    /* Finally, lets look for VARIABLE VALUE pairs */
    if ( (sscanf(line, "%s %s", variable, value)) < 1 ){
      fprintf(stderr, "ppaSetBase: couldn't match a pair in %s... I will try to continue\n", line);
    }
    if (!strncmp(variable,"TYPE",4)) {
      if(!strncmp(value, "710", 3)) {
	settings -> type = TYPE_HP710;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "712", 3)) {
	settings -> type = TYPE_HP712;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "720", 3)) {
	settings -> type = TYPE_HP720;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "722", 3)) {
	settings -> type = TYPE_HP722;
	defaultSettingsForPrinter(settings, 0);
      } else if(!strncmp(value, "820", 3)) {
	settings -> type = TYPE_HP820;
	defaultSettingsForPrinter(settings, 1);
      } else if(!strncmp(value, "1000", 4)) {
	settings -> type = TYPE_HP1000;
	defaultSettingsForPrinter(settings, 2);
      } else {
	fprintf(stderr, "ppaSetBase: unrecognized value %s for variable %s... I will try to continue\n", value, variable);
      }
    } else if (!strncmp(variable,"COLOR",5)) {
      if(!strncmp(value, "ppmraw", 6)) {
	settings->color = COLOR_COLOR;
      } else if(!strncmp(value, "pgmraw", 6)) {
	settings->color = COLOR_GREYSCALE;
      } else if(!strncmp(value, "pbmraw", 6)) {
	settings->color = COLOR_BLACKWHITE;
      }      
    } else if (!strncmp(variable,"DITHER",6)) {
      if(!strncmp(value, "--fd", 4)) {
	settings->dither = DITHER_FAST;
      } else {
	settings->dither = DITHER_FS;
      }
    } else if (!strncmp(variable,"MODE",4)) {
      if(!strncmp(value, "--eco", 5)) {
	settings->mode = MODE_ECO;
      } else if(!strncmp(value, "-p", 2)) {
	settings->mode = MODE_PHOTO;
      } else {
	settings->mode = MODE_NORMAL;
      }      
    } else if (!strncmp(variable,"PAPER",5)) {
      if(!strncmp(value, "a4", 2)) {
	settings->paper = PAPER_A4;
      } else if(!strncmp(value, "a3", 2)) {
	settings->paper = PAPER_A3;
      } else if(!strncmp(value, "legal", 5)) {
	settings->paper = PAPER_LEGAL;
      } else if(!strncmp(value, "letter", 6)) {
	settings->paper = PAPER_LETTER;
      }
    } else if (!strncmp(variable,"XOFFSET",7)) {
      settings -> xOffset = atoi(value);
    } else if (!strncmp(variable,"YOFFSET",7)) {
      settings -> yOffset = atoi(value);
    } else if (!strncmp(variable,"TOPMARGIN",9)) {
      settings -> topMargin = atoi(value);
    } else if (!strncmp(variable,"BOTTOMMARGIN",11)) {
      settings -> bottomMargin = atoi(value);
    } else if (!strncmp(variable,"LEFTMARGIN",10)) {
      settings -> leftMargin = atoi(value);
    } else if (!strncmp(variable,"RIGHTMARGIN",11)) {
      settings -> rightMargin = atoi(value);
    } else if (!strncmp(variable,"BIUNI",5)) {
      if(!strncmp(value, "--bi", 4)) {
	settings->biUni = BIUNI_BI;
      } else if(!strncmp(value, "--uni", 5)) {
	settings->biUni = BIUNI_UNI;
      }      
    } else if (!strncmp(variable,"BLACKNESS",9)) {
      settings -> blackness = atoi(value);
    } else if (!strncmp(variable,"GAMMA",5)) {
      if(!strncmp(value, "--noGamma", 9)) {
	settings->gammaMode = GAMMA_NONE;
      } else if(!strncmp(value, "-F", 2)) {
	settings->gammaMode = GAMMA_FILE;
	sscanf(line, "%s %s %s", variable, value, settings->gammaFile);
      } else {
	settings->gammaMode = GAMMA_DEFAULT;
	// Other gamma modes will be set when parsing ppaSetRC.conf file
      }
    } else {
      fprintf(stderr, "ppaSetBase: unrecognized variable %s... I will try to continue\n", variable);
    }
  }

  /******************************/
  /* Let's parse .ppaSetRC.conf */
  /******************************/
  while (1) { // We will use "break"
    if (fgets(line, MAXLINELEN, rcConfFile) == NULL){
      /* This is an EOF, so exit*/
      break;
    }
    /* Let's remove comments, if available */
    commented = 0;
    for(i=0; i<MAXLINELEN; i++){
      if (commented == 1)
	line[i] = 0;
      else if (line[i] == '#'){
	line[i] = 0;
	commented = 1;
      }
      else if (line[i] == '\n'){
	line[i] = 0;
	commented = 1; // Fine... it's not a comment, but...
      }
    }
    /* If the resulting is a blank line, let's do nothing */
    if (line[0] == 0)
      continue;
    //    printf("%s\n", line);

    /* Finally, lets look for VARIABLE VALUE pairs */
    if ( (sscanf(line, "%s %s", variable, value)) < 1 ){
      fprintf(stderr, "ppaSetBase: couldn't match a pair in %s... I will try to continue\n", line);
    }
    if (!strncmp(variable,"version", 7)) {
      // This is not actually needed, but we need to check it.
    } else if (!strncmp(variable,"blackshear",10)) {
      settings -> bwShearing = atoi(value);
    } else if (!strncmp(variable,"colorshear",10)) {
      settings -> colorShearing = atoi(value);
    } else if (!strncmp(variable,"ColOffsX",8)) {
      settings -> xOffsetColor = atoi(value);
    } else if (!strncmp(variable,"ColOffsY",8)) {
      settings -> yOffsetColor = atoi(value);
    } else if (!strncmp(variable, "GammaR", 6)) {
      settings->gammaMode = GAMMA_ENH;
      settings->gammaEnhR = atof(value);
    } else if (!strncmp(variable, "GammaG", 6)) {
      settings->gammaMode = GAMMA_ENH;
      settings->gammaEnhG = atof(value);
    } else if (!strncmp(variable, "GammaB", 6)) {
      settings->gammaMode = GAMMA_ENH;
      settings->gammaEnhB = atof(value);
    } else if (!strncmp(variable, "RedGammaIdx", 11)) {
      settings->gammaMode = GAMMA_INDEX;
      settings->gammaIndexR = atof(value);
    } else if (!strncmp(variable, "GreenGammaIdx", 11)) {
      settings->gammaMode = GAMMA_INDEX;
      settings->gammaIndexG = atof(value);
    } else if (!strncmp(variable, "BlueGammaIdx", 11)) {
      settings->gammaMode = GAMMA_INDEX;
      settings->gammaIndexB = atof(value);
    }
  }
  fclose(rcFile);
  fclose(rcConfFile);
  return 0;
}

int removePpaSetRC(){
  char *home;
  char rcString[MAXPATHLEN];
  char rcConfString[MAXPATHLEN];
  char rc1String[MAXPATHLEN];
  char rc1ConfString[MAXPATHLEN];
  char rcCalString[MAXPATHLEN];
  char rcCalConfString[MAXPATHLEN];

  if ( (home = getenv("HOME")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't get HOME enviroment variable\n");
    return -1;
  }
  strncpy(rcString, home, MAXPATHLEN);
  strcat(rcString, "/.ppaSetRC");
  strncpy(rcConfString, home, MAXPATHLEN);
  strcat(rcConfString, "/.ppaSetRC.conf");
  strncpy(rc1String, home, MAXPATHLEN);
  strcat(rc1String, "/.ppaSet1RC");
  strncpy(rc1ConfString, home, MAXPATHLEN);
  strcat(rc1ConfString, "/.ppaSet1RC.conf");
  strncpy(rcCalString, home, MAXPATHLEN);
  strcat(rcCalString, "/.ppaSetCalRC");
  strncpy(rcCalConfString, home, MAXPATHLEN);
  strcat(rcCalConfString, "/.ppaSetCalRC.conf");

  if (remove(rcString) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  if (remove(rcConfString) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  if (remove(rc1String) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  if (remove(rc1ConfString) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  if (remove(rcCalString) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  if (remove(rcCalConfString) == -1){
    if (errno != ENOENT){
      perror("ppaSetBase");
      return -1;
    }
  }
  
  return 0;
}

int createPnm2ppa_conf(ppaSet_t settings){
  FILE *output;

  if ( (output = fopen("/tmp/pnm2ppa.conf", "w")) == NULL ){
    fprintf(stderr, "ppaSetBase: couldn't open /tmp/pnm2ppa.conf for writing\n");
    return -1;
  }

  fprintf(output,"#File generated by ppaSetBase (ppaSet)\n");
  fprintf(output,"#Do not edit unless you know what you are doing\n");

  if (settings.type == TYPE_HP710){
    fprintf(output, "version 710\n");
  } else if (settings.type == TYPE_HP712){
    fprintf(output, "version 712\n");
  } else if (settings.type == TYPE_HP720){
    fprintf(output, "version 720\n");
  } else if (settings.type == TYPE_HP722){
    fprintf(output, "version 722\n");
  } else if (settings.type == TYPE_HP820){
    fprintf(output, "version 820\n");
  } else if (settings.type == TYPE_HP1000){
    fprintf(output, "version 1000\n");
  }
  fprintf(output, "blackshear %i\n", settings.bwShearing);
  fprintf(output, "colorshear %i\n", settings.colorShearing);
  fprintf(output, "ColOffsX %i\n", settings.xOffsetColor);
  fprintf(output, "ColOffsY %i\n", settings.yOffsetColor);

  if (settings.gammaMode == GAMMA_DEFAULT){
  } else if (settings.gammaMode == GAMMA_NONE){
    fprintf(stderr, "ppaSetBase: WARNNING: none-gamma-mode hasn't meaning in the system wide pnm2ppa.conf file\n");
  } else if (settings.gammaMode == GAMMA_ENH){
    fprintf(output, "GammaR %f\n", settings.gammaEnhR);
    fprintf(output, "GammaG %f\n", settings.gammaEnhG);
    fprintf(output, "GammaB %f\n", settings.gammaEnhB);
  } else if (settings.gammaMode == GAMMA_INDEX){
    fprintf(output, "RedGammaIdx %i\n", settings.gammaIndexR);
    fprintf(output, "GreenGammaIdx %i\n", settings.gammaIndexG);
    fprintf(output, "BlueGammaIdx %i\n", settings.gammaIndexB);
  } else if (settings.gammaMode == GAMMA_FILE){
    fprintf(stderr, "ppaSetBase: WARNNING: none-gamma-mode hasn't meanning in the system wide pnm2ppa.conf file\n");
  }
    fprintf(stderr, "ppaSetBase: WARNNING: Color-mode hasn't meanning in the system wide pnm2ppa.conf file.\n");
    fprintf(stderr, "ppaSetBase: WARNNING: Dithering-mode hasn't meanning in the system wide pnm2ppa.conf file.\n");
    fprintf(stderr, "ppaSetBase: WARNNING: Quality-mode hasn't meanning in the system wide pnm2ppa.conf file.\n");
    fprintf(stderr, "ppaSetBase: WARNNING: Paper-size only has meanning in the system wide pnm2ppa.conf file in order to make the calibration page. It must has been set in the filter too, as an option to gs.\n");
  if (settings.paper == PAPER_A4){
    fprintf(output, "papersize a4\n");
  } else if (settings.paper == PAPER_A3){
    fprintf(output, "papersize a3\n");
  } else if (settings.paper == PAPER_LEGAL){
    fprintf(output, "papersize legal\n");
  } else if (settings.paper == PAPER_LETTER){
    fprintf(output, "papersize letter\n");
  }
  fprintf(output, "xoffset %i\n", settings.xOffset);
  fprintf(output, "y offset %i\n", settings.yOffset);
  fprintf(output, "topmargin %i\n", settings.topMargin);
  fprintf(output, "bottommargin %i\n", settings.bottomMargin);
  fprintf(output, "leftmargin %i\n", settings.leftMargin);
  fprintf(output, "rightmargin %i\n", settings.rightMargin);
  if (settings.biUni == BIUNI_BI){

  } else if (settings.biUni == BIUNI_UNI){
    fprintf(output, "unimode 1\n");
  }
  fprintf(output, "blackness %i\n", settings.blackness);

  fclose(output);
  return 0;
}

int createGammaRef(){
  int pid;
  int status;
  pid = fork();
  if (pid == -1){
    perror("ppaSetBase");
    return -1;
  } else if (pid == 0) { // child
    if (execl(GAMMAREF, GAMMAREF, NULL) == -1){
      perror("ppaSetBase");
      exit(1);
    }
    exit(0);
  } else { // parent
    // wait for the children to exit
    waitpid(pid, &status, 0);
    if (WIFEXITED(status)){
      // Do nothing, process was successfully executed
    } else {
      fprintf(stderr, "ppaSetBase: error in child process\n");
      return -1;
    }
  }
  return 0;
}

int printNoGamma(ppaSet_t settings){
  int pid;
  int status;
  // Create the *Cal files, so the ppa.if filter is actually the same
  if (createPpaSetCalRC(settings) == -1){
    fprintf(stderr, "ppaSetBase: error creating file in HOME\n");
  }
  // And now, print the gamma page
  pid = fork();
  if (pid == -1){
    perror("ppaSetBase");
    return -1;
  } else if (pid == 0) { // child
    if (execl(NOGAMMA, NOGAMMA, NULL) == -1){
      perror("ppaSetBase");
      exit(1);
    }
    exit(0);
  } else { // parent
    // wait for the children to exit
    waitpid(pid, &status, 0);
    if (WIFEXITED(status)){
      // Do nothing, process was successfully executed
    } else {
      fprintf(stderr, "ppaSetBase: error in child process\n");
      return -1;
    }
  }
  return 0;
}

int printTestPage(ppaSet_t settings){
  int pid;
  int status;
  // Create the *1 files, so the ppa.if filter is actually the same
  if (createPpaSet1RC(settings) == -1){
    fprintf(stderr, "ppaSetBase: error creating file in HOME\n");
  }
  // And now, print the test page
  pid = fork();
  if (pid == -1){
    perror("ppaSetBase");
    return -1;
  } else if (pid == 0) { // child
    if (execl(TEST, TEST, NULL) == -1){
      perror("ppaSetBase");
      exit(1);
    }
    exit(0);
  } else { // parent
    // wait for the children to exit
    waitpid(pid, &status, 0);
    if (WIFEXITED(status)){
      // Do nothing, process was successfully executed
    } else {
      fprintf(stderr, "ppaSetBase: error in child process\n");
      return -1;
    }
  }
return 0;
}

int printCalibrationPages(ppaSet_t settings){
  int pid;
  int status;
  // Create the *Cal files, so the ppa.if filter is actually the same
  if (createPpaSetCalRC(settings) == -1){
    fprintf(stderr, "ppaSetBase: error creating file in HOME\n");
  }
  // And now, print the calibration pages
  pid = fork();
  if (pid == -1){
    perror("ppaSetBase");
    return -1;
  } else if (pid == 0) { // child
    if (execl(CALIBRATION, CALIBRATION, NULL) == -1){
      perror("ppaSetBase");
      exit(1);
    }
    exit(0);
  } else { // parent
    // wait for the children to exit
    waitpid(pid, &status, 0);
    if (WIFEXITED(status)){
      // Do nothing, process was successfully executed
    } else {
      fprintf(stderr, "ppaSetBase: error in child process\n");
      return -1;
    }
  }
  return 0;
}

int cleanHeads(ppaSet_t settings){
  int pid;
  int status;
  // Create the *Cal files, so the ppa.if filter is actually the same
  if (createPpaSetCalRC(settings) == -1){
    fprintf(stderr, "ppaSetBase: error creating file in HOME\n");
  }
  // And now, print the calibration pages
  pid = fork();
  if (pid == -1){
    perror("ppaSetBase");
    return -1;
  } else if (pid == 0) { // child
    if (execl(CLEAN, CLEAN, NULL) == -1){
      perror("ppaSetBase");
      exit(1);
    }
    exit(0);
  } else { // parent
    // wait for the children to exit
    waitpid(pid, &status, 0);
    if (WIFEXITED(status)){
      // Do nothing, process was successfully executed
    } else {
      fprintf(stderr, "ppaSetBase: error in child process\n");
      return -1;
    }
  }
  return 0;
}
