#
# Copyright 2007 Fedora Unity
#
# Jonathan Steffan <jon a fedoraunity.org>
# Jeroen van Meeuwen <kanarip a fedoraunity.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 only
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import sys
import os
from revisor.constants import *
from revisor.errors import *
import shutil
import subprocess

from ConfigParser import SafeConfigParser

import yum
from yum.constants import *
import yum.Errors
import logging
from rhpl.translate import _, N_, getDefaultLangs
import rhpl.translate as translate

import pykickstart
import pykickstart.errors
import pykickstart.version

class ConfigStore:
    """ This ConfigStore holds all options available in Revisor, including Kickstart Data and Handlers,
    so that these are available to both the GUI and the CLI mode."""
    def __init__(self, cli_options, log):
        self.log = log
        self.cli_options = cli_options
        self.check_plugins()

    def check_plugins(self):
        plugins = ['cobbler','delta']
        self.plugins = {}

        for plugin in plugins:
            try:
                self.plugins[plugin] = True
                exec("import revisor.%s" % plugin)
            except:
                self.plugins[plugin] = False

    def setup_cfg(self):
        self.options_set_defaults()
        if self.revisorUseCLI or self.revisorUseGUI:
            self.options_set_from_config(self.cli_options)
            self.options_set_from_commandline(self.cli_options)

        self.required_pkgs =              [
                                            'grub',
                                            'kernel',
                                            'mailx',
                                            'nano',
                                            'selinux-policy',
                                            'selinux-policy-targeted',
                                            'yum'
                                          ]

        self.required_pkgs_installation = [
                                            'acl',
                                            'anaconda',
                                            'anaconda-runtime',
                                            'attr',
                                            'authconfig',
                                            'busybox',
                                            'busybox-anaconda',
                                            'dejavu-lgc-fonts',
                                            'dosfstools',
                                            'firstboot',
                                            'fontconfig',
                                            'gfs2-utils',
                                            'glibc',
                                            'gnome-python2-gtkhtml2',
                                            'grub',
                                            'hdparm',
                                            'iscsi-initiator-utils',
                                            'jfsutils',
                                            'joe',
                                            'kernel',
                                            'kernel-xen',
                                            'man',
                                            'mdadm',
                                            'memtest86+',
                                            'nano',
                                            'openssl',
                                            'pango',
                                            'policycoreutils',
                                            'reiserfs-utils',
                                            'selinux-policy',
                                            'selinux-policy-targeted',
                                            'syslinux',
                                            'system-config-keyboard',
                                            'xfsprogs',
                                            'xorg-x11-drivers',
                                            'xorg-x11-fonts-base',
                                            'xorg-x11-fonts-ISO8859-1-75dpi',
                                            'yum'
                                          ]

        self.required_pkgs_live =         [
                                            'authconfig',
                                            'cpuspeed',
                                            'dhclient',
                                            'eject',
                                            'fedora-logos',
                                            'file',
                                            'grub',
                                            'kernel',
                                            'mailx',
                                            'man',
                                            'nano',
                                            'openssh-clients',
                                            'passwd',
                                            'rootfiles',
                                            'rpm',
                                            'rsync',
                                            'selinux-policy',
                                            'selinux-policy-targeted',
                                            'sendmail',
                                            'shadow-utils',
                                            'sudo',
                                            'syslinux',
                                            'tree',
                                            'wget',
                                            'xterm',
                                            'yum'
                                          ]

        self.instgroups = []
        self.removedgroups = []
        self.removed = []
        self.installed = []
        self.updated = []
        self.obsoleted = []
        self.depremoved = []
        self.depinstalled = []
        self.depupdated = []

        # Nasty, but it saves us from requiring system-config-display/firstboot
        self.default_xorg_config = """

Section "ServerLayout"
        Identifier     "Default Layout"
        Screen      0  "Screen0" 0 0
        InputDevice    "Keyboard0" "CoreKeyboard"
EndSection

Section "InputDevice"
        Identifier  "Keyboard0"
        Driver      "kbd"
        Option      "XkbModel" "pc105"
        Option      "XkbLayout" "%s"
EndSection

Section "Device"
        Identifier  "Videocard0"
        Driver      "%s"
EndSection

Section "Screen"
        Identifier "Screen0"
        Device     "Videocard0"
        DefaultDepth     24
        SubSection "Display"
                Viewport   0 0
                Depth     24
        EndSubSection
EndSection

"""


    def check_package_selection(self, only_clear_data=False):
        """ A function to check some key things about our selected packages. """

        self.package_Xorg = False
        self.package_windowmanager_gnome = False
        self.package_windowmanager_kde = False
        self.package_windowmanager_xfce = False

        if not only_clear_data:
            pkgs = self.yumobj.tsInfo.getMembers()
            for pk in pkgs:
                if pk.name in ('xorg-x11-server-Xorg',):
                    self.package_Xorg = True
                elif pk.name in ('kdebase',):
                    self.package_windowmanager_kde = True
                elif pk.name in ('gnome-session',):
                    self.package_windowmanager_gnome = True
                elif pk.name in ('xfce-session',):
                    self.package_windowmanager_xfce = True

##
##
## Functions that can load and test regular configuration files
##
##

    def set_revisor_config(self,val):
        self.revisor_config = val

    def set_revisor_model(self,val):
        self.revisor_model = val

    def set_main_conf(self,val):
        self.main_conf = val

    def check_working_directory(self):
        """From the Revisor Configuration Files and other settings,
        check for the directory hierarchy"""

        self.log.debug(_("Checking working directories"))
        complain = False
        if os.access(os.path.join(self.working_directory,"pungi-revisor"), os.R_OK):
            complain = True
        if os.access(os.path.join(self.working_directory,"revisor-pungi"), os.R_OK):
            complain = True
# FIXME
# Clean out metadata, not all of it
#
        if os.access(os.path.join(self.working_directory,"revisor-yumcache"), os.R_OK):
            complain = True
        if os.access(os.path.join(self.working_directory,"revisor"), os.R_OK):
            complain = True
        if os.access(os.path.join(self.working_directory,"revisor-livecd"), os.R_OK):
            complain = True

        if complain:
            self.log.warning(_("The directories Revisor uses in %s already exist. This could possibly hold data from a previous run. Please remove or move them to a safe location. Then confirm to continue. If you do not move or remove the files, Revisor will simply delete them" % self.working_directory))
            complain = False
            if os.access(os.path.join(self.working_directory,"pungi-revisor"), os.R_OK):
                shutil.rmtree(os.path.join(self.working_directory,"pungi-revisor"))
            if os.access(os.path.join(self.working_directory,"revisor-pungi"), os.R_OK):
                shutil.rmtree(os.path.join(self.working_directory,"revisor-pungi"))
# FIXME
# Clean out metadata, not all of it
#
            if os.access(os.path.join(self.working_directory,"revisor-yumcache"), os.R_OK):
                shutil.rmtree(os.path.join(self.working_directory,"revisor-yumcache"))
            if os.access(os.path.join(self.working_directory,"revisor"), os.R_OK):
                if os.path.ismount("%s%s" % (self.working_directory,"revisor")):
                    for f in ["/sys", "/proc", "/dev/pts", "/dev", "/selinux", "/yum-cache", ""]:
                        if os.path.ismount("%s%s%s" % (self.working_directory,"revisor",f)):
                            try:
                                rc = subprocess.call(["/bin/umount", "%s%s%s" % (self.working_directory,"revisor",f)])
                            except OSError, e:
                                self.log.error(_("Unable to unmount %s%s%s with error:\n\n%s\n\nPlease resolve the issue and continue.") % (self.working_directory,"revisor",f,e))
                                return False

                try:
                    shutil.rmtree("%s%s" % (self.working_directory,"revisor"))
                except OSError, e:
                    self.log.error(_("Unable to unmount %s%s with error:\n\n%s\n\nPlease resolve the issue and continue.") % (self.working_directory,"revisor",e))
                    return False

            if os.access(os.path.join(self.working_directory,"revisor-livecd"), os.R_OK):
                shutil.rmtree(os.path.join(self.working_directory,"revisor-livecd"))

        return True

    def check_destination_directory(self):
        """From the Revisor Configuration Files and other settings,
        check for the directory hierarchy"""

        self.log.debug(_("Checking destination directories"))
        complain = False
        if os.access(os.path.join(self.destination_directory,"iso"), os.R_OK):
            complain = True
        if os.access(os.path.join(self.destination_directory,"live"), os.R_OK):
            complain = True
        if os.access(os.path.join(self.destination_directory,"os"), os.R_OK):
            complain = True

        if complain:
            self.log.warning(_("The directories Revisor uses in %s already exist. This could possibly hold data from a previous run. Please remove or move them to a safe location. Then confirm to continue. If you do not move or remove the files, Revisor will simply delete them" % self.destination_directory))
            complain = False
            if os.access(os.path.join(self.destination_directory,"iso"), os.R_OK):
                shutil.rmtree(os.path.join(self.destination_directory,"iso"))
            if os.access(os.path.join(self.destination_directory,"live"), os.R_OK):
                shutil.rmtree(os.path.join(self.destination_directory,"live"))
            if os.access(os.path.join(self.destination_directory,"os"), os.R_OK):
                shutil.rmtree(os.path.join(self.destination_directory,"os"))

            os.makedirs(os.path.join(self.destination_directory,"iso"))
            os.makedirs(os.path.join(self.destination_directory,"live"))
            os.makedirs(os.path.join(self.destination_directory,"os"))

        else:
            if not os.access(os.path.join(self.destination_directory,"iso"), os.R_OK) and self.media_installation:
                while not os.access(os.path.join(self.destination_directory,"iso"), os.R_OK):
                    try:
                        os.makedirs(os.path.join(self.destination_directory,"iso"))
                    except:
                        self.log.error(_("Cannot access %s, please check the permissions so we can try again." % os.path.join(self.destination_directory,"iso")))

            if not os.access(os.path.join(self.destination_directory,"live"), os.R_OK) and self.media_live:
                while not os.access(os.path.join(self.destination_directory,"live"), os.R_OK):
                    try:
                        os.makedirs(os.path.join(self.destination_directory,"live"))
                    except:
                        self.log.error(_("Cannot access %s, please check the permissions so we can try again." % os.path.join(self.destination_directory,"live")))

            if not os.access(os.path.join(self.destination_directory,"os"), os.R_OK) and self.media_installation_tree:
                while not os.access(os.path.join(self.destination_directory,"os"), os.R_OK):
                    try:
                        os.makedirs(os.path.join(self.destination_directory,"os"))
                    except:
                        self.log.error(_("Cannot access %s, please check the permissions so we can try again." % os.path.join(self.destination_directory,"os")))

##
##
##  Yum related stuff
##
##

    def setup_yum(self):
        # You do like having your own package sack, don't you?
        self.yumobj = yum.YumBase()

        if self.arch == "i386":
            self.arch_list = yum.rpmUtils.arch.getArchList('i686')
            self.yumobj.compatarch = 'i686'
        elif self.arch == "ppc":
            self.arch_list = yum.rpmUtils.arc.getArchList('ppc64')
            self.yumobj.compatarch = 'ppc64'
        else:
            self.arch_list = yum.rpmUtils.arch.getArchList(self.arch)
            self.yumobj.compatarch = self.arch

    def reposSetup(self, callback = None, thisrepo = None):
        # We shouldn't run for all keys(), but for all repos
        tmpparser = SafeConfigParser()
        tmpparser.read(self.main_conf)
        for item in tmpparser._sections:
            # Maybe if item in self.repos.keys()... but that fuck up matching development against
            # extras-development... Right?
            try:
                r = self.repos[item]
                if r:
                    self.yumobj.repos.enableRepo(item)
                else:
                    self.yumobj.repos.disableRepo(item)
            except:
                pass

        if callback:
            self.yumobj.repos.callback = callback
            callback.num_tasks += 10

        self.yumobj.doLock(YUM_PID_FILE)

        if callback: callback.next_task()
        try:
            self.yumobj.doRepoSetup(thisrepo)
        except yum.Errors.RepoError, e:
            raise RevisorDownloadError, e

        if callback: callback.next_task()

        try:
            self.yumobj.doGroupSetup()
        except yum.Errors.GroupsError:
            self.log.warning(_("No groups present!"))
        except yum.Errors.RepoError, e:
            raise RevisorDownloadError, e

        if callback: callback.next_task(next = 5) # hack... next should be long

        try:
            self.yumobj.doSackSetup()
        except yum.Errors.RepoError, e:
            raise RevisorDownloadError, e

        if callback: callback.next_task(incr=3)

        if callback: self.yumobj.repos.callback = None

##
##
## Kickstart related stuff
##
##

    def reset_ks(self):
        self.ksparser._reset()

        try:
            self.kshandler = pykickstart.version.makeVersion(pykickstart.version.DEVEL)
        except pykickstart.errors.KickstartVersionError:
            self.log.error(_("Could not handle the kickstart data or something... Log a bug"))

        self.ksparser = pykickstart.parser.KickstartParser(self.kshandler)

    def test_ks(self,file):
        kshandler = pykickstart.version.makeVersion(pykickstart.version.DEVEL)
        ksparser = pykickstart.parser.KickstartParser(kshandler,None)

        try:
            ksparser.readKickstart(file)
            return True
        except:
            return False

    def set_ks_authconfig(self):
        self.kshandler.authconfig.authconfig = self.nis_auth + self.ldap_auth + self.kerberos_auth + self.hesiod_auth + self.samba_auth + self.nscd_auth

    def setup_ks(self):
        try:
            self.kshandler = pykickstart.version.makeVersion(pykickstart.version.DEVEL)
        except pykickstart.errors.KickstartVersionError:
            self.log.error(_("Could not handle the kickstart data or something... Log a bug"))

        self.ksparser = pykickstart.parser.KickstartParser(self.kshandler)

    def load_kickstart(self,fn,packages=True):
        """Function loads a kickstart from a file, yes it does.
        Set the 'packages' parameter to include or exclude the
        package manifest"""
        self.kickstart_file = fn
        self.ksparser.readKickstart(self.kickstart_file)

    def set_kickstart_file(self,val):
        self.kickstart_file = val

    def config_to_kickstart(self,ks):
        """This would be a function that writes the configstore
        out to a pykickstart object"""
        pass

    def set_authconfig(self):
        self.kshandler.authconfig.authconfig = self.nis_auth + self.ldap_auth + self.kerberos_auth + self.hesiod_auth + self.samba_auth + self.nscd_auth

##
##
## Revisor Configuration related stuff
##
##

    def options_set_defaults(self):
        """Sets the default configuration options for Revisor"""

        self.log.debug(_("Setting default options"))

        # Media Type defaults for Advanced Configuration
        self.media_installation = False
        self.media_live = False
        self.media_virtualization = False
        self.media_utility = False

        # Defaults for Installation Media
        self.media_installation_cd = False
        self.media_installation_dvd = False
        self.media_installation_dvd_duallayer = False
        self.media_installation_tree = False
        self.media_installation_unified = False
        self.media_installation_pxe = False

        # Defaults for Live Media
        self.media_live_optical = False
        self.media_live_thumb = False
        self.media_live_hd = False

        # Defaults for Virtualization Media
        self.media_virtual_vmware_appliance = False
        self.media_virtual_vmware_guest = False
        self.media_virtual_xen = False
        self.media_virtual_kvm = False

        # Defaults for Utility Media
        self.media_utility_rescue = False

        # Default configuration file. Customize via BASE_CONF_DIR in
        # revisor.constants.py
        self.revisor_config = os.path.join(BASE_CONF_DIR,"revisor.conf")

        # Default destination directory
        self.destination_directory = "/srv/revisor/"

        # Default working directory
        self.working_directory = "/var/tmp/"

        # Sets which models we want to use for spinning
        # Either have an empty string here (takes 1 line) or have a lot
        # of useless code doing hasattr() and try/except (You'll see this often)
        self.main_conf = ""
        self.revisor_model = None

        # Show repos, hide repos
        self.repos_enablesource = False
        self.repos_enabledebuginfo = False
        self.repos_enabletesting = False
        self.repos_enabledevelopment = False

        # Options! More options!
        self.kickstart_file = os.path.join(BASE_CONFD_DIR,"sample-ks.cfg")
        self.use_kickstart_file = True

        # Use the package manifest from the kickstart file you provided earlier
        self.kickstart_manifest = True
        # Customize that package manifest. Works like a charm if packages have
        # become obsolete or fail dependencies.
        self.kickstart_manifest_customize = False
        self.kickstart_options_customize = False
        self.kickstart_include = False
        self.kickstart_default = False

        # Show advanced configuration. Allows for manual package version
        # selection, kickstart options override, chroot shell, etc.
        self.advanced_configuration = False

        # Use fedora-release by default
        self.release_pkgs = "^fedora-release.*$"
        self.release_files = "eula.txt fedora.css GPL README-BURNING-ISOS-en_US.txt RELEASE-NOTES-en_US.html ^RPM-GPG img images stylesheet-images"

        # Live Media options
        self.lm_use_shadow = True
        self.lm_use_md5 = True
        self.lm_use_nis = False
        self.lm_use_ldap = False
        self.lm_use_kerberos = False
        self.lm_use_hesiod = False
        self.lm_use_samba = False
        self.lm_use_nscd = False

        self.lm_user_configuration = False
        self.lm_user_name = "live"
        self.lm_user_comment = "Fedora Unity - Revisor"
        self.lm_user_password = "live"

        self.lm_user_auto_login = False
        self.lm_user_wheel = True
        self.lm_wheel_sudo_no_passwd = True
        self.lm_dump_current_profile = False

        self.lm_fs_label = "fc"
        self.lm_skip_fs_compression = False

        self.lm_bootloader_options = "rhgb quiet"
        self.lm_hostname = "livecd.fedoraunity.org"

        self.lm_default_runlevel = 3

        # Repo store for dialog population
        self.repos = {}

        # Default architecture we're running for
        self.arch = "i386"

        # Still reading?
        self.i_did_all_this = False
        self.nis_auth = ""
        self.ldap_auth = ""
        self.kerberos_auth = ""
        self.hesiod_auth = ""
        self.samba_auth = ""
        self.nscd_auth = ""

        self.tasks = []
        self.built_iso_images = []

        ##
        ## Debug
        ##
        self.lift_off_called = 0

        # Plugin defaults
        self.plugins = {}

        self.hasCobbler = False
        self.hasDelta = False

        # Cobbler Actions
        self.cobbler_add_distro = ""
        self.cobbler_add_profile = ""

        # Delta Options
        self.delta_old_image = ""

        # Init package booleans
        self.check_package_selection(only_clear_data=True)

    def options_reload_from_config(self):
        config = SafeConfigParser()
        config.read(self.revisor_config)

        if config.has_section("revisor"):
            for (key,val) in config.items("revisor"):
                self.log.debug("setting %s to %s" % (key,val))
                exec("self.%s = %s" % (key,val))

    def model_options_reload_from_config(self, config_file=None):
        config = SafeConfigParser()
        if config_file:
            config.read(self.config_file)
            key = "default"
        else:
            config.read(self.revisor_config)
            key = self.revisor_model

        if config.has_option(key,"main"):
            self.main_conf = config.get(key,"main")

        allowed_config = ['product_name', 'product_path', 'iso_basename', 'comps', 'architecture',
                          'version', 'getsource', 'release_pkgs', 'release_files', 'copy_dir', 'strip_copy_dir']
        repo_config = ['repos_enabledevelopment', 'repos_enabletesting', 'repos_enabledebuginfo', 'repos_enablesource']

        if config.has_section(key):
            for item in allowed_config:
                if config.has_option(key,item):
                    self.log.debug(_("Setting %s to %s" % (item,config.get(key,item))))
                    exec("self.%s = '%s'" % (item,config.get(key,item)))
                else:
                    self.log.debug(_("Emptying setting %s because no setting is available for this directive" % item))
                    exec("self.%s = ''" % (item))

            for item in repo_config:
                if config.has_option(key,item):
                    self.log.debug(_("Setting %s to %s" % (item,config.getint(key,item))))
                    exec("self.%s = %d" % (item,config.getint(key,item)))
                else:
                    self.log.debug(_("Emptying setting %s because no setting is available for this directive" % item))
                    exec("self.%s = False" % (item))

        # Use fedora-release by default
        if self.release_pkgs == "": self.release_pkgs = "^fedora-release.*$"
        if self.release_files == "": self.release_files = "eula.txt fedora.css GPL README-BURNING-ISOS-en_US.txt RELEASE-NOTES-en_US.html ^RPM-GPG img images stylesheet-images"

    def options_set_from_config(self, cli_options):
        """Sets the default configuration options for Revisor from a
        configuration file. Configuration file may be customized using
        the --config CLI option"""

        self.log.debug(_("Setting options from configuration file"))

        # Check from which configuration file we should get the defaults
        if cli_options.revisor_config:
            self.revisor_config = cli_options.revisor_config

        config = SafeConfigParser()
        config.read(self.revisor_config)

        if config.has_section("revisor"):
            if config.has_option("revisor","model"):
                if config.has_section(config.get("revisor","model")):
                    self.revisor_model = config.get("revisor","model")

            for (key,val) in config.items("revisor"):
                self.log.debug(_("Setting %s to %s") % (key,val))
                exec("self.%s = %s" % (key,val))

        if not self.revisor_model == None and not self.revisor_model == "":
            if config.has_option(self.revisor_model,"main"):
                if os.access(config.get(self.revisor_model,"main"), os.R_OK):
                    self.main_conf = config.get(self.revisor_model,"main")
                else:
                    self.log.error(_("Invalid model %s, configuration file %s not readable" % (self.revisor_model,config.get(self.revisor_model,"main"))))
            else:
                self.log.error(_("Invalid model %s, no configuration file %s" % (self.revisor_model,"main")))

            if config.has_option(self.revisor_model,"pungi"):
                # If this directive is here... We're using old configuration files
                self.log.error(_("You may be using an old configuration file, since there is a 'pungi' configuration directive in %s" % self.revisor_model))
            else:
                # Read the configuration directives from the general revisor configuration file's model section
                self.model_options_reload_from_config()

# FIXME
        # We should probably check the system's arch versus the arch
        # set in configuration and have some compatibility check
        # is yum.rpmUtils.arch.getArchList() suitable?
        if config.has_option(self.revisor_model,"arch"):
            self.arch = config.get(self.revisor_model,"arch")

        if config.has_option(self.revisor_model,"repos_enabledevelopment"):
            self.repos_enabledevelopment = config.get(self.revisor_model,"repos_enabledevelopment")

        if config.has_option(self.revisor_model,"repos_enabletesting"):
            self.repos_enabletesting = config.get(self.revisor_model,"repos_enabletesting")

        if config.has_option(self.revisor_model,"repos_enablesource"):
            self.repos_enablesource = config.get(self.revisor_model,"repos_enablesource")

        if config.has_option(self.revisor_model,"repos_enabledebuginfo"):
            self.repos_enabledebuginfo = config.get(self.revisor_model,"repos_enabledebuginfo")

        config = ""

    def options_set_from_commandline(self, cli_options):
        """Overrides default options from the CLI"""

        self.log.debug(_("Setting options from command-line"))

        if self.revisorUseCLI:
            if cli_options.revisor_model:
                self.revisor_model = cli_options.revisor_model
            elif not self.revisor_model == None and not self.revisor_model == "":
                pass
            else:
                self.log.error(_("No model specified."), recoverable = False)

            config = SafeConfigParser()
            config.read(self.revisor_config)

            if not config.has_section(self.revisor_model):
                self.log.error(_("Invalid model specified: %s does not exist in %s") % (self.revisor_model,self.revisor_config), recoverable = False)

        if self.revisorUseCLI:
            # Add options that need to be saved after setting defaults here:
            self.media_installation_dvd = cli_options.media_installation_dvd
            self.media_installation_cd = cli_options.media_installation_cd
            self.media_live_optical = cli_options.media_live_optical
            self.media_live_thumb = cli_options.media_live_thumb
            self.media_live_hd = cli_options.media_live_hd
            self.media_installation_tree = cli_options.media_installation_tree
# FIXME
# Disabled cobbler and delta
#            self.cobbler_add_distro = cli_options.cobbler_add_distro
#            self.cobbler_add_profile = cli_options.cobbler_add_profile
#            self.delta_old_image = cli_options.delta_old_image

            if self.media_installation_dvd or self.media_installation_cd or self.media_installation_dvd_duallayer or self.media_installation_tree or self.media_installation_unified:
                self.media_installation = True
            else:
                self.media_installation = False

            if self.media_live_thumb or self.media_live_optical or self.media_live_hd:
                self.media_live = True
            else:
                self.media_live = False

        if cli_options.kickstart_file:
            if os.access(cli_options.kickstart_file, os.R_OK):
                self.use_kickstart_file = True
                self.kickstart_file = cli_options.kickstart_file
            elif cli_options.revisorUseCLI:
                self.log.error(_("Kickstart file %s fails to load" % cli_options.kickstart_file), recoverable = False)
            else:
                self.log.error(_("Kickstart file %s fails to load, continuing with defaults" % cli_options.kickstart_file))
        elif cli_options.revisorUseCLI:
            self.log.error(_("No kickstart file specified"), recoverable = False)

        if cli_options.revisorUseCLI and not self.media_installation and not self.media_live:
            self.log.error(_("No media specified"), recoverable = False)
