#
# Copyright 2007 Fedora Unity
#
# Jonathan Steffan <jon a fedoraunity.org>
# Jeroen van Meeuwen <kanarip a fedoraunity.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 only
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import subprocess
import pypungi.pungi
import logging
import os
import sys
sys.path.append('/usr/lib/anaconda-runtime')
import splittree
import shutil
import re

# Translation
from rhpl.translate import _, N_, getDefaultLangs
import rhpl.translate as translate

class RevisorPungi(pypungi.pungi.Pungi):
    def __init__(self, config, cfg, log):
        pypungi.pungi.Pungi.__init__(self, config)
        self.cfg = cfg
        self.log = log

    def _doRunCommand(self, command, rundir=None, output=subprocess.PIPE, error=subprocess.STDOUT, inshell=False):
        """Run a command and log the output."""

        if rundir == None:
            self.log.debug(_("Setting rundir to %s") % rundir)
            rundir = os.path.join(self.cfg.working_directory,"revisor")

        if not os.access(rundir, os.R_OK):
            self.log.error(_("Directory %s is not accessible. Aborting"), recoverable = False)

        self.log.debug(_("Running command %s") % ' '.join(command))

        if output == subprocess.PIPE:
            p1 = subprocess.Popen(command, cwd=rundir, stdout=output, stderr=error, shell=inshell)

            while p1.poll() == None:
                try:
                    returncode = int(p1.poll())
                except:
                    if output == subprocess.PIPE:
                        try:
                            line = p1.stdout.readline()
                            if not line == "":
                                self.callback.parse_line(command, line)
                        except:
                            pass
        else:
            p1 = subprocess.Popen(command, cwd=rundir, stdout=output, stderr=error)
            (out, err) = p1.communicate()

    def setCallback(self, callback=None):
        self.callback = callback

    def doCreateRepo(self):

        # create repodata for the tree
        createrepo = ['/usr/bin/createrepo']
        createrepo.append('-v')
        createrepo.append('--database')

        createrepo.append('--groupfile')
        createrepo.append(self.config.get('default', 'comps'))

        createrepo.append(self.topdir)

        # run the command
        self._doRunCommand(createrepo, rundir=self.topdir)

    def doBuildinstall(self):
        """Run anaconda-runtime's buildinstall on the tree."""

        # setup the buildinstall call
        buildinstall = ['/usr/lib/anaconda-runtime/buildinstall']
        #buildinstall.append('TMPDIR=%s' % self.workdir) # TMPDIR broken in buildinstall

        buildinstall.append('--product')
        buildinstall.append(self.config.get('default', 'product_name'))

        if not self.config.get('default', 'flavor') == "":
            buildinstall.append('--variant')
            buildinstall.append(self.config.get('default', 'flavor'))

        buildinstall.append('--version')
        buildinstall.append(self.config.get('default', 'version'))

        buildinstall.append('--release')
        buildinstall.append('"%s %s"' % (self.config.get('default', 'product_name'), self.config.get('default', 'version')))

        buildinstall.append('--prodpath')
        buildinstall.append(self.config.get('default', 'product_path'))

        if self.config.has_option('default', 'bugurl'):
            buildinstall.append('--bugurl')
            buildinstall.append(self.config.get('default', 'bugurl'))

        buildinstall.append(self.topdir)

        # run the command
        self._doRunCommand(buildinstall)

    def doGetRelnotes(self):
        """Get extra files from packages in the tree to put in the topdir of
           the tree."""

        self.log.debug(_("Getting relnotes..."))

        docsdir = os.path.join(self.workdir, 'docs')
        os.makedirs(docsdir)

        # Expload the packages we list as relnote packages
        pkgs = os.listdir(os.path.join(self.topdir, self.cfg.product_path))

        self.log.debug(str(pkgs))
        rpm2cpio = ['/usr/bin/rpm2cpio']
        cpio = ['cpio', '-imud']

        for pkg in pkgs:
            for pattern in self.cfg.release_pkgs.split():
                if re.match(pattern, pkg):
                    extraargs = [os.path.join(self.topdir, self.cfg.product_path, pkg)]
                    try:
                        p1 = subprocess.Popen(rpm2cpio + extraargs, cwd=docsdir, stdout=subprocess.PIPE)
                        (out, err) = subprocess.Popen(cpio, cwd=docsdir, stdin=p1.stdout, stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE, universal_newlines=True).communicate()
                    except:
                        self.log.error(_("An error occured extracting the release files: %s") % err)

                    self.log.debug(out)

        # Walk the tree for our files
        for dirpath, dirname, filelist in os.walk(docsdir):
            for directory in dirname:
                for pattern in self.cfg.release_files.split():
                    if re.match(pattern,directory) and not os.path.exists(os.path.join(self.topdir, directory)):
                        self.log.info(_("Copying %s") % directory)
                        shutil.copytree(os.path.join(dirpath, directory), os.path.join(self.topdir, directory))

            for filename in filelist:
                for pattern in self.cfg.release_files.split():
                    if re.match(pattern,filename) and not os.path.exists(os.path.join(self.topdir, filename)):
                        self.log.info("Copying release note file %s" % filename)
                        shutil.copy(os.path.join(dirpath, filename), os.path.join(self.topdir, filename))
                        self.common_files.append(filename)

        self.log.debug(str(self.common_files))

    def doCopyDir(self, copy_dir=None):
        if not copy_dir:
            return

        if not self.cfg.strip_copy_dir == "":
            shutil.copytree(copy_dir, "%s%s%s" % (self.topdir, "/files/", copy_dir.replace(self.cfg.strip_copy_dir,"")))
        else:
            shutil.copytree(copy_dir, "%s%s%s" % (self.topdir, "/files/", copy_dir))

    def doSplittree(self, media_size = 685, discdir = ""):
        """Use anaconda-runtime's splittree to split the tree into appropriate
        sized chunks."""

        timber = splittree.Timber()
        timber.arch = self.config.get('default', 'arch')
        timber.target_size = float(media_size * 1024 * 1024)
        timber.total_discs = self.config.getint('default', 'discs')
        timber.bin_discs = self.config.getint('default', 'discs')
        timber.src_discs = 0
        timber.release_str = '%s %s' % (self.config.get('default', 'product_name'), self.config.get('default', 'version'))
        timber.package_order_file = os.path.join(self.workdir, 'pkgorder-%s' % self.config.get('default', 'arch'))
        timber.dist_dir = "%s-%s" % (self.topdir,discdir)
        self.log.debug("dist_dir is %s" % timber.dist_dir)
        timber.src_dir = os.path.join(self.config.get('default', 'destdir'), self.config.get('default', 'version'), 'source', 'SRPMS')
        timber.product_path = self.config.get('default', 'product_path')
        timber.common_files = self.common_files
        #timber.reserve_size =

        output = timber.main()
        del timber

    def doCreateSplitrepo(self, discdir = ""):
        """Create the split metadata for the isos"""

        if self.config.getint('default', 'discs') > 1:
            discinfo = open('%s-%s-disc1/.discinfo' % (self.topdir,discdir), 'r').readlines()
        else:
            discinfo = open(os.path.join(self.topdir, '.discinfo'), 'r').readlines()
        mediaid = discinfo[0].rstrip('\n')

        # set up the process
        createrepo = ['/usr/bin/createrepo']
        createrepo.append('-v')
        createrepo.append('--database')

        createrepo.append('--groupfile')
        createrepo.append(self.config.get('default', 'comps'))

        createrepo.append('--baseurl')
        createrepo.append('media://%s' % mediaid)

        createrepo.append('--outputdir')
        if self.config.getint('default', 'discs') == 1:
            os.makedirs('%s-%s-disc1' % (self.topdir,discdir))
        createrepo.append('%s-%s-disc1' % (self.topdir,discdir))

        createrepo.append('--basedir')
        if self.config.getint('default', 'discs') == 1:
            createrepo.append(self.topdir)
            createrepo.append(self.topdir)
        else:
            createrepo.append('%s-%s-disc1' % (self.topdir,discdir))

        if self.config.getint('default', 'discs') > 1:
            createrepo.append('--split')

            for disc in range(1, self.config.getint('default', 'discs') + 1):
                createrepo.append('%s-%s-disc%s' % (self.topdir, discdir, disc))

        # run the command
        self._doRunCommand(createrepo)

    def doCreateIsos(self, discdir = ""):
        """Create isos from the various split directories."""

        anaruntime = '/usr/lib/anaconda-runtime/boot'
        discinfofile = os.path.join(self.topdir, '.discinfo') # we use this a fair amount

        try:
            os.makedirs(self.isodir)
        except:
            pass

        # setup the base command
        mkisofs = ['/usr/bin/mkisofs']
        mkisofs.extend(['-v', '-U', '-J', '-R', '-T', '-f']) # common mkisofs flags

        x86bootargs = ['-b', 'isolinux/isolinux.bin', '-c', 'isolinux/boot.cat',
            '-no-emul-boot', '-boot-load-size', '4', '-boot-info-table']

        ia64bootargs = ['-b', 'images/boot.img', '-no-emul-boot']

        ppcbootargs = ['-part', '-hfs', '-r', '-l', '-sysid', 'PPC', '-no-desktop', '-allow-multidot', '-chrp-boot']

        ppcbootargs.append('-map')
        ppcbootargs.append(os.path.join(anaruntime, 'mapping'))

        ppcbootargs.append('-magic')
        ppcbootargs.append(os.path.join(anaruntime, 'magic'))

        ppcbootargs.append('-hfs-bless') # must be last

        # We're gonna do some tricks here

#
# Create Unified ISO
#

        if discdir == "":
            self.log.debug(_("Creating Unified ISO"))
            isoname = '%s-%s-%s.iso' % (self.config.get('default', 'iso_basename'), self.config.get('default', 'version'), self.config.get('default', 'arch'))
            isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)
            extraargs = []
            if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                extraargs.extend(x86bootargs)
            elif self.config.get('default', 'arch') == 'ia64':
                extraargs.extend(ia64bootargs)
            elif self.config.get('default', 'arch') == 'ppc':
                extraargs.extend(ppcbootargs)
                extraargs.append(os.path.join('%s' % (self.topdir), "ppc/mac"))

            extraargs.append('-V')
            extraargs.append('"%s %s %s"' % (self.config.get('default', 'product_name'), self.config.get('default', 'version'), self.config.get('default', 'arch')))

            extraargs.append('-o')
            extraargs.append(isofile)

            extraargs.append(os.path.join('%s' % (self.topdir)))

            # Add this ISO image to the built list
            self.cfg.built_iso_images.append(isofile)

            # run the command
            self._doRunCommand(mkisofs + extraargs)

            # implant md5 for mediacheck on all but source arches
            if not self.config.get('default', 'arch') == 'source':
                self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])


            # shove the sha1sum into a file
            sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
            self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
            sha1file.close()

#
# Creating CDs
#

        if discdir == "cd":
            self.log.debug("Creating %s images" % (discdir))
            if self.cfg.num_cd > 1:
                for disc in range(1, self.config.getint('default', 'discs') + 1): # cycle through the CD isos
                    isoname = '%s-%s-%s-cd%s.iso' % (self.config.get('default', 'iso_basename'), self.config.get('default', 'version'),
                        self.config.get('default', 'arch'), disc)
                    isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)

                    extraargs = []

                    if disc == 1: # if this is the first disc, we want to set boot flags
                        if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                            extraargs.extend(x86bootargs)
                        elif self.config.get('default', 'arch') == 'ia64':
                            extraargs.extend(ia64bootargs)
                        elif self.config.get('default', 'arch') == 'ppc':
                            extraargs.extend(ppcbootargs)
                            extraargs.append(os.path.join('%s-disc%s' % (self.topdir, disc), "ppc/mac"))

                    extraargs.append('-V')
                    extraargs.append('"%s %s %s CD %s"' % (self.config.get('default', 'product_name'),
                        self.config.get('default', 'version'), self.config.get('default', 'arch'), disc))

                    extraargs.append('-o')
                    extraargs.append(isofile)

                    extraargs.append(os.path.join('%s-%s-disc%s/' % (self.topdir, discdir, disc)))

                    # Add this ISO image to the built list
                    self.cfg.built_iso_images.append(isofile)

                    # run the command
                    self._doRunCommand(mkisofs + extraargs)

                    # implant md5 for mediacheck on all but source arches
                    if not self.config.get('default', 'arch') == 'source':
                        self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])

                    # shove the sha1sum into a file
                    sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
                    self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
                    sha1file.close()

            else:
                isoname = '%s-%s-%s-CD.iso' % (self.config.get('default', 'iso_basename'), self.config.get('default', 'version'), self.config.get('default', 'arch'))
                isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)
                extraargs = []
                if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                    extraargs.extend(x86bootargs)
                elif self.config.get('default', 'arch') == 'ia64':
                    extraargs.extend(ia64bootargs)
                elif self.config.get('default', 'arch') == 'ppc':
                    extraargs.extend(ppcbootargs)
                    extraargs.append(os.path.join('%s' % (self.topdir), "ppc/mac"))

                extraargs.append('-V')
                extraargs.append('"%s %s %s CD"' % (self.config.get('default', 'product_name'), self.config.get('default', 'version'), self.config.get('default', 'arch')))

                extraargs.append('-o')
                extraargs.append(isofile)

                extraargs.append(os.path.join('%s' % (self.topdir)))

                # Add this ISO image to the built list
                self.cfg.built_iso_images.append(isofile)

                # run the command
                self._doRunCommand(mkisofs + extraargs)

                # implant md5 for mediacheck on all but source arches
                if not self.config.get('default', 'arch') == 'source':
                    self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])

                # shove the sha1sum into a file
                sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
                self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
                sha1file.close()
#
# Creating DVDs
#

        if discdir == "dvd":
            self.log.debug("Creating %s images" % (discdir))
            if self.cfg.num_dvd > 1:
                # We have done splittree
                for disc in range(1, self.config.getint('default', 'discs') + 1):
                    print "Running DISC in range %s" % disc

                    isoname = '%s-%s-%s-DVD%s.iso' % (self.config.get('default', 'iso_basename'), self.config.get('default', 'version'),
                        self.config.get('default', 'arch'), disc)
                    isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)

                    extraargs = []

                    if disc == 1: # if this is the first disc, we want to set boot flags
                        if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                            extraargs.extend(x86bootargs)
                        elif self.config.get('default', 'arch') == 'ia64':
                            extraargs.extend(ia64bootargs)
                        elif self.config.get('default', 'arch') == 'ppc':
                            extraargs.extend(ppcbootargs)
                            extraargs.append(os.path.join('%s-%s-disc%s' % (self.topdir, discdir, disc), "ppc/mac"))

                    extraargs.append('-V')
                    extraargs.append('"%s %s %s DVD %s"' % (self.config.get('default', 'product_name'),
                        self.config.get('default', 'version'), self.config.get('default', 'arch'), disc))

                    extraargs.append('-o')
                    extraargs.append(isofile)

                    if self.config.getint('default', 'discs') > 1:
                        extraargs.append(os.path.join('%s-%s-disc%s/' % (self.topdir, discdir, disc)))
                    else:
                        extraargs.append(os.path.join('%s-%s/' % (self.topdir, discdir) ))

                    # Add this ISO image to the built list
                    self.cfg.built_iso_images.append(isofile)

                    # run the command
                    self._doRunCommand(mkisofs + extraargs)

                    # implant md5 for mediacheck on all but source arches
                    if not self.config.get('default', 'arch') == 'source':
                        self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])

                    # shove the sha1sum into a file
                    sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
                    self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
                    sha1file.close()

            elif self.cfg.num_cd > 1:
                isoname = '%s-%s-%s-DVD.iso' % (self.config.get('default', 'iso_basename'), self.config.get('default', 'version'), self.config.get('default', 'arch'))
                isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)
                extraargs = []
                if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                    extraargs.extend(x86bootargs)
                elif self.config.get('default', 'arch') == 'ia64':
                    extraargs.extend(ia64bootargs)
                elif self.config.get('default', 'arch') == 'ppc':
                    extraargs.extend(ppcbootargs)
                    extraargs.append(os.path.join('%s' % (self.topdir), "ppc/mac"))

                extraargs.append('-V')
                extraargs.append('"%s %s %s DVD"' % (self.config.get('default', 'product_name'), self.config.get('default', 'version'), self.config.get('default', 'arch')))

                extraargs.append('-o')
                extraargs.append(isofile)

                extraargs.append(os.path.join('%s' % (self.topdir)))

                # run the command
                self._doRunCommand(mkisofs + extraargs)

                # Add this ISO image to the built list
                self.cfg.built_iso_images.append(isofile)

                # implant md5 for mediacheck on all but source arches
                if not self.config.get('default', 'arch') == 'source':
                    self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])

                # shove the sha1sum into a file
                sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
                self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
                sha1file.close()

            else:
                self.log.info(_("Not creating a DVD ISO Image, Installation Media fits on a CD"))

#
# Create Rescue Image
#

        if discdir == "rescue":
            # Now make rescue images
            if not self.config.get('default', 'arch') == 'source' and \
                os.path.exists('/usr/lib/anaconda-runtime/mk-rescueimage.%s' % self.config.get('default', 'arch')):
                isoname = '%s-%s-%s-rescuecd.iso' % (self.config.get('default', 'iso_basename'),
                    self.config.get('default', 'version'), self.config.get('default', 'arch'))
                isofile = os.path.join(self.cfg.destination_directory,"iso",isoname)

                # make the rescue tree
                rescue = ['/usr/lib/anaconda-runtime/mk-rescueimage.%s' % self.config.get('default', 'arch')]
                rescue.append(self.topdir)
                rescue.append(self.workdir)
                rescue.append(self.config.get('default', 'iso_basename'))
                rescue.append(self.config.get('default', 'product_path'))

                # run the command
                self._doRunCommand(rescue)

                # write the iso
                extraargs = []

                if self.config.get('default', 'arch') == 'i386' or self.config.get('default', 'arch') == 'x86_64':
                    extraargs.extend(x86bootargs)
                elif self.config.get('default', 'arch') == 'ia64':
                    extraargs.extend(ia64bootargs)
                elif self.config.get('default', 'arch') == 'ppc':
                    extraargs.extend(ppcbootargs)
                    extraargs.append(os.path.join(self.workdir, "%s-rescueimage" % self.config.get('default', 'arch'), "ppc/mac"))

                extraargs.append('-V')
                extraargs.append('"%s %s %s Rescue"' % (self.config.get('default', 'product_name'),
                        self.config.get('default', 'version'), self.config.get('default', 'arch')))

                extraargs.append('-o')
                extraargs.append(os.path.join(self.cfg.destination_directory,"iso",isofile))

                extraargs.append(os.path.join(self.workdir, "%s-rescueimage" % self.config.get('default', 'arch')))

                # run the command
                self._doRunCommand(mkisofs + extraargs)

                # implant md5 for mediacheck on all but source arches
                if not self.config.get('default', 'arch') == 'source':
                    self._doRunCommand(['/usr/lib/anaconda-runtime/implantisomd5', isofile])

                # shove the sha1sum into a file
                sha1file = open(os.path.join(self.isodir, 'SHA1SUM'), 'a')
                self._doRunCommand(['/usr/bin/sha1sum', isoname], rundir=self.isodir, output=sha1file)
                sha1file.close()

# FIXME
#        # Do some clean up
#        dirs = os.listdir(self.archdir)
#
#        for directory in dirs:
#            if directory.startswith('os-disc') or directory.startswith('SRPM-disc'):
#                shutil.move(os.path.join(self.archdir, directory), os.path.join(self.workdir, directory))
#
