/***************************************************************************
    smb4kbrowserwidget.cpp  -  Network browser widget class of Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qheader.h>
#include <qvaluelist.h>
#include <qlabel.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdialogbase.h>
#include <kglobalsettings.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kdebug.h>

// application specific includes
#include "smb4kbrowserwidget.h"
#include "smb4kpreviewdialog.h"
#include "smb4kmountdialog.h"
#include "smb4kcustomoptionsdialog.h"
#include "smb4kprintdialog.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kdefs.h"
#include "../core/smb4kglobal.h"
using namespace Smb4KGlobal;



Smb4KBrowserWidget::Smb4KBrowserWidget( QWidget *parent, const char *name ) : KListView( parent, name )
{
  setFrameShape( KListView::PopupPanel );
  setFrameShadow( KListView::Sunken );
  setRootIsDecorated( true );
  setAllColumnsShowFocus( false );
  setMouseTracking( true );

  addColumn( i18n( "Network" ), -1 );
  addColumn( i18n( "Type" ), -1 );
  addColumn( i18n( "IP Address" ), -1 );
  addColumn( i18n( "Comment" ), -1 );

  m_collection = new KActionCollection( this, "BrowserWidget_ActionCollection", KGlobal::instance() );

  m_menu = new Smb4KBrowserActionMenu( Smb4KBrowserActionMenu::Full, m_collection, QString::null, QIconSet(), this, "BrowserWidget_ActionMenu" );

  m_tooltip = NULL;

  initActions();

  // Widget specific connections.
  connect( this, SIGNAL( executed( QListViewItem * ) ),
           this, SLOT( slotItemExecuted( QListViewItem * ) ) );
  connect( this, SIGNAL( expanded( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( collapsed( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( selectionChanged( QListViewItem * ) ),
           this, SLOT( slotSelectionChanged( QListViewItem * ) ) );
  connect( this, SIGNAL( pressed( QListViewItem * ) ),
           this, SLOT( slotMouseButtonPressed( QListViewItem * ) ) );
  connect( this, SIGNAL( rightButtonPressed( QListViewItem *, const QPoint &, int ) ),
           this, SLOT( slotRightButtonPressed( QListViewItem *, const QPoint &, int ) ) );

  // External connections.
  connect( smb4k_core->scanner(), SIGNAL( workgroups( const QValueList<Smb4KWorkgroupItem *> & ) ),
           this,                  SLOT( slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( members( const QString &, const QValueList<Smb4KHostItem *> & ) ),
           this,                  SLOT( slotWorkgroupMembers( const QString &, const QValueList<Smb4KHostItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( shares( const QString &, const QValueList<Smb4KShareItem *> & ) ),
           this,                  SLOT( slotShares( const QString &, const QValueList<Smb4KShareItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( state( int ) ),
           this,                  SLOT( slotReceiveScannerState( int ) ) );
  connect( smb4k_core->scanner(), SIGNAL( ipAddress( Smb4KHostItem * ) ),
           this,                  SLOT( slotAddIPAddress( Smb4KHostItem * ) ) );
  connect( smb4k_core->scanner(), SIGNAL( info( Smb4KHostItem * ) ),
           this,                  SLOT( slotAddInfo( Smb4KHostItem * ) ) );
  connect( smb4k_core->scanner(), SIGNAL( failed() ),
           this,                  SLOT( slotFailed() ) );
  connect( smb4k_core->mounter(), SIGNAL( updated() ),
           this,                  SLOT( slotMarkShares() ) );

  connect( kapp,                  SIGNAL( iconChanged( int ) ),
           this,                  SLOT( slotIconChanged( int ) ) );
}


Smb4KBrowserWidget::~Smb4KBrowserWidget()
{
  delete m_menu;
  delete m_tooltip;

  uint index = 0;

  while ( index < m_collection->count() )
  {
    delete m_collection->action( index++ );
  }

  m_collection->clear();
}


void Smb4KBrowserWidget::initActions()
{
  m_preview_action = m_menu->previewAction();
  m_preview_action->setGroup( "BrowserWidget" );
  m_preview_action->setEnabled( false );
  connect( m_preview_action, SIGNAL( activated() ), this, SLOT( slotPreview() ) );

  m_askpass_action = m_menu->askpassAction();
  m_askpass_action->setGroup( "BrowserWidget" );
  m_askpass_action->setEnabled( false );
  connect( m_askpass_action, SIGNAL( activated() ), this, SLOT( slotAskPass() ) );

  m_print_action = m_menu->printAction();
  m_print_action->setGroup( "BrowserWidget" );
  m_print_action->setEnabled( false );
  connect( m_print_action, SIGNAL( activated() ), this, SLOT( slotPrint() ) );

  m_mount_action = m_menu->mountAction();
  m_mount_action->setGroup( "BrowserWidget" );
  m_mount_action->setEnabled( false );
  connect( m_mount_action, SIGNAL( activated() ), this, SLOT( slotMountShare() ) );

  m_bookmark_action = m_menu->bookmarkAction();
  m_bookmark_action->setGroup( "BrowserWidget" );
  m_bookmark_action->setEnabled( false );
  connect( m_bookmark_action, SIGNAL( activated() ), this, SLOT( slotAddBookmark() ) );

  m_rescan_action = m_menu->rescanAction();
  m_rescan_action->setGroup( "BrowserWidget" );
  m_rescan_action->setEnabled( true );
  connect( m_rescan_action, SIGNAL( activated() ), this, SLOT( slotRescan() ) );

  m_abort_action = m_menu->abortAction();
  m_abort_action->setGroup( "BrowserWidget" );
  m_abort_action->setEnabled( false );
  connect( m_abort_action, SIGNAL( activated() ), this, SLOT( slotAbort() ) );

  m_search_action = m_menu->searchAction();
  m_search_action->setGroup( "BrowserWidget" );
  m_search_action->setEnabled( true );
  connect( m_search_action, SIGNAL( activated() ), this, SLOT( slotSearch() ) );

  m_manual_action = m_menu->manualMountAction();
  m_manual_action->setGroup( "BrowserWidget" );
  m_manual_action->setEnabled( true );
  connect( m_manual_action, SIGNAL( activated() ), this, SLOT( slotManualMount() ) );

  m_custom_action = m_menu->customOptionsAction();
  m_custom_action->setGroup( "BrowserWidget" );
  m_custom_action->setEnabled( false );
  connect( m_custom_action, SIGNAL( activated() ), this, SLOT( slotCustomOptions() ) );
}


void Smb4KBrowserWidget::setOpen( QListViewItem *item, bool open )
{
  if ( open )
  {
    switch ( item->depth() )
    {
      case 0:
      {
        Smb4KWorkgroupItem *i = ((Smb4KBrowserWidgetItem *)item)->workgroupItem();
        smb4k_core->scanner()->getWorkgroupMembers( i->workgroup(), i->master(), i->ip() );
        break;
      }
      case 1:
      {
        Smb4KHostItem *i = ((Smb4KBrowserWidgetItem *)item)->hostItem();
        smb4k_core->scanner()->getShares( i->workgroup(), i->name(), i->ip() );
        break;
      }
      default:
        break;
    }
  }
  else
  {
    switch ( item->depth() )
    {
      case 1:
      {
        // Remove all children:
        QListViewItem *child = item->firstChild();

        while ( child )
        {
          delete child;
          child = item->firstChild();
        }

        break;
      }
      default:
        break;
    }
  }

  KListView::setOpen( item, open );
}


void Smb4KBrowserWidget::insertItem( Smb4KHostItem *item )
{
  if ( item )
  {
    smb4k_core->scanner()->addHost( item );
  }
  else
  {
    return;
  }

  if ( smb4k_core->scanner()->getWorkgroup( item->workgroup() ) == 0 )
  {
    smb4k_core->scanner()->appendWorkgroup( new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
  }

  // Now put the host into the list.
  QListViewItem *workgroupItem = findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

  if ( workgroupItem )
  {
    QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( !hostItem )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );

      emit itemInserted( item );
    }
    else if ( hostItem && hostItem->parent() /* Strange: Under some circumstances it might occurr
              that a workgroup is recognized as a server. (reported by Giuseppe Della Bianca) */ &&
              QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) != 0 )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );

      emit itemInserted( item );
    }
    else
    {
      // Do not need to do anything here.
    }
  }
  else
  {
    // Add the host to the list of hosts (Smb4KScanner):
    smb4k_core->scanner()->addHost( item );

    // Put the host in the list:
    Smb4KBrowserWidgetItem *groupItem = new Smb4KBrowserWidgetItem( this, new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
    groupItem->setExpandable( true );

    Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( groupItem, item );
    hostItem->hostItem()->setMaster( true );  // pseudo master
    hostItem->setExpandable( true );

    emit itemInserted( item );
  }
}


void Smb4KBrowserWidget::readOptions()
{
  // Read the configuration.
  config()->setGroup( "User Interface" );
  bool showType = config()->readBoolEntry( "Show Type", true );
  bool showComment = config()->readBoolEntry( "Show Comment", true );
  bool showIP = config()->readBoolEntry( "Show IP", true );
  m_hidden = config()->readBoolEntry( "Show Hidden", true );
  m_ipc = config()->readBoolEntry( "Show IPC", false );
  m_admin = config()->readBoolEntry( "Show ADMIN", false );
  m_printer = config()->readBoolEntry( "Show Printer", true );
  m_show_tooltip = config()->readBoolEntry( "Show Network Tooltip", true );

  // Now put everything back in according to the wishes of the user.
  if ( showIP )
  {
    setColumnWidth( IP, 10 );
    setColumnWidthMode( IP, QListView::Maximum );
  }
  else
  {
    setColumnWidth( IP, 0 );
    setColumnWidthMode( IP, QListView::Manual );
  }

  if ( showType )
  {
    setColumnWidth( Type, 10 );
    setColumnWidthMode( Type, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Type, 0 );
    setColumnWidthMode( Type, QListView::Manual );
  }

  if ( showComment )
  {
    setColumnWidth( Comment, 10 );
    setColumnWidthMode( Comment, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Comment, 0 );
    setColumnWidthMode( Comment, QListView::Manual );
  }

  // Adjust the columns:
  for ( int col = 0; col < columns(); col++ )
  {
    if ( columnWidth( col ) != 0 )
    {
      adjustColumn( col );
    }
  }
}


void Smb4KBrowserWidget::contentsMouseMoveEvent( QMouseEvent *e )
{
  m_pos = viewport()->mapFromGlobal( e->globalPos() );

  Smb4KBrowserWidgetItem *item = (Smb4KBrowserWidgetItem *)itemAt( m_pos );

  if ( item && m_show_tooltip )
  {
    // Check if we are on the root decoration.
    bool on_root = true;

    if ( m_pos.x() > header()->sectionPos( header()->mapToIndex( 0 ) ) +
         treeStepSize() * ( item->depth() + ( rootIsDecorated() ? 1 : 0 ) ) + itemMargin() ||
         m_pos.x() < header()->sectionPos( header()->mapToIndex( 0 ) ) )
    {
      on_root = false;
    }

    // Show the tool tip if there is an item, the list view has the mouse,
    // and the user wants to see it. Additionally, make sure that it is only shown,
    // if we are on the executable area (here: first column) and not on the root
    // decoration.
    if ( hasMouse() && isExecuteArea( m_pos ) /* first column */ && !on_root )
    {
      if ( !m_tooltip )
      {
        m_tooltip = new Smb4KNetworkItemTooltip( item, e->globalPos() );

        QTimer::singleShot( 2000, this, SLOT( slotShowTooltip() ) );
      }
      else if ( m_tooltip )
      {
        if ( m_tooltip->item() != item )
        {
          // If we get here, the user moved away from the previous
          // item. So, we will kill the process, that should retrieve
          // the additional info for it.
          if ( smb4k_core->scannerIsRunning() &&
               smb4k_core->scannerState() == SCANNER_RETRIEVING_INFO )
          {
            smb4k_core->scanner()->abort();
          }

          delete m_tooltip;
          m_tooltip = NULL;
        }
      }
    }
    else
    {
      if ( m_tooltip )
      {
        delete m_tooltip;
        m_tooltip = NULL;
      }
    }
  }
  else
  {
    if ( m_tooltip )
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }

  KListView::contentsMouseMoveEvent( e );
}


void Smb4KBrowserWidget::changeIcons()
{
  QListViewItemIterator it( this );
  QListViewItem *item;

  // Reload the icon of each item:
  while ((item = it.current()) != 0 )
  {
    ++it;
    ((Smb4KBrowserWidgetItem *)item)->setIcon();
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KBrowserWidget::slotShowTooltip()
{
  if ( m_tooltip && m_tooltip->item() == (Smb4KBrowserWidgetItem *)itemAt( m_pos ) )
  {
    if ( m_tooltip->item()->depth() == 1 )
    {
      // We do not need to do additional checks here, because
      // the scanner will only start a process, if this item wasn't
      // checked for additional information already.
      smb4k_core->scanner()->getInfo( m_tooltip->item()->hostItem()->workgroup(),
                                      m_tooltip->item()->hostItem()->name(),
                                      m_tooltip->item()->hostItem()->ip() );
    }

    if ( hasMouse() && isExecuteArea( m_pos ) )
    {
      m_tooltip->update();
      m_tooltip->showTip( viewport()->mapToGlobal( m_pos ) );
    }
    else
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }
  else
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }
}


void Smb4KBrowserWidget::slotItemExecuted( QListViewItem *item )
{
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( item )
  {
    if ( item->depth() == 2 )
    {
      if ( item->text( Type ).contains( "Printer" ) == 0 )
      {
        slotMountShare();
      }
      else
      {
        slotPrint();
      }
    }
    else
    {
      setOpen( item, !item->isOpen() );
    }
  }
}


void Smb4KBrowserWidget::slotItemExpandedCollapsed( QListViewItem *item )
{
  setSelected( item, true );
}


void Smb4KBrowserWidget::slotSelectionChanged( QListViewItem *item )
{
  if ( item && item->depth() == 2 )
  {
    m_preview_action->setEnabled( true );
    m_askpass_action->setEnabled( true );

    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      m_bookmark_action->setEnabled( true );
      m_preview_action->setEnabled( true );
      m_mount_action->setEnabled( true );
      m_print_action->setEnabled( false );
      m_custom_action->setEnabled( true );
    }
    else
    {
      m_bookmark_action->setEnabled( false );
      m_preview_action->setEnabled( false );
      m_mount_action->setEnabled( false );
      m_print_action->setEnabled( true );
      m_custom_action->setEnabled( false );
    }
  }
  else if ( item && item->depth() == 1 )
  {
    m_bookmark_action->setEnabled( false );
    m_askpass_action->setEnabled( true );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_custom_action->setEnabled( true );
  }
  else
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );
    m_custom_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotRightButtonPressed( QListViewItem *item, const QPoint &point, int )
{
  if ( !item )
  {
    m_menu->popupMenu()->changeTitle( 0, SmallIcon( "network" ), i18n( "Network" ) );
  }
  else
  {
    m_menu->popupMenu()->changeTitle( 0, *(item->pixmap( 0 )), item->text( 0 ) );

    if ( item->depth() == 0 )
    {
      m_rescan_action->setText( i18n( "Scan Wo&rkgroup" ) );
    }
    else
    {
      m_rescan_action->setText( i18n( "Scan Compute&r" ) );
    }
  }

  m_menu->popupMenu()->exec( point, 0 );

  m_rescan_action->setText( i18n( "Scan Netwo&rk" ) );
}


void Smb4KBrowserWidget::slotMouseButtonPressed( QListViewItem *item )
{
    if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( !item && !selectedItem() )
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );
    m_custom_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list )
{
  if ( !list.isEmpty() )
  {
    // Check if the workgroups in the list view are still
    // valid. Remove obsolete ones and add new ones.

    if ( childCount() > 0 )
    {
      QListViewItemIterator it( this );

      while ( it.current() )
      {
        // We only want to check workgroup items:
        if ( it.current()->depth() == 0 )
        {
          QValueList<Smb4KWorkgroupItem *>::ConstIterator i;

          for ( i = list.begin(); i != list.end(); ++i )
          {
            if ( QString::compare( it.current()->text( Network ), (*i)->workgroup() ) == 0 )
            {
              QString old_master = ((Smb4KBrowserWidgetItem *)it.current())->workgroupItem()->master();

              // Found the workgroup item in the new list. Update it and stop here.
              ((Smb4KBrowserWidgetItem *)it.current())->update( *i );

              // We update the master as well, if it changed and the workgroup item is open.
              // In the case the item is closed, Smb4KScanner::getWorkgroupMembers() will be
              // called by setOpen() and an update will be done by slotWorkgroupMembers().
              if ( QString::compare( old_master, (*i)->master() ) != 0 && it.current()->isOpen() )
              {
                // Get the list view items:
                Smb4KBrowserWidgetItem *oldMasterItem = (Smb4KBrowserWidgetItem *)findItem( old_master, Network, CaseSensitive|ExactMatch );
                Smb4KBrowserWidgetItem *newMasterItem = (Smb4KBrowserWidgetItem *)findItem( (*i)->master(), Network, CaseSensitive|ExactMatch );

                // Get the host item of the new master from the scanner. The old master
                // has been removed from the internal host list, so we cannot search for
                // it!
                Smb4KHostItem *newMaster = smb4k_core->scanner()->getHost( (*i)->master(), (*i)->workgroup() );

                if ( oldMasterItem )
                {
                  // The old master item is still present, so update and tell it
                  // that it is no master anymore.
                  // Note: We cannot decide here whether the old master has
                  // to be removed because it vanished from the network. This
                  // has to be done by slotWorkgroupMembers()!
                  oldMasterItem->hostItem()->setMaster( false );
                }
                else
                {
                  // Huh? It vanished...?
                }

                if ( newMasterItem )
                {
                  // Tell the list view item that it is the new master.
                  // Note: Do not use Smb4KBrowserWidgetItem::update() here,
                  // because the will remove the comment and maybe also the
                  // IP address (depending of the lookup method the user chose).
                  // The update will be done by slotWorkgroupMembers().
                  newMasterItem->hostItem()->setMaster( true );
                }
                else
                {
                  // We do not need to check if newMaster is NULL, because it won't be.
                  Smb4KBrowserWidgetItem *master = new Smb4KBrowserWidgetItem( it.current(), newMaster );
                  master->setExpandable( true );
                }
              }

              break;
            }
            else
            {
              // Is the list entry in the list view? If not, add it to the list
              // view. (If it is, it will be found and updated by the code above.)
              if ( findItem( (*i)->workgroup(), Network, CaseSensitive|ExactMatch ) == 0 )
              {
                Smb4KBrowserWidgetItem *workgroup = new Smb4KBrowserWidgetItem( this, *i );
                workgroup->setExpandable( true );
              }

              continue;
            }
          }

          // The workgroup has vanished. Delete it from
          // the list view:
          if ( i == list.end() )
          {
            delete it.current();
          }
        }

        ++it;
      }
    }
    else
    {
      // Put the items in the empty list view:
      for ( QValueList<Smb4KWorkgroupItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
      {
        Smb4KBrowserWidgetItem *workgroup = new Smb4KBrowserWidgetItem( this, *it );
        workgroup->setExpandable( true );
      }
    }
  }
  else
  {
    // Nothing was found clear the list view:
    clear();
  }

  // Adjust the columns:
  for ( int col = 0; col < columns(); col++ )
  {
    if ( columnWidth( col ) != 0 )
    {
      adjustColumn( col );
    }
  }
}



void Smb4KBrowserWidget::slotWorkgroupMembers( const QString &workgroup, const QValueList<Smb4KHostItem *> &list )
{
  // Get the workgroup item:
  Smb4KBrowserWidgetItem *workgroupItem = NULL;

  if ( !workgroup.isEmpty() )
  {
    workgroupItem = (Smb4KBrowserWidgetItem *)findItem( workgroup, Network, ExactMatch|CaseSensitive );
  }
  else
  {
    return;
  }

  if ( workgroupItem )
  {
    if ( !list.isEmpty() )
    {
      // Check if the host items are still valid. Update
      // them if they are and remove them if they are not:
      if ( workgroupItem->childCount() > 0 )
      {
        // Traverse though the hosts:
        QListViewItemIterator it( this );

        while ( it.current() )
        {
          // We will only take action if we have a host item that belongs
          // to the workgroup 'workgroup':
          if ( it.current()->depth() == 1 &&
               QString::compare( it.current()->parent()->text( Network ), workgroupItem->text( Network ) ) == 0 )
          {
            QValueList<Smb4KHostItem *>::ConstIterator i;

            for ( i = list.begin(); i != list.end(); ++i )
            {
              if ( QString::compare( it.current()->text( Network ), (*i)->name() ) == 0 )
              {
                // The host is already in the workgroup. Update it:
                ((Smb4KBrowserWidgetItem *)it.current())->update( *i );

                break;
              }
              else
              {
                // Is the list entry in the workgroup? If not, add it to it.
                // (If it is, it will be found and updated by the code above.)
                // Also: In case the whole list of known hosts is emitted by the
                // scanner, we need omit all entries that do not belong to this
                // workgroup.
                if ( QString::compare( (*i)->workgroup(), workgroupItem->text( Network ) ) == 0 &&
                     findItem( (*i)->name(), Network, CaseSensitive|ExactMatch ) == 0 )
                {
                  Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( workgroupItem, *i );
                  hostItem->setExpandable( true );
                }

                continue;
              }
            }

            // The host has vanished. Delete it from the
            // workgroup:
            if ( i == list.end() )
            {
              delete it.current();
            }
          }

          ++it;
        }
      }
      else
      {
        // Add the children to the childless host item:
        for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
        {
          // In case the whole list of known hosts is emitted by the scanner,
          // we need omit all entries that do not belong to this workgroup.
          if ( QString::compare( (*it)->workgroup(), workgroupItem->text( Network ) ) == 0 )
          {
            Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( workgroupItem, *it );
            hostItem->setExpandable( true );

            continue;
          }
          else
          {
            continue;
          }
        }
      }
    }
    else
    {
      // Delete all host items in this workgroup:
      setOpen( workgroupItem, false );

      QListViewItem *child = workgroupItem->firstChild();

      while ( child )
      {
        delete child;
        child = workgroupItem->firstChild();
      }
    }

    // Adjust the columns:
    for ( int col = 0; col < columns(); col++ )
    {
      if ( columnWidth( col ) != 0 )
      {
        adjustColumn( col );
      }
    }
  }
  else
  {
    // The workgroup item could not be found. So, do nothing.
  }
}


void Smb4KBrowserWidget::slotShares( const QString &host, const QValueList<Smb4KShareItem *> &list )
{
  // Get the host item:
  Smb4KBrowserWidgetItem *hostItem = NULL;

  if ( !host.isEmpty() )
  {
    hostItem = (Smb4KBrowserWidgetItem *)findItem( host, Network, ExactMatch|CaseSensitive );
  }
  else
  {
    return;
  }

  if ( hostItem )
  {
    if ( !list.isEmpty() )
    {
      // Expand the host item, if it is collapsed:
      if ( !hostItem->isOpen() )
      {
        setOpen( hostItem, true );
      }

      if ( hostItem->childCount() > 0 )
      {
        QListViewItemIterator it( this );

        while( it.current() )
        {
          // We only take action, if the shares belong to hostItem:
          if ( it.current()->depth() == 2 &&
               QString::compare( it.current()->parent()->text( Network ), hostItem->text( Network ) ) == 0 )
          {
            QValueList<Smb4KShareItem *>::ConstIterator i;

            for ( i = list.begin(); i != list.end(); ++i )
            {
              if ( QString::compare( it.current()->text( Network ), (*i)->name() ) == 0 )
              {
                // Found the share. Check, if should still be shown and
                // update or remove it:
                Smb4KBrowserWidgetItem *shareItem = (Smb4KBrowserWidgetItem *)it.current();

                if ( !shareItem->shareItem()->isHidden() )
                {
                  if ( !shareItem->shareItem()->isPrinter() )
                  {
                    // There are no restrictions for this kind of
                    // share. Update it.
                    shareItem->update( *i );
                  }
                  else
                  {
                    // Does the user want to see printer shares
                    // or not? If not, delete the entry.
                    if ( !m_printer )
                    {
                      delete shareItem;
                    }
                    else
                    {
                      shareItem->update( *i );
                    }
                  }
                }
                else
                {
                  // Delete or update the item depending on the
                  // settings the user chose:
                  if ( !m_hidden )
                  {
                    delete shareItem;
                  }
                  else
                  {
                    // Can we have hidden printers?
                    if ( shareItem->shareItem()->isPrinter() )
                    {
                      if ( !m_printer )
                      {
                        delete shareItem;
                      }
                      else
                      {
                        shareItem->update( *i );
                      }

                      break;
                    }

                    if ( shareItem->shareItem()->isIPC() )
                    {
                      if ( !m_ipc )
                      {
                        delete shareItem;
                      }
                      else
                      {
                        shareItem->update( *i );
                      }

                      break;
                    }

                    if ( shareItem->shareItem()->isADMIN() )
                    {
                      if ( !m_admin )
                      {
                        delete shareItem;
                      }
                      else
                      {
                        shareItem->update( *i );
                      }

                      break;
                    }
                  }
                }

                break;
              }
              else
              {
                // Does the host carry the list entry? If not, add it to it.
                // (If it is, it will be found and updated by the code above.)
                if ( findItem( (*i)->name(), Network, CaseSensitive|ExactMatch ) == 0 )
                {
                  // Respect the settings the user chose to use:
                  if ( !(*i)->isHidden() )
                  {
                    if ( !(*i)->isPrinter() || (m_printer && (*i)->isPrinter()) )
                    {
                      (void) new Smb4KBrowserWidgetItem( hostItem, *i );
                    }
                    else
                    {
                      // This share won't make it into the list view.
                    }

                    continue;
                  }
                  else
                  {
                    if ( m_hidden )
                    {
                      if ( !(*i)->isIPC() && !(*i)->isADMIN() && !(*i)->isPrinter() )
                      {
                        // The share is no IPC$ and no ADMIN$ share. Include it.
                        (void) new Smb4KBrowserWidgetItem( hostItem, *i );

                        continue;
                      }
                      else
                      {
                        if ( (m_ipc && (*i)->isIPC()) || (m_admin && (*i)->isADMIN()) || (m_printer && (*i)->isPrinter()) )
                        {
                          // We are allowed to put the IPC$/ADMIN$ share in:
                          (void) new Smb4KBrowserWidgetItem( hostItem, *i );

                          continue;
                        }
                        else
                        {
                          // The user does not want to see this item.

                          continue;
                        }
                      }
                    }
                    else
                    {
                      // This item won't be included.

                      continue;
                    }
                  }
                }

                continue;
              }
            }

            // The share has vanished. Delete it from the
            // host item:
            if ( i == list.end() )
            {
              delete it.current();
            }
          }

          ++it;
        }
      }
      else
      {
        // Add the children to the childless host item:
        for ( QValueList<Smb4KShareItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
        {
          // Respect the settings the user chose to use:
          if ( !(*it)->isHidden() )
          {
            if ( !(*it)->isPrinter() || (m_printer && (*it)->isPrinter()) )
            {
              (void) new Smb4KBrowserWidgetItem( hostItem, *it );
            }
            else
            {
              // This share won't make it into the list view.
            }

            continue;
          }
          else
          {
            if ( m_hidden )
            {
              if ( !(*it)->isIPC() && !(*it)->isADMIN() && !(*it)->isPrinter() )
              {
                // The share is no IPC$ and no ADMIN$ share. Include it.
                (void) new Smb4KBrowserWidgetItem( hostItem, *it );

                continue;
              }
              else
              {
                if ( (m_ipc && (*it)->isIPC()) || (m_admin && (*it)->isADMIN()) || (m_printer && (*it)->isPrinter()) )
                {
                  // We are allowed to put the IPC$/ADMIN$ share in:
                  (void) new Smb4KBrowserWidgetItem( hostItem, *it );

                  continue;
                }
                else
                {
                  // The user does not want to see this item.

                  continue;
                }
              }
            }
            else
            {
              // This item won't be included.

              continue;
            }
          }
        }
      }
    }
    else
    {
      // Delete all shares of this host:
      setOpen( hostItem, false );

      QListViewItem *child = hostItem->firstChild();

      while ( child )
      {
        delete child;
        child = hostItem->firstChild();
      }
    }

    // Adjust the columns:
    for ( int col = 0; col < columns(); col++ )
    {
      if ( columnWidth( col ) != 0 )
      {
        adjustColumn( col );
      }
    }
  }
  else
  {
    // The host item could not be found. So, do nothing.
  }
}


void Smb4KBrowserWidget::slotMountShare()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    smb4k_core->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ) );
  }
}


void Smb4KBrowserWidget::slotAskPass()
{
  if ( currentItem() && currentItem()->depth() == 1 )
  {
    QString workgroup = currentItem()->parent()->text( Network );
    QString host = currentItem()->text( Network );
    QString share = QString::null;

    passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData, this, "AskPass" );
  }
  else if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );

    passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData, this, "AskPass" );
  }
}


void Smb4KBrowserWidget::slotPreview()
{
  if ( currentItem()->depth() == 2 )
  {
    Smb4KPreviewDialog *dlg = new Smb4KPreviewDialog( ((Smb4KBrowserWidgetItem *)currentItem())->shareItem(),
                                                      this,
                                                      "PreviewDialog" );
    if ( dlg->initialized() )
    {
      dlg->show();
    }
    else
    {
      delete dlg;
    }
  }
}


void Smb4KBrowserWidget::slotAddBookmark()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString ip = currentItem()->parent()->text( IP );
    QString type = currentItem()->text( Type );

    Smb4KBookmark *bookmark = new Smb4KBookmark( host, share, workgroup, ip, type );
    smb4k_core->bookmarkHandler()->addBookmark( bookmark );
  }
}


void Smb4KBrowserWidget::slotRescan()
{
  bool viewport_has_mouse;

  if ( m_pos.x() <= 0 || viewport()->width() <= m_pos.x() ||
       m_pos.y() <= 0 || viewport()->height() <= m_pos.y() )
  {
    viewport_has_mouse = false;
  }
  else
  {
    viewport_has_mouse = true;
  }

  if ( currentItem() && selectedItem() && viewport_has_mouse )
  {
    if ( currentItem()->depth() == 0 )
    {
      if ( !currentItem()->isOpen() )
      {
        setOpen( currentItem(), true );
      }
      else
      {
        Smb4KWorkgroupItem *item = ((Smb4KBrowserWidgetItem *)currentItem())->workgroupItem();
        smb4k_core->scanner()->getWorkgroupMembers( item->workgroup(), item->master(), item->ip() );
      }
    }
    else if ( currentItem()->depth() == 1 )
    {
      if ( !currentItem()->isOpen() )
      {
        setOpen( currentItem(), true );
      }
      else
      {
        Smb4KHostItem *item = ((Smb4KBrowserWidgetItem *)currentItem())->hostItem();
        smb4k_core->scanner()->getShares( item->workgroup(), item->name(), item->ip() );
      }
    }
    else
    {
      Smb4KHostItem *item = ((Smb4KBrowserWidgetItem *)currentItem()->parent())->hostItem();
      smb4k_core->scanner()->getShares( item->workgroup(), item->name(), item->ip() );
    }
  }
  else
  {
    smb4k_core->scanner()->rescan();
  }
}


void Smb4KBrowserWidget::slotAddIPAddress( Smb4KHostItem *item )
{
  if ( item )
  {
    Smb4KBrowserWidgetItem *workgroup = (Smb4KBrowserWidgetItem *)findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

    if ( workgroup && QString::compare( workgroup->text( Network ), item->workgroup() ) == 0 )
    {
      Smb4KWorkgroupItem *wg = smb4k_core->scanner()->getWorkgroup( item->workgroup() );

      if ( wg )
      {
        workgroup->update( wg );
      }
    }

    Smb4KBrowserWidgetItem *hostItem = (Smb4KBrowserWidgetItem *)findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( hostItem && hostItem->parent() && QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) == 0 )
    {
      hostItem->update( item );

      if ( columnWidth( IP ) != 0 )
      {
        adjustColumn( IP );
      }
    }
  }
}


void Smb4KBrowserWidget::slotPrint()
{
  if ( currentItem() && currentItem()->depth() == 2 &&
       QString::compare( ((Smb4KBrowserWidgetItem *)currentItem())->shareItem()->plainType(), "Printer" ) == 0 )
  {
    Smb4KPrintDialog *dlg = new Smb4KPrintDialog( ((Smb4KBrowserWidgetItem *)currentItem())->shareItem(),
                                                  this,
                                                  "PrintDialog" );

    dlg->exec();

    // The dialog will be closed destructively.
  }
}


void Smb4KBrowserWidget::slotMarkShares()
{
  config()->setGroup( "User Interface" );
  bool show_all = config()->readBoolEntry( "Show All Shares", false );

  QListViewItemIterator it( this );
  QListViewItem *item;

  while ((item = it.current()) != 0 )
  {
    ++it;

    if ( item->depth() == 2 )
    {
      Smb4KShare *share = smb4k_core->mounter()->findShareByName( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) );

      if ( smb4k_core->mounter()->isMounted( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) ) &&
           (!share->isForeign() || show_all) )
      {
        if ( !((Smb4KBrowserWidgetItem *)item)->isMounted() )
        {
          ((Smb4KBrowserWidgetItem *)item)->setMounted( true );
        }

        continue;
      }
      else
      {
        ((Smb4KBrowserWidgetItem *)item)->setMounted( false );
        continue;
      }
    }
    else
    {
      continue;
    }
  }
}


void Smb4KBrowserWidget::slotSearch()
{
  emit searchRequest();
}


void Smb4KBrowserWidget::slotInsertItem( Smb4KHostItem *item )
{
  insertItem( item );
}


void Smb4KBrowserWidget::slotAbort()
{
  smb4k_core->scanner()->abort();
}


void Smb4KBrowserWidget::slotManualMount()
{
  // Since this slot is also invoked by the system tray icon,
  // we need to do something special with the dialog:
  Smb4KMountDialog *dlg = new Smb4KMountDialog( kapp->mainWidget() && kapp->mainWidget()->isShown() ?
                                                kapp->mainWidget() : 0,
                                                "ManualMountDialog" );

  dlg->exec();

  // The dialog will be closed destructively.
}


void Smb4KBrowserWidget::slotReceiveScannerState( int state )
{
  if ( state != SCANNER_STOP )
  {
    if ( state != SCANNER_SEARCHING  )
    {
      m_rescan_action->setEnabled( false );
    }

    m_abort_action->setEnabled( true );
  }
  else
  {
    m_rescan_action->setEnabled( true );
    m_abort_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotAddInfo( Smb4KHostItem *item )
{
  if ( item )
  {
    Smb4KBrowserWidgetItem *host = (Smb4KBrowserWidgetItem *)findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( host && QString::compare( host->parent()->text( Network ), item->workgroup() ) == 0 )
    {
      host->update( item );
    }

    if ( m_tooltip && m_tooltip->item() == host )
    {
      m_tooltip->update();
    }
  }
}


void Smb4KBrowserWidget::slotCustomOptions()
{
  if ( currentItem() )
  {
    Smb4KCustomOptionsDialog *dlg = NULL;

    if ( currentItem()->depth() == 1 )
    {
      dlg = new Smb4KCustomOptionsDialog( ((Smb4KBrowserWidgetItem *)currentItem())->hostItem(),
                                          this,
                                          "CustomOptionsDialog" );
    }
    else if ( currentItem()->depth() == 2 )
    {
      dlg = new Smb4KCustomOptionsDialog( ((Smb4KBrowserWidgetItem *)currentItem())->shareItem(),
                                          this,
                                          "CustomOptionsDialog" );
    }
    else
    {
      delete dlg;

      return;
    }

    if ( dlg->initialized() )
    {
      dlg->exec();
    }
    else
    {
      delete dlg;
    }

    // The dialog will be closed destructively.
  }
}


void Smb4KBrowserWidget::slotFailed()
{
  if ( !m_tooltip && currentItem() && currentItem()->isOpen() )
  {
    setOpen( currentItem(), false );

    // Remove all children, if there are any:
    if ( currentItem()->depth() == 0 || currentItem()->depth() == 1 )
    {
      QListViewItem *item = currentItem()->firstChild();

      while ( item )
      {
        delete item;
        item = currentItem()->firstChild();
      }
    }
  }
}


void Smb4KBrowserWidget::slotIconChanged( int )
{
  changeIcons();
}

#include "smb4kbrowserwidget.moc"
