/*
 * tmms.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <locale.h>
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <getopt.h>
#include <signal.h>
#include <ncurses.h>
#include <pwd.h>
#include <sys/types.h>
#include "terminal.h"
#include "tmms.h"
#include "cfgfile.h"
#include "xmem.h"
#include "printlog.h"
#include "playlist.h"
#include "playerwin.h"
#include "playeractions.h"
#include "playlistwin.h"
#include "loadsave.h"
#include "search.h"
#include "jump.h"
#include "control.h"
#include "modules.h"
#include "cda_mod.h"
#include "mp3_mod.h"
#include "ogg_mod.h"



#if DEBUG_LEVEL!=0
#define LOGFILE "tmms.log"
extern FILE     * logfile;
static int        lgfile = FALSE;
#endif

static char     * all_locale;
static char     * playlist_loc = NULL;
static char     * config_file = NULL;
static int        uses_color = TRUE;

/* Global variables */
pthread_mutex_t   mutex_current;
pthread_mutex_t   mutex_request;
pthread_mutex_t   mutex_status;
pthread_mutex_t   mutex_display;
char            * output_device = NULL;
TimeData          timedata = ELAPSED;
Mode              repeat_mode = REPEAT_NOT;


/* Default colors */
static ColorDefinitions color[NB_COLOR_ELEMENTS] = {
        /* FOREGROUND   BACKGROUND      HIGHLIGHT */
        /* NormalColor */
        { White,        -1,             0, NULL},
        /* BordersColor */
        { White,        -1,             1, NULL},
        /* WindowTitleColor */
        { Yellow,       -1,             1, NULL},
        /* TimeColor */
        { Green,        -1,             1, NULL},
        /* BitrateColor */
        { Magenta,      -1,             1, NULL},
        /* StatusColor */
        { Cyan,         -1,             1, NULL},
        /* PlayedSongColor */
        { White,        -1,             1, NULL},
        /* NumberColor */
        { Green,        -1,             0, NULL},
        /* MoveCursorColor */
        { Cyan,         -1,             0, NULL},
        /* SelectedColor */
        { Magenta,      -1,             0, NULL},
        /* ListSongColor */
        { White,        -1,             0, NULL}
};


/* Default commands */
static Cmdslst commands[] = {
        {KEY_RESIZE, redisp},

        {KEY_UP,     previous},
        {KEY_DOWN,   next},
        {KEY_PPAGE,  pgup},
        {KEY_NPAGE,  pgdn},
        {KEY_HOME,   first},
        {KEY_END,    last},
        {KEY_ENTER,  enter},
        {10,         enter},
        {KEY_DC,     remove_entry},

        {'p', play_pause},
        {'s', stop},
        {'f', fforward},
        {'r', frewind},
        {'n', next_file},
        {'b', prev_file},
        {'j', jump_to},
        {'J', jump_to_selected},
        {'m', switch_repeat_mode},
        {'t', time_data},
        {'L', reload_list},
        {'l', load_fileordir},
        {'S', save_fileordir},
        {'a', add_fileorstream},
        {'i', add_playlist},
        {'R', toggle_pm},
        {' ', select_entry},
        {'A', sel_all},
        {'U', unsel_all},
        {'I', invert_sel},
        {'M', move_entry},
        {'/', search_new},
        {'!', search_next},
        {'-', search_prev},
        {'q', quit},
        {0, NULL}
};


/* Options from config. file */
static options options_list[] =
{
        /* Command line */
        {"Output",                   OPT_STR,  &output_device},
        {"DisplayList",              OPT_BOOL, &playlist_display},
        {"PlayList",                 OPT_STR,  &playlist_loc},
        /* Colors */
        {"TimeColor",                OPT_STR,  &(color[TimeColor].str)},
        {"BitrateColor",             OPT_STR,  &(color[BitrateColor].str)},
        {"StatusColor",              OPT_STR,  &(color[StatusColor].str)},
        {"PlayedSongColor",          OPT_STR,  &(color[PlayedSongColor].str)},
        {"NumberColor",              OPT_STR,  &(color[NumberColor].str)},
        {"MoveCursorColor",          OPT_STR,  &(color[MoveCursorColor].str)},
        {"SelectedColor",            OPT_STR,  &(color[SelectedColor].str)},
        {"ListSongColor",            OPT_STR,  &(color[ListSongColor].str)},
        {"BordersColor",             OPT_STR,  &(color[BordersColor].str)},
        {"WindowTitleColor",         OPT_STR,  &(color[WindowTitleColor].str)},
        /* Keybindings */
        {"KBind_play_pause",         OPT_INT,  NULL},
        {"KBind_stop",               OPT_INT,  NULL},
        {"KBind_fforward",           OPT_INT,  NULL},
        {"KBind_frewind",            OPT_INT,  NULL},
        {"KBind_next_file",          OPT_INT,  NULL},
        {"KBind_prev_file",          OPT_INT,  NULL},
        {"KBind_switch_repeat_mode", OPT_INT,  NULL},
        {"KBind_time_data",          OPT_INT,  NULL},
        {"KBind_reload_list",        OPT_INT,  NULL},
        {"KBind_load_fileordir",     OPT_INT,  NULL},
        {"KBind_save_fileordir",     OPT_INT,  NULL},
        {"KBind_add_fileorstream",   OPT_INT,  NULL},
        {"KBind_add_playlist",       OPT_INT,  NULL},
        {"KBind_toggle_pm",          OPT_INT,  NULL},
        {"KBind_quit",               OPT_INT,  NULL},
        {"KBind_select_entry",       OPT_INT,  NULL},
        {"KBind_sel_all",            OPT_INT,  NULL},
        {"KBind_unsel_all",          OPT_INT,  NULL},
        {"KBind_invert_sel",         OPT_INT,  NULL},
        {"KBind_move_entry",         OPT_INT,  NULL},
        /* End of options */
        {NULL,                       0,        NULL}
};



char *robust_home(const char *user)
{
        if (! user)
        {
                if (getenv("HOME"))
                {
                        return getenv("HOME");
                }
                else if (getenv("USER") && getpwnam(getenv("USER")))
                {
                        return getpwnam(getenv("USER"))->pw_dir;
                }
                else if (getenv("LOGNAME") && getpwnam(getenv("LOGNAME")))
                {
                        return getpwnam(getenv("LOGNAME"))->pw_dir;
                }
                else if (getpwuid(getuid()))
                {
                        return getpwuid(getuid())->pw_dir;
                }
        }
        else
        {
                if (getpwnam(user))
                {
                        return getpwnam(user)->pw_dir;
                }
        }
        return "/";
}


int redisp()
{
        endwin();
        init_curses(uses_color, NB_COLOR_ELEMENTS, color);
        display_player();
        display_playlist();
        return CONT;
}


int quit()
{
        if (get_status() == PLAYING)
        {
                stop_player();
        }
        return QUIT;
}


int reload_list()
{
        playlist_free();
        load_playlist(playlist_loc);
        return redisp();
}


void interact()
{
        int cont = CONT;

        while (cont == CONT)
        {
                cont = command_loop(commands);
        }
}


void usage(int help)
{
        if (help)
        {
                printf("Usage : " PROGRAM
                                " [-h] [-v] [-P] [-o device] [-p playlist]\n"
                       " -h          : this help\n"
                       " -v          : program version\n"
#if DEBUG_LEVEL!=0
                       " -d          : debug mode (print to " LOGFILE ")\n"
#endif
                       " -P          : don't display playlist\n"
                       " -o device   : output device (oss, esd...)\n"
                       " -p playlist : load playlist file or directory\n"
                       "\n");
        }
        else
        {
                printf(PACKAGE " v" VERSION
                       " (" OSTYPE "-" MACHTYPE "),  Thomas Nemeth 2004.\n");
        }
}


void parse_argv(int argc, char *argv[])
{
        int option;

#if DEBUG_LEVEL!=0
        while ((option = getopt(argc, argv, "hvdPo:p:f:")) != -1)
#else
        while ((option = getopt(argc, argv, "hvPo:p:f:")) != -1)
#endif
        {
                switch (option)
                {
                        case 'h' :
                                usage(1);
                                exit(0);
                                break;
                        case 'v' :
                                usage(0);
                                exit(0);
                                break;
#if DEBUG_LEVEL!=0
                        case 'd' :
                                lgfile = TRUE;
                                break;
#endif
                        case 'P' :
                                playlist_display = FALSE;
                                break;
                        case 'o' :
                                output_device = xstrdup(optarg);
                                break;
                        case 'p' :
                                playlist_loc = xstrdup(optarg);
                                break;
                        case 'f' :
                                config_file = xstrdup(optarg);
                        default :
                                usage(1);
                                exit( -1);
                                break;
                }
        }
}


void sig_hand(int sig)
{
        if (sig == SIGWINCH)
        {
                printlog(1, "New screen size = %dLx%dC\n", LINES, COLS);
                redisp();
        }
        else
        {
                printlog(1, "Signal error (%d)\n", sig);
        }
        signal(SIGWINCH, sig_hand);
}


void tmms_init()
{
        pthread_mutexattr_t mutex_attr;

        /* Load configuration file */
        if (!config_file)
        {
                char * path = robust_home(NULL);
                int    len  = strlen(path) + strlen(DEFAULT_CFGFILE) + 3;

                config_file = xmalloc(len);
                snprintf(config_file, len, "%s/%s", path, DEFAULT_CFGFILE);
        }
        load_config(options_list, config_file);
        convert_colors();

        /* Mutex initialization */
        pthread_mutexattr_init(&mutex_attr);
        pthread_mutex_init(&mutex_request, &mutex_attr);
        pthread_mutex_init(&mutex_status,  &mutex_attr);
        pthread_mutex_init(&mutex_current, &mutex_attr);
        pthread_mutex_init(&mutex_display, &mutex_attr);
        pthread_mutex_lock(&mutex_request);

#if DEBUG_LEVEL!=0
        /* Debug logfile */
        if (lgfile)
        {
                logfile = fopen(LOGFILE, "w");
        }
#endif

        /* Modules initialization */
        mod_list = NULL;
        register_mp3();
        register_ogg();
        register_cda();

#ifdef COLORIZED
        uses_color = TRUE;
#else
        uses_color = FALSE;
#endif
        /* activate and save current locales */
        all_locale = setlocale(LC_ALL, "");

        /* where to find locale files */
        /*bindtextdomain(PACKAGE, LOCALEDIR); */
        /* set locale domain */
        /*textdomain(PACKAGE); */

        /* Init libncurses */
        init_curses(uses_color, NB_COLOR_ELEMENTS, color);

        /* Terminal resizing */
        signal(SIGWINCH, sig_hand);
}


void tmms_deinit()
{
#if DEBUG_LEVEL!=0
        if (lgfile && logfile)
        {
                fclose(logfile);
        }
#endif

        pthread_mutex_destroy(&mutex_request);
        pthread_mutex_destroy(&mutex_status);
        pthread_mutex_destroy(&mutex_current);
        pthread_mutex_destroy(&mutex_display);

        playlist_free();
        unregister_modules();
        clear();
        refresh();
        endwin();
}


int main(int argc, char *argv[])
{
        pthread_t pthread_id;

        parse_argv(argc, argv);
        tmms_init();
        load_playlist(playlist_loc);

        if (pthread_create(&pthread_id, NULL, (void*)thread_control, NULL) == -1)
        {
                fprintf(stderr, "Error creating player control thread.\n");
        }
        else
        {
                display_player();
                display_playlist();
                interact();
        }

        tmms_deinit();

        return 0;
}

