/*
 * Copyright (c) 2003 Patrick van Staveren <pvanstav@cs.wmich.edu>
 * Copyright (c) 2005 Kemal Ilgar Eroglu <kieroglu@math.washington.edu>
 * Copyright (c) 2005 Mario Streiber <mario.streiber@gmx.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfce4panel/xfce-panel-plugin.h>

#include "preferences.h"
#include "xmms_plugin.h"
#include "playerctrl.h"


/************************************/
/* callback to change the font size */
/************************************/
static void set_font_size(GtkSpinButton *spin, gpointer data) {
	plugin_data *pd     = (plugin_data*) data;
	PangoAttrSize *attr = (PangoAttrSize*) pd->labelattr;
	pd->titletextsize   = gtk_spin_button_get_value_as_int(spin);

	if (pd->titletextsize < MIN_TITLE_SIZE) pd->titletextsize = 1;
	if (pd->titletextsize > MAX_TITLE_SIZE) pd->titletextsize = MAX_TITLE_SIZE;
	attr->size = pd->titletextsize * PANGO_SCALE;
	gtk_label_set_attributes(GTK_LABEL(pd->label), pd->labelattrlist);
	gtk_widget_hide(pd->viewport);
	gtk_widget_show(pd->viewport);
}


/***************************************/
/* callback to change the scroll speed */
/***************************************/
static void set_scroll_speed(GtkSpinButton *spin, gpointer data) {
	plugin_data *pd  = (plugin_data*) data;
	pd->scroll_speed = gtk_spin_button_get_value_as_int(spin);

	if (pd->scroll_speed < MIN_SCROLL_SPEED) pd->scroll_speed = MIN_SCROLL_SPEED;
	if (pd->scroll_speed > MAX_SCROLL_SPEED) pd->scroll_speed = MAX_SCROLL_SPEED;
	pd->timer_reset = TRUE;
}


/********************************************/
/* callback to change the scroll step width */
/********************************************/
static void set_scroll_step(GtkSpinButton *spin, gpointer data) {
	plugin_data *pd             = (plugin_data*) data;
	gint s                      = gtk_spin_button_get_value_as_int(spin);

	if (s < MIN_SCROLL_STEP) s  = MIN_SCROLL_STEP;
	if (s > MAX_SCROLL_STEP) s  = MAX_SCROLL_STEP;
	pd->scroll_step = s;
	if (s == 0) pd->title_scroll_position = 0;
}


/***************************************/
/* callback to change the scroll delay */
/***************************************/
static void set_scroll_delay(GtkSpinButton *spin, gpointer data) {
	plugin_data *pd             = (plugin_data*) data;
	gint d                      = gtk_spin_button_get_value_as_int(spin);

	if (d < MIN_SCROLL_DELAY) d = MIN_SCROLL_DELAY;
	if (d > MAX_SCROLL_DELAY) d = MAX_SCROLL_DELAY;
	pd->scroll_delay            = d;
	pd->step_delay              = d * pd->scroll_speed * pd->scroll_step;
	pd->title_scroll_position   = 0;
}


/***************************************************************/
/* callback to toggle the visibility of the scrolled songtitle */
/***************************************************************/
static void show_title(GtkToggleButton *check, gpointer data) {
	plugin_data *pd             = (plugin_data*) data;
	pd->show_scrolledtitle      = gtk_toggle_button_get_active(check);

	/* show/hide the event box parent of the scrolled title */
	if (pd->show_scrolledtitle)   gtk_widget_show_all(pd->viewport);
	else                          gtk_widget_hide_all(pd->viewport);
}


/***************************************/
/* callback to change quit xmms option */
/***************************************/
static void quit_xmms_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd = (plugin_data*) data;
	pd->quit_xmms   = gtk_toggle_button_get_active(button);
}


/******************************************/
/* callback to change simple title option */
/******************************************/
static void simple_title_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd    = (plugin_data*) data;
	pd->simple_title   = gtk_toggle_button_get_active(button);
	set_song_title(pd);
}

/*********************************************************/
/* callback for show track postion (pbar_visible) option */
/*********************************************************/
static void pbar_visible_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd   = (plugin_data*) data;
	pd->pbar_visible  = gtk_toggle_button_get_active(button);

	if(pd->pbar_visible)
		gtk_widget_show_all (pd->pbar);
	else
		gtk_widget_hide_all (pd->pbar);
}

/******************************************************/
/* callback for show volume (vol_pbar_visible) option */
/******************************************************/
static void vol_pbar_visible_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd        = (plugin_data*) data;
	pd->vol_pbar_visible   = gtk_toggle_button_get_active(button);

	if(pd->vol_pbar_visible)
		gtk_widget_show_all (pd->vol_pbar);
	else
		gtk_widget_hide_all (pd->vol_pbar);
}

/********************************************/
/* callback for "hor_vol_if_vertical" option */
/*********************************************/
static void hor_vol_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd  = (plugin_data*) data;
	pd->hor_vol_if_vertical   = gtk_toggle_button_get_active(button);
	adjust_vol_pbar(pd);
}

/**
 * callback for "Show menu button" option
 */
static void menu_btn_visible_toggled(GtkToggleButton *button, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	pd->menu_btn_visible = gtk_toggle_button_get_active(button);

	if (pd->menu_btn_visible)
		gtk_widget_show(pd->menu_btn);
	else
		gtk_widget_hide(pd->menu_btn);
}

/**
 * callback for "Media Player" option
 */
static void player_selection_changed(GtkComboBox *widget, gpointer data) {
	plugin_data *pd = (plugin_data *) data;
	GtkTreeIter iter;
	GValue val = {0};

	gtk_combo_box_get_active_iter(widget, &iter);
	gtk_tree_model_get_value(gtk_combo_box_get_model(widget), &iter, 0, &val);
	pd->player = g_value_get_int(&val);
	g_value_unset(&val);
	player_control_close(pd->pctrl);
	pd->pctrl = player_control_get_instance(pd->player);
}

/*****************************************************/
/* add label to the left column of the options table */
/*****************************************************/
static void add_label(GtkWidget *table, gchar *text, gint pos) {
	gint att_opts               = GTK_SHRINK | GTK_EXPAND | GTK_FILL;
	GtkWidget *label            = gtk_label_new(text);

	gtk_misc_set_alignment        (GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach              (GTK_TABLE(table), label, 0, 1, pos, pos+1, att_opts, att_opts, 0, 0);
}


/*************************************************************/
/* add spin button to the right columns of the options table */
/*************************************************************/
static void add_spin(GtkWidget *table, gint min, gint max, gint value, GCallback cb, gpointer data, gint pos) {
	gint att_opts               = GTK_SHRINK/* | GTK_EXPAND | GTK_FILL*/;
	GtkWidget *spin             = gtk_spin_button_new_with_range(min, max, 1.0);

	gtk_spin_button_set_value     (GTK_SPIN_BUTTON(spin), value);
	gtk_table_attach              (GTK_TABLE(table), spin, 1, 2, pos, pos+1, att_opts, att_opts, 0, 0);
	g_signal_connect              (G_OBJECT(spin), "value-changed", cb, data);
}

/********************/
/* add check button */
/********************/
static void add_check(GtkWidget *parent, gchar *title, gboolean active, GCallback cb, gpointer data, const char* tip_title) {
	GtkWidget *check            = gtk_check_button_new_with_label(title);

	gtk_toggle_button_set_active  (GTK_TOGGLE_BUTTON(check), active);
	gtk_box_pack_start            (GTK_BOX(parent), check, DOEXPAND, DOFILL, PADDING);
	if(tip_title)
		gtk_tooltips_set_tip    (((plugin_data *) data)->tooltip, check, tip_title, NULL);
	g_signal_connect              (G_OBJECT(check), "toggled", cb, data);
}

/**
 * add media player selection combo box
 */
static void add_player_selection_box(GtkWidget *parent, plugin_data *pd) {
	GtkListStore *store;
	GtkTreeIter iter, a_iter;
	GtkCellRenderer *renderer;
	GtkWidget *combo, *ebox;

	store = gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
	combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(store));
	g_signal_connect(G_OBJECT(combo), "changed",
			G_CALLBACK(player_selection_changed), pd);

	// invisible column stores player ID
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo), renderer, "text", 0, NULL);
	g_object_set (renderer, "visible", FALSE, NULL);

	// player name column
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, FALSE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo), renderer, "text", 1, NULL);

	// add items
	if (player_control_has_library("xmms")) {
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, PLAYER_XMMS, 1, "XMMS", -1);
		if (pd->player == PLAYER_XMMS)
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combo), &iter);
	}
	if (player_control_has_library("beep")) {
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, PLAYER_BEEP, 1, "BeepMediaPlayer", -1);
		if (pd->player == PLAYER_BEEP)
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combo), &iter);
	}
	if (player_control_has_library("audacious")) {
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter, 0, PLAYER_AUDACIOUS, 1, "Audacious", -1);
		if (pd->player == PLAYER_AUDACIOUS)
			gtk_combo_box_set_active_iter(GTK_COMBO_BOX(combo), &iter);
	}

	ebox = gtk_event_box_new();
	gtk_tooltips_set_tip(pd->tooltip, ebox, _(
				"Notice:\n"
				"Do not change this property from XMMS or BeepMediaPlayer to the other one when either one of these is running. It will crash because they use same socket address. For the same reason, you cannot run the other player when this plugin is running."), NULL);
	gtk_container_add(GTK_CONTAINER(ebox), combo);
	gtk_widget_show(combo);
	gtk_box_pack_start(GTK_BOX(parent), ebox, DOEXPAND, DOFILL, PADDING);
	gtk_widget_show(ebox);
}

/***************************/
/* options dialog response */
/***************************/
static void
options_dialog_response (GtkWidget *dlg, int reponse, plugin_data *pd)
{
	gtk_widget_destroy (dlg);
	xfce_panel_plugin_unblock_menu (pd->base);
	xmms_plugin_write_config(pd->base,pd);
}

/**********************/
/* preferences dialog */
/**********************/
void xmms_plugin_create_options (XfcePanelPlugin *plugin, plugin_data *pd) {

	GtkWidget *vbox, *table, *label, *size, *speed, *step, *delay, *hbox;
	gint att_opts = GTK_SHRINK | GTK_EXPAND | GTK_FILL;

	GtkWidget *dlg, *header;


	gtk_tooltips_disable(pd->tooltip);
	gtk_tooltips_enable(pd->tooltip);

	xfce_panel_plugin_block_menu (plugin);

	dlg = gtk_dialog_new_with_buttons (_("Properties"), 
			GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (plugin))),
			GTK_DIALOG_DESTROY_WITH_PARENT |
			GTK_DIALOG_NO_SEPARATOR,
			GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
			NULL);

	g_signal_connect (dlg, "response", G_CALLBACK (options_dialog_response),
			pd);

	gtk_container_set_border_width (GTK_CONTAINER (dlg), 2);

	header = xfce_create_header (NULL, _("Xfce4 XMMS Plugin Options"));
	gtk_widget_set_size_request (GTK_BIN (header)->child, 200, 32);
	gtk_container_set_border_width (GTK_CONTAINER (header), 6);
	gtk_widget_show (header);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), header,
			FALSE, TRUE, 0);

	vbox  = gtk_vbox_new      (FALSE, 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), vbox,
			FALSE, TRUE, 0);

	/* media player selection */
	hbox = gtk_hbox_new(FALSE, 6);
	label = gtk_label_new(_("Media Player"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, PADDING);
	add_player_selection_box(hbox, pd);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, DOEXPAND, DOFILL, PADDING);

	/* separator */
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), DOEXPAND, DOFILL, PADDING);

	table = gtk_table_new     (4, 2, FALSE);

	/* add table */
	gtk_box_pack_start        (GTK_BOX(vbox), table, DOEXPAND, DOFILL, PADDING);

	/* put labels into the left column of our table */
	add_label(table, _("Font Size"),        0);
	add_label(table, _("Scroll Speed"),     1);
	add_label(table, _("Scroll Stepwidth"), 2);
	add_label(table, _("Scroll Delay"),     3);

	/* put spin buttons into the right column */
	add_spin(table, MIN_TITLE_SIZE,   MAX_TITLE_SIZE,   pd->titletextsize, G_CALLBACK(set_font_size),    pd, 0);
	add_spin(table, MIN_SCROLL_SPEED, MAX_SCROLL_SPEED, pd->scroll_speed,  G_CALLBACK(set_scroll_speed), pd, 1);
	add_spin(table, MIN_SCROLL_STEP,  MAX_SCROLL_STEP,  pd->scroll_step,   G_CALLBACK(set_scroll_step),  pd, 2);
	add_spin(table, MIN_SCROLL_DELAY, MAX_SCROLL_DELAY, pd->scroll_delay,  G_CALLBACK(set_scroll_delay), pd, 3);

	/*new separator*/
	gtk_box_pack_start (GTK_BOX(vbox), gtk_hseparator_new(), DOEXPAND, DOFILL, PADDING);

	/* add check buttons for the scrolled title, progressbar and volume bar */
	add_check(vbox, _("Show scrolled song title"), pd->show_scrolledtitle, G_CALLBACK(show_title), pd, NULL);
	add_check(vbox, _("Show track position"), pd->pbar_visible, G_CALLBACK(pbar_visible_toggled), pd, NULL);
	add_check(vbox, _("Show volume level"), pd->vol_pbar_visible, G_CALLBACK(vol_pbar_visible_toggled), pd, NULL);
	/* add check button for show menu button option */
	add_check(vbox, _("Show menu button"), pd->menu_btn_visible,
			G_CALLBACK(menu_btn_visible_toggled), pd, NULL);
	/* add check button for simple title option */
	add_check(vbox, _("Simple song title format"), pd->simple_title, G_CALLBACK(simple_title_toggled), pd, NULL);
	/* add check button for hor_vol_if_vertical option */
	add_check(vbox, _("Horizontal volume bar"), 
			pd->hor_vol_if_vertical, G_CALLBACK(hor_vol_toggled), pd, NULL);  
	/* add check button for quit xmms option */
	add_check(vbox, _("Quit the player when plugin terminates"), pd->quit_xmms, G_CALLBACK(quit_xmms_toggled), pd, NULL);

	gtk_widget_show_all(GTK_WIDGET(dlg));

}

/*
 * Store/Restore preferences
 */

/**************************/
/* read plugin parameters */
/**************************/
void xmms_plugin_read_config(plugin_data *pd) {

	XfceRc *rc;
	char *file;
	gint n;

	if ((file = xfce_panel_plugin_lookup_rc_file (pd->base)) != NULL)
	{
		rc = xfce_rc_simple_open (file, TRUE);
		g_free (file);

		if (rc != NULL)
		{
			/* load title font size */
			n = xfce_rc_read_int_entry (rc,"textsize",10);
			if ((n >= MIN_TITLE_SIZE) && (n <= MAX_TITLE_SIZE))
				pd->titletextsize = n;

			/* load scroll speed */
			n = xfce_rc_read_int_entry (rc, "scroll_speed",5);
			if ((n >= MIN_SCROLL_SPEED) && (n <= MAX_SCROLL_SPEED)) {
				pd->scroll_speed = n;
				pd->timer_reset = TRUE;
			}

			/* load scroll step width */
			n = xfce_rc_read_int_entry (rc,"scroll_step", 2);
			if ((n >= MIN_SCROLL_STEP) && (n <= MAX_SCROLL_STEP)) pd->scroll_step = n;

			/* load scroll delay */
			n = xfce_rc_read_int_entry (rc,"scroll_delay", 0);
			if ((n >= MIN_SCROLL_DELAY) && (n <= MAX_SCROLL_DELAY)) pd->scroll_delay = n;

			/* load xmms window visibility */
			pd->xmmsvisible = xfce_rc_read_bool_entry (rc, "xmms_visible", TRUE);

			/* load visibility of scrolled sing title */
			pd->show_scrolledtitle = xfce_rc_read_bool_entry (rc, "title_visible", TRUE);

			/* load quit xmms option */
			pd->quit_xmms = xfce_rc_read_bool_entry (rc, "quit_xmms", FALSE);

			/* load simple title option */
			pd->simple_title = xfce_rc_read_bool_entry (rc, "simple_title", TRUE);

			/* load progressbar visibility option */
			pd->pbar_visible = xfce_rc_read_bool_entry (rc, "pbar_visible", TRUE);


			/* load volume progressbar visibility option */
			pd->vol_pbar_visible =xfce_rc_read_bool_entry(rc,"vol_pbar_visible",TRUE);   	  

			/* load horizontal volume bar if vertical panel option */
			pd->hor_vol_if_vertical = 
				xfce_rc_read_bool_entry (rc, "hor_vol_if_vertical",TRUE);

			/* load show menu button option*/
			pd->menu_btn_visible = xfce_rc_read_bool_entry(rc, "menu_btn_visible", TRUE);

			/* load player option */
			n = xfce_rc_read_int_entry(rc, "player", PLAYER_XMMS);
			if ((n >= PLAYER_XMMS) && (n <= PLAYER_AUDACIOUS))
				pd->player = n;

			xfce_rc_close(rc);
		}	

	}

}


/**************************/
/* save plugin parameters */
/**************************/
void xmms_plugin_write_config(XfcePanelPlugin *plugin, plugin_data *pd) {

	XfceRc *rc;
	char *file;

	g_return_if_fail(plugin!=NULL);

	if (!(file = xfce_panel_plugin_save_location (plugin, TRUE)))
		return;

	rc = xfce_rc_simple_open (file, FALSE);
	g_free (file);

	if (!rc)
		return;


	/* save title font size */
	xfce_rc_write_int_entry (rc,"textsize",pd->titletextsize);

	/* save scroll speed */
	xfce_rc_write_int_entry (rc, "scroll_speed",pd->scroll_speed);

	/* save scroll step width */
	xfce_rc_write_int_entry (rc,"scroll_step", pd->scroll_step);

	/* save scroll delay */
	xfce_rc_write_int_entry (rc,"scroll_delay", pd->scroll_delay);

	/* save xmms window visibility */
	xfce_rc_write_bool_entry (rc, "xmms_visible", pd->xmmsvisible);

	/* save visibility of scrolled sing title */
	xfce_rc_write_bool_entry (rc, "title_visible", pd->show_scrolledtitle);

	/* save quit xmms option */
	xfce_rc_write_bool_entry (rc, "quit_xmms", pd->quit_xmms);

	/* save simple title option */
	xfce_rc_write_bool_entry (rc, "simple_title", pd->simple_title);

	/* save progressbar visibility option */
	xfce_rc_write_bool_entry (rc, "pbar_visible", pd->pbar_visible);  

	/* save volume progressbar visibility option */
	xfce_rc_write_bool_entry (rc, "vol_pbar_visible", pd->vol_pbar_visible);  

	/* save horizontal volume bar if vertical panel option */
	xfce_rc_write_bool_entry (rc, "hor_vol_if_vertical", pd->hor_vol_if_vertical);

	/* save show menu button option */
	xfce_rc_write_bool_entry(rc, "menu_btn_visible", pd->menu_btn_visible);

	/* save player option */
	xfce_rc_write_int_entry(rc, "player", pd->player);

	xfce_rc_close(rc);   
}
