/* queue.c --
 * Copyright 2007 Red Hat Inc., Durham, North Carolina.
 * All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Authors:
 *      Steve Grubb <sgrubb@redhat.com>
 */

#include "config.h"
#include <stdlib.h>
#include <pthread.h>
#include <syslog.h>
#include "queue.h"

static volatile event_t **q;
static pthread_mutex_t queue_lock;
static pthread_cond_t queue_nonempty;
static unsigned int q_next, q_last, q_depth;


int init_queue(unsigned size)
{
	unsigned int i;

	q_next = 0;
	q_last = 0;
	q_depth = size;
	q = malloc(q_depth * sizeof(event_t *));
	if (q == NULL)
		return -1;

	for (i=0; i<q_depth; i++) 
		q[i] = NULL;

	/* Setup IPC mechanisms */
	pthread_mutex_init(&queue_lock, NULL);
	pthread_cond_init(&queue_nonempty, NULL);

	return 0;
}

void enqueue(event_t *e)
{
	unsigned int n, retry_cnt = 0;

retry:
	// We allow 1 retry and then its over
	if (retry_cnt > 1) {
		syslog(LOG_ERR, "queue is full - dropping event");
		return;
	}
	pthread_mutex_lock(&queue_lock);

	// OK, have lock add event
	n = q_next%q_depth;
	if (q[n] == NULL) {
		q[n] = e;
		q_next = (n+1) % q_depth;
		pthread_mutex_unlock(&queue_lock);
		pthread_cond_signal(&queue_nonempty);
	} else {
		pthread_mutex_unlock(&queue_lock);
		pthread_yield(); // Let dequeue thread run to clear queue
		retry_cnt++;
		goto retry;
	}
}

event_t *dequeue(void)
{
	event_t *e;
	unsigned int n;

	// Wait until its got something in it
	pthread_mutex_lock(&queue_lock);
	n = q_last%q_depth;
	if (q[n] == NULL) {
		pthread_cond_wait(&queue_nonempty, &queue_lock);
		n = q_last%q_depth;
	}

	// OK, grab the next event
	if (q[n] != NULL) {
		e = q[n];
		q[n] = NULL;
		q_last = (n+1) % q_depth;
	} else
		e = NULL;

	pthread_mutex_unlock(&queue_lock);

	// Process the event
	return e;
}

void nudge_queue(void)
{
	pthread_cond_signal(&queue_nonempty);
}

void destroy_queue(void)
{
	unsigned int i;

	for (i=0; i<q_depth; i++)
		free(q[i]);

	free(q);
}

