/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2006  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "device-store.h"

#include "marshal.h"

#define RSSI_AS_PROGRESS 1

static GtkTreeModel *store = NULL;
static GtkTreeModel *model = NULL;
static GtkWidget *button;
static GtkWidget *cancel;
static GtkWidget *periodic;
static GtkWidget *sort;
static GtkWidget *tree;
static GtkWidget *statusbar;
static GtkWidget *progress;
static GtkWidget *treeview;
static GtkWidget *iconview;

static gboolean periodic_running = FALSE;

static gboolean message_remove(gpointer data)
{
	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), 0);

	return FALSE;
}

static void delete_callback(GtkWidget *window, GdkEvent *event,
						gpointer user_data)
{
	gtk_widget_destroy(GTK_WIDGET(window));

	gtk_main_quit();
}

static void clicked_callback(GtkWidget *button, gpointer user_data)
{
	DBusGConnection *conn = user_data;
	DBusGProxy *object;
	GError *error = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Adapter");

	dbus_g_proxy_call(object, "DiscoverDevices", &error,
				G_TYPE_INVALID, G_TYPE_INVALID);

	if (error != NULL) {
		gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0,
							error->message);
		g_timeout_add(3000, message_remove, NULL);
		g_error_free(error);
	}
}

static void cancel_callback(GtkWidget *button, gpointer user_data)
{
	DBusGConnection *conn = user_data;
	DBusGProxy *object;
	GError *error = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Adapter");

	dbus_g_proxy_call(object, "CancelDiscovery", &error,
				G_TYPE_INVALID, G_TYPE_INVALID);

	if (error != NULL) {
		gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0,
							error->message);
		g_timeout_add(3000, message_remove, NULL);
		g_error_free(error);
	}
}

static void periodic_callback(GtkWidget *widget, gpointer user_data)
{
	DBusGConnection *conn = user_data;
	DBusGProxy *object;
	GError *error = NULL;
	gboolean mode;

	mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

	if (mode == periodic_running)
		return;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Adapter");

	if (mode == TRUE) {
		dbus_g_proxy_call(object, "StartPeriodicDiscovery", &error,
					G_TYPE_INVALID, G_TYPE_INVALID);

		if (error != NULL) {
			gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0,
							error->message);
			g_timeout_add(3000, message_remove, NULL);
			g_error_free(error);

			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget),
									FALSE);
		}
	} else {
		dbus_g_proxy_call(object, "StopPeriodicDiscovery", &error,
					G_TYPE_INVALID, G_TYPE_INVALID);

		if (error != NULL) {
			gtk_statusbar_push(GTK_STATUSBAR(statusbar), 0,
							error->message);
			g_timeout_add(3000, message_remove, NULL);
			g_error_free(error);

			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget),
									TRUE);
		}
	}
}

static void toggled_callback(GtkWidget *button, gpointer user_data)
{
	gboolean mode;

	mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	if (mode == TRUE) {
		gtk_widget_hide(GTK_WIDGET(treeview));
		gtk_widget_show(GTK_WIDGET(iconview));
		gtk_widget_hide(GTK_WIDGET(sort));
	} else {
		gtk_widget_show(GTK_WIDGET(sort));
		gtk_widget_hide(GTK_WIDGET(iconview));
		gtk_widget_show(GTK_WIDGET(treeview));
	}
}

static void sort_callback(GtkWidget *button, gpointer user_data)
{
	gboolean mode;

	mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	if (mode == TRUE)
		gtk_tree_view_set_model(GTK_TREE_VIEW(tree), model);
	else
		gtk_tree_view_set_model(GTK_TREE_VIEW(tree), store);
}

static void class_to_text(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	guint32 class;
	gboolean valid;
	gchar *str;

	gtk_tree_model_get(model, iter, 2, &class, 6, &valid, -1);

	if (class != 0 && valid == TRUE) {
		str = g_strdup_printf("0x%06x", class);
		g_object_set(cell, "text", str, NULL);
		g_free(str);
	}

	g_object_set(cell, "visible", valid, NULL);
}

#if RSSI_AS_PROGRESS
static void rssi_to_value(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gint rssi, value;
	gchar *str;

	gtk_tree_model_get(model, iter, 3, &rssi, -1);

	if (rssi == 0) {
		g_object_set(cell, "text", "", NULL);
		g_object_set(cell, "visible", FALSE, NULL);
	} else {
		str = g_strdup_printf("%d", rssi);
		g_object_set(cell, "text", str, NULL);
		g_free(str);

		g_object_set(cell, "visible", TRUE, NULL);
	}

	value = (rssi < 0) ? 100 + rssi : 100 - rssi;

	if (value > 100)
		value = 100;

	if (value < 0 || rssi == 0)
		value = 0;

	g_object_set(cell, "value", value, NULL);
}
#endif

static void create_window(DBusGConnection *conn)
{
	GtkWidget *window;
	GtkWidget *scrolled;
	GtkWidget *icon;
	GtkWidget *vbox;
	GtkWidget *toggle;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	gtk_window_set_title(GTK_WINDOW(window), "Bluetooth proximity");

	gtk_window_set_icon_name(GTK_WINDOW(window), "stock_bluetooth");

	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

	gtk_window_set_default_size(GTK_WINDOW(window), 480, 360);

	vbox = gtk_vbox_new(FALSE, 7);

	scrolled = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(vbox), scrolled);

	tree = gtk_tree_view_new();

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), TRUE);

	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree), TRUE);

	column = gtk_tree_view_column_new();

	gtk_tree_view_column_set_title(column, "Device");

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_set_spacing(column, 4);
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer, "icon-name", 4);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_set_attributes(column, renderer, "text", 1, NULL);
	gtk_tree_view_column_add_attribute(column, renderer, "sensitive", 6);

	gtk_tree_view_append_column(GTK_TREE_VIEW(tree), column);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree),
			-1, "Class", renderer, class_to_text, NULL, NULL);

#if RSSI_AS_PROGRESS
	renderer = gtk_cell_renderer_progress_new();
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree),
			-1, "RSSI", renderer, rssi_to_value, NULL, NULL);
#else
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(tree),
		-1, "RSSI", gtk_cell_renderer_text_new(), "text", 3, NULL);
#endif

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 0);

	gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 280);

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 1);

	gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 100);

	column = gtk_tree_view_get_column(GTK_TREE_VIEW(tree), 2);

	gtk_tree_view_column_set_min_width(GTK_TREE_VIEW_COLUMN(column), 80);

	store = GTK_TREE_MODEL(bluetooth_device_store_new());

	model = gtk_tree_model_sort_new_with_model(store);

	gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model), 3,
			bluetooth_device_store_rssi_compare_func, NULL, NULL);

	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model),
						3, GTK_SORT_ASCENDING);

	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), GTK_TREE_MODEL(store));

	gtk_widget_grab_focus(GTK_WIDGET(tree));

	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	treeview = scrolled;

	scrolled = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(vbox), scrolled);

	icon = gtk_icon_view_new_with_model(GTK_TREE_MODEL(store));

	gtk_icon_view_set_text_column(GTK_ICON_VIEW(icon), 1);

	gtk_icon_view_set_pixbuf_column(GTK_ICON_VIEW(icon), 5);

	gtk_icon_view_set_item_width(GTK_ICON_VIEW(icon), 120);

	gtk_container_add(GTK_CONTAINER(scrolled), icon);

	iconview = scrolled;

	sort = gtk_check_button_new_with_label(_("Sort by RSSI"));

	g_signal_connect(G_OBJECT(sort), "toggled",
				G_CALLBACK(sort_callback), NULL);

	gtk_box_pack_start(GTK_BOX(vbox), sort, FALSE, FALSE, 0);

	toggle = gtk_check_button_new_with_label(_("Icon view"));

	g_signal_connect(G_OBJECT(toggle), "toggled",
				G_CALLBACK(toggled_callback), NULL);

	gtk_box_pack_start(GTK_BOX(vbox), toggle, FALSE, FALSE, 0);

	periodic = gtk_check_button_new_with_label(_("Periodic discovery"));

	g_signal_connect(G_OBJECT(periodic), "toggled",
				G_CALLBACK(periodic_callback), conn);

	gtk_box_pack_start(GTK_BOX(vbox), periodic, FALSE, FALSE, 0);

	progress = gtk_progress_bar_new();

	gtk_box_pack_start(GTK_BOX(vbox), progress, FALSE, FALSE, 2);

	button = gtk_button_new_with_label(_("Search for devices"));

	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);

	gtk_button_set_focus_on_click(GTK_BUTTON(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 4);

        g_signal_connect(G_OBJECT(button), "clicked",
                                G_CALLBACK(clicked_callback), conn);

	cancel = gtk_button_new_with_label(_("Stop discovery"));

	gtk_button_set_relief(GTK_BUTTON(cancel), GTK_RELIEF_NONE);

	gtk_button_set_focus_on_click(GTK_BUTTON(cancel), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), cancel, FALSE, FALSE, 4);

        g_signal_connect(G_OBJECT(cancel), "clicked",
                                G_CALLBACK(cancel_callback), conn);

	statusbar = gtk_statusbar_new();

	gtk_box_pack_start(GTK_BOX(vbox), statusbar, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(window), "delete-event",
				G_CALLBACK(delete_callback), conn);

	gtk_widget_show_all(window);

	gtk_widget_hide(cancel);

	gtk_widget_hide(GTK_WIDGET(progress));

	gtk_widget_hide(GTK_WIDGET(iconview));
}

static void remote_device_found(DBusGProxy *object, const char *address,
		const unsigned int class, const int rssi, gpointer user_data)
{
	DBusGConnection *conn = user_data;

	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(progress));

	bluetooth_device_store_add_device(BLUETOOTH_DEVICE_STORE(store),
							address, class, rssi);

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Test");

	dbus_g_proxy_call(object, "AuditRemoteDevice", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID);	
}

static void remote_device_disappeared(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	DBusGConnection *conn = user_data;

	bluetooth_device_store_del_device(BLUETOOTH_DEVICE_STORE(store),
								address);

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Test");

	dbus_g_proxy_call(object, "CancelAuditRemoteDevice", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID);	
}

static void remote_name_updated(DBusGProxy *object, const char *address,
					const char *name, gpointer user_data)
{
	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(progress));

	bluetooth_device_store_add_name(BLUETOOTH_DEVICE_STORE(store),
								address, name);
}

static void discovery_started(DBusGProxy *object, gpointer user_data)
{
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_widget_hide(GTK_WIDGET(button));

	gtk_widget_show(GTK_WIDGET(cancel));

	gtk_widget_show(GTK_WIDGET(progress));

	gtk_progress_bar_pulse(GTK_PROGRESS_BAR(progress));

	bluetooth_device_store_invalidate_devices(BLUETOOTH_DEVICE_STORE(store));
}

static void discovery_completed(DBusGProxy *object, gpointer user_data)
{
	gtk_widget_hide(GTK_WIDGET(progress));

	gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);

	gtk_widget_hide(GTK_WIDGET(cancel));

	gtk_widget_show(GTK_WIDGET(button));
}

static void periodic_discovery_started(DBusGProxy *object, gpointer user_data)
{
	periodic_running = TRUE;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(periodic), TRUE);
}

static void periodic_discovery_stopped(DBusGProxy *object, gpointer user_data)
{
	periodic_running = FALSE;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(periodic), FALSE);
}

static void setup_dbus(DBusGConnection *conn)
{
	DBusGProxy *object;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
				"/org/bluez/hci0", "org.bluez.Adapter");

	dbus_g_object_register_marshaller(marshal_VOID__STRING_UINT_INT,
				G_TYPE_NONE, G_TYPE_STRING, G_TYPE_UINT,
						G_TYPE_INT, G_TYPE_INVALID);

	dbus_g_proxy_add_signal(object, "RemoteDeviceFound",
		G_TYPE_STRING, G_TYPE_UINT, G_TYPE_INT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "RemoteDeviceFound",
				G_CALLBACK(remote_device_found), conn, NULL);

	dbus_g_object_register_marshaller(marshal_VOID__STRING_STRING,
						G_TYPE_NONE, G_TYPE_STRING,
						G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_add_signal(object, "RemoteDeviceDisappeared",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "RemoteDeviceDisappeared",
			G_CALLBACK(remote_device_disappeared), conn, NULL);

	dbus_g_proxy_add_signal(object, "RemoteNameUpdated",
				G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "RemoteNameUpdated",
				G_CALLBACK(remote_name_updated), NULL, NULL);

	dbus_g_proxy_add_signal(object, "DiscoveryStarted", G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "DiscoveryStarted",
				G_CALLBACK(discovery_started), NULL, NULL);

	dbus_g_proxy_add_signal(object, "DiscoveryCompleted", G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "DiscoveryCompleted",
				G_CALLBACK(discovery_completed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "PeriodicDiscoveryStarted",
							G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "PeriodicDiscoveryStarted",
			G_CALLBACK(periodic_discovery_started), NULL, NULL);

	dbus_g_proxy_add_signal(object, "PeriodicDiscoveryStopped",
							G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "PeriodicDiscoveryStopped",
			G_CALLBACK(periodic_discovery_stopped), NULL, NULL);
}

int main(int argc, char *argv[])
{
	DBusGConnection *conn;
	GError *error = NULL;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	gtk_init(&argc, &argv);

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr(_("Connecting to system bus failed: %s\n"),
							error->message);
		g_error_free(error);
		exit(EXIT_FAILURE);
	}

	setup_dbus(conn);

	create_window(conn);

	gtk_main();

	g_object_unref(model);

	g_object_unref(store);

	dbus_g_connection_unref(conn);

	return 0;
}
