//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_AUDIO_HH
#define BMP_AUDIO_HH

#ifndef BMP_PLUGIN_BUILD

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <cstring>
#include <glibmm.h>

#include <boost/variant.hpp>
#include <boost/optional.hpp>

#include <gst/gst.h>
#include <gst/gstelement.h>
#include <gst/interfaces/mixer.h>
#include <gst/interfaces/mixertrack.h>
#include <gst/interfaces/mixeroptions.h>

#include "uri++.hh"
#include "util.hh"

namespace Bmp
{
  namespace Audio
  {

    enum Sink
    {
        SINK_ALSA,
        SINK_OSS,
        SINK_SUNAUDIO,
        SINK_ESD,
        SINK_HAL,
    };

#include "exception.hh"

    EXCEPTION(PipelineError)
    EXCEPTION(InvalidUriError)

    /** Elements enum
     */
    enum Elements
    {
        SOURCE,	
        DECODER,
        TEE,
        QUEUE,
        CONVERT,
        VOLUME,
        PUID,
        SINK,
        LAME,
        FLACENC,
        VORBISENC,
        OGGMUX,
            
        N_ELEMENTS
    };

    enum Caps
    {
        CAPS_NONE     = 0,
        CAPS_CDDA     = 1 << 0,
        CAPS_HTTP     = 1 << 1,
        CAPS_MMS      = 1 << 2,
    };

    /** Returns the current stream type playback capabilities 
    * @returns Playback Capabilities, see @link Bmp::Audio::Caps@endlink
    */
    Caps
    get_caps ();

    /** Names of the elements, corresponding to @link Bmp::Audio::Elements@endlink 
     */
    namespace Element  
    {
      namespace Variant
      {
          /** boost::variant type for bool, int, double and string values
           */
          typedef boost::variant<bool, int, double, std::string> Variant;

          /** Enum specifying the type of the variant 
           *  FIXME: This might not be needed
           */
          enum Type 
          {
            BOOL,
            INTEGER,
            DOUBLE,
            STRING,
          };
      }

      /** An attribute holds a @link Bmp::Audio::Element::Variant::Variant@endlink,  
       *  a @link Bmp::Audio::Element::Variant::Type@endlink and a name (std::string)
       *
       */
      struct Attribute
      {
        Variant::Variant  value;
        Variant::Type	    type;
        std::string	      name;

        /** Default ctor
         */
        Attribute () {};

        /** Ctor taking the value, type and the name 
         */
        Attribute (std::string name, Variant::Variant value) : value (value), name (name) { type = Bmp::Audio::Element::Variant::Type(value.which ()); }; 
      };

      /** std::vector typedef of an Attribute
       */
      typedef std::vector<Attribute> Attributes;

      /** An element consists of a a 'name' (std::string) and a list of attributes (@link Bmp::Audio::Element::Attributes@endlink)
       */
      struct Element
      {
        std::string	    name;
        Attributes	    attributes;

        /** Default ctor
         */
        Element () {};

         /** Ctor that takes the name of the element
         */
        Element (std::string name) : name (name) {};

        /** Ctor that takes the name and a reference to a vector of @link Bmp::Audio::Element::Attributes@endlink
         */
        Element (std::string name, const Attributes& attributes) : name (name), attributes (attributes) {};

         /** Adds an attribute to the list of element attributes
         * @param attribute An @link Bmp::Audio::Element::Attribute@endlink
         * @returns void
         */
        void
        add (const Attribute& attribute)
        {
          attributes.push_back (attribute);
        }
      };

    }//Element

  }//Audio

}//Bmp

namespace Bmp
{ 
  namespace Audio
  {
    /** Current state of the audio processing unit
     */ 
    enum ProcessorState
    {
      STATE_STOPPED,
      STATE_PAUSED,
      STATE_RUNNING,
    };

    /** Stream audioproperties 
      *
      */
    struct StreamProperties
    {
      unsigned int bitrate;
      unsigned int samplerate;
      std::string  title;

      StreamProperties () : bitrate (0), samplerate (0), title ("") {}
    };

    /** sigc signal typedef for the stream position
     */
    typedef sigc::signal<void, int>
	    SignalPosition;

    /** sigc signal typedef signalizing end-of-stream 
     */
    typedef sigc::signal<void>
	    SignalEos;

    /** sigc signal typedef signalizing error 
     */
    typedef sigc::signal<void, Glib::ustring>
	    SignalError;

    /** sigc signal typedef for stream properties
     */
    typedef sigc::signal<void, StreamProperties&>
	    SignalStreamProperties;

    /** This is the base class for the audio processors
      *
      */
    class ProcessorBase : public Glib::Object
    {
      public:

        ProcessorBase ();
        virtual ~ProcessorBase ();

        /** PropertyProxy to get or set the stream time.
         *  Setting the stream time equals to a seek.
         */ 
        virtual Glib::PropertyProxy<unsigned int>
        prop_stream_time ();

        /** PropertyProxy to get or set the time interval in millisecond at which to report the current stream time via SignalPosition
         */
        virtual Glib::PropertyProxy<unsigned int>
        prop_stream_time_report_interval ();

        /** ProcessorState PropertyProxy. Allows to read the current processor state, or set it which equals to one of ::stop (),
         *  ::run (), or ::pause ()
         */
        virtual Glib::PropertyProxy<ProcessorState>
        prop_state ();

        virtual Glib::PropertyProxy<unsigned int>
        prop_length();

        /** Volume, Range 0-100
         *
         */
        virtual Glib::PropertyProxy<int>
        prop_volume();

        virtual SignalPosition&
        signal_position();

        virtual SignalEos&
        signal_eos();

        virtual SignalStreamProperties&
        signal_stream_properties ();

        virtual SignalError&
        signal_error ();

        /** Starts the processor
         */
        virtual GstStateChangeReturn
        run   ();

              /** Stops the processor
         */
        virtual GstStateChangeReturn 
        stop  ();

        /** Puts the processor into pause mode
         */
        virtual GstStateChangeReturn 
        pause (); 

        /** Taps into the processor's pipeline
         * 
         * @returns The 'tee' element of the pipeline to connect a Processor_Source_* to 
         *
         */
        virtual GstElement*
        tap ();

      protected:

        virtual bool
        verify_pipeline ();

        GstElement *pipeline;

        Glib::Property<unsigned int>    prop_stream_time_;
        Glib::Property<unsigned int>	  prop_stream_time_report_interval_;
        Glib::Property<unsigned int>	  prop_length_;
        Glib::Property<int>             prop_volume_;
        Glib::Property<ProcessorState>	prop_state_;

        SignalPosition		              signal_position_;
        SignalEos		                    signal_eos_;
        SignalStreamProperties	        signal_stream_properties_;
        SignalError           	        signal_error_;

        sigc::connection                conn_position; 
        sigc::connection                conn_state; 

        StreamProperties                stream_properties;

        /** Creates a pipeline as specified by the current source and sink elements
         */
        virtual void
        create_pipeline () = 0;

        /** Preforms a position query on the current pipeline and emits the current stream position via @link SignalPosition@endlink
         */
        virtual bool
        emit_stream_position ();

        /** Stops emission of the current stream position 
         */
        virtual void
        position_send_stop ();

        /** Starts emission of the current stream position in intervals specified by prop_stream_time_report_interval()
         */
        void
        position_send_start ();

        /** Handler of state changes of the processor's state property 
         */
        void
        prop_state_changed ();

        /** Handler of state changes of the processor's volume property 
         */
        void
        prop_volume_changed ();

        /** Handler of changes to the interval for sending the stream's position 
         */
        void
        position_send_interval_change ();

        static void
        link_pad  (GstElement *element,
                   GstPad     *pad,
                   gboolean    last,
                   gpointer    data);

        static gboolean
        bus_watch (GstBus     *bus,
                   GstMessage *message,
                   gpointer    data);

        static gboolean
        foreach_structure (GQuark	 field_id,
                           const GValue	*value,
                           gpointer	 data);
    };


    /** This class can be used to play a stream file
     *
     */
    class ProcessorURISink : public ProcessorBase 
    {
      public:

        ProcessorURISink ();
        virtual ~ProcessorURISink ();

        /** Will set the uri to the ProcessorURISink
         * @param uri The uri to use. The processor will automatically adapt to the protocol of 
         *            the URI to construct an appropriate pipeline
         *
         * @param sink The @link Bmp::Audio::Element::Element@endlink to specify the sink
         */
        void
        set_uri (Glib::ustring const&                uri,
                 const Bmp::Audio::Element::Element& sink);

      private:

        virtual void
        create_pipeline ();

        Bmp::URI::Protocol current_protocol;
        Glib::ustring	     stream;

        Element::Element   source;	
        Element::Element   sink;
        
    };

#ifdef HAVE_OFA
    /** This class uses GstPUID to determine a file PUID 
     *
     */
    class ProcessorPUID : public ProcessorBase 
    {
      public:

        ProcessorPUID ();
        virtual ~ProcessorPUID ();

        /** Will set the uri to the ProcessorPUID
         * @param uri The uri to use
         */
        void
        set_uri (Glib::ustring const& uri);

        boost::optional<Glib::ustring>
        get_puid () const;

        boost::optional<Glib::ustring>
        get_artist () const;

        boost::optional<Glib::ustring>
        get_title () const;

      private:

        Glib::ustring	m_stream;

        boost::optional<Glib::ustring> m_puid;
        boost::optional<Glib::ustring> m_artist;
        boost::optional<Glib::ustring> m_title;

        void
        stream_eos ();

        virtual void
        create_pipeline ();

    };
#endif

    /** This class is a processor to rip an audiocd track to an mp3 file 
     *
     */
    class ProcessorCDDA_MP3 : public ProcessorBase 
    {
      public:
        /** ProcessorCDDA_MP3 ctor
         * @param path Destination path for the file
         * @param track CDDA Track Number 
         */
        ProcessorCDDA_MP3 (std::string const& path, unsigned int track, std::string const& device, int quality);
        virtual ~ProcessorCDDA_MP3 ();
      private:
        std::string   m_path;
        unsigned int  m_track;
        std::string   m_device;
        int           m_quality;
        virtual void
        create_pipeline ();
    };

    /** This class is a processor to rip an audiocd track to an FLAC file 
     *
     */
    class ProcessorCDDA_FLAC : public ProcessorBase 
    {
      public:
        /** ProcessorCDDA_FLAC ctor
         * @param path Destination path for the file
         * @param track CDDA Track Number 
         */
        ProcessorCDDA_FLAC (std::string const& path, unsigned int track, std::string const& device, int quality);
        virtual ~ProcessorCDDA_FLAC ();
      private:
        std::string   m_path;
        unsigned int  m_track;
        std::string   m_device;
        int           m_quality;
        virtual void
        create_pipeline ();
    };

    /** This class is a processor to rip an audiocd track to an Ogg Vorbis file 
     *
     */
    class ProcessorCDDA_Vorbis : public ProcessorBase 
    {
      public:
        /** ProcessorCDDA_Vorbis ctor
         * @param path Destination path for the file
         * @param track CDDA Track Number 
         */
        ProcessorCDDA_Vorbis (std::string const& path, unsigned int track, std::string const& device, int quality);
        virtual ~ProcessorCDDA_Vorbis ();
      private:
        std::string   m_path;
        unsigned int  m_track;
        std::string   m_device;
        int           m_quality;
        virtual void
        create_pipeline ();
    };

  }
}
#endif

namespace Bmp
{
  namespace Audio
  {
    bool typefind (const std::string& filename, std::string& type);
  }
}

#endif //BMP_AUDIO_HPP

