/* dbus_service.h
 * D-BUS Service Utilities
 *
 * Copyright © 2006 Red Hat, Inc.  All rights reserved.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions of
 * the GNU General Public License v.2.  This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY expressed or
 * implied, including the implied warranties of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.  You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA. Any Red Hat trademarks that are incorporated in the source code or
 * documentation are not subject to the GNU General Public License and may
 * only be used or replicated with the express permission of Red Hat, Inc.
 *
 * Red Hat Author(s):  Jason Vas Dias
 *                     David Cantrell
 */

#ifndef D_BUS_SERVER_UTILITIES_H
#define D_BUS_SERVER_UTILITIES_H

#include <stdint.h>
#include <stdarg.h>

typedef struct dbcs_s* DBUS_SVC;

typedef enum {
    HANDLED,
    NOT_HANDLED,
    HANDLED_NOW 
} dbus_svc_HandlerResult;

typedef enum {
    INVALID,
    CALL,
    RETURN,
    ERROR,
    SIGNAL 
} dbus_svc_MessageType;

typedef enum {
    DBUS_SESSION,
    DBUS_SYSTEM, 
    DBUS_STARTER 
} dbus_svc_DBUS_TYPE;

/* D-BUS Protocol Type Codes / Signature Chars */
typedef enum {
    TYPE_INVALID     = (int)'\0',
    TYPE_BYTE        = (int)'y',
    TYPE_BOOLEAN     = (int)'b',
    TYPE_INT16       = (int)'n',
    TYPE_UINT16      = (int)'q',
    TYPE_INT32       = (int)'i',
    TYPE_UINT32      = (int)'u',
    TYPE_INT64       = (int)'x',
    TYPE_UINT64      = (int)'t',
    TYPE_DOUBLE      = (int)'d',
    TYPE_STRING      = (int)'s',
    TYPE_OBJECT_PATH = (int)'o',
    TYPE_SIGNATURE   = (int)'g',
    TYPE_ARRAY       = (int)'a',
    TYPE_VARIANT     = (int)'v',
    TYPE_STRUCT      = (int)'r',
    TYPE_DICT_ENTRY  = (int)'e',
    STRUCT_BEGIN     = (int)'(',
    STRUCT_END       = (int)')',
    DICT_ENTRY_BEGIN = (int)'{',
    DICT_ENTRY_END   = (int)'}'
} dbus_svc_DataType;

typedef struct DBusMessage* dbus_svc_MessageHandle;

typedef dbus_svc_HandlerResult 
(*dbus_svc_MessageHandler)
( DBUS_SVC,
  dbus_svc_MessageType type,
  uint8_t  reply_expected,    /* 1 / 0                                      */
  uint32_t serial,            /* serial number of message; needed to reply  */
  char *destination,          /* D-BUS connection name / destination        */
  char *path,                 /* D-BUS Object Path                          */
  char *member,               /* D-BUS Object Member                        */
  char *interface,            /* D-BUS Object interface                     */
  char *if_suffix,            /* remainder of interface prefixed by         */
                              /*     ifPrefix                               */
  char *sender,               /* Senders' connection destination            */
  char *signature,            /* Signature String composed of Type Codes    */
  dbus_svc_MessageHandle msg, /* Message pointer: call                      */
                              /*     dbus_svc_get_args(msg,...) to get data */
  char *prefix,               /* If non-null, this is the root prefix for   */
                              /*     this sub-path message                  */
  char *suffix,               /* If non-null, this is the suffix of this    */
                              /*     sub-path message                       */
  void *prefixObject,         /* If non-null, this is the object that was   */
                              /*     registered for the prefix              */
  void *object                /* If non-null, this is the object that was   */
                              /*     registered for the complete path       */
); /* Message Handler function prototype */

#define DBusMsgHandlerArgs      \
    DBUS_SVC dbus,              \
    dbus_svc_MessageType type,  \
    uint8_t  reply_expected,    \
    uint32_t serial,            \
    char *destination,          \
    char *path,                 \
    char *member,               \
    char *interface,            \
    char *if_suffix,            \
    char *sender,               \
    char *signature,            \
    dbus_svc_MessageHandle msg, \
    char *prefix,               \
    char *suffix,               \
    void *prefixObject,         \
    void *object             

/* Error Handler function prototype - handle FATAL errors from D-BUS calls */
typedef int (*dbus_svc_ErrorHandler) (const char *errorFmt, ...);

/*
 * Obtains connection to DBUS_BUS_STARTER and registers "name".
 * "eh" will be called for all errors from this server session.
 */
extern DBUS_SVC dbus_svc_init(dbus_svc_DBUS_TYPE bus, char *name,
                              dbus_svc_ErrorHandler eh,
                              dbus_svc_ErrorHandler dh); 

/*
 * Close connections and clean up. 
 * DBUS_SVC pointer is invalid after this.
 */
extern void dbus_svc_shutdown(DBUS_SVC);

/*
 * Registers message handler with D-BUS to handle all messages sent 
 * to rootPath root Object Path.
 * If isPrefix > 1, this path may prefix other handled paths; ie. 
 * a "Fallback" handler is registered, and when a message is sent
 * to a path prefixed by the rootPath, if no sub-path handler has been 
 * registered  with "dbus_svc_add_handler", the message-handler is called
 * with "prefix" non-null, and "path" set to the suffix (sub-path); otherwise
 * the sub-path handler is called. 
 */
extern uint8_t dbus_svc_add_path_handler(DBUS_SVC, char *rootPath,
                                         char *ifPrefix,
                                         dbus_svc_MessageHandler mh,
                                         void *object, uint8_t isPrefix);

extern uint8_t dbus_svc_add_default_handler(DBUS_SVC,
                                            dbus_svc_MessageHandler mh,
                                            void *object);

/*
 * Registers with the 'dbus_svc' module a sub-path handler for objectPath
 * prefixed by an existing rootPath registered with dbus_svc_add_path_handler.
 * The registered handler will be called with "path" set to the complete
 * path, "prefix" set to the prefix portion, "suffix" set to the suffix,
 * "object" set to the this registered object pointer, and "prefixObject" set
 * to the object registered for the prefix path.
 */
extern uint8_t dbus_svc_add_handler(DBUS_SVC, char *rootPath,
                                    char *objectPath,
                                    dbus_svc_MessageHandler mh, void *object);

/*
 * Removes the root Object Path handler and unregisters it with D-BUS.
 * All sub-path handlers for this path are also removed.
 * If objectP is non-null, the address of the object stored with
 * dbus_svc_add_path_handler will be stored in it.
 */
extern uint8_t dbus_svc_remove_path_handler(DBUS_SVC, char *rootPath,
                                            void **objectP);

/*
 * Removes sub-path handler for objectPath under rootPath
 * and returns stored object in *objectP if not null.
 */
extern uint8_t dbus_svc_remove_handler(DBUS_SVC, char *rootPath,
                                       char *objectPath, void **objectP);

/*
 * get arguments from message
 */
extern uint8_t dbus_svc_get_args(DBUS_SVC, dbus_svc_MessageHandle,
                                 dbus_svc_DataType, ...);

/*
 * get arguments from message
 */
extern uint8_t dbus_svc_get_args_va(DBUS_SVC, dbus_svc_MessageHandle,
                                    dbus_svc_DataType, va_list);


/*
 * Unregisters the message handler
 */
extern uint8_t dbus_svc_remove_message_handler(DBUS_SVC, char *objectPath);

/*
 * Registers message handler to FILTER all messages sent to objectPath
 * before they are Handled.
 * If n_matches is > 0, then the variable argument list can contain n_matches
 * "const char*"s that are "match expressions" to pass to dbus_bus_add_match()  
 */
extern uint8_t dbus_svc_add_message_filter(DBUS_SVC,
                                           dbus_svc_MessageHandler mh,
                                           int n_matches, ...);

/*
 * Unregisters the message filter
 */
extern uint8_t dbus_svc_remove_message_filter(DBUS_SVC,
                                              dbus_svc_MessageHandler mh);

/*
 * sends messages / replies to "destination"
 * pointer, { (dbus_svc_DataType, pointer )...}
 */
extern uint8_t dbus_svc_send(DBUS_SVC, dbus_svc_MessageType type,
                             int32_t reply_serial, uint32_t *new_serial,
                             char *destination, char *path, char *member,
                             char *interface, dbus_svc_DataType firstType,
                             ...);

/*
 * sends messages / replies to "destination"
 */
extern uint8_t dbus_svc_send_va(DBUS_SVC cs, dbus_svc_MessageType type,
                                int32_t reply_serial, uint32_t *new_serial,
                                char *destination, char *path, char *member,
                                char *interface, dbus_svc_DataType firstType,
                                va_list va);

/*
 * constructs message, sends it, returns reply
 */
extern dbus_svc_MessageHandle dbus_svc_call(DBUS_SVC cs, char *destination,
                                            char *path, char *member,
                                            char *interface,
                                            dbus_svc_DataType firstType, ...);

extern dbus_svc_MessageHandle dbus_svc_new_message(DBUS_SVC cs,
                                                   dbus_svc_MessageType type,
                                                   int32_t reply_serial,
                                                   char *destination,
                                                   char *path,
                                                   char *member,
                                                   char *interface);

extern uint8_t dbus_svc_send_message(DBUS_SVC, dbus_svc_MessageHandle,
                                     uint32_t *); 

extern uint8_t dbus_svc_message_append_args(DBUS_SVC, dbus_svc_MessageHandle,
                                            dbus_svc_DataType, ...);

#endif
