// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999-2001 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include <cstdio>
#include "global.h"
#include "catalog.h"
#include "gui.h"
#include "buildwin.h"

#ifdef HAVE_LIBXML
#include <libxml/xmlmemory.h>
#include <libxml/parser.h>
#include <libxml/encoding.h>
#if (LIBXML_VERSION < 20000)
extern "C" xmlCharEncodingHandlerPtr
  xmlNewCharEncodingHandler(const char *name, xmlCharEncodingInputFunc input,
			    xmlCharEncodingOutputFunc output);
#endif
#ifdef USE_ICONV
#include <iconv.h>
#include <errno.h>
#endif
#else
#include "tinyxml.h"
#endif
#include "xmlsub.h"
#ifdef HAVE_LANGINFO_H
#include <langinfo.h>
#endif

#define DOC reinterpret_cast<xmlDocPtr>(preference_private)

NATIVE_STRING PREFERENCE::GetStringValue(const char* path){
  xmlDocPtr doc=reinterpret_cast<xmlDocPtr>(preference_private);
  xmlNodePtr np=XML_GetNodeFromXPath(doc,path);
  if (np!=NULL){
    return XML_GetString(doc,np);
  }
  return NATIVE_STRING("");
}

int PREFERENCE::GetIntValue(const char* path){
  std::vector<xmlNodePtr> vnp=XML_GetNodesFromXPath(DOC,path);
  if (vnp.size()>0){
    return XML_GetInt(DOC,vnp[0]);
  }
  return 0;
}

bool PREFERENCE::GetBoolValue(const char* path){
  std::vector<xmlNodePtr> vnp=XML_GetNodesFromXPath(DOC,path);
  if (vnp.size()>0){
    return XML_GetBool((xmlDocPtr)preference_private,vnp[0]);
  }
  return 0;
}

bool PREFERENCE::SetStringValue(const char* path,const NATIVE_STRING& str){
  xmlNodePtr node=XML_CreateNodesFromXPath(DOC,path);
  if (node==NULL) {
    return false;
  }
  return XML_SetString(DOC,node,str);
}

bool PREFERENCE::SetIntValue(const char* path,int val){
  char buf[1024];
  snprintf(buf,1024,"%d",val);
  NATIVE_STRING str=buf;
  return SetStringValue(path,str);
}

bool PREFERENCE::SetBoolValue(const char* path,bool val){
  NATIVE_STRING str="true";
  if (val!=true) str="false";
  return SetStringValue(path,str);
}

bool PREFERENCE::Read(const char* fileName){
  bool isCatalogs=false;
  bool isConfig=false;
  char fname[PATH_MAX+1];
  if (fileName!=NULL) {
    strncpy(fname,fileName,PATH_MAX);
  } else {
    if (getenv("HOME")!=NULL){
      strncpy(fname,getenv("HOME"),PATH_MAX);
      if (strlen(fname)>0 && fname[strlen(fname)-1]=='/') {
	fname[strlen(fname)-1]=0;
      }
    } else {
      fname[0]=0;
    }
    strncat(fname,"/.ebdic-preference",PATH_MAX);
  }
  fname[PATH_MAX]=0;
  FILE* fp=fopen(fname,"r");
  if (fp==NULL){
    if (mainWindow!=NULL){
      CreateMessageBox(mainWindow,
		       gtk_label_new(_("Prefenence File Not Found!")),NULL);
    }
    return false;
  } 
  int ch=getc(fp);
  if (0<=ch && ch<0x07) isCatalogs=true;
  if (ch=='<' || ch==0x1f) isConfig=true;
  fclose(fp);

  if (isCatalogs){
    char* fname=strdup(fileName);
    char* ptr=strrchr(fname,'/');
    if (ptr!=NULL) *ptr=0;
    catalogs.Append(fname,_("Dictionary"),0,0,4,0,"");
    free(fname);
    if (mainWindow!=NULL) SetDicMenu();
    return true;
  } 
  if (!isConfig) return false;

  xmlDocPtr doc=xmlParseFile(fname);

  xmlNodePtr cur;
  if (doc==NULL) return false;
  preference_private=reinterpret_cast<void*>(doc);

  currentDictNum=GetIntValue(PP "current-dictionary-number");


  cur=XML_GetNodeFromXPath(DOC,"/ebdic/dictionary");
  if (cur==NULL) return false;
  for(xmlNodePtr cur2=cur->xmlChildrenNode;cur2!=NULL;cur2=cur2->next){
    if (xmlIsBlankNode(cur2)) continue;
    if (XML_Match(cur2,"book")){
      NATIVE_STRING path,name,disableSubbook;
      int charOffset=0,charExtend=0,gaijiOffset=0;
      bool autoImageDisplay=false,disable=false;
      for(xmlNodePtr cur3=cur2->xmlChildrenNode;
	  cur3!=NULL;cur3=cur3->next){
	if (xmlIsBlankNode(cur3)) continue;
	if (XML_Match(cur3,"path")){
	  path=XML_GetString(doc,cur3);
	} else if (XML_Match(cur3,"name")){
	  name=XML_GetString(doc,cur3);
	} else if (XML_Match(cur3,"char-offset")){
	  charOffset=XML_GetInt(doc,cur3);
	} else if (XML_Match(cur3,"char-extend")){
	  charExtend=XML_GetInt(doc,cur3);
	} else if (XML_Match(cur3,"gaiji-offset")){
	  gaijiOffset=XML_GetInt(doc,cur3);
	} else if (XML_Match(cur3,"auto-image-display")){
	  autoImageDisplay=XML_GetBool(doc,cur3);
	} else if (XML_Match(cur3,"disable")){
	  disable=XML_GetBool(doc,cur3);
	} else if (XML_Match(cur3,"disable-subbook")){
	  disableSubbook=XML_GetString(doc,cur3);
	} else {
		// Unknow Element
	}
      }
      if (path.length()>0) {
	catalogs.Append(path,_EC(_UE(name)),charOffset,gaijiOffset,
			charExtend,autoImageDisplay,disable,disableSubbook);
      }
    } else {
    }
  }
  if (mainWindow!=NULL) SetDicMenu();
  return true;
};

bool PREFERENCE::ClearDictionaryList(){
  xmlDocPtr doc=(xmlDocPtr) preference_private;
  if (doc==NULL) return false;
  xmlNodePtr cur=xmlDocGetRootElement(doc);
  if (cur==NULL) return false;
  if (xmlStrcmp(cur->name,(const xmlChar *)"ebdic")!=0) return false;
  cur=cur->xmlChildrenNode;
  if (cur==NULL) return false;
  while(cur!=NULL && xmlStrcmp(cur->name,(const xmlChar*)"dictionary")!=0) {
    cur=cur->next;
  }
  if (cur==NULL) return false;
  xmlUnlinkNode(cur);
  xmlFreeNode(cur);
  return true;
};

static const xmlChar* IntString(char* buf,int len, int num){
  snprintf(buf,len,"%d",num);
  return (xmlChar*)buf;
}

static const xmlChar* BoolString(bool b){
  if (b) return (const xmlChar*)"true"; else return (const xmlChar*)"false";
}

bool PREFERENCE::CreateDictionaryList(){
  char buf[10];
  xmlDocPtr doc=(xmlDocPtr) preference_private;
  if (doc==NULL) return false;
  xmlNodePtr cur=xmlDocGetRootElement(doc);
  if (cur==NULL) return false;
  if (xmlStrcmp(cur->name,(const xmlChar *)"ebdic")!=0) return false;
  if (cur==NULL) return false;
  xmlNodePtr diclist=xmlNewChild(cur,NULL,(const xmlChar*)"dictionary",NULL);
  for(std::vector<BOOK>::iterator bk=catalogs.books.begin();
      bk!=catalogs.books.end();++bk){
    xmlNodePtr book=xmlNewChild(diclist,NULL,(const xmlChar*)"book",NULL);
    xmlNewChild(book,NULL,(const xmlChar*)"path",
		(const xmlChar*) (*bk).path.c_str());
    xmlNewChild(book,NULL,(const xmlChar*)"name",
		(const xmlChar*) _EU(_CE((*bk).title)).c_str());
    xmlNewChild(book,NULL,(const xmlChar*)"char-offset",
		IntString(buf,10,(*bk).charOffset));
    xmlNewChild(book,NULL,(const xmlChar*)"char-extend",
		IntString(buf,10,(*bk).charExtend));
    xmlNewChild(book,NULL,(const xmlChar*)"gaiji-offset",
		IntString(buf,10,(*bk).gaijiOffset));
    xmlNewChild(book,NULL,(const xmlChar*)"auto-image-display",
		BoolString((*bk).autoImage));
    xmlNewChild(book,NULL,(const xmlChar*)"disable",
		BoolString((*bk).disable));
    xmlNewChild(book,NULL,(const xmlChar*)"disable-subbook",
		(const xmlChar*) (*bk).disableSubbook.c_str());
  }
  return true;
};

bool PREFERENCE::Write(const char* fileName){
  char fname[PATH_MAX+1];
  if (fileName!=NULL) {
    strncpy(fname,fileName,PATH_MAX);
  } else {
    if (getenv("HOME")!=NULL){
      strncpy(fname,getenv("HOME"),PATH_MAX);
      if (strlen(fname)>0 && fname[strlen(fname)-1]=='/') {
	fname[strlen(fname)-1]=0;
      }
    } else {
      fname[0]=0;
    }
    strncat(fname,"/.ebdic-preference",PATH_MAX);
  }
  fname[PATH_MAX]=0;

  FILE* fp=fopen(fname,"r");
  if (fp!=NULL){
    fclose(fp);
    CreateYesNoBox(mainWindow,_("File already exist!"),_("Over Write ?"),
		   _("Yes"),_("No"),false);
    if (!yesno) return false;
  }

  if (mainWindow==NULL) exit(1);

  int x,y,w,h,hw;

  gdk_window_get_size(mainWindow->window,&w,&h);
  gdk_window_get_origin(mainWindow->window,&x,&y);
  hw=GTK_PANED(hpaned)->child1_size;

  SetIntValue(PP "geometry/x",x);
  SetIntValue(PP "geometry/y",y);
  SetIntValue(PP "geometry/w",w);
  SetIntValue(PP "geometry/h",h);
  SetIntValue(PP "geometry/hit-window-width",hw);
  SetIntValue(PP "current-dictionary-number",currentDictNum);

  ClearDictionaryList();
  CreateDictionaryList();

  bool saveCurrentLocale=false;
#if (LIBXML_VERSION>20000) && defined(HAVE_NL_LANGINFO) && defined(CODESET)
  const char* code=nl_langinfo(CODESET);
  if (code!=NULL){
    xmlSaveFileEnc(fname,(xmlDocPtr)preference_private,code);
    saveCurrentLocale=true;
  }
#endif
  if (!saveCurrentLocale) {
    xmlSaveFile(fname,DOC);
  }
  return true;
};

#ifdef HAVE_LIBXML
int EucToUTF8(unsigned char* out, size_t* outlen, 
	      unsigned char* in, size_t* inlen){
#ifdef USE_ICONV
  static iconv_t cv=(iconv_t)-1;
  if (cv==(iconv_t)-1) cv=iconv_open("UTF-8", "EUC-JP");
  char* outbuf=(char*)out;
  char* inbuf=(char*)in;
  size_t ilen=*inlen;
  size_t olen=*outlen;
  int len=iconv(cv,&inbuf,&ilen,&outbuf,&olen);
  if (len<0 && errno!=E2BIG) {
    return -1;
  }
  *inlen-=ilen;
  *outlen-=olen;
  return 0;
#else
  if (*outlen<*inlen) return -1;
  memcpy(out,in,*inlen);
  *outlen=*inlen;
  return 0;
#endif
};

int UTF8ToEuc(unsigned char* out, size_t* outlen, 
	      unsigned char* in, size_t* inlen){
#ifdef USE_ICONV
  static iconv_t cv=(iconv_t)-1;
  if (cv==(iconv_t)-1) cv=iconv_open("EUC-JP","UTF-8");
  char* outbuf=(char*)out;
  char* inbuf=(char*)in;
  size_t len=iconv(cv,&inbuf,inlen,&outbuf,outlen);
  if (len<0) return -1;
  return len;
#else
  if (*outlen<*inlen) return -1;
  memcpy(out,in,*inlen);
  *outlen=*inlen;
  *inlen=0;
  return *outlen;
#endif
};
#endif

bool PREFERENCE::Reset(void){
  static bool initialized=false;
  if (!initialized){
    LIBXML_TEST_VERSION;
    xmlKeepBlanksDefault(0);
#if defined(HAVE_LIBXML) && (LIBXML_VERSION < 20000)
    xmlNewCharEncodingHandler("EUC-JP", (xmlCharEncodingInputFunc)EucToUTF8,
 			      (xmlCharEncodingOutputFunc)UTF8ToEuc);
#endif
    initialized=true;
  }

  if (preference_private!=NULL){ 
    xmlFreeDoc((xmlDocPtr)preference_private);
  }
  xmlDocPtr p=xmlParseDoc((xmlChar*)
		"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		"<ebdic></ebdic>\n");
  preference_private=(void*) p;
  SetIntValue(PP "geometry/x",10);
  SetIntValue(PP "geometry/y",20);
  SetIntValue(PP "geometry/w",780);
  SetIntValue(PP "geometry/h",560);
  SetIntValue(PP "geometry/hit-window-width",100);
  SetIntValue(PP "dict-list-type",0);
  SetBoolValue(PP "split-vertical",false);
  SetBoolValue(PP "full-menu-mode",false);
  SetIntValue(PP "current-dictionary-number",0);
  currentDictNum=0;

  SetStringValue("/ebdic/fontset/normal","-misc-variable-medium-r-normal--17-*-iso10646-1,-*-courier-medium-r-normal--17-*,-*-fixed-medium-r-normal--16-*,*");
  SetStringValue("/ebdic/fontset/bold","-*-courier-bold-r-normal--17-*,-*-*-bold-r-normal--16-*,-misc-variable-medium-r-normal--17-*-iso10646-1,-*-fixed-medium-r-normal--16-*,*");
  SetStringValue("/ebdic/fontset/italic","-*-courier-medium-o-normal--17-*,-*-*-medium-i-normal--16-*,-misc-variable-medium-r-normal--17-*-iso10646-1,-*-fixed-medium-r-normal--16-*,*");
  SetStringValue("/ebdic/fontset/bold-italic","-*-courier-bold-o-normal--17-*,-*-*-bold-i-normal--16-*,-misc-variable-medium-r-normal--17-*-iso10646-1,-*-fixed-medium-r-normal--16-*,*");
  SetStringValue("/ebdic/fontset/script","-*-courier-medium-r-normal--8-*,-*-fixed-medium-*--8-*,-misc-biwidth-medium-r-normal--13-*-iso10646-1,*");
  SetStringValue("/ebdic/fontset/iso8859-1","-misc-fixed-medium-r-normal--14-*-iso8859-1,-*-medium-r-normal--14-*-iso8859-1,-*-iso8859-1,*");


  return true;
}
