// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999-2000 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "bfile.h"
#include "ndtp.h"
#include <cstring>

class DicCache {
  int maxLength;
  int length;
  int* cacheBlockNumber;
  byte** cachePointer;
  byte* cache;
public:
  DicCache(int size);
  ~DicCache();
  void Clear();
  void Append(int block,byte* data);
  const byte* Read(int block);
};

DicCache::DicCache(int size){
  maxLength=size;
  cache=new byte[size*cBlockSize];
  cacheBlockNumber=new int[size];
  cachePointer=new byte*[size];
  Clear();
}

void DicCache::Clear(){
  length=0;
  for(int i=0;i<maxLength;i++){ 
    cacheBlockNumber[i]=0; 
    cachePointer[i]=cache+i*cBlockSize;
  };
}

DicCache::~DicCache(){
  delete [] cache;
  delete [] cacheBlockNumber;
  delete [] cachePointer;
}

const byte* DicCache::Read(int block){
  int i;
  for(i=0;i<length;i++){
    if (cacheBlockNumber[i]==block) break;
  }
  if (i==length) return NULL;
  int t=cacheBlockNumber[i];
  byte* b=cachePointer[i];
  for(int j=i;j>0;j--){
    cacheBlockNumber[j]=cacheBlockNumber[j-1];
    cachePointer[j]=cachePointer[j-1];
  }
  cachePointer[0]=b;
  cacheBlockNumber[0]=t;
  return b;
}

void DicCache::Append(int block,byte* data){
  if (length>=maxLength) length=maxLength-1;
  byte* b=cachePointer[length];
  for(int i=length;i>0;--i){
    cachePointer[i]=cachePointer[i-1];
    cacheBlockNumber[i]=cacheBlockNumber[i-1];
  }
  cacheBlockNumber[0]=block;
  cachePointer[0]=b;
  memcpy(b,data,cBlockSize);
  length++;
}
  
BlockIO::BlockIO() :
  type(BFileTypeError),
  currentPoint(TAG()),
  cache(new DicCache(cCacheBlocks))
{
};

BlockIO::~BlockIO(){
  delete cache;
};

bool BlockIO::Seek(const TAG& t){
  if (!Initialized()) return false;
  if (t.Block()==currentPoint.Block()) {
    currentPoint=t; 
    return true;
  }
  if (t.Block()<=0) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO::Seek Invalid Block\n");
    return false;
  }
  currentPoint=t; 
  const byte* p=cache->Read(t.Block());
  if (p==NULL){
    if (SeekWithoutCache(t.Block())) {
      cache->Append(t.Block(),ibuffer);
      return true;
    } else {
      Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO::Seek Seek Error\n");
      currentPoint=TAG();
      return false;
    }
  } else {
    memcpy(ibuffer,p,cBlockSize);
    return true;
  }
}

bool BlockIO::Seek(int block,int offset) {
  return Seek(TAG(block,offset));
}  

bool BlockIO::Seek(int offset) {
  return Seek(TAG(offset/cBlockSize+1,offset % cBlockSize));
}  

const TAG& BlockIO::Tell(){
  return currentPoint;
}

int BlockIO::LTell(){
  return currentPoint.Block()*2048+currentPoint.Offset()-2048;
}

int BlockIO::Read(byte* buffer,int len) {
  int ilen=0;
  while(len>0){
    if (cBlockSize-currentPoint.Offset()>len){
      if (buffer!=NULL) memcpy(buffer,ibuffer+currentPoint.Offset(),len);
      currentPoint.Offset(currentPoint.Offset()+len);
      ilen+=len;
      len=0;
    } else {
      int tlen=cBlockSize-currentPoint.Offset();
      if (buffer!=NULL) {
	memcpy(buffer,ibuffer+currentPoint.Offset(),tlen);
	buffer+=tlen;
      }
      len-=tlen;
      ilen+=tlen;
      if (Seek(currentPoint.Block()+1,0)==false) return ilen;
    }
  }
  return ilen;
}

byte BlockIO::GetByte(){
  byte x;
  Read(&x,1);
  return x;
}

word BlockIO::GetWord(){
  byte x[2];
  Read(x,2);
  return ::GetWord(x);
}

dword BlockIO::GetDWord(){
  byte x[4];
  Read(x,4);
  return ::GetDWord(x);
}

word BlockIO::GetRWord(){
  byte x[2];
  Read(x,2);
  return ::GetRWord(x);
}

dword BlockIO::GetRDWord(){
  byte x[4];
  Read(x,4);
  return ::GetRDWord(x);
}

int BlockIO::GetBCD(int nDigits){
  int nBytes=(nDigits-1)/2+1;
  byte* x=new byte[nBytes];
  Read(x,nBytes);
  int r=::GetBCD(x,nDigits);
  delete x;
  return r;
}

TAG BlockIO::GetTag(){
  dword block=GetDWord();
  return TAG(block,GetWord());
}

TAG BlockIO::GetBCDTag(){
  dword block=GetBCD(8);
  return TAG(block,GetBCD(4));
}

bool BlockIO::ReadBlock(int block,byte* buf){
  if (!Seek(TAG(block,0))) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO::ReadBlock Seek Error\n");
    return false;
  }
  return Read(buf,2048);
}

int BlockIO::GetLine(byte* buf,int len){
  int l=1;
  while(l<len){
    *buf=GetByte();
    if (*buf==0 || *buf==0x0a || *buf==0x0d) {
      break;
    }
    buf++;
    l++;
  }
  *buf=0;
  return l-1;
}

DiskIO::DiskIO(const char* fileName){
  Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO Initialize\n");
  pgLib=new EBZip(FILENAME::FuzzyFind(fileName).c_str());
  if (!pgLib->Initialized()) {
    delete pgLib;
    pgLib=NULL;
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO Initialize Fail\n");
    return;
  }
  type=BFileTypeDisk;
  currentPoint.Block(-1);  // Force flush buffer
  Seek(TAG(1,0));
};

DiskIO::~DiskIO(){
  type=BFileTypeError;
  if (pgLib!=NULL) delete pgLib;
};

bool DiskIO::SeekWithoutCache(int block) {
  Debug::DebugOut(Debug::LOW_LEVEL_FILE,"BlockIO::SeekWithoutCache %d\n",block);
  return pgLib->ReadBlock(block,ibuffer);
}  

// =========================================
BlockIO* OpenDict(const char* path){
  if (path==NULL) return NULL;
  if (strlen(path)>5 && strncasecmp(path,"ndtp:",5)==0) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"NDTP Open '%s'\n",path);
    NdtpPath* np=new NdtpPath(path);
    NdtpIO* pf=new NdtpIO(np->host,np->port);
    pf->SelectDict(np->path);
    return pf;
  } else {
    if (strlen(path)>5 && strncasecmp(path,"file:",5)==0) path+=5;
    char* p=strdup(path);
    if (strlen(path)>4 && strncasecmp(path+strlen(path)-4,".ebz",4)==0){
      p[strlen(path)-4]=0;
    } 
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"DISK Open '%s'\n",p);
    BlockIO* pf=new DiskIO(p);
    if (!pf->Initialized() && strlen(p)>1 && p[strlen(p)-1]=='2'){
      // Try to open honmon.ebz instead of honmon2
      delete pf;
      p[strlen(p)-1]=0;
      pf=new DiskIO(p);
    }
    free(p);
    return pf;
  }
}


