// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999-2000 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "def.h"
#include "eb_sup.h"
#include "debug.h"
#include <cstdio>
#include <cstring>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "ebzip.h"

EBZip::EBZip() : file(-1) 
{
  zip=new Zio;
};

EBZip::EBZip(const char* fileName) : file(-1)
{
  zip=new Zio;
  Open(fileName);
};

EBZip::~EBZip() 
{
  if (file>=0){
    eb_sup_zclose(zip);
  }
  file=-1;
  delete zip;
};

bool
EBZip::Open(const char* fileName){
  if (zip==NULL) return false;
  if (file!=-1) return false;
  if (fileName==NULL) return false;
  file=eb_sup_zopen(zip,fileName);
  Debug::DebugOut(Debug::LOW_LEVEL_FILE,"EBZip::Open file=%d\n",file);
  if (file==-1) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"File Open Error %s\n",fileName);
    return false;
  }
  return true;
}

bool 
EBZip::Close(){
  if (file<0) return false;
  eb_sup_zclose(zip);
  file=-1;
  return true;
}

off_t
EBZip::LSeek(off_t offset,int whence){
  if (file<0) return false;
  Debug::DebugOut(Debug::LOW_LEVEL_FILE,"EBZip::LSeek %d %d %d\n",file,offset,whence);
#ifdef HAVE_LIBEB
  off_t cur=zio_lseek(zip,offset,whence);
#else
  off_t cur=lseek(file,offset,whence);
#endif
  Debug::DebugOut(Debug::LOW_LEVEL_FILE,"EBZip::LSeek Status %d %d\n",(cur!=offset),(cur==offset));
  return cur;
}

bool 
EBZip::Read(char* buf, size_t size){
  if (file<0) return false;
#ifdef HAVE_LIBEB
  size_t len=zio_read(zip,buf,size);
  if (len!=size) return false;
  return true;
#else
  size_t len=0;
  do {
    size_t len2=read(file,buf,size);
    if (len2==0) return false;
    len+=len2;
    size-=len2;
  } while(size>0);
  return true;
#endif
}

bool 
EBZip::ReadBlock(int sect,char* buf){
  sect--;
  if (sect*2048!=LSeek(sect*2048,SEEK_SET)) {
    Debug::DebugOut(Debug::LOW_LEVEL_FILE,"EBZip::ReadBlock Seek Error\n");
    return false;
  }
  return Read(buf,2048);
}

