/**
 * @file person.c Galago Person message processor
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "filter.h"
#include "meta-person.h"
#include "utils.h"
#include <unistd.h>

static const char *
galagod_get_uid(void)
{
	static char id[64];
	static int serial = 0;

	g_snprintf(id, sizeof(id), "%08lx-%04lx-%04lx-%04x",
			   (unsigned long)time(NULL),
			   (unsigned long)getpid(),
			   (unsigned long)g_random_int_range(0, 65535),
			   serial++);

	return id;
}

static void
_galagod_dbus_person_emit(GalagoPerson *person, const char *signal_name)
{
	DBusMessage *message;

	g_return_if_fail(person != NULL);

	message = dbus_message_new_signal(
		galago_object_get_dbus_path(GALAGO_OBJECT(person)),
		GALAGO_DBUS_PERSON_INTERFACE,
		signal_name);
	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static void
_galagod_dbus_person_emit_with_object(GalagoPerson *person,
									  const char *signal_name, void *obj)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(person != NULL);
	g_return_if_fail(obj != NULL && GALAGO_IS_OBJECT(obj));

	message = dbus_message_new_signal(
		galago_object_get_dbus_path(GALAGO_OBJECT(person)),
		GALAGO_DBUS_PERSON_INTERFACE,
		signal_name);

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(obj));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static DBusHandlerResult
person_set_attribute(DBusConnection *dbus_conn, DBusMessage *message,
					 GalagodCoCo *coco, GalagoPerson *person)
{
	GalagoPerson *main_person =
		galagod_meta_person_get_person(GALAGOD_META_PERSON(person));

	return common_object_set_attribute(dbus_conn, message,
									   GALAGO_DBUS_PERSON_INTERFACE,
									   GALAGO_OBJECT(person),
									   GALAGO_OBJECT(main_person));
}

static DBusHandlerResult
person_remove_attribute(DBusConnection *dbus_conn, DBusMessage *message,
						GalagodCoCo *coco, GalagoPerson *person)
{
	GalagoPerson *main_person =
		galagod_meta_person_get_person(GALAGOD_META_PERSON(person));

	return common_object_remove_attribute(dbus_conn, message,
										  GALAGO_DBUS_PERSON_INTERFACE,
										  GALAGO_OBJECT(person),
										  GALAGO_OBJECT(main_person));
}

static DBusHandlerResult
person_get_attribute(DBusConnection *dbus_conn, DBusMessage *message,
					 GalagodCoCo *coco, GalagoPerson *person)
{
	GalagoPerson *main_person =
		galagod_meta_person_get_person(GALAGOD_META_PERSON(person));

	return common_object_get_attribute(dbus_conn, message,
									   GALAGO_OBJECT(main_person));
}

static DBusHandlerResult
person_get_attributes(DBusConnection *dbus_conn, DBusMessage *message,
					  GalagodCoCo *coco, GalagoPerson *person)
{
	GalagoPerson *main_person =
		galagod_meta_person_get_person(GALAGOD_META_PERSON(person));

	return common_object_get_attributes(dbus_conn, message,
										GALAGO_OBJECT(main_person));
}

static DBusHandlerResult
person_get_account(DBusConnection *dbus_conn, DBusMessage *message,
				   GalagodCoCo *coco, GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;
	GalagoAccount *account;
	GalagoService *service;
	const char *username;
	DBusMessageIter iter;
	DBusMessage *reply;

	dbus_message_iter_init(message, &iter);
	service = galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_SERVICE);
	dbus_message_iter_next(&iter);
	dbus_message_iter_get_basic(&iter, &username);

	meta_person = GALAGOD_META_PERSON(person);
	main_person = galagod_meta_person_get_person(meta_person);

	account = galago_person_get_account(main_person, service, username, FALSE);

	if (account == NULL)
	{
		const char *person_id = galago_person_get_id(main_person);

		if (person_id != NULL)
		{
			reply = galagod_object_not_found_error_new(message,
				"The account '%s' on person '%s' could not be found.",
				username, person_id);
		}
		else
		{
			reply = galagod_object_not_found_error_new(message,
				"The account '%s' on person could not be found.",
				username );
		}
	}
	else
	{
		reply = dbus_message_new_method_return(message);
		dbus_message_iter_init_append(reply, &iter);
		galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(account));
	}

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
person_get_accounts(DBusConnection *dbus_conn, DBusMessage *message,
					GalagodCoCo *coco, GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;
	GList *list;
	DBusMessageIter iter;
	DBusMessage *reply;

	meta_person = GALAGOD_META_PERSON(person);
	main_person = galagod_meta_person_get_person(meta_person);

	list = galago_person_get_accounts(main_person, FALSE);

	reply = dbus_message_new_method_return(message);
	dbus_message_iter_init_append(reply, &iter);

	galago_dbus_message_iter_append_object_list(&iter, GALAGO_TYPE_ACCOUNT,
												list);

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
person_get_accounts_count(DBusConnection *dbus_conn, DBusMessage *message,
						  GalagodCoCo *coco, GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;
	DBusMessageIter iter;
	DBusMessage *reply;
	size_t count;

	meta_person = GALAGOD_META_PERSON(person);
	main_person = galagod_meta_person_get_person(meta_person);

	count = g_list_length(
		galago_person_get_accounts(main_person, FALSE));

	reply = dbus_message_new_method_return(message);
	dbus_message_iter_init_append(reply, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT32, &count);
	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
set_unset_photo(DBusConnection *dbus_conn, DBusMessage *message,
				GalagodCoCo *coco, GalagoPerson *person, GalagoImage *photo)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;

	meta_person = GALAGOD_META_PERSON(person);
	main_person = galagod_meta_person_get_person(meta_person);

	if (photo != galago_person_get_photo(main_person, FALSE))
	{
		galago_person_set_photo(main_person, photo);

		galagod_dbus_send_empty_reply(dbus_conn, message);

		if (photo == NULL)
			_galagod_dbus_person_emit(main_person, "PhotoUnset");
		else
		{
			_galagod_dbus_person_emit_with_object(main_person, "PhotoSet",
												  photo);
		}
	}

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult
person_set_photo(DBusConnection *dbus_conn, DBusMessage *message,
				 GalagodCoCo *coco, GalagoPerson *person)
{
	GalagoImage *photo;
	DBusMessageIter iter;

	dbus_message_iter_init(message, &iter);
	photo = galago_dbus_message_iter_get_object(&iter, GALAGO_TYPE_IMAGE);

	return set_unset_photo(dbus_conn, message, coco, person, photo);
}

static DBusHandlerResult
person_unset_photo(DBusConnection *dbus_conn, DBusMessage *message,
				   GalagodCoCo *coco, GalagoPerson *person)
{
	return set_unset_photo(dbus_conn, message, coco, person, NULL);
}

static DBusHandlerResult
person_get_photo(DBusConnection *dbus_conn, DBusMessage *message,
				 GalagodCoCo *coco, GalagoPerson *person)
{
	GalagodMetaPerson *meta_person;
	GalagoPerson *main_person;
	GalagoImage *photo;
	DBusMessage *reply;
	DBusMessageIter iter;

	meta_person = GALAGOD_META_PERSON(person);
	main_person = galagod_meta_person_get_person(meta_person);

	photo = galago_person_get_photo(main_person, FALSE);

	if (photo == NULL)
	{
		const char *person_id = galago_person_get_id(main_person);

		if (person_id != NULL)
		{
			reply = galagod_object_not_found_error_new(message,
				"The photo for person '%s' could not be found.",
				galago_person_get_id(main_person));
		}
		else
		{
			reply = galagod_object_not_found_error_new(message,
				"The photo for person could not be found.");
		}
	}
	else
	{
		reply = dbus_message_new_method_return(message);
		dbus_message_iter_init_append(reply, &iter);
		galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(photo));
	}

	dbus_connection_send(dbus_conn, reply, NULL);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static const GalagodCommand commands[] =
{
	GALAGOD_COMMAND("SetAttribute",     person_set_attribute),
	GALAGOD_COMMAND("RemoveAttribute",  person_remove_attribute),
	GALAGOD_COMMAND("GetAttribute",     person_get_attribute),
	GALAGOD_COMMAND("GetAttributes",    person_get_attributes),
	GALAGOD_COMMAND("GetAccount",       person_get_account),
	GALAGOD_COMMAND("GetAccounts",      person_get_accounts),
	GALAGOD_COMMAND("GetAccountsCount", person_get_accounts_count),
	GALAGOD_COMMAND("SetPhoto",         person_set_photo),
	GALAGOD_COMMAND("GetPhoto",         person_get_photo),
	GALAGOD_COMMAND("UnsetPhoto",       person_unset_photo),
	GALAGOD_COMMAND_LAST
};

char *
galagod_person_generate_path(GalagoPerson *person)
{
	GalagodCoCo *coco = g_object_get_data(G_OBJECT(person), "coco");

	return g_strdup_printf("%s/people/%s",
		(coco == NULL
		 ? galago_context_get_obj_path_prefix()
		 : galago_object_get_dbus_path(GALAGO_OBJECT(coco))),
		galagod_dbus_normalize_name(galagod_get_uid()));
}

void
galagod_person_add_filter(GalagoPerson *person)
{
	galagod_filters_add(person, commands, GALAGO_DBUS_PERSON_INTERFACE);
}

void
galagod_person_remove_filter(GalagoPerson *person)
{
	galagod_filters_remove(person);
}
