/**
 * @file service-list.c Service list
 *
 * Copyright (C) 2004-2006 Christian Hammond.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 */
#include "service-list.h"
#include "utils.h"

static GHashTable *meta_services_table = NULL;
static GList *meta_services = NULL;
static size_t meta_service_count = 0;

static void
_galagod_dbus_services_emit_service_added(GalagoService *service)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(service != NULL);

	message = dbus_message_new_signal(GALAGO_DBUS_CORE_OBJECT,
									  GALAGO_DBUS_CORE_INTERFACE,
									  "ServiceAdded");

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(service));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static void
_galagod_dbus_services_emit_service_removed(GalagoService *service)
{
	DBusMessage *message;
	DBusMessageIter iter;

	g_return_if_fail(service != NULL);

	message = dbus_message_new_signal(GALAGO_DBUS_CORE_OBJECT,
									  GALAGO_DBUS_CORE_INTERFACE,
									  "ServiceRemoved");

	dbus_message_iter_init_append(message, &iter);
	galago_dbus_message_iter_append_object(&iter, GALAGO_OBJECT(service));

	dbus_connection_send(galago_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

void
galagod_services_add(GalagoService *service)
{
	GalagodMetaService *meta_service;
	const char *id;

	g_return_if_fail(service != NULL);

	id = galago_service_get_id(service);

	if ((meta_service = galagod_services_get_meta_service(id)) == NULL)
	{
		meta_service = galagod_meta_service_new(id,
			galago_service_get_name(service),
			galago_service_get_flags(service));
	}

	galagod_meta_service_add_service(meta_service, service);
}

void
galagod_services_remove(GalagoService *service)
{
	GalagodMetaService *meta_service;

	g_return_if_fail(service != NULL);

	meta_service = GALAGOD_META_SERVICE(service);
	g_return_if_fail(meta_service != NULL);

	galagod_meta_service_remove_service(meta_service, service);

	if (galagod_meta_service_get_services(meta_service) == NULL)
		galagod_meta_service_destroy(meta_service);
}

void
galagod_services_remove_with_coco(GalagodCoCo *coco)
{
	const GList *l;

	g_return_if_fail(coco != NULL);

	galago_context_push(galagod_coco_get_context(coco));

	for (l = galago_context_get_services(GALAGO_REMOTE); l != NULL; l = l->next)
		galagod_services_remove((GalagoService *)l->data);

	galago_context_pop();
}

void
galagod_services_add_meta_service(GalagodMetaService *meta_service)
{
	GalagoService *main_service;
	const char *id;

	g_return_if_fail(meta_service != NULL);

	main_service = galagod_meta_service_get_service(meta_service);
	id = galago_service_get_id(main_service);

	meta_services = g_list_append(meta_services, meta_service);

	g_hash_table_insert(meta_services_table,
						g_ascii_strdown(id, -1), meta_service);

	meta_service_count++;

	_galagod_dbus_services_emit_service_added(
		galagod_meta_service_get_service(meta_service));
}

void
galagod_services_remove_meta_service(GalagodMetaService *meta_service)
{
	const char *id;

	g_return_if_fail(meta_service != NULL);

	id = galago_service_get_id(
		galagod_meta_service_get_service(meta_service));

	g_hash_table_remove(meta_services_table, id);
	meta_services = g_list_remove(meta_services, meta_service);

	_galagod_dbus_services_emit_service_removed(
		galagod_meta_service_get_service(meta_service));

	meta_service_count--;
}

const GList *
galagod_services_get_meta_services(void)
{
	return meta_services;
}

size_t
galagod_services_get_meta_service_count(void)
{
	return meta_service_count;
}

GalagodMetaService *
galagod_services_get_meta_service(const char *id)
{
	GalagodMetaService *meta_service;
	char *temp;

	g_return_val_if_fail(id != NULL, NULL);

	temp = g_ascii_strdown(id, -1);

	meta_service = g_hash_table_lookup(meta_services_table, temp);

	free(temp);

	return meta_service;
}

void
galagod_services_init(void)
{
	meta_services_table = g_hash_table_new_full(g_str_hash, g_str_equal,
												g_free, NULL);
}

void
galagod_services_uninit(void)
{
	g_hash_table_destroy(meta_services_table);
	meta_services_table = NULL;

	g_list_free(meta_services);
	meta_services = NULL;

	meta_service_count = 0;
}
