#!/usr/bin/env python

import unittest
import time
import galago


def makeDummyPerson(id="dummy-person"):
    return galago.create_person(id)

def makeDummyService():
    return galago.create_service("dummy-service", "Dummy Service", 0)

def makeDummyAccount():
    return makeDummyService().create_account(makeDummyPerson(),
                                             "dummy-account")

def makeDummyPresence():
    return makeDummyAccount().create_presence()

class BaseTestCase(unittest.TestCase):
    def setUp(self):
        self.failUnless(galago.init("check-galago-python", galago.INIT_FEED))

    def tearDown(self):
        galago.uninit()
        self.failIf(galago.is_initted())


class InitTestCase(BaseTestCase):
    def setUp(self):
        # Don't init the feed yet.
        pass

    def testInitReinit(self):
        self.failUnless(galago.init("check-galago-python"))
        galago.uninit()
        self.failUnless(galago.init("check-galago-python"))


class ObjectTestCase(BaseTestCase):
    def testCreatePerson(self):
        self.__checkObject(galago.create_person("dummy-person"))
        self.__checkObject(galago.create_person())

    def testCreateService(self):
        self.__checkObject(galago.create_service("dummy-service-1",
                                                 "Dummy Service 1", 0))
        self.__checkObject(galago.create_service("dummy-service-2",
                                                 "Dummy Service 2"))

    def testCreateAccount(self):
        person = galago.create_person("dummy-person-x")
        service = galago.create_service("dummy-service-x", "Dummy Service")
        self.__checkObject(person)
        self.__checkObject(service)
        self.__checkObject(service.create_account(person, "dummy-account"))

    def testCreatePresence(self):
        self.__checkObject(makeDummyPresence())

    def __checkObject(self, obj):
        self.failIf(obj == None)
        self.failIf(obj.get_dbus_path() == "")


class ServiceTestCase(BaseTestCase):
    def testNormalize(self):
        self.__testNormalizeWith(0, "joebobsmith/home")
        self.__testNormalizeWith(galago.PRESERVE_SPACES, "joebob  smith/home")
        self.__testNormalizeWith(galago.PRESERVE_CASE,   "JoeBobSmith/Home")
        self.__testNormalizeWith(galago.STRIP_SLASH,     "joebobsmith")
        self.__testNormalizeWith(galago.PRESERVE_SPACES |
                                 galago.PRESERVE_CASE |
                                 galago.STRIP_SLASH,
                                 "JoeBob  Smith")

    def __testNormalizeWith(self, flags, expectedStr):
        service = galago.create_service("test-service", "Test Service", flags)
        username = service.normalize("JoeBob  Smith/Home")
        self.failIf(username != expectedStr)
        service.destroy()

class PersonTestCase(BaseTestCase):
    def testPriorityAccounts(self):
        person = makeDummyPerson()
        service = makeDummyService()

        account1 = service.create_account(person, "account-1")
        presence = account1.create_presence()
        presence.set_idle(True)

        account2 = service.create_account(person, "account-2")
        presence = account2.create_presence()

        priorityAccount = person.get_priority_account()

        self.failIf(priorityAccount == None)
        self.failIf(priorityAccount != account2)

        # Test with a dummy handler
        connId = galago.get_core().connect("calc_priority_account",
                                           self.__dummyCalcPriorityAccount)
        priorityAccount = person.get_priority_account()
        self.failIf(priorityAccount == None)
        self.failIf(priorityAccount != account2)
        galago.get_core().disconnect(connId)

        # Test with a real handler
        connId = galago.get_core().connect("calc_priority_account",
                                           self.__customCalcPriorityAccount)
        priorityAccount = person.get_priority_account()
        self.failIf(priorityAccount == None)
        self.failIf(priorityAccount != account1)
        galago.get_core().disconnect(connId)


    def __dummyCalcPriorityAccount(self, core, person):
        return None

    def __customCalcPriorityAccount(self, core, person):
        accounts = person.get_accounts()
        self.failIf(accounts == None)
        return accounts[0]


def suite():
    return unittest.TestSuite()

if __name__ == '__main__':
    unittest.main()
    print "Done"
