/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2003 Bert Vermeulen
    Copyright (C) 2002-2003 Peter Stamfest <peter@stamfest.at>

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: server-browse.c,v 1.8 2003/11/02 17:23:49 stamfest Exp $ */

#include <glib.h>
#include <gtk/gtk.h>


#include <errno.h>
#include <string.h>
#include <stdio.h>			/* snprintf */

#include <config.h>

#include "server-browse.h"
#include "dn-browse.h"
#include "prefs.h"			/* create_edit_server_window */
#include "util.h"			/* get_suffixes */

#include "browse-export.h"
#include "errorchain.h"
#include "i18n.h"

#ifdef BROWSER_DND
#include "browse-dnd.h"
#endif

/*  #include "../icons/warning.xpm" */

/*
 *  Really add a single suffix to the tree 
 */
static void add_suffix(server_browse_entry *entry, 
		       GtkCTree *ctreeroot, GtkCTreeNode *node,
		       char *suffix)
{
     GtkCTreeNode *new_item;
     char *labels[] = { suffix, NULL };
     browse_entry *new_entry = new_dn_browse_entry(suffix);
     
     new_item = gtk_ctree_insert_node(ctreeroot,
				      node, NULL,
				      labels, 
				      0,
				      NULL, NULL, NULL, NULL,
				      FALSE, FALSE);
     
     gtk_ctree_node_set_row_data_full(ctreeroot,
				      new_item,
				      new_entry,
				      (GtkDestroyNotify) destroy_browse_entry);

     /* add dummy node to have something to expand */
     labels[0] = "DUMMY";
     
     new_item = gtk_ctree_insert_node(ctreeroot,
				      new_item, NULL,
				      labels, 
				      0,
				      NULL, NULL, NULL, NULL,
				      TRUE, FALSE);
}

static void server_browse_entry_destroy(browse_entry *entry)
{
     assert(IS_SERVER_ENTRY(entry));
     ldapserver_unref(SERVER_BROWSE_ENTRY(entry)->server);

     g_free(entry);
}

static void server_browse_entry_expand(browse_entry *e, 
				       int error_context,
				       GtkCTree *ctree, GtkCTreeNode *node,
				       struct tab *tab)
{
     GList *suffixes = NULL, *next;
     server_browse_entry *entry;

     assert(IS_SERVER_ENTRY(e));
     entry = SERVER_BROWSE_ENTRY(e);

     if (!entry->once_expanded) {
/*  	  printf("expanding %s\n", entry->server->name); */
	
	  while (GTK_CTREE_ROW(node)->children) {
	       gtk_ctree_remove_node(ctree, GTK_CTREE_ROW(node)->children);
	  }
	  entry->once_expanded = 1;
	  
	  suffixes = get_suffixes(error_context, entry->server);
	  
	  gtk_clist_freeze(GTK_CLIST(ctree));

	  for (next = suffixes ; next ; next = g_list_next(next) ) {
	       add_suffix(entry, ctree, node, next->data);
	       g_free(next->data);
	       next->data = NULL;
	  }

	  gtk_clist_thaw(GTK_CLIST(ctree));

	  g_list_free(suffixes);
     }
}

/*
 * a server was selected in the tree widget.
 *
 * put up some server info.
 */
static void server_browse_entry_selected(browse_entry *be,
					 int error_context,
					 GtkCTree *ctree,
					 GtkCTreeNode *node,
					 struct tab *tab)
{
     GtkWidget *pane2_scrwin, *pane2_vbox, *label, *e;
     GtkWidget *table;
     char *server_name;
     int row = 0;
     char buf[128];
     LDAP *ld;
     server_browse_entry *entry;

     assert(IS_SERVER_ENTRY(be));
     entry = SERVER_BROWSE_ENTRY(be);

     ld = open_connection(error_context, entry->server);
     if (!ld) return;

     server_name = entry->server->name; /* dn_by_node(node); */
     record_path(tab, (browse_entry *) entry, ctree, node);

     pane2_scrwin = BROWSETAB(tab)->pane2_scrwin;
     pane2_vbox = BROWSETAB(tab)->pane2_vbox;

     /*  	  gtk_widget_destroy(pane2_vbox); */
     /* remove the viewport of the scrolled window. This should
	_really_ destroy the widgets below it. The pane2_scrwin
	is a GtkBin Object and thus has only one child, use this
	to obtain the viewport */

     gtk_container_remove(GTK_CONTAINER(pane2_scrwin),
			  GTK_BIN(pane2_scrwin)->child);

     pane2_vbox = gtk_vbox_new(FALSE, 2);
     BROWSETAB(tab)->pane2_vbox = pane2_vbox;

     gtk_widget_show(pane2_vbox);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(pane2_scrwin),
					   pane2_vbox);

     table = gtk_table_new(5, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(pane2_vbox), table, FALSE, FALSE, 5);

     /* Nickname */
     label = gtk_label_new(_("Nickname"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), server_name);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);
     row++;

     /* Host name */
     label = gtk_label_new(_("Hostname"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), entry->server->ldaphost);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);
     row++;

     /* Port */
     label = gtk_label_new(_("Port"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     snprintf(buf, sizeof(buf), "%d", entry->server->ldapport);
     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), buf);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);
     row++;

     /* Connection caching */
     label = gtk_label_new(_("Connection caching"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     snprintf(buf, sizeof(buf), "%s", 
	      entry->server->cacheconn ? _("on") : _("off"));
     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), buf);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     row++;

     /* TLS */
     label = gtk_label_new(_("TLS"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     snprintf(buf, sizeof(buf), "%s", 
	      entry->server->enabletls ? _("on") : _("off"));
     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), buf);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     row++;

#if HAVE_LDAP_CLIENT_CACHE
     label = gtk_label_new(_("Client-side caching"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     snprintf(buf, sizeof(buf), "%s", 
	      (entry->server->local_cache_timeout >= 0) ? _("on") : _("off"));
     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), buf);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     row++;
#endif

     /* Connections so far */
     label = gtk_label_new(_("Connections so far"));
     gtk_widget_show(label);
     gtk_table_attach(GTK_TABLE(table),
		      label,
		      0, 1, row, row+1,
		      GTK_SHRINK,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     snprintf(buf, sizeof(buf), "%d", 
	      entry->server->incarnation);
     e = gtk_entry_new();
     gtk_entry_set_text(GTK_ENTRY(e), buf);
     gtk_widget_set_sensitive(e, FALSE);
     gtk_widget_show(e);
     gtk_table_attach(GTK_TABLE(table),
		      e,
		      1, 2, row, row+1,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		      0, 0);

     row++;

     if (ld) {
	  int intdata;
	  int rc;
	  /*  	       void *optdata; */
	  char *rootDSEattr[] = {
	       "vendorName",    _("Vendor Name"),	/* RFC 3045 */
	       "vendorVersion", _("Vendor Version"),	/* RFC 3045 */
	       "altServer", _("Alternative Server(s)"), /* RFC 2251 */
	       "supportedLDAPVersion", _("Supported LDAP Version"), /* RFC 2251 */
	       "supportedSASLMechanisms", _("Supported SASL Mechanisms"), /* RFC 2251 */
	       NULL
	  };
	  LDAPMessage *res, *ee; 
	  BerElement *berptr;
	  char *attr;
	  char **vals;
	  int i, msg;

	  rc = ldap_get_option(ld, LDAP_OPT_PROTOCOL_VERSION, &intdata);

	  /* LDAP protocol version */
	  label = gtk_label_new(_("LDAP protocol version"));
	  gtk_widget_show(label);
	  gtk_table_attach(GTK_TABLE(table),
			   label,
			   0, 1, row, row+1,
			   GTK_SHRINK,
			   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			   0, 0);
	       
	  snprintf(buf, sizeof(buf), "%d", intdata);
	  e = gtk_entry_new();
	  gtk_entry_set_text(GTK_ENTRY(e), buf);
	  gtk_widget_set_sensitive(e, FALSE);
	  gtk_widget_show(e);
	  gtk_table_attach(GTK_TABLE(table),
			   e,
			   1, 2, row, row+1,
			   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			   GTK_EXPAND | GTK_SHRINK | GTK_FILL,
			   0, 0);
	       
	  row++;

	  /* read some Information from the root DSE */
	  for (i = 0 ; rootDSEattr[i] && ld != NULL ; i += 2) {
	       char *attrs[2] = { rootDSEattr[i], NULL };

	       msg = ldap_search_ext_s(ld, "", 
				       LDAP_SCOPE_BASE,	/* scope */
				       "(objectclass=*)", /* filter */
				       attrs,		/* attrs */
				       0,		/* attrsonly */
				       NULL,		/* serverctrls */
				       NULL,		/* clientctrls */
				       NULL,		/* timeout */
				       LDAP_NO_LIMIT,	/* sizelimit */
				       &res);

	       if(msg != LDAP_SUCCESS) {
		    if (msg == LDAP_SERVER_DOWN) {
			 close_connection(entry->server, FALSE);
			 ld = open_connection(error_context, entry->server);
		    }
		    statusbar_msg("%s", ldap_err2string(msg));
	       } else {
		    if(res == NULL) continue;
		    ee = ldap_first_entry(ld, res);

		    if (ee == NULL) {
			 ldap_msgfree(res);
			 continue;
		    }
		    attr = ldap_first_attribute(ld, res, &berptr);

		    if (attr == NULL) {
			 ldap_msgfree(res);
#ifndef HAVE_OPENLDAP_12
			 if(berptr) ber_free(berptr, 0);
#endif
			 continue;
		    }
		    vals = ldap_get_values(ld, res, attr);
		    if (vals) {
			 int j;
			 for (j = 0 ; vals[j] ; j++) ;

			 label = gtk_label_new(rootDSEattr[i + 1]);
			 gtk_widget_show(label);
			 gtk_table_attach(GTK_TABLE(table),
					  label,
					  0, 1, row, row+j,
					  GTK_SHRINK,
					  GTK_EXPAND | GTK_SHRINK | GTK_FILL,
					  0, 0);
			 
			 for (j = 0 ; vals[j] ; j++) {
			      snprintf(buf, sizeof(buf), "%s", vals[j]);
			      e = gtk_entry_new();
			      gtk_entry_set_text(GTK_ENTRY(e), buf);
			      gtk_widget_set_sensitive(e, FALSE);
			      gtk_widget_show(e);
			      gtk_table_attach(GTK_TABLE(table),
					       e,
					       1, 2, row, row+1,
					       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
					       GTK_EXPAND | GTK_SHRINK | GTK_FILL,
					       0, 0);
			      row++;
			 }

			 
			 ldap_value_free(vals);
		    }
		    
		    ldap_memfree(attr);
#ifndef HAVE_OPENLDAP_12
		    if(berptr) ber_free(berptr, 0);
#endif
		    ldap_msgfree(res);
	       }
	  }
	  close_connection(entry->server, FALSE);
     }

     /*  	  gtk_box_pack_start(GTK_BOX(pane2_vbox), label, FALSE, FALSE, 0); */
}






static void server_browse_entry_refresh(browse_entry *e,
					int error_context,
					GtkCTree *ctree,
					GtkCTreeNode *node,
					struct tab *tab)
{
     server_browse_entry *entry;

     assert(IS_SERVER_ENTRY(e));
     entry = SERVER_BROWSE_ENTRY(e);
     
     entry->once_expanded = 0;

     gtk_clist_freeze(GTK_CLIST(ctree));

     server_browse_entry_selected(e, error_context, ctree, node, tab);

     /* toggle expansion twice to fire the expand callback and to
	return to the current expansion state */
     gtk_ctree_toggle_expansion(ctree, node);
     gtk_ctree_toggle_expansion(ctree, node);

/*       server_browse_entry_expand(entry, ctree, node, tab); */

     gtk_clist_thaw(GTK_CLIST(ctree));

}


static char* server_browse_entry_get_name(browse_entry *entry,
					  gboolean long_form)
{
     assert(IS_SERVER_ENTRY(entry));

     return g_strdup(SERVER_BROWSE_ENTRY(entry)->server->name);
}

struct edit_server_cb_data {
     struct ldapserver *server;
     struct tab *tab;
};

static void free_edit_server_cb_data(struct edit_server_cb_data *cbd)
{
     ldapserver_unref(cbd->server);
     g_free(cbd);
}

static void edit_server_activated(struct edit_server_cb_data *cbd)
{
     create_edit_server_window(cbd->server, cbd->tab->win->mainwin);
}

static void dump_server(GtkWidget *widget, struct tab *tab)
{
     GtkCTree *ctree;
     GtkCTreeNode *node;
     browse_entry *e;
     struct ldapserver *server;
     GList *bases = NULL;
     GList *to_export = NULL, *I;
     struct dn_on_server *dos;
     int error_context;

     ctree = BROWSETAB(tab)->ctreeroot;
     node = BROWSETAB(tab)->tree_row_popped_up;
     e = (browse_entry *) gtk_ctree_node_get_row_data(ctree, node);

     assert(IS_SERVER_ENTRY(e));

     server = server_from_node(ctree, node);

     if (e == NULL || server == NULL)
	  return;

     error_context = error_new_context(_("Exporting server to LDIF"),
				       tab->win->mainwin);

     bases = get_suffixes(error_context, ((server_browse_entry *)e)->server);

     /* turn suffixes list into a list of dn_on_server objects
	(impedance match) */
     for (I = g_list_first(bases) ; I ; I = g_list_next(I) ) {
	  dos = new_dn_on_server(I->data, server);
	  dos->flags = LDAP_SCOPE_SUBTREE; /* default is LDAP_SCOPE_BASE */
	  to_export = g_list_append(to_export, dos);
	  g_free(I->data);
	  I->data = NULL;
     }
     g_list_free(bases);

     export_many(error_context, tab->win->mainwin, to_export);

     error_flush(error_context);
}

static void server_browse_entry_popup(browse_entry *e,
				      GtkWidget *menu, 
				      GtkWidget *ctreeroot,
				      GtkCTreeNode *ctree_node,
				      struct tab *tab) 
{
     GtkWidget *menu_item;
     struct ldapserver *server;
     struct edit_server_cb_data *cbd;
     server_browse_entry *entry;

     assert(IS_SERVER_ENTRY(e));
     entry = SERVER_BROWSE_ENTRY(e);

     server = server_from_node(GTK_CTREE(ctreeroot), ctree_node);

     /* Edit Server settings */
     menu_item = gtk_menu_item_new_with_label(_("Edit Server"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_widget_show(menu_item);

     cbd = (struct edit_server_cb_data *)
	  g_malloc0(sizeof(struct edit_server_cb_data));
     cbd->server = server;
     cbd->tab = tab;

     ldapserver_ref(server); /* not strictly necessary, but ... */

     gtk_signal_connect_object(GTK_OBJECT(menu_item), "activate",
			       GTK_SIGNAL_FUNC(edit_server_activated),
			       (gpointer) cbd);

     /* explicitly attach cbd to assure call to destructor */
     gtk_object_set_data_full(GTK_OBJECT(menu_item), "cbd", 
			      cbd, (GtkDestroyNotify)free_edit_server_cb_data);

     gtk_widget_show(menu_item);

     if (server == NULL) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
     
     /* Export to LDIF */
     menu_item = gtk_menu_item_new_with_label(_("Export to LDIF"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(dump_server),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     /* Close connection */
     menu_item = gtk_menu_item_new_with_label(_("Close Connection"));
     gtk_menu_append(GTK_MENU(menu), menu_item);
     gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
			GTK_SIGNAL_FUNC(tree_row_close_connection),
			(gpointer) tab);
     gtk_widget_show(menu_item);

     if (server == NULL) {
	  gtk_widget_set_sensitive(menu_item, FALSE);
     }
}

static struct browse_entry_vtab server_vtab = {
     	server_browse_entry_destroy,	/* destroy */
     	server_browse_entry_expand,	/* expand */
     	server_browse_entry_selected,	/* select */
     	server_browse_entry_refresh,	/* refresh */
     	server_browse_entry_get_name,	/* get_name */
     	server_browse_entry_popup,	/* popup */
};


browse_entry *new_server_browse_entry(struct ldapserver *server) 
{
     server_browse_entry *e;
     e = g_malloc0(sizeof(server_browse_entry));

     e->type = SERVER_BROWSE_ENTRY_ID;
     e->base_methods = &server_vtab;

     e->server = server;
     ldapserver_ref(server);

     return (browse_entry *) e;
}



/* 
   Local Variables:
   c-basic-offset: 5
   End:
*/
