/*
 * IIIMF-Canna, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Shingo Akagaki <dora@momonga-linux.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: im_util.c,v 1.2 2002/09/25 20:10:59 famao Exp $ */

/* vi:set ts=4 sw=4: */


/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

#ifdef	WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include "csconv.h"
#include "im_util.h"

#define CSC_PATH	"/usr/lib/im/csconv/csconv.so"
#define CSC_OPEN_LOCALE	"csconv_open_locale"
#define CSC_OPEN	"csconv_open"
#define CSC_CONV	"csconv"
#define CSC_CLOSE	"csconv_close"

#define CSC_FAILED -1
#define CSC_UNLOADED 0
#define CSC_UNOPENED 1
#define CSC_LOADED 2

#define CSC_CODESET_LOCALE "ja_JP.eucJP"

typedef csconv_t	(* csc_open_locale_t)(const char *,
					      const char *, const char *);
typedef csconv_t	(* csc_open_t)(const char *, const char *);
typedef size_t		(* csc_conv_t)(csconv_t, const char **, size_t *,
				       char **, size_t *);
typedef int		(* csc_close_t)(csconv_t);

static void *			csc_handle;	
static csc_open_locale_t	csc_open_locale;
static csc_open_t		csc_open;
static csc_conv_t		csc_conv;
static csc_close_t		csc_close;
static csconv_t                 csconv_cd = NULL;

static void dlopen_csconv(void);
static int csconv_status(void);
static void csconv_open_conv(void);
static int UTFCHAR_buffer_size(int str_len);

static IMFeedbackList * create_feedback(iml_session_t *s, int size);
static IMFeedbackList * create_feedback2(iml_session_t *s, int size);
static int get_feedback(IMFeedbackList *fbl);
static void set_feedback_private( IMFeedbackList * fbl, int normalfeedback,
                                  int fg, int bg, int underline);
static void set_im_feedback_1(IMFeedbackList *fbl, int feedback_type);
static void set_im_feedback(IMFeedbackList *fbl, int feedback_type, int st,
                            int end);
static IMFeedbackList * create_im_feedback(iml_session_t *s, int size,
                                           int normalfeedback, int fg, int bg);

static IMText* create_IMText(iml_session_t *s, int len);

/*
  CSConv:
*/

static void
dlopen_csconv(void)
{
      csc_handle = dlopen(CSC_PATH, RTLD_LAZY);
      if (NULL == csc_handle) {
        csc_handle = (void *)(-1);
	return;
      }

      csc_open_locale = (csc_open_locale_t)dlsym(csc_handle, CSC_OPEN_LOCALE);
      csc_open = (csc_open_t)dlsym(csc_handle, CSC_OPEN);
      csc_conv = (csc_conv_t)dlsym(csc_handle, CSC_CONV);
      csc_close = (csc_close_t)dlsym(csc_handle, CSC_CLOSE);

      if ((NULL == csc_open_locale) || (NULL == csc_open) ||
	  (NULL == csc_conv) || (NULL == csc_close)) {
        dlclose(csc_handle);
	csc_handle = (void *)(-1);
	return;
      }
}

static int
csconv_status(void)
{
    if ((csc_handle == (void *)(-1))
	|| (csconv_cd == (csconv_t)(-1)))
	return CSC_FAILED;
    else if (csc_handle == NULL)
	return CSC_UNLOADED;
    else if (csconv_cd == NULL)
	return CSC_UNOPENED;

    return CSC_LOADED;
}

static void
csconv_open_conv(void)
{
    if (csconv_status() == CSC_UNOPENED)
	csconv_cd = (csc_open_locale)(CSC_CODESET_LOCALE,
				      "UTF-16",
				      "MultiByte");
    if (csconv_cd <= 0)
	fprintf(stderr, "csconv_open_conv(): Cannot Open csconv - %s\n",
		CSC_CODESET_LOCALE);
}

Bool
im_util_setup(void)
{
    if (csconv_status() == CSC_UNLOADED)
	dlopen_csconv();
    if (csconv_status() == CSC_UNOPENED)
        csconv_open_conv();
    if (csconv_status() == CSC_FAILED)
        return False;
    return True;
}

/*
  String conversion:
*/

static int 
UTFCHAR_buffer_size(int str_len)
{
    return str_len * sizeof (UTFCHAR);
}

UTFCHAR*
im_string_to_UTFCHAR(unsigned char *str)
{
    int ret, clen, ulen;
    UTFCHAR *p,*ustr;

    clen = strlen(str);
    ulen = UTFCHAR_buffer_size(clen + 1);
    ustr = p = (UTFCHAR*) malloc(ulen);
    ret = csc_conv(csconv_cd,
		   (const char **)&str, &clen,
		   (char **) &p, &ulen);
    *p = 0;
    return ustr;
}

static IMFeedbackList *
create_feedback(iml_session_t *s, int size)
{
    int i;
    IMFeedbackList *feedback;
    IMFeedback *fb;
   
    if (!s) return NULL;

    feedback = ((IMFeedbackList *)
                s->If->m->iml_new(s, sizeof(IMFeedbackList) * size));
    for (i = 0; i < size; i++) {
        IMFeedbackList *fbl = &feedback[i];
        fbl->count_feedbacks = 1;
        fb = ((IMFeedback *) s->If->m->iml_new(s, sizeof(IMFeedback) * 4));
        fbl->feedbacks = fb;
        memset(fbl->feedbacks, 0, sizeof(IMFeedback) * 4);
    }
    return feedback;
}

static IMFeedbackList *
create_feedback2(iml_session_t *s, int size)
{
    int i;
    IMFeedbackList *feedback;
    IMFeedback *fb;
   
    if (!s) return NULL;

    feedback = ((IMFeedbackList *)
                s->If->m->iml_new2(s, sizeof(IMFeedbackList) * size));
    for (i = 0; i < size; i++) {
        IMFeedbackList *fbl = &feedback[i];
        fbl->count_feedbacks = 1;
        fb = ((IMFeedback *) s->If->m->iml_new2(s, sizeof(IMFeedback) * 4));
        fbl->feedbacks = fb;
        memset(fbl->feedbacks, 0, sizeof(IMFeedback) * 4);
    }
    return feedback;
}

static int
get_feedback(IMFeedbackList *fbl)
{
    /* returns IM_DECORATION_FEEDBACK */
    IMFeedback *fb = &fbl->feedbacks[0];
    return IM_FEEDBACK_VALUE(fb);
}

static void
set_feedback_private(
    IMFeedbackList * fbl,
    int normalfeedback,
    int fg,
    int bg,
    int underline
)
{
    int count = 0;
    IMFeedback *fb;

    fb = &fbl->feedbacks[count];
    IM_FEEDBACK_TYPE(fb) = IM_DECORATION_FEEDBACK;
    IM_FEEDBACK_VALUE(fb) = normalfeedback;
    count++;

#ifdef  USE_COLOR_FEEDBACK

    if (fg != -1) {
        fb = &fbl->feedbacks[count];
        IM_FEEDBACK_TYPE(fb) = IM_FOREGROUND_RGB_FEEDBACK;
        IM_FEEDBACK_VALUE(fb) = fg;
        count++;
    }
    if (bg != -1) {
        fb = &fbl->feedbacks[count];
        IM_FEEDBACK_TYPE(fb) = IM_BACKGROUND_RGB_FEEDBACK;
        IM_FEEDBACK_VALUE(fb) = bg;
        count++;
    }
    if (underline != -1) {
        fb = &fbl->feedbacks[count];
        IM_FEEDBACK_TYPE(fb) = IM_UNDERLINE_RGB_FEEDBACK;
        IM_FEEDBACK_VALUE(fb) = underline;
        count++;
    }

#endif
    IM_FEEDBACK_COUNT(fbl) = count;
}

static void
set_im_feedback_1(IMFeedbackList *fbl, int feedback_type)
{

    switch(feedback_type) {
    case IM_Feedback_Input:
        set_feedback_private(fbl,
                             IMUnderline,
                             IM_RGB_COLOR(0, 0, 255),     /* FG: blue */
                             IM_RGB_COLOR(255, 255, 255), /* BG: white */
                             1);                          /* Underline */
        break;
    case IM_Feedback_Strong:
        set_feedback_private(fbl,
                             IMReverse,
                             IM_RGB_COLOR(255, 255, 255), /* FG: white */
                             IM_RGB_COLOR(0, 0, 255),     /* BG: blue */
                             -1);
        break;
    case IM_Feedback_Normal:
    default:
        set_feedback_private(fbl, IMNormal, -1, -1, -1);
        break;
    }
    return;

}

static void
set_im_feedback(IMFeedbackList *fbl, int feedback_type,
                   int st, int end)
{
    for (;st < end;st++)
        set_im_feedback_1((fbl + st), feedback_type);
    return;
}


static IMFeedbackList *
create_im_feedback(iml_session_t *s,
                      int size,
                      int normalfeedback,
                      int fg,
                      int bg)
{
    int i;
    IMFeedbackList *feedback;
    feedback = (IMFeedbackList *) create_feedback(s, size);
    for (i = 0; i < size; i++) {
        IMFeedbackList *fbl = &feedback[i];
        set_feedback_private(fbl, normalfeedback, fg, bg, -1);
    }

    return feedback;
}

/*
  IMText operations:
*/

static IMText*
create_IMText(iml_session_t *s, int len)
{
    IMText *p;

    if (!s) return NULL;

    p = (IMText *) s->If->m->iml_new(s, sizeof(IMText));
    memset(p, 0, sizeof(IMText));
    p->encoding = UTF16_CODESET;
    p->text.utf_chars = ((UTFCHAR *)
                         s->If->m->iml_new(s, sizeof(UTFCHAR) * (len + 1)));
    p->char_length = len;
    p->feedback = create_feedback(s, len);

    return p;
}

IMText*
im_UTFCHAR_to_IMText(iml_session_t *s, UTFCHAR *p)
{
    IMText *pit;
    UTFCHAR *p2 = p;
    int len;

    for (len = 0;*p2;p2++) len++;
    pit = create_IMText(s, len);
    if (!pit) return NULL;
    memcpy(pit->text.utf_chars, p, (len + 1) * sizeof(UTFCHAR));

    return pit;
}

IMText*
im_string_to_IMText(iml_session_t *s,
                       int nseg, int *nb,
                       unsigned char **strs,
                       int *feedback_type)
{
    UTFCHAR *ustr, *p;
    IMText *pit;
    char *from;
    int from_size;
    int i, ret, clen, ulen, uidx;
    int *idices;

    clen = 0;
    for (i = 0;i < nseg;i++) {
        clen += nb[i];
    }
    ulen = UTFCHAR_buffer_size(clen + 1);
    ustr = p = (UTFCHAR*) alloca(ulen);
    idices = (int*) alloca(sizeof(int) * (nseg + 1));

    for (i = 0;i < nseg;i++) {
        from = strs[i];
        from_size = nb[i];
        idices[i] = p - ustr;
        ret = csc_conv(csconv_cd,
                       (const char **)&from, &from_size,
                       (char **) &p, &ulen);
    }
    *p = 0;
    uidx = p - ustr;
    idices[nseg] = uidx;
    pit = create_IMText(s, uidx);

    if (!pit) return NULL;
    memcpy(pit->text.utf_chars, ustr, (p - ustr + 1) * sizeof(UTFCHAR));
    if (feedback_type) {
        /* set feedback */
        for (i = 0;i < nseg;i++) {
            set_im_feedback(pit->feedback, feedback_type[i],
                               idices[i], idices[i + 1]);
        }
    }
    return pit;
}
