/***************************************************************************
 *   Copyright (C) 2005 by Peter Müller   *
 *   pmueller@cs.tu-berlin.de   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kglcometen3.h"
#include "cometenmath.h"



KGLCometen3::KGLCometen3(QWidget *parent, const char *name)
: QGLWidget(parent, name)
{
	cometen3 = new Cometen3();
	settings = new Setting[SE_NUM];
	textures = new GLuint[T_NUM];
}

KGLCometen3::~KGLCometen3() {
	delete cometen3;
	delete settings;
	delete textures;
}

void KGLCometen3::start(KConfig* config) 
{
	config->setGroup( "Settings" );
	settings[SE_FREECAMERA].intValue = 
		config->readBoolEntry( "freecamera", true ) ? 1 : 0;
	settings[SE_FREECAMERASPEED].doubleValue = 
		clamp<double>(config->readDoubleNumEntry("freecamspeed", 1.5), 0.0, 10.0);
	settings[SE_DOBLITZ].intValue =
		config->readBoolEntry( "lightning", true ) ? 1 : 0;
	settings[SE_DOSPLITCOMET].intValue =
		config->readBoolEntry( "splitcomet", true ) ? 1 : 0;
	settings[SE_DOROTATECOMET].intValue =
		config->readBoolEntry( "rotatecomet", true ) ? 1 : 0;
	settings[SE_DOCURVECOMET].intValue =
		config->readBoolEntry( "curvecomet", true ) ? 1 : 0;
	settings[SE_DOMATRIX].intValue = 
		config->readBoolEntry( "matrix", true ) ? 1 : 0;
	settings[SE_DOSLOWMOTION].intValue = 
		config->readBoolEntry( "slowmotion", true ) ? 1 : 0;
	settings[SE_CREATEINTERVAL].doubleValue = 
		clamp<double>(config->readDoubleNumEntry("interval", 1.25), 0.1, 10.0);
	settings[SE_TIMESCALE].doubleValue = 
		clamp<double>(config->readDoubleNumEntry("timescale", 1.0), 0.1, 5.0);
	settings[SE_SIZESCALE].doubleValue = 
		clamp<double>(config->readDoubleNumEntry("sizescale", 2.5), 0.5, 5.0);
	
	settings[SE_COLORS].intValue = config->readNumEntry("color", 0);
	
	bgtype = config->readNumEntry("bgtype", 0);
	bgfile = config->readPathEntry("bgfile","");
	bgsize = config->readNumEntry("bgsize", 2);
	bgdir = config->readPathEntry("bgdir","");
	maxfps = clamp<int>(10, 999, config->readNumEntry("maxfps", 60));
	
	startTimer(1000/maxfps);
}

void KGLCometen3::initializeGL() {
	QImage image;
	
	srand(time(NULL));

	if (bgtype == 2) {
		QString file = randomFileFromDir(bgdir);
		if (!image.load(file)) {
			image = QPixmap::grabWindow(QApplication::desktop()->winId());
		}
	}
	else if (bgtype == 1) {
		if (!image.load(bgfile)) {
			image = QPixmap::grabWindow(QApplication::desktop()->winId());
		}
	}
	else {
		image = QPixmap::grabWindow(QApplication::desktop()->winId());
	}
	
	switch (bgsize) {
	case 0:
		image = image.smoothScale(128, 128, QImage::ScaleFree);
		break;
	case 1:
		image = image.smoothScale(256, 256, QImage::ScaleFree);
		break;
	case 2:
		image = image.smoothScale(512, 512, QImage::ScaleFree);
		break;
	case 3:
	default:
		break;
	}
		
	image = QGLWidget::convertToGLFormat(image);
	
	glEnable(GL_TEXTURE_2D);
	glGenTextures(1, &textures[T_BACKGROUND]);
	glBindTexture(GL_TEXTURE_2D, textures[T_BACKGROUND]); 
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	
	glTexImage2D(GL_TEXTURE_2D, 0, 3, image.width(), image.height(),
	             0, GL_RGBA, GL_UNSIGNED_BYTE, image.bits());	
	
	textures[T_PARTICLE] = loadTexture("particle.png");
	textures[T_LIGHTMAP] = loadTexture("lightmap.png");
	
	t.start();
	settings[SE_DOMATRIX].intValue = 1;
	cometen3->init(settings, textures, 640, 480, t.elapsed() / 1000.0);
}

void KGLCometen3::resizeGL(int res_x, int res_y) {
	cometen3->resize(res_x, res_y);
}

void KGLCometen3::paintGL() {
	cometen3->render();
}

void KGLCometen3::timerEvent(QTimerEvent * event) {
	cometen3->process(t.elapsed() / 1000.0);
	updateGL();
}

QString KGLCometen3::randomFileFromDir(const QString& dirname) {
	QDir dir(dirname, "*.jpg;*.jpeg;*.png;*.gif",
	         QDir::Unsorted, QDir::Files | QDir::Readable);
	
	if (dir.count() < 1) {
		return QString("");
	}
	
	int num = rand() % dir.count();
	
	return dir.absPath() + "/" + dir[num];
}

GLuint KGLCometen3::loadTexture(const char* file) {
	QImage image;
	GLuint tex = 0;

	if (image.load(KGlobal::dirs()->findResource("data", 
	      QString("kcometen3/pixmaps/") + file))) {
		image = QGLWidget::convertToGLFormat(image);
		
		glEnable(GL_TEXTURE_2D);
		glGenTextures(1, &tex);
		glBindTexture(GL_TEXTURE_2D, tex); 
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
		
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, image.width(), image.height(),
		             0, GL_RGBA, GL_UNSIGNED_BYTE, image.bits());
	}

	return tex;
}
		
		

#include "kglcometen3.moc"
