// -*- c++ -*-

/*
 *
 * Copyright (C) 2002 Richard Moore <rich@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify 
 *   it under the terms of the GNU General Public License as published by 
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#ifndef ACTIONS_H
#define ACTIONS_H

#include <qptrlist.h>
#include <qwidget.h>
#include <qslider.h>
#include <kaction.h>

class QDockWindow;
class QLCDNumber;
class KMainWindow;
class Kdetv;

//
// Actions for displaying custom QWidgets.
//

/**
 * A KAction that displays a number using the @ref QLCDNumber class.
 *
 * @author Richard Moore, rich@kde.org
 */
class LCDNumberAction : public KAction
{
    Q_OBJECT

public:
    /**
     * Creates a LCDNumberAction.
     *
     * @param text  The label for the LCD. Currently this is displayed
     *              as a tooltip.
     */
    LCDNumberAction( const QString &text, int accel,
                     QObject *receiver, const char *slot,
                     QObject *parent = 0, const char *name = 0 );

    virtual int plug( QWidget *widget, int index = -1 );
    virtual void unplug( QWidget *widget );

    int intValue() const { return val.toInt(); }
    QString value() const { return val; }

    int numDigits() const { return numDig; }

public slots:
    void display( int num );
    void display( const QString &s );

    void setNumDigits( int nDigits );

    signals:
    void valueChanged( const QString & );

protected:
    virtual QLCDNumber *createWidget( QWidget *parent, const char *name=0 );

private:
    int numDig;
    QString val;
};


class KdetvSlider : public QSlider
{
    Q_OBJECT

public:
    KdetvSlider( int minValue, int maxValue, int pageStep, int value, Orientation,
                 QWidget *parent, const char* name = 0 );
    virtual ~KdetvSlider();

public slots:
    virtual void setOrientation( Orientation );
    void updateOrientation();

    signals:
    void orientationChanged(Orientation o);
};


/**
 * A KAction that displays a number using a QSlider. Applications
 * can use this class to add a QSlider widget to their toolbars (eg. a
 * volume control).
 *
 * @author Richard Moore, rich@kde.org
 */
class SliderAction : public KAction
{
    Q_OBJECT

public:
    SliderAction( int min, int max, int step, int val,
                  const QString &text,
                  QObject *parent = 0, const char *name = 0 );

    virtual int plug( QWidget *widget, int index = -1 );
    virtual void unplug( QWidget *widget );

    /**
     * Returns the current value of the slider.
     */
    int value() const;

    /**
     * A factory method that creates the QSlider object.
     */
    virtual KdetvSlider *createWidget( QWidget *parent, const char *name=0 );

public slots:
    /**
     * Sets the value of the slider to the specified value.
     */
    void setValue( int num );

    /**
     * Sets the interval between the tickmarks of the slider.
     *
     * @ref QSlider::setTickInteral(int)
     */
    void setTickInterval( int ticks );

signals:
    /**
     * Emitted when the value of the slider is changed. Note the 
     * orientation of the slider is automatically accounted for
     * when this signal is emitted.
     */
    void valueChanged( int );    

protected slots:
    /**
     * Sets the value of the slider action and emits @p valueChanged signal.
     *
     * @ref QSlider::setValue(int)
     */
    void setValueInternal( int num );

    /**
     * Same as above except it inverts the given value and calls 
     * @p setValueInternal.
     *
     * @ref #setValue(int)
     */
    void setInverseInternal( int num );
    

    void setOrientation( Orientation o );
    void removeSlider( QObject* o );

private:
    QPtrList<KdetvSlider> _sliders;
    bool inhibitRecursion;
    int min;
    int max;
    int step;
    int val;
    int tickStep;
};

/**
 * A KAction for channel import/export
 *
 * @author Dirk Ziegelmeier
 */
class ImpExChannelsAction : public KAction
{
    Q_OBJECT

public:
    ImpExChannelsAction( const QString &text, int accel, const QString& fmt, Kdetv* ktv,
                         int direction, QObject *parent = 0, const char *name = 0);
    virtual ~ImpExChannelsAction();

private slots:
    virtual void slotActivated();

private:
    QString _fmt;
    Kdetv*  _ktv;
    int     _direction;
};


/**
 * A KAction for setting a device
 *
 * @author Dirk Ziegelmeier
 */
class DeviceAction : public KToggleAction
{
    Q_OBJECT

public:
    DeviceAction( const QString &text, int accel, const QString& dev, Kdetv* ktv,
                  QObject *parent = 0, const char *name = 0);
    virtual ~DeviceAction();

private slots:
    virtual void slotActivated();

private:
    QString _dev;
    Kdetv*  _ktv;
};


/**
 * A KAction for changing audio modes
 *
 * @author Dirk Ziegelmeier
 */
class AudioModeAction : public KToggleAction
{
    Q_OBJECT

public:
    AudioModeAction( const QString &text, int accel, const QString& mode, Kdetv* ktv,
                     QObject *parent = 0, const char *name = 0);
    virtual ~AudioModeAction();

private slots:
    virtual void slotActivated();

private:
    QString _mode;
    Kdetv*  _ktv;
};

#endif // ACTIONS_H

