/***************************************************************************
                          kpkabcontact.cpp  -  description
                             -------------------
    begin                : Fre Jan 17 2003
    copyright            : (C) 2003 by Bert Speckels
    email                : bert.speckels@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <iostream>
#include <list>

#include <klocale.h>
#include <kprocess.h>
#include <krun.h>
#include <kabc/addressbook.h>
#include <kabc/phonenumber.h>
#include <kabc/stdaddressbook.h>

#include "kickpim.h"
#include "kpkabcontact.h"

using namespace std;

const int numAddressTypes = 7;
const int AddressTypes[numAddressTypes] =
      { Address::Dom,  Address::Intl, Address::Postal, Address::Parcel,
        Address::Home, Address::Work, Address::Pref
      };
const QString AddressTypeNames[numAddressTypes]=
      { I18N_NOOP("Domestic Address"), I18N_NOOP("International Address"), I18N_NOOP("Postal Address"),
        I18N_NOOP("Parcel Address"),   I18N_NOOP("Home Address"),     I18N_NOOP("Work Address"),
        I18N_NOOP("Preferred Address")
      };

const int numPhoneTypes = 14;

const int PhoneTypes[numPhoneTypes] =
{
    PhoneNumber::Home,  PhoneNumber::Work,  PhoneNumber::Msg,
    PhoneNumber::Pref,  PhoneNumber::Voice, PhoneNumber::Fax,
    PhoneNumber::Cell,  PhoneNumber::Video, PhoneNumber::Bbs,
    PhoneNumber::Modem, PhoneNumber::Car,   PhoneNumber::Isdn,
    PhoneNumber::Pcs,   PhoneNumber::Pager
};

const QString PhoneTypeNames[numPhoneTypes]=
{
      I18N_NOOP("home phone"),        I18N_NOOP("work phone"),     I18N_NOOP("Messaging"),
      I18N_NOOP("Preferred Number"),  I18N_NOOP("Voice"),          I18N_NOOP("Fax Number"),
      I18N_NOOP("mobile phone"),      I18N_NOOP("Video phone"),    I18N_NOOP("Mailbox"),
      I18N_NOOP("Modem"),             I18N_NOOP("Car Phone"),      I18N_NOOP("ISDN connection"),
      I18N_NOOP("Personal Communication Service"),            I18N_NOOP("Pager")
};


KPKabContact::KPKabContact(bool autoInsert)
{
  m_addr = new Addressee();
  m_toBeInserted = autoInsert;
}

KPKabContact::KPKabContact(Addressee& addr)
{
  m_addr = &addr;
  m_toBeInserted = false;
}


KPKabContact::~KPKabContact()
{
  m_addr = 0;
}


bool KPKabContact::isEmpty() const
{
  return m_addr->isEmpty();
}



QStringMap KPKabContact::names() const
{
  // "formattedName", "realName", "familyName", "givenName", "nickName", "organizationName"

  if (!m_addr) return QStringMap();

  QStringMap map;
  map["formattedName"]    = m_addr->formattedName();
  map["familyName"]       = m_addr->familyName();
  map["givenName"]        = m_addr->givenName();
  map["nickName"]         = m_addr->nickName();
  map["organizationName"] = m_addr->organization();
  return map;
}

QAddressMap KPKabContact::addresses() const
{
  if (isEmpty()) return QAddressMap();

  QAddressMap map;
  Address addr;

  for (int i=0;i<numAddressTypes;i++)
  {
    addr = m_addr->address( AddressTypes[i] );
    if (!addr.isEmpty())
    {
      KPContactAddress address;
      address.setStreet        ( addr.street()        );
      address.setCountry       ( addr.country()       );
      address.setLocality      ( addr.locality()      );
      address.setPostOfficeBox ( addr.postOfficeBox() );
      address.setPostalCode    ( addr.postalCode()    );
      address.setRegion        ( addr.region()        );

      QString key = AddressTypeNames[i];
      map[ key ] = address;
    }
  }

  return map;
}

QStringList KPKabContact::emailAddresses() const
{
  if (isEmpty()) return QStringList();

  return m_addr->emails();
}

QStringMap KPKabContact::phoneNumbers() const
{
  if (isEmpty()) return QStringMap();

  QStringMap  map;
  PhoneNumber phone;

  for (int i=0;i<numPhoneTypes;i++)
  {
    phone = m_addr->phoneNumber( PhoneTypes[i] );
    if (!phone.number().isEmpty())
    {
      QString key = PhoneTypeNames[i];
      map[ key ]  = phone.number();
    }
  }

  return map;
}

QStringList KPKabContact::categories() const
{
  if (isEmpty()) return QStringList();

  QStringList list = m_addr->categories();

  return m_addr->categories();
}

void KPKabContact::setCategories ( QStringList list )
{
  m_addr->setCategories( list );
}

QDate KPKabContact::birthday() const
{
  if (isEmpty()) return QDate(99,99,99); // invalid date
  return m_addr->birthday().date();
}

#include <kmessagebox.h>

QDate KPKabContact::anniversary() const
{
  if (isEmpty()) return QDate(99,99,99); // invalid date

  QString dateStr = m_addr->custom( "KADDRESSBOOK" , "X-Anniversary" );

  return QDate::fromString( dateStr, Qt::ISODate );
}

QStringMap KPKabContact::webpageURLs() const
{
  if (isEmpty()) return QStringMap();
  return QStringMap();
}

QString KPKabContact::note() const
{
  if (isEmpty()) return QString();
  return m_addr->note();
}

void KPKabContact::setName(QString name, QString key)
{
  if (!m_addr) return;

  if      (key==QString("formattedName"))    m_addr->setFormattedName(name);
  else if (key==QString("familyName"))       m_addr->setFamilyName(name);
  else if (key==QString("givenName"))        m_addr->setGivenName(name);
  else if (key==QString("nickName"))         m_addr->setNickName(name);
  else if (key==QString("organizationName")) m_addr->setOrganization(name);
}

bool KPKabContact::setNote(QString& note)
{
  if (!m_addr) return false;

  m_addr->setNote(note);

  return true;
}


void KPKabContact::clearEmailAddresses()
{
  if (!m_addr) return;
  QStringList list = m_addr->emails();
  QStringList::iterator itor = list.begin();
  while ( itor != list.end() )
  {
    m_addr->removeEmail( *itor );
    ++itor;
  }
}

void KPKabContact::addEmailAddress(QString emailAddr, bool preferred)
{
  if (m_addr && !emailAddr.isEmpty())
  {
    m_addr->insertEmail(emailAddr, preferred);
  }
};



void KPKabContact::clearPhoneNumbers()
{
  if (!m_addr) return;
  PhoneNumber::List list = m_addr->phoneNumbers();
  PhoneNumber::List::iterator itor = list.begin();
  while ( itor != list.end() )
  {
    m_addr->removePhoneNumber( *itor );
    ++itor;
  }

};

void KPKabContact::addPhoneNumber(QString number, QString key)
{
  if (m_addr && !number.isEmpty())
  {
    for (int i=0;i<numPhoneTypes;i++)
    {
      if (PhoneTypeNames[i] == key)
      {
        PhoneNumber phone(number,AddressTypes[i]);
        m_addr->insertPhoneNumber( phone );
      }
    }
  }
};

void KPKabContact::setBirthday    ( QDate date )
{
  if (!m_addr) return;
  m_addr->setBirthday(date);
};

void KPKabContact::setAnniversary ( QDate date )
{
  if (!m_addr) return;
  QString dateStr = date.toString( Qt::ISODate );
  m_addr->insertCustom( "KADDRESSBOOK", "X-Anniversary" , dateStr );
};


bool KPKabContact::edit()
{
  if (!m_addr) return false;

  return KPContact::edit();
}


bool KPKabContact::remove()
{
  AddressBook* book = KABC::StdAddressBook::self();
  if (!book) return false;
  
  book->removeAddressee(*m_addr);

  changed();

  return true;
}


void KPKabContact::changed( )
{
  if (m_toBeInserted)
  {
    AddressBook* book = StdAddressBook::self();
    if (book)
    {
      QString s = m_addr->givenName() + " " + m_addr->familyName();
      if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_CONTACTS,"KPKabContact.changed. Inserted name="+s);

      book->insertAddressee(*m_addr);
      m_toBeInserted=false;
    }
  }
  KABC::StdAddressBook::save();
};
