/*
   This is taken from KBiff (Copyright (C) 1999-2001 Kurt Granroth <granroth@kde.org>)
   -------------------------------------------------------------

   Copyright (C) 2000-2001 Dawit Alemayehu <adawit@kde.org>
   Copyright (C) 2001 Rik Hemsley (rikkus) <rik@kde.org>

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU Lesser General Public License (LGPL) as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   This KMD5 class is based upon a C++ implementation of "RSA
   Data Security, Inc. MD5 Message-Digest Algorithm" by Mordechai
   T. Abzug,	Copyright (c) 1995.  This implementation passes the
   test-suite supplied with RFC 1321.

   RFC 1321 "MD5 Message-Digest Algorithm" Copyright (C) 1991-1992,
   RSA Data Security, Inc. Created 1991. All rights reserved.

   The encoding and decoding utilities in KickPimMailCodecs with the exception of
   quoted-printable were ported from the HTTPClient java package by Ronald
   Tschalr Copyright (C) 1996-1999. The quoted-printable codec as described
   in RFC 2045, section 6.7. is by Rik Hemsley (C) 2001.
*/

#include <string.h>

#include "kickpimmailcodec.h"

class CodecPrivate
{
public:
  static char Base64EncMap[64];
  static char Base64DecMap[128];

static void base64Encode( const QByteArray& in, QByteArray& out)
{
    bool insertLFs(false);

    // clear out the output buffer
    out.resize( 0 );
    if ( in.isEmpty() )
        return;

    unsigned int sidx = 0, didx = 0;
    const char* buf = in.data();
    const unsigned int len = in.size();

    unsigned int out_len = ((len+2)/3)*4;

    // Deal with the 76 characters or less per
    // line limit specified in RFC 2045 on a
    // pre request basis.
    insertLFs = (insertLFs && out_len > 76);
    if ( insertLFs )
      out_len += (out_len/76);

    int count = 0;
    out.resize( out_len );

    // 3-byte to 4-byte conversion + 0-63 to ascii printable conversion
    if ( len > 1 )
    {
        while (sidx < len-2)
        {
            if ( insertLFs )
            {
                if ( count && (count%76) == 0 )
                    out[didx++] = '\n';
                count += 4;
            }
            out[didx++] = Base64EncMap[(buf[sidx] >> 2) & 077];
            out[didx++] = Base64EncMap[(buf[sidx+1] >> 4) & 017 |
                                       (buf[sidx] << 4) & 077];
            out[didx++] = Base64EncMap[(buf[sidx+2] >> 6) & 003 |
                                       (buf[sidx+1] << 2) & 077];
            out[didx++] = Base64EncMap[buf[sidx+2] & 077];
            sidx += 3;
        }
    }

    if (sidx < len)
    {
        out[didx++] = Base64EncMap[(buf[sidx] >> 2) & 077];
        if (sidx < len-1)
        {
            out[didx++] = Base64EncMap[(buf[sidx+1] >> 4) & 017 |
                                       (buf[sidx] << 4) & 077];
            out[didx++] = Base64EncMap[(buf[sidx+1] << 2) & 077];
        }
        else
        {
            out[didx++] = Base64EncMap[(buf[sidx] << 4) & 077];
        }
    }

    // Add padding
    while (didx < out.size())
    {
        out[didx] = '=';
        didx++;
    }
}

static void base64Decode( const QByteArray& in, QByteArray& out )
{
    out.resize(0);
    if ( in.isEmpty() )
        return;

    unsigned int count = 0;
    unsigned int len = in.size(), tail = len;
    const char* in_buf = in.data();

    // Deal with possible *nix "BEGIN" marker!!
    while ( count < len && (in_buf[count] == '\n' || in_buf[count] == '\r' ||
            in_buf[count] == '\t' || in_buf[count] == ' ') )
        count++;

    if ( strncasecmp(in_buf, "begin", 5) == 0 )
    {
        count += 5;
        while ( count < len && in_buf[count] != '\n' && in_buf[count] != '\r' )
            count++;
        in_buf += (++count);
        tail = (len -= count);
    }

    // Find the tail end of the actual encoded data even if
    // there is/are trailing CR and/or LF.
    while ( in_buf[tail-1] == '=' || in_buf[tail-1] == '\n' ||
            in_buf[tail-1] == '\r' )
        if ( in_buf[--tail] != '=' ) len = tail;

    unsigned int outIdx = 0;
    out.resize( (count=len) );
    for (unsigned int idx = 0; idx < count; idx++)
    {
        // Adhere to RFC 2045 and ignore characters
        // that are not part of the encoding table.
        char ch = in_buf[idx];
        if ((ch > 47 && ch < 58) || (ch > 64 && ch < 91) ||
            (ch > 96 && ch < 123) || ch == '+' || ch == '/' || ch == '=')
        {
            out[outIdx++] = Base64DecMap[(int)ch];
        }
        else
        {
            len--;
            tail--;
        }
    }

    // 4-byte to 3-byte conversion
    len = (tail>(len/4)) ? tail-(len/4) : 0;
    unsigned int sidx = 0, didx = 0;
    if ( len > 1 )
    {
      while (didx < len-2)
      {
          out[didx] = (((out[sidx] << 2) & 255) | ((out[sidx+1] >> 4) & 003));
          out[didx+1] = (((out[sidx+1] << 4) & 255) | ((out[sidx+2] >> 2) & 017));
          out[didx+2] = (((out[sidx+2] << 6) & 255) | (out[sidx+3] & 077));
          sidx += 4;
          didx += 3;
      }
    }

    if (didx < len)
        out[didx] = (((out[sidx] << 2) & 255) | ((out[sidx+1] >> 4) & 003));

    if (++didx < len )
        out[didx] = (((out[sidx+1] << 4) & 255) | ((out[sidx+2] >> 2) & 017));

    // Resize the output buffer
    if ( len == 0 || len < out.size() )
      out.resize(len);
}
};

// static constants for base64
char CodecPrivate::Base64EncMap[64] = {
                         0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
                         0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
                         0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
                         0x59, 0x5A, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
                         0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E,
                         0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76,
                         0x77, 0x78, 0x79, 0x7A, 0x30, 0x31, 0x32, 0x33,
                         0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x2B, 0x2F
                         };

char CodecPrivate::Base64DecMap[128] = {
                          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00, 0x3F,
                          0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B,
                          0x3C, 0x3D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
                          0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
                          0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
                          0x17, 0x18, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00,
                          0x00, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
                          0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
                          0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30,
                          0x31, 0x32, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
                          };

/***************************************************************************/
QString KickPimMailCodecs::base64Encode( const QString& str )
{
    if ( str.isEmpty() )
        return QString::fromLatin1("");
    QByteArray in, out;
    const unsigned int len = str.length();
    in.resize( len );
    memcpy( in.data(), str.latin1(), len );
    CodecPrivate::base64Encode( in, out );
    return QString( out );
}

QString KickPimMailCodecs::base64Decode( const QString& str )
{
    if ( str.isEmpty() )
        return QString::fromLatin1("");

    QByteArray in, out;
    const unsigned int len = str.length();
    in.resize( str.length() );
    memcpy( in.data(), str.latin1(), len );
    CodecPrivate::base64Decode( in, out );
    return QString( out );
}
