/*
   kickpimmailmonitor.h
   
   This is taken from KBiff:
      Copyright (C) 1999-2001 Kurt Granroth <granroth@kde.org>

  -------------------------------------------------------------

   Copyright (C) 2000-2001 Dawit Alemayehu <adawit@kde.org>
   Copyright (C) 2001 Rik Hemsley (rikkus) <rik@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License (LGPL) as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   This KMD5 class is based on a C++ implementation of
   "RSA Data Security, Inc. MD5 Message-Digest Algorithm" by
   Mordechai T. Abzug,	Copyright (c) 1995.  This implementation
   passes the test-suite as defined by RFC 1321.

   RFC 1321 "MD5 Message-Digest Algorithm" Copyright (C) 1991-1992.
   RSA Data Security, Inc. Created 1991. All rights reserved.

   The encoding and decoding utilities in KCodecs with the exception of
   quoted-printable were ported from the HTTPClient java package by Ronald
   Tschalr Copyright (C) 1996-1999. The quoted-printable codec as described
   in RFC 2045, section 6.7. is by Rik Hemsley (C) 2001.

 */
#ifndef KICKPIMMAILMONITOR_H
#define KICKPIMMAILMONITOR_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h> // Needed on some systems.
#endif

#include <sys/time.h>
#include <qthread.h>
#include <qobject.h>
#include <qlist.h>
#include <qptrlist.h>
#include <qevent.h>
#include <qfileinfo.h>

class KPMailUrl;
class KPMailAccount;
class QString;
class QLabel;
class KickPimMailMonitorThread;

/**
 * @internal
 * Internal class to store UIDL list
 */
class KickPimMailUidlList : public QList<QString>
{
protected:
    int compareItems(Item s1, Item s2)
    {
        QString *str1, *str2;
        str1 = (QString *)s1;
        str2 = (QString *)s2;
        if((*str1) == (*str2))
            return 0;
        else
            return -1;
    }
};

/**
 * @internal
 */

#define SOCKET_TIMEOUT 5

class KickPimMailSocket
{
public:
    KickPimMailSocket();
    virtual ~KickPimMailSocket();

    bool connectSocket(const QString& host, unsigned short int port);
    bool active();

    bool isAsync();
    void setAsync(bool on);

    int numberOfMessages();
    int numberOfNewMessages();

    void close();

protected:
    QString readLine();
    int writeLine(const QString& line);

    bool    async;

    struct timeval socketTO;

    int socketFD;
    fd_set socketFDS;
    int messages;
    int newMessages;
};

/**
 * @internal
 */
class KickPimMailImap : public KickPimMailSocket
{
public:
    virtual ~KickPimMailImap();

    bool command(const QString& line, unsigned int seq);
    QString mungeUserPass(const QString& old_user);
    void resetNumbers();
};

/**
 * @internal
 */
class KickPimMailPop : public KickPimMailSocket
{
public:
    virtual ~KickPimMailPop();

    bool command(const QString& line);
    KickPimMailUidlList getUidlList() const;

    void close();

protected:
    KickPimMailUidlList  uidlList;    
};

/**
 * @internal
 */
class KickPimMailNntp : public KickPimMailSocket
{
public:
    virtual ~KickPimMailNntp();

    bool command(const QString& line);
    int first() const;
    int last() const;
protected:
    int firstMsg;
    int lastMsg;
};

typedef enum
{
    NewMail = 0,
    NoMail = 1,
    OldMail = 2,
    NoConn = 4,
    UnknownState = 5
} KickPimMailState;


class KickPimMailMonitor;

class KickPimMailMonitorThread : public QThread
{
public:

  KickPimMailMonitorThread( KickPimMailMonitor* monitor );
  virtual ~KickPimMailMonitorThread( );

  void setSkipMailchecks( bool on );
  void checkMailNow();

  virtual void run();
  virtual void terminateThread();

  KickPimMailMonitor* monitor();

protected:

  void logState();

private:

  QString name;
  bool    forceCheck;
  bool    skipChecks;
  bool    terminate;
  KickPimMailMonitor* m_monitor;
  QWaitCondition pollIntervalWait;
  QWaitCondition terminateWait;
};

class MailMonitorEvent : public QCustomEvent
{
public:
  static const int TYPE = QEvent::User + 6245; /* M.A.I.L */

  static const int NEW_MAIL      = 1;
  static const int NO_MAIL       = 2;
  static const int OLD_MAIL      = 3;
  static const int NO_CONNECTION = 4;
  static const int INVALID_LOGIN = 5;

  MailMonitorEvent( int id ) : QCustomEvent( TYPE ) {
    m_id = id;
  }

  int id() { return m_id; }

private:

  int m_id;
};

/**
 * A class that can monitor local and remote mailboxes for new
 * mail. KickPimMailMonitor currently supports six protocols.
 */
class KickPimMailMonitor : public QObject
{
    Q_OBJECT
public:

    /**
     * Constructor.
     */
    KickPimMailMonitor( KPMailAccount* accountData, QObject* eventListener );

    /**
     * Destructor.
     */
    virtual ~KickPimMailMonitor();

    /**
     * Sets new monitor data.
     */
    void setData( KPMailAccount* accountData );

    void initMailbox();

    /**
     * Returns the current state of the mailbox (NewMail, OldMail, NoMail, or
     * UnknownState)
     */
    KickPimMailState getMailState() const { return mailState; }

    /**
     * Returns the simpleURL of current mailbox being monitored
     */
    const QString getMailbox() const { return simpleURL; }

    /**
     * Returns <CODE>true</CODE> is KickPimMailMonitor is currently monitoring
     * a mailbox.
     */
    bool isRunning() { return m_started; }

    /**
     * Returns the number of new messages for the current mailbox
     */
    int allMessageCount()     { return m_allCount; }
    int newMessageCount()     { return m_newCount; }
    

    KPMailAccount* account() { return m_mailAccount; }

    void saveConfig();
    void readConfig();

    void postEvent( MailMonitorEvent* event );

public slots:

    void setMailLabels(QLabel* newMails, QLabel* allMails);

    //void start();
    //void stop();

    void resetMailCount();

    /**
     * Fakes KickPimMailMonitor into thinking that the mailbox was just read
     */
    void setMailboxIsRead();

    /**
     * Forces a mailbox check
     */
    void checkMailNow();

signals:

    /**
     * This will get <CODE>emit</CODE>ed everytime the mailbox
     * should be checked (determined by @ref #setPollInterval)
     */
    void signal_checkMail();

protected:

protected slots:
    void checkLocal();
    void checkMbox();
    void checkPop();
    void checkMaildir();
    void checkImap();
    void checkMHdir();
    void checkNntp();

protected:
    // protected (non-slot) functions
    void determineState(unsigned int size, const QDateTime& last_read,
                        const QDateTime& last_modified);
    void determineState(unsigned int size);

    void determineState(KickPimMailUidlList uidl_list);
    void determineState(KickPimMailState state);
    void onStateChanged();
    int  mboxMessages();

    void updateLabels();        

    void invalidLogin();

private:

    QObject* m_eventListener;
    // General stuff    
    bool    m_started;    
    bool    m_firstRun;
    QString mailbox;
    QString simpleURL;

    QString fetchCommand;
    unsigned short int port;
    bool    preauth;
    bool    keepalive;

    // New state cache
    unsigned int   new_lastSize;
    QDateTime      new_lastRead;
    QDateTime      new_lastModified;
    KickPimMailUidlList  new_uidlList;
    bool           b_new_lastSize;
    bool           b_new_lastRead;
    bool           b_new_lastModified;
    bool           b_new_uidlList;
    // State variables
    QString          mailStateMessage;
    KickPimMailState mailState;
    unsigned int   lastSize;
    QDateTime      lastRead;
    QDateTime      lastModified;
    KickPimMailUidlList  uidlList;

    // Socket protocols
    KickPimMailImap      *imap;
    KickPimMailPop       *pop;
    KickPimMailNntp      *nntp;

    // labels
    QLabel* m_newMailsLabel;
    QLabel* m_allMailsLabel;

    KPMailAccount* m_mailAccount;

    bool m_pause;
    int  m_allCount;  // complete amount of waiting messages
    int  m_newCount;  // new arrived messages since the last mail check    
    int  m_oldCount;
};

typedef QPtrList<KickPimMailMonitorThread> KPMailMonitorThreadList;

#endif // KICKPIMMAILMONITOR_H
