/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "kmobiletools_devicepart.h"

#include <kinstance.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kabc/stdaddressbook.h>
#include <kabc/resourcefile.h>
#include <kapplication.h>
#include <qtabwidget.h>

#include <qfile.h>
#include <qtextstream.h>
#include <qmultilineedit.h>
#include <qlabel.h>
#include <kpushbutton.h>
#include <kprogress.h>
#include <klistview.h>
#include <qsplitter.h>
#include <kparts/partmanager.h>
#include <klibloader.h>
#include <kmessagebox.h>
#include <qwidgetstack.h>
#include <khtmlview.h>
#include <qtimer.h>
#include <kactivelabel.h>
#include <kabc/phonenumber.h>
#include <qlayout.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <knuminput.h>
#include <klineedit.h>
#include <qvbox.h>
#include <qlistview.h>
#include <kdeversion.h>
#include <qlayout.h>
#include <kstandarddirs.h>
#include <dcopclient.h>
#include <kparts/statusbarextension.h>
#include <kstatusbar.h>
#include <kwin.h>
#include <klibloader.h>
#include <kparts/factory.h>
#include <kparts/part.h>
#include <kresources/manager.h>
#include <knotifyclient.h>
#include <kcombobox.h>
#include <kplugininfo.h>

#include "config.h"
#ifdef HAVE_KCAL
#include <libkcal/resourcelocal.h>
#endif

#include "kmobiletools_cfg.h"
#include "kmobiletools_devices.h"
#include "aboutdata.h"
#include "kmobiletoolsengine.h"
#include "statusbar.h"
#include "addressdetails.h"
#include "editaddressee.h"
#include "smspart.h"
#include "importphonebookdlg.h"
#include "exportphonebookdlg.h"
#include "popupaddressee.h"
#include "popupsms.h"
#include "contactptrlist.h"
#include "newsmsdlg.h"
#include "smslistviewitem.h"
#include "calldialogimpl.h"
#include "mainIFace_stub.h"

#define INDEX_WIDGET_ID 0
#define PHONEBOOK_WIDGET_ID 2
#define SMS_WIDGET_ID 1
#define CAL_WIDGET_ID 3

// #define ABCLIST_ID_COL 2

#define PIDFILE KGlobal::dirs()->resourceDirs("tmp").first() + ".kmobiletools-" + name + ".pid"

using namespace KMobileTools;

bool ContactsSearchLine::itemMatches (const QListViewItem *item, const QString &s) const
{
    if(KListViewSearchLine::itemMatches(item, s)) return true;
    ContactListViewItem *p_item=(ContactListViewItem*) item;
    // Search telephone numbers
    KABC::PhoneNumber::List list=p_item->contact()->phoneNumbers();
    for(QValueListIterator<KABC::PhoneNumber> it=list.begin(); it!=list.end(); ++it)
        if( (*it).number().find(s, 0, caseSensitive () ) >=0) return true;
    return false;
}

KListViewSearchLine * ContactsSearchLineWidget::createSearchLine (KListView *listView)
{
    if ( searchLine()) return searchLine();
    p_searchline= new ContactsSearchLine(this, listView);
    return p_searchline;
}

ContactListViewItem::ContactListViewItem(QListView *parent, KABC::Addressee *contact, bool readOnly)
    : KListViewItem(parent)
{
    p_contact=contact;
    b_ro=readOnly;
    setText(0, contact->formattedName());
}



kmobiletoolsDevicePart::kmobiletoolsDevicePart( QWidget *parentWidget, const char *widgetName,
        kmobiletoolsMainPart *parent, const char *name )
    : DCOPObject(name), QObject( parent, name )
{
    p_mainPart=parent;
    statusBarExtension=0;
    memslotSelected=0;
    smsTypeSelected=0;
    smsnotifynum=0;
    networkLabel=0;
    batteryProgress=0;
    devNameLabel=0;
    voidLabel=0;
    signalProgress=0;
    batteryLabel=0;
    signalLabel=0;
    statusBarBox=0;
    suspends_count=0;
    b_dialing=false;
    b_dialing_called=false;
    engine=0;
    m_widget = new mainWidget( parentWidget, widgetName );

    setupWidgets();
    home->printInfoPage( 2,  DeviceConfigInstance( name )->devicename(), 0);
    QTimer::singleShot( 1000, this, SLOT(loadEngine() ) );
    disableWidgets();

    connect(m_widget->contactsTab, SIGNAL(currentChanged ( QWidget * )), this, SLOT(contactsTabChanged()));
    connect(EnginesList::instance(), SIGNAL(phonebookUpdated()), this, SLOT(updateSMSList() ) );
    // Setup actions
    l_actionList.append( new KAction(i18n("New SMS"), "mail_generic" ,0,this,SLOT(slotNewSMS()), parent->actionCollection(), "sms_new") );
    l_actionList.append( new KAction(i18n("Export SMS List"), "exportsms" ,0,this,SLOT(slotExportSMSList()), parent->actionCollection(), "exportsms") );
    l_actionList.append( new KAction(i18n("Export SMS List to CSV"), "mail_get" ,0,this,SLOT(slotExportSMSListToCSV()), parent->actionCollection(), "exportcsv") );
    QTimer::singleShot( 1000, this, SLOT(slotStatusBar() ) );
    updateAllContacts();
}

#include "kmobiletools_devicepart.moc"

void kmobiletoolsDevicePart::setupWidgets()
{
    QVBoxLayout *layout, *layout2;
//#if KDE_IS_VERSION( 3, 4, 0 )
    slwidget=new ContactsSearchLineWidget( m_widget->phonebookListView, m_widget->abc_sl_frame);
    KListViewSearchLineWidget *sms_slwidget=new KListViewSearchLineWidget( m_widget->SMSListView, m_widget->sms_lv_src);
/*#else
    ContactsSearchLine *slwidget=new ContactsSearchLine( m_widget->abc_sl_frame, m_widget->phonebookListView);
    ContactsSearchLine *sms_slwidget=new ContactsSearchLine( m_widget->sms_lv_src, m_widget->SMSListView);
#endif*/
//     m_widget->phonebookListView->setColumnWidth(0,999);
    m_widget->phonebookListView->setShadeSortColumn(false);
    m_widget->phonebookListViewFull->setShadeSortColumn(false);
//     m_widget->phonebookListView->resize( 60, m_widget->phonebookListView->height() );
    layout=new QVBoxLayout(m_widget->abc_sl_frame);
    layout->addWidget(slwidget);
    layout->setSpacing(2);
    layout->setMargin(2);
    layout2=new QVBoxLayout(m_widget->sms_lv_src);
    layout2->addWidget(sms_slwidget);
    layout2->setSpacing(2);
    layout2->setMargin(2);
    p_addressDetails=new addressDetails(m_widget->detailsFrame, name() , this, name() );
    p_smsPart=new smsPart(m_widget->smsFrame, "smsPart", this, DeviceConfigInstance(name() )->devicename() );
    layout=new QVBoxLayout(m_widget->detailsFrame);
    layout->addWidget(p_addressDetails->view() );
    layout->setSpacing(2);
    layout->setMargin(2);
    layout=new QVBoxLayout(m_widget->smsFrame);
    layout->addWidget(p_smsPart->view() );
    layout->setSpacing(2);
    layout->setMargin(2);
    home=new homepagePart(m_widget, "homepage");
    m_widget->widgetStack->removeWidget(m_widget->widgetStack->widget(0) );
    m_widget->widgetStack->addWidget(home->view(), 0);
    m_widget->widgetStack->raiseWidget(INDEX_WIDGET_ID);
    m_widget->setFocusPolicy(QWidget::ClickFocus);

    p_listViewItem=new DeviceListViewItem(p_mainPart->listview() , DeviceConfigInstance(name() )->devicename());
    p_listViewItem->setDeviceName( name() );
    p_listViewItem->setOpen(true);
    p_listViewItem->setPixmap(0,DeviceConfigInstance(name() )->deviceTypeIcon(DeviceConfigInstance(name() )->currentGroup(), KIcon::NoGroup, KIcon::SizeSmall ));
//     (new KListViewItem(p_listViewItem, "Calls") )
//             ->setPixmap(0,KGlobal::iconLoader()->loadIcon("kaddressbook", KIcon::NoGroup, KIcon::SizeSmall) );
    (new KListViewItem(p_listViewItem, i18n("PhoneBook") ) )
            ->setPixmap(0,KGlobal::iconLoader()->loadIcon("kontact_contacts", KIcon::NoGroup, KIcon::SizeSmall) );
    p_smsItem = new KListViewItem(p_listViewItem, i18n("SMS") );
    p_smsItem->setPixmap(0,KGlobal::iconLoader()->loadIcon("mail_generic", KIcon::NoGroup, KIcon::SizeSmall) );

    // SMS Tree ListView
    SMSFolderListViewItem *sms_inbox= new SMSFolderListViewItem(m_widget->SMSFolderView, SMS::SIM | SMS::Phone, SMS::Unread | SMS::Read, i18n("Inbox") );
    SMSFolderListViewItem *sms_inbox_phone= new SMSFolderListViewItem(sms_inbox, SMS::Phone, SMS::Unread | SMS::Read, i18n("Phone") );
    SMSFolderListViewItem *sms_inbox_sim= new SMSFolderListViewItem(sms_inbox, SMS::SIM, SMS::Unread | SMS::Read, i18n("SIM") );
    SMSFolderListViewItem *sms_outbox= new SMSFolderListViewItem(m_widget->SMSFolderView, SMS::SIM | SMS::Phone, SMS::Sent | SMS::Unsent, i18n("Outgoing") );
    SMSFolderListViewItem *sms_outbox_phone= new SMSFolderListViewItem(sms_outbox, SMS::Phone, SMS::Sent | SMS::Unsent, i18n("Phone") );
    SMSFolderListViewItem *sms_outbox_sim= new SMSFolderListViewItem(sms_outbox, SMS::SIM, SMS::Sent | SMS::Unsent, i18n("SIM") );
    sms_inbox->setPixmap(0,KGlobal::iconLoader()->loadIcon("mail_get", KIcon::NoGroup, KIcon::SizeSmall) );
    sms_outbox->setPixmap(0,KGlobal::iconLoader()->loadIcon("mail_send", KIcon::NoGroup, KIcon::SizeSmall) );
    sms_inbox_phone->setPixmap(0,KGlobal::iconLoader()->loadIcon("kmobiletools", KIcon::NoGroup, KIcon::SizeSmall) );
    sms_outbox_phone->setPixmap(0,KGlobal::iconLoader()->loadIcon("kmobiletools", KIcon::NoGroup, KIcon::SizeSmall) );
    sms_inbox_sim->setPixmap(0,KGlobal::iconLoader()->loadIcon("simcard", KIcon::NoGroup, KIcon::SizeSmall) );
    sms_outbox_sim->setPixmap(0,KGlobal::iconLoader()->loadIcon("simcard", KIcon::NoGroup, KIcon::SizeSmall) );

    sms_inbox->setOpen(true);
    sms_outbox->setOpen(true);

    m_widget->SMSListView->setSorting(2, false);
    connect(home, SIGNAL(infopage(int)), SLOT(printInfoPage(int) ) );
    connect(m_widget->phonebookListView, SIGNAL(clicked ( QListViewItem *)), SLOT(pb_clicked( QListViewItem *)) );
    connect(m_widget->phonebookListViewFull, SIGNAL(clicked ( QListViewItem *)), SLOT(pb_clicked( QListViewItem *)) );
    connect(m_widget->phonebookListView, SIGNAL(currentChanged ( QListViewItem *)), SLOT(pb_clicked( QListViewItem *)) );
    connect(m_widget->phonebookListViewFull, SIGNAL(currentChanged ( QListViewItem *)), SLOT(pb_clicked( QListViewItem *)) );
    connect(p_smsPart, SIGNAL(writeNew() ), this, SLOT(slotNewSMS() ) );
    connect(p_smsPart, SIGNAL(exportList()), this, SLOT(slotExportSMSList() ) );
    connect(p_smsPart, SIGNAL(exportList()), this, SLOT(slotExportSMSListToCSV()() ) );
    connect(p_smsPart, SIGNAL(reply(const QString &)), this, SLOT(slotNewSMS( const QString& ) ) );
    connect(p_smsPart, SIGNAL(send( SMS* ) ), this, SLOT(slotSendStoredSMS( SMS* ) ) );
    connect(m_widget->SMSListView, SIGNAL(clicked ( QListViewItem * ) ), SLOT(smsSelected(QListViewItem *) ) );
    connect(m_widget->SMSListView, SIGNAL(currentChanged ( QListViewItem * )), SLOT(smsSelected(QListViewItem *) ) );
    connect(p_addressDetails, SIGNAL(exportPB() ), this, SLOT(slotSavePhonebook() ) ); // Fetching addressbook
    connect(p_addressDetails, SIGNAL(importPB() ), this, SLOT(slotUploadAddressBook() ) ); // Uploading addressbook
    connect(p_addressDetails, SIGNAL(addContact() ), this, SLOT(slotAddContact() ) );
    connect(p_addressDetails, SIGNAL(delContact() ), this, SLOT(slotDeleteContact()) );
    connect(p_addressDetails, SIGNAL(refreshClicked() ), this, SLOT(slotFetchPhonebook() ) );
    connect(p_addressDetails, SIGNAL(dial(const QString &)), this, SLOT(slotDialNumber(const QString &) ) );
    connect(p_addressDetails,SIGNAL(editClicked(KABC::Addressee* )), this, SLOT(slotEditContact(KABC::Addressee* ) )) ;
    connect(m_widget->phonebookListView, SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int ) ), 
            this, SLOT(addresseeListRightClick(QListViewItem*, const QPoint&, int ) ) );
    connect(m_widget->phonebookListViewFull, SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int ) ), 
            this, SLOT(addresseeListRightClick(QListViewItem*, const QPoint&, int ) ) );
    connect(m_widget->SMSListView, SIGNAL(rightButtonPressed(QListViewItem*, const QPoint&, int ) ), 
            this, SLOT(smsListRightClick(QListViewItem*, const QPoint&, int ) ) );
    connect(m_widget->SMSFolderView, SIGNAL(clicked( QListViewItem*)), this, SLOT(smsFolderClicked( QListViewItem*)));
    connect(home, SIGNAL(deviceCMD( const KURL& )), this, SLOT(openURL( const KURL& )) );
    connect(m_widget->b_dial, SIGNAL(clicked()), this, SLOT(slotDial()));
#ifdef HAVE_KCAL
    if(DeviceConfigInstance(name() )->calendar() )
    {
        (new KListViewItem(p_listViewItem, i18n("Calendar") ) )
                ->setPixmap(0,KGlobal::iconLoader()->loadIcon("date", KIcon::NoGroup, KIcon::SizeSmall) );
        KParts::Factory *pfactory=(KParts::Factory *) (KLibLoader::self()->factory("libkorganizerpart") );
        QVBoxLayout *l1=new QVBoxLayout(m_widget->korg_frame);
        if(pfactory)
        {
            korgpart=static_cast<KParts::ReadOnlyPart*>(pfactory->createPart( m_widget->korg_frame, "KOrganizerPart" , m_widget->korg_frame, "KOrganizerPart", "KParts::ReadOnlyPart"));
            if(korgpart)
            {
                l1->addWidget( korgpart->widget() );
                p_calendar=new KCal::CalendarLocal("");
                if(QFile::exists( KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcs" ).arg(name() ) ) )
                    p_calendar->load(KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcs" ).arg(name() ) );
                else p_calendar->save(KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcs" ).arg(name() ) );
                korgpart->openURL( KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcs" ).arg(name() ) );
                l_actionList.append( new KAction(i18n("Fetch Calendar"), "date" ,0,this,SLOT(slotFetchCalendar() ), parent()->actionCollection(), "get_cal") );
            }
        }
    }
#endif
}

void kmobiletoolsDevicePart::loadEngine()
{
    QString libName;
    KLibFactory *factory;
    devIsConnected=false;
    KPluginInfo *infos=EnginesList::instance()->engineInfo( DeviceConfigInstance(name() )->engine() );
    if(!infos)
    {
        engine=0;
        KMessageBox::error(m_widget, i18n("The selected engine could not be found. Please reinstall KMobileTools.") );
        emit deleteThis( name() );
        return;
    }
    libName=infos->service()->library();
    kdDebug() << "Trying to load library \"" << libName << "\"\n";
    factory=KLibLoader::self()->factory(libName);
    if (factory)
    {
        engine = static_cast<kmobiletoolsEngine *>(factory->create(this, name(), "kmobiletoolsEngine" ));
    }
    else
    {
        engine=0;
        kdDebug() << KLibLoader::self()->lastErrorMessage() << endl;
        KMessageBox::error(m_widget, i18n("Could not load the device %1.\nIf this error persists, please restart KMobileTools.").arg(name() ) );
        emit deleteThis( name() );
        return;
    }
    connect(engine, SIGNAL(connected()), this, SLOT(devConnected()) );
    connect(engine, SIGNAL(disconnected()), this, SLOT(devDisconnected() ) );
    connect(engine, SIGNAL(connected() ), this, SLOT(enableWidgets() ) );
    connect(engine, SIGNAL(disconnected() ), this, SLOT(disableWidgets() ) );
    connect(engine, SIGNAL(updateInfoPage(int)), this, SLOT( printInfoPage(int) ));  
    connect(engine, SIGNAL(addressBookToUpdate() ), this, SLOT(slotFetchPhonebook() ) );
    connect(engine, SIGNAL(phoneBookUpdated() ), SLOT(updatePB()) );
    connect(engine, SIGNAL(phoneBookUpdated(int, const ContactPtrList& ) ), SLOT(updatePB(int, const ContactPtrList& ) ) );
    connect(engine, SIGNAL(smsListUpdated() ), SLOT(updateSMSList()) );
    connect(engine, SIGNAL(addSMSFolders() ), SLOT(addSMSFolders()) );
    connect(engine, SIGNAL(smsAdded( const QCString& )), SLOT(smsAdded( const QCString&) ) );
    connect(engine, SIGNAL(smsDeleted( const QCString& )), SLOT(smsRemoved(const QCString&) ) );
    connect(engine, SIGNAL(smsModified( const QCString& )), SLOT(smsModified( const QCString& )) );
    connect(engine, SIGNAL(isRinging( bool )), this, SLOT(slotRing( bool ) ) );
    connect(engine, SIGNAL(fullPhonebook()), this, SLOT(fullPhonebook()) );
    connect(p_smsPart, SIGNAL(getSMSList() ), engine, SLOT( retrieveSMSList() ) );
    connect(p_smsPart, SIGNAL(remove( SMS* ) ), engine, SLOT(slotDelSMS( SMS* ) ) );
    connect(engine->smsList(), SIGNAL(updated()), this, SLOT(updateSMSList() ) );
    connect(engine, SIGNAL(jobFinished(kmobiletoolsJob::JobType)), this, SLOT(jobDone(kmobiletoolsJob::JobType)));
#ifdef HAVE_KCAL
    connect(engine, SIGNAL(calendarParsed() ), this, SLOT(slotCalendarFetched() ) );
#endif
    home->printInfoPage( 2, engine );
    engine->probePhone();
    if( DeviceConfigInstance(name() )->status_poll() && DeviceConfigInstance(name() )->status_pollTimeout() > 0 )
    {
        statusPollTimer=new QTimer(this);
        connect(statusPollTimer, SIGNAL(timeout() ), engine, SLOT(pollPhoneStatus()) );
        statusPollTimer->start( (int) DeviceConfigInstance(name() )->status_pollTimeout() * 1000, false);
    }
    if( DeviceConfigInstance(name() )->smsPoll() && DeviceConfigInstance(name() )->sms_pollTimeout() > 0 )
    {
        smsPollTimer=new QTimer(this);
        connect(smsPollTimer, SIGNAL(timeout() ), engine, SLOT(retrieveSMSList()) );
        uint smsTimeout=(int) DeviceConfigInstance(name() )->sms_pollTimeout() * 1000;
        smsPollTimer->start( smsTimeout, false);
        if(smsTimeout > 30*1000)
            // if we've a long timer for sms just retrieve them earlier the first time.
            QTimer::singleShot(15*1000, engine, SLOT(retrieveSMSList()));
    }
    updateSMSCount();
}


const QString kmobiletoolsDevicePart::friendlyName()
{
    return DeviceConfigInstance(name())->devicename();
}

kmobiletoolsDevicePart::~kmobiletoolsDevicePart()
{
    kdDebug() << "kmobiletoolsDevicePart::~kmobiletoolsDevicePart()\n";
    EnginesList::instance()->unlock(name() );
//     f_pidfile.remove();
//     delete m_widget;
//     delete m_statusbar;
}

void kmobiletoolsDevicePart::setupStatusBar()
{
    if(!devNameLabel || !networkLabel || !voidLabel || !batteryLabel ||
        !batteryProgress || !signalLabel || !signalProgress || !statusBarBox )
    {
        QTimer::singleShot( 70, this, SLOT(setupStatusBar() ) );
        return;
    }
    statusBarExtension->addStatusBarItem( devNameLabel, 0, true );
    statusBarExtension->addStatusBarItem( networkLabel, 0, true );
    statusBarExtension->addStatusBarItem( voidLabel, 1, true );

    statusBarExtension->addStatusBarItem( batteryLabel, 0, true );

    statusBarExtension->addStatusBarItem( batteryProgress, 1, true);
    statusBarExtension->addStatusBarItem( signalLabel, 0, true );

    statusBarExtension->addStatusBarItem( signalProgress, 1, true);
    statusBarExtension->addStatusBarItem( statusBarBox, 0, true );
}

void kmobiletoolsDevicePart::clearStatusBar()
{
    if(!statusBarExtension) return;
    if(devNameLabel) statusBarExtension->removeStatusBarItem(devNameLabel);
    if(networkLabel) statusBarExtension->removeStatusBarItem(networkLabel);
    if(voidLabel) statusBarExtension->removeStatusBarItem(voidLabel);
    if(batteryLabel) statusBarExtension->removeStatusBarItem(batteryLabel);
    if(batteryProgress) statusBarExtension->removeStatusBarItem(batteryProgress);
    if(signalLabel) statusBarExtension->removeStatusBarItem(signalLabel);
    if(signalProgress) statusBarExtension->removeStatusBarItem(signalProgress);
    if(statusBarBox) statusBarExtension->removeStatusBarItem(statusBarBox);
}

void kmobiletoolsDevicePart::addSMSFolders()
{
    QStringList list( engine->smsFolders() );

    for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) 
    {
        int fn = 1;
        KListViewItem* i = new KListViewItem(p_smsItem, *it );
        i->setPixmap(0,KGlobal::iconLoader()->loadIcon("mail_generic", KIcon::NoGroup, KIcon::SizeSmall) );
        i->setText( 1, "SMSFolder" );
        i->setText( 2, QString::number( fn ) );
     }

}

/*!
    \fn kmobiletoolsDevicePart::clicked ( QListViewItem * item )
 */
void kmobiletoolsDevicePart::clicked ( QListViewItem * item )
{
    if (! item) return;
    if ( item->text(0) == i18n("PhoneBook") )
    {
        m_widget->widgetStack->raiseWidget(PHONEBOOK_WIDGET_ID);
        return;
    }
    if ( item->text(0) == i18n("SMS") || item->text(1) == "SMSFolder" )
    {
        m_widget->widgetStack->raiseWidget(SMS_WIDGET_ID);
        return;
    }
    if (item->text(0) == i18n("Calendar") )
    {
        m_widget->widgetStack->raiseWidget(CAL_WIDGET_ID);
        return;
    }
    m_widget->widgetStack->raiseWidget(INDEX_WIDGET_ID);
}

void kmobiletoolsDevicePart::updateAllContacts(ContactPtrList *addressBook)
{
    KABC::Addressee *addressee;
    QPtrListIterator<KABC::Addressee> it_addressee(*addressBook);
    while( (addressee=it_addressee.current()) )
    {
        ++it_addressee;
        if(!addressee->phoneNumbers().count()) continue;
        new ContactListViewItem(m_widget->phonebookListViewFull, addressee, true);
    }
}
void kmobiletoolsDevicePart::updateAllContacts()
{
    m_widget->phonebookListViewFull->clear();
    ContactPtrList *abc=new ContactPtrList(KABC::StdAddressBook::self()->allAddressees() );
    if(!abc) return;
    updateAllContacts(abc);
    QPtrListIterator<kmobiletoolsEngine> it_engine ( *EnginesList::instance() );
    kmobiletoolsEngine *p_engine;
    while( (p_engine=it_engine.current()) )
    {
        ++it_engine;
        if( engine && QString(p_engine->name())==QString(engine->name())) continue;
        kdDebug() << "DevicePart " << name() << ": adding contacts from engine " << p_engine->name() << endl;
        updateAllContacts( p_engine->addresseeList() );
    }
}

void kmobiletoolsDevicePart::jobDone(kmobiletoolsJob::JobType jobtype)
{
    if(jobtype==kmobiletoolsJob::fetchAddressBook) emit phonebookUpdated();
    int newsmscnt=engine->smsList()->count(SMS::Unread, SMS::SIM | SMS::Phone );

    if(newsmscnt && engine->ThreadWeaver()->isEmpty() && engine->ThreadWeaver()->isIdle() && newsmscnt!=smsnotifynum) {
        smsnotifynum=newsmscnt;
        QString eventString(i18n("<qt>%1 New Messages.<br>Mobile Phone: %2</qt>").arg(newsmscnt).arg(DeviceConfigInstance(name() )->devicename() ) );
        KNotifyClient::event( KMobileTools::KMobiletoolsHelper::instance()->systray()->winId(), "kmobiletools_sms", eventString );
    }
}


void kmobiletoolsDevicePart::updatePB()
{
//     updating without partial updates should ALWAYS clear the listview...
    m_widget->phonebookListView->clear();
//     p_addressbook->clear();
    ContactPtrList *phoneBook =engine->addresseeList();
    /*    if (!(phoneBook->count() ) ) return;*/
//     p_addressDetails->stopFetch();
//     m_widget->phonebookListView->clear();
    for ( phoneBook->first(); phoneBook->current(); phoneBook->next() )
    {
        new ContactListViewItem( m_widget->phonebookListView, phoneBook->current() );
//         p_addressbook->insertAddressee( *phoneBook->current() );
    }
    home->printInfoPage( home->currentInfoPage(), engine );
    enableWidgets(false);
    slotSaveAddressBook();
    emit phonebookUpdated();
}

void kmobiletoolsDevicePart::updatePB(int , const ContactPtrList &p_phoneBook)
{
    ContactPtrList *phoneBook = (ContactPtrList *) &p_phoneBook;
    /*    if (!(phoneBook->count() ) ) return;*/
//     p_addressDetails->stopFetch();
//     m_widget->phonebookListView->clear();
    for ( phoneBook->first(); phoneBook->current(); phoneBook->next() )
    {
        new ContactListViewItem( m_widget->phonebookListView, phoneBook->current() );
//         p_addressbook->insertAddressee( *phoneBook->current() );
    }
    home->printInfoPage( home->currentInfoPage(), engine );
    enableWidgets(false);
//     slotSaveAddressBook();
//     emit phonebookUpdated();
}

const QStringList kmobiletoolsDevicePart::parseAddressee(KABC::Addressee *addressee)
{
    QStringList retval;
    QValueListIterator<KABC::PhoneNumber> it;
    QValueList<KABC::PhoneNumber> p_list=addressee->phoneNumbers();
    for( it=p_list.begin(); it!=p_list.end() ;++it )
        retval+=(*it).number();
    retval+=addressee->emails();

    return retval;
}

/*!
    \fn kmobiletoolsDevicePart::pb_clicked ( QListViewItem * item )
 */
void kmobiletoolsDevicePart::pb_clicked ( QListViewItem * item )
{
    if(!item)return;
    ContactListViewItem* c_item=((ContactListViewItem*) item);
    p_addressDetails->showAddressee( c_item->contact(), c_item->readOnly() );
}

void kmobiletoolsDevicePart::devDisconnected()
{
    devIsConnected=false;
    emit disconnected();
    home->printInfoPage(0, this->engine);
}

void kmobiletoolsDevicePart::devConnected()
{
//     kdDebug() << "kmobiletoolsDevicePart::devConnected()" << endl;
    devIsConnected=true;
    emit connected();
    DeviceConfigInstance(name() )->setLastpath(engine->currentDeviceName());
    home->printInfoPage(0, this->engine);
}

/*!
    \fn kmobiletoolsDevicePart::enableWidgets()
 */
void kmobiletoolsDevicePart::enableWidgets( bool fetch)
{
    if (!fetch) return;
    // It seems that having a retrieveAddressBook too early it's confusing.. so let's start it after some secs
    QTimer::singleShot( 5000, this, SLOT(slotFetchPhonebook() ) );
}


/*!
    \fn kmobiletoolsDevicePart::disableWidgets()
 */
void kmobiletoolsDevicePart::disableWidgets()
{
//     m_widget->pb_PutPhoneBook->setEnabled(false);
//     m_widget->pb_getPhoneBook->setEnabled(false);
//     m_widget->pb_add->setEnabled(false);
//     m_widget->pb_del->setEnabled(false);
}


/*!
    \fn kmobiletoolsDevicePart::smsSelected(QListViewItem *smsItem)
 */
void kmobiletoolsDevicePart::smsSelected(QListViewItem *smsItem)
{
        if (! smsItem) return;
        SMSListViewItem *smsLVItem=(SMSListViewItem*) smsItem;
        p_smsPart->show( smsLVItem->sms() );
        smsLVItem->selected();
}


/*!
    \fn kmobiletoolsDevicePart::slotAddContact()
 */
void kmobiletoolsDevicePart::slotAddContact()
{
    editAddressee *new_contact=new editAddressee( engine->manufacturer(), engine->availPbSlots() );
    if (new_contact->exec() == QDialog::Accepted )
    {
        QValueList<KABC::Addressee> *abclist=new QValueList<KABC::Addressee>();
        KABC::Addressee *addressee=new_contact->getAddressee();
        abclist->append( *addressee );
        engine->slotAddAddressee( abclist );
    }
}

void kmobiletoolsDevicePart::slotEditContact(KABC::Addressee *p_addressee)
{
    editAddressee *new_contact=new editAddressee( p_addressee, engine->manufacturer(), engine->availPbSlots() );
    if (new_contact->exec() == QDialog::Accepted )
        engine->slotEditAddressee(p_addressee, new_contact->getAddressee() );
}


/*!
    \fn kmobiletoolsDevicePart::slotDeleteContact()
 */
void kmobiletoolsDevicePart::slotDeleteContact()
{
    QPtrList<QListViewItem> lst;
    QStringList(sl_toDelete);
    QListViewItemIterator l_it( m_widget->phonebookListView, QListViewItemIterator::Selected );
    if( ! l_it.current()  ) return;
    sl_toDelete+= l_it.current()->text(0); // Adding the first item to delete.
    bool multiple_selected= (bool) *( ++l_it );
    while ( l_it.current() )
    {
        sl_toDelete+= l_it.current()->text(0);
        ++l_it;
    }

    int retval;
    if( multiple_selected )
        retval=KMessageBox::warningYesNoList(NULL,
                                             i18n("<qt>This will permanently delete the following contacts.\nContinue?</qt>"),
                                             sl_toDelete,
                                             "KMobileTools");
    else
        retval=KMessageBox::warningYesNo(NULL,
                                         i18n("<qt>This will permanently delete the contact <b>%1</b> from the phone addressbook.\nContinue?</qt>")
                                                 .arg( sl_toDelete.first() ),
                                         "KMobileTools" );

    if(retval!=KMessageBox::Yes) return;
    QValueList<KABC::Addressee> *toDelete=new QValueList<KABC::Addressee>();

    l_it=QListViewItemIterator ( m_widget->phonebookListView, QListViewItemIterator::Selected );

    while ( l_it.current() ) {
        toDelete->append(*((ContactListViewItem*)(*l_it))->contact());
        ++l_it;
    }
    engine->slotDelAddressee( toDelete);
}

void kmobiletoolsDevicePart::slotFetchPhonebook()
{
//     p_addressbook->clear();
    disableWidgets();
    m_widget->phonebookListView->clear();
    engine->retrieveAddressBook();
    p_addressDetails->showHP();
//     p_addressDetails->startFetch();
}

void kmobiletoolsDevicePart::slotSavePhonebook()
{
    exportPhonebookDlg *exportDlg=new exportPhonebookDlg(engine->addresseeList() );
    exportDlg->exec();
}



/*!
    \fn kmobiletoolsDevicePart::slotUploadAddressBook()
 */
void kmobiletoolsDevicePart::slotUploadAddressBook()
{
    importPhonebookDlg *abcDialog=new importPhonebookDlg( engine->availPbSlots() );
    if (abcDialog->exec() != QDialog::Accepted ) return;
    if(abcDialog->deletePhoneBook() )
    {
        QValueList<KABC::Addressee> *toDeleteList=new QValueList<KABC::Addressee>;
        ContactPtrList *phoneBook =engine->addresseeList();
        for ( phoneBook->first(); phoneBook->current(); phoneBook->next() )
            toDeleteList->append( *(phoneBook->current() ) );
        engine->slotDelAddressee( toDeleteList);
    }
    engine->slotAddAddressee( abcDialog->addresseeList() );
}

void kmobiletoolsDevicePart::stopDevice()
{
    if( ! suspends_count ) engine->stopDevice();
    suspends_count++;
}
void kmobiletoolsDevicePart::resumeDevice()
{
    suspends_count--;
    if ( ! suspends_count ) engine->resumeDevice();
}



/*!
    \fn kmobiletoolsDevicePart::addresseeListRightClick(QListViewItem *item, const QPoint &point, int column)
 */
void kmobiletoolsDevicePart::addresseeListRightClick(QListViewItem *item, const QPoint &point, int )
{
    if(!item) return;
    ContactListViewItem *c_item=(ContactListViewItem*)item;
    popupAddressee *popup=new popupAddressee( c_item->contact(), m_widget->phonebookListView, name(), c_item->readOnly() );
    connect(popup, SIGNAL(editClicked(KABC::Addressee* ) ), this, SLOT(slotEditContact(KABC::Addressee* ) ) );
    connect(popup, SIGNAL(delContact() ), this, SLOT(slotDeleteContact() ) );
    popup->exec(point);
}

void kmobiletoolsDevicePart::errNotConnected()
{
    KMessageBox::error( m_widget, i18n("Please connect your mobile phone first."));
}

/*!
    \fn kmobiletoolsDevicePart::smsListRightClick(QListViewItem *item, const QPoint &point, int column)
 */
void kmobiletoolsDevicePart::smsListRightClick(QListViewItem *item, const QPoint &point, int )
{
    if(!item) return;
    popupSMS *popup=new popupSMS( ((SMSListViewItem*)item)->sms(), m_widget->SMSListView);
//     connect(popup, SIGNAL(editClicked(KABC::Addressee* ) ), this, SLOT(slotEditContact(KABC::Addressee* ) ) );
//     connect(popup, SIGNAL(delContact() ), this, SLOT(slotDeleteContact() ) );
    connect(popup, SIGNAL(remove( SMS* )), engine, SLOT(slotDelSMS(SMS*)) );
    popup->exec(point);
}


/*!
    \fn kmobiletoolsDevicePart::raiseDevice()
 */
void kmobiletoolsDevicePart::raiseDevice()
{
    QWidget *widget=kapp->mainWidget();
    if(widget)
    {
        // Thanks to boren@ #kde-devel for these infos ;-)
//         KWin::setCurrentDesktop(KWin::windowInfo( widget->winId() ).desktop() );
        KWin::activateWindow( widget->winId() );
        widget->show();
        widget->raise();
        widget->setFocus();
        widget->setActiveWindow();
    }
    (new MainIFace_stub("kmobiletools", "KMobileTools"))->switchPart(name() );
}


/*!
    \fn kmobiletoolsDevicePart::smsModified(const QCString& smsUID)
 */
void kmobiletoolsDevicePart::smsModified(const QCString& smsUID)
{
    kdDebug( ) << "kmobiletoolsDevicePart::smsModified(" << smsUID << ")\n";
}


/*!
    \fn kmobiletoolsDevicePart::smsAdded(const QCString& smsUID)
 */
void kmobiletoolsDevicePart::smsAdded(const QCString& smsUID)
{
    updateSMSCount();
    home->printInfoPage( home->currentInfoPage(), engine );
    kdDebug( ) << "kmobiletoolsDevicePart::smsAdded(" << smsUID << ")\n";
    SMSList *smsList = engine->smsList();
    int newSMSIndex=smsList->find( smsUID );
    if(newSMSIndex<0) return;
    SMS *newSMS=smsList->at(newSMSIndex);
    if( !(newSMS->slot() & memslotSelected) || ! (newSMS->type() & smsTypeSelected)  ) return;
    new SMSListViewItem( m_widget->SMSListView, newSMS, engine->addresseeList() );
}

void kmobiletoolsDevicePart::raisePage(int page)
{
    if(!m_widget->widgetStack->widget(page)) return;
    m_widget->widgetStack->raiseWidget(page);
}

QString kmobiletoolsDevicePart::currentDeviceName() const
{
    return engine->currentDeviceName();
}

/*!
    \fn kmobiletoolsDevicePart::smsRemoved(const QCString& smsUID)
 */
void kmobiletoolsDevicePart::smsRemoved(const QCString& smsUID)
{
//     kdDebug( ) << "kmobiletoolsDevicePart::smsRemoved(" << smsUID << ")\n";
//     updateSMSList();
//     return;
    updateSMSCount();
    home->printInfoPage( home->currentInfoPage(), engine );
    SMSList *smsList = engine->smsList();
    int oldSMSIndex=smsList->find( smsUID );
    if(oldSMSIndex<0) return;
    SMS *oldSMS=smsList->at(oldSMSIndex);
    if( !(oldSMS->slot() & memslotSelected) || ! (oldSMS->type() & smsTypeSelected)  ) return;

    QListViewItemIterator it(m_widget->SMSListView);
    SMSListViewItem *tempItem;
    while( (tempItem=((SMSListViewItem*)it.current())) !=0 )
    {
        if( tempItem->sms()->uid() == smsUID )
        {
            delete tempItem;
//             return;
        }
        ++it;
    }
//     smsList->dump();
}


/*!
    \fn kmobiletoolsDevicePart::smsFolderClicked( QListViewItem * item )
 */
void kmobiletoolsDevicePart::smsFolderClicked( QListViewItem * item )
{
    if(!item) return;
    SMSFolderListViewItem *sms_item=(SMSFolderListViewItem*) item;
    memslotSelected=sms_item->memSlot(); smsTypeSelected=sms_item->smsType();
    updateSMSList();
}

void kmobiletoolsDevicePart::updateSMSList()
{
    SMSList *smsList = engine->smsList();
    m_widget->SMSListView->clear();
    SMS *newSMS;
//     KListViewItem *newSMSItem;
//     smsList->dump();
    for ( smsList->first(); smsList->current(); smsList->next() )
    {
        newSMS=smsList->current();
        if( !(newSMS->slot() & memslotSelected) || ! (newSMS->type() & smsTypeSelected)  ) continue;
        new SMSListViewItem( m_widget->SMSListView, newSMS, engine->addresseeList() );
    }
    updateSMSCount();

    if( (smsTypeSelected & SMS::Unread) || (smsTypeSelected & SMS::Read) )
    {
        m_widget->SMSListView->adjustColumn(0);
        m_widget->SMSListView->hideColumn(1);
    } else
    {
        m_widget->SMSListView->adjustColumn(1);
        m_widget->SMSListView->hideColumn(0);
    }
}

void kmobiletoolsDevicePart::slotNewSMS(const QString &number)
{
    newSMSDlg *dialogNewSMS = new newSMSDlg(m_widget, name() );
    if(number!=QString::null) dialogNewSMS->addNumber( number );
    if(! dialogNewSMS->exec()) return;
    if( dialogNewSMS->action() & newSMSDlg::Send )
        engine->slotSendSMS(dialogNewSMS->getSMSItem() );
    if( dialogNewSMS->action() & newSMSDlg::Store )
        engine->slotStoreSMS(dialogNewSMS->getSMSItem() );
}


void kmobiletoolsDevicePart::sendSMS(const QString& number, const QString& text)
{
    engine->slotSendSMS( number, text );
}
void kmobiletoolsDevicePart::storeSMS(const QString& number, const QString& text)
{
    engine->slotStoreSMS( number, text);
}


/*!
    \fn kmobiletoolsDevicePart::openURL(const KURL &url)
 */
void kmobiletoolsDevicePart::openURL(const KURL &url)
{
    kdDebug() << "Parsing url " << url << endl;
    if(url.path() == "sms") m_widget->widgetStack->raiseWidget(SMS_WIDGET_ID);
    if(url.path() == "phonebook") m_widget->widgetStack->raiseWidget(PHONEBOOK_WIDGET_ID);
    if(url.path() == "tryconnect")
    {
        if(suspends_count) resumeDevice();
        else engine->probePhone();
        home->printInfoPage( 2, engine);
    }
    if(url.path() == "configure")
    {
        emit command( QString("configure:") + name() );
        kdDebug() << "emitted command(" << QString("configure:") + name() << ")\n";
    }
}


/*!
    \fn kmobiletoolsDevicePart::updateSMSCount()
 */
void kmobiletoolsDevicePart::updateSMSCount()
{
    if(!engine) return;
    engine->smsList()->calcSMSNumber();

    QListViewItemIterator it( m_widget->SMSFolderView );
    SMSFolderListViewItem *tempItem;
    while ( it.current() ) {
        tempItem=(SMSFolderListViewItem*) it.current();
        tempItem->setText(1, QString::number(engine->smsList()->count( ( tempItem->smsType() & (SMS::Unread | SMS::Unsent) ), tempItem->memSlot() ) ) );
        tempItem->setText(2,QString::number(engine->smsList()->count( tempItem->smsType(), tempItem->memSlot() ) ) );
        ++it;
    }
}

// This method is partially derived from akregator - articlelistview.cpp, See copyright statement below
//    Copyright (C) 2007 Frank Osterfeld <frank.osterfeld at kdemail.net>


void SMSFolderListViewItem::paintCell( QPainter * p, const QColorGroup & cg,
                                 int column, int width, int align )

{
    if(column != 1 || text(column).toInt() <=0 )
    {
        KListViewItem::paintCell(p,cg,column,width,align);
        return;
    }
    QColorGroup cg2(cg);
    if( i_smsType & SMS::Unsent) cg2.setColor(QColorGroup::Text, Qt::red);
    else cg2.setColor(QColorGroup::Text, Qt::blue);
    KListViewItem::paintCell( p, cg2, column, width, align );
}



/*!
    \fn kmobiletoolsDevicePart::fullPhonebook()
 */
void kmobiletoolsDevicePart::fullPhonebook()
{
    KMessageBox::error( m_widget, i18n("Could not import all contacts. Phonebook is full.") );
}


/*!
    \fn kmobiletoolsDevicePart::slotFetchCalendar()
 */
void kmobiletoolsDevicePart::slotFetchCalendar()
{
    engine->fetchCalendar();
}


/*!
    \fn kmobiletoolsDevicePart::slotCalendarFetched()
 */
void kmobiletoolsDevicePart::slotCalendarFetched()
{
#ifdef HAVE_KCAL
    QString savefile=KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcs" ).arg(name() );

    korgpart->closeURL();
    kdDebug() << "kmobiletoolsDevicePart::slotCalendarFetched()\n";
    Calendar *engineCal=engine->calendar();
    p_calendar->deleteAllEvents ();
    for(Calendar::Iterator it=engineCal->begin(); it!=engineCal->end(); ++it)
        p_calendar->addEvent(*it);
    p_calendar->save(savefile );
    if(engineCal->count() ==0 )
    {
        p_calendar->close();
        delete p_calendar;
        QFile::remove( savefile );
        p_calendar=new KCal::CalendarLocal("");
        p_calendar->save(savefile );
    } else
    {
        KRES::Manager<KCal::ResourceLocal> *manager=new KRES::Manager<KCal::ResourceLocal>("calendar");
        KConfig *config=new KConfig("kresources/calendar/stdrc");
        manager->readConfig( config );
        KRES::Manager<KCal::ResourceLocal>::Iterator it;
        bool isPresent=false;
        for(it=manager->begin(); it!=manager->end(); it++)
            if( (*it)->fileName() == savefile) isPresent=true;
        if(!isPresent)
        {
            KCal::ResourceLocal *res=new KCal::ResourceLocal(savefile);
            res->setResourceName( DeviceConfigInstance(name() )->devicename() );
//            res->setValue("Format", "vcal");
            res->setReadOnly(true);
            manager->add(res);
            manager->writeConfig(config);
        }
        delete config;
        delete manager;
    }
    korgpart->openURL( savefile );
#endif
}


/*!
    \fn kmobiletoolsDevicePart::slotExportSMSList()
 */
void kmobiletoolsDevicePart::slotExportSMSList()
{
//     if ( KMobileTools::MainConfig::self()->maildir() )
    KMobileTools::KMobiletoolsHelper::createMailDir( name() );
    engine->smsList()->saveToMailBox();
    kdDebug() << "STARTING SMS EXPORT\n";
    KMessageBox::information( m_widget, i18n("<qt>SMS List for the mobile phone <b>%1</b> was exported to KMail default directory (%2).<br>To view exported messages, close and reopen KMail.</qt>").arg( DeviceConfigInstance(name() )->devicename() ).arg(DeviceConfigInstance(name() )->maildir_path() ), i18n("SMS List Exported."), "smslistexported_infobox" );
}

/*!
    \fn kmobiletoolsDevicePart::slotExportSMSListToCSV()
 */
void kmobiletoolsDevicePart::slotExportSMSListToCSV()
{
    int result;
    
    kdDebug() << "STARTING SMS EXPORT TO CSV\n";
    result = engine->smsList()->saveToCSV();
    if (result >= 1) {
        KMessageBox::information( m_widget, i18n("<qt>SMS List for the mobile phone <b>%1</b> was exported to the selected Directory.</qt>").arg( DeviceConfigInstance(name() )->devicename() ), i18n("SMS List Exported."), "smslistexportedtocsv_infobox" );
    }
}


/*!
    \fn kmobiletoolsDevicePart::setupStatusBar()
 */
void kmobiletoolsDevicePart::slotStatusBar()
{
    statusBarExtension=((kmobiletoolsMainPart *)parent())->statusBarExtension();

    signalProgress=new KProgress(100, statusBarExtension->statusBar(), "signal");
    batteryProgress=new KProgress(100, statusBarExtension->statusBar(), "battery");
    networkLabel=new KStatusBarLabel("", 1, statusBarExtension->statusBar() );
    signalProgress->setMaximumSize(70,16);
    batteryProgress->setMaximumSize(70, 16);
//     statusBarExtension->statusBar()->clear();
    statusBarBox=new StatusBarProgressBox( statusBarExtension->statusBar(), widget() );
    devNameLabel=new KStatusBarLabel(DeviceConfigInstance(name())->devicename(), 2, statusBarExtension->statusBar() );
    voidLabel=new KStatusBarLabel("", 3, statusBarExtension->statusBar() );
    batteryLabel=new KStatusBarLabel( i18n("Battery"), 4, statusBarExtension->statusBar() );
    signalLabel=new KStatusBarLabel( i18n("Signal"), 5, statusBarExtension->statusBar() );
    if(!engine) return;
    connect(engine, SIGNAL(charge(int ) ), batteryProgress, SLOT(setProgress(int ) ) );
    connect(engine, SIGNAL(signal(int ) ), signalProgress, SLOT(setProgress(int ) ) );
    connect(engine, SIGNAL(networkName( const QString &) ), networkLabel, SLOT(setText(const QString& ) ) );
    connect(engine, SIGNAL( jobEnqueued(kmobiletoolsJob *) ), statusBarBox, SLOT(slotJobEnqueued(kmobiletoolsJob* ) ) );
}


/*!
    \fn kmobiletoolsDevicePart::slotSaveAddressBook()
 */
void kmobiletoolsDevicePart::slotSaveAddressBook()
{
    if( engine->addresseeList()->isEmpty() ) return;
    KABC::AddressBook *p_addressbook;
    KABCResFile *p_resourcefile;
    QString filename=KGlobal::dirs()->saveLocation( "data", "kmobiletools", true).append( "%1.vcf" ).arg(name() );
    p_resourcefile=0;
    KRES::Manager<KABCResFile> *manager=new KRES::Manager<KABCResFile>("contact");
    KConfig *config=new KConfig("kresources/contact/stdrc");
    manager->readConfig( config );
    KRES::Manager<KABCResFile>::Iterator it;
    bool found=false;
    for(it=manager->begin(); it!=manager->end(); it++)
        if( (*it)->fileName() == filename )
    {
        p_resourcefile=(*it);
        found=true;
        break;
    }
    if(!found)
        p_resourcefile=new KABCResFile( filename );
    p_addressbook=new KABC::AddressBook();
    p_resourcefile->setResourceName( DeviceConfigInstance(name() )->devicename() );

    p_addressbook->addResource( p_resourcefile);
    p_addressbook->load();
    p_addressbook->clear();
    QPtrListIterator<KABC::Addressee> pbit( *engine->addresseeList() );
    while(pbit.current() )
    {
        p_addressbook->insertAddressee( *pbit.current() );
        ++pbit;
    }
    p_resourcefile->setReadOnly(false);
    KABC::Ticket *ticket=p_addressbook->requestSaveTicket( p_resourcefile );
    if(!ticket)
    {
        kdDebug() << "Error: Unable to save to KAddressBook (engine " << name() << ", " << DeviceConfigInstance(name() )->devicename() << "; filename: " << p_resourcefile->fileName() << ")\n";
        return;
    }
    p_addressbook->save( ticket );
    p_resourcefile->setReadOnly(true);

    if( !found && ! p_addressbook->allAddressees().isEmpty() )
    {
        manager->add(p_resourcefile);
        manager->writeConfig(config);
    }
    delete p_addressbook;
//     delete p_resourcefile;
    delete config;
//     delete manager;
}

void kmobiletoolsDevicePart::slotSendStoredSMS( SMS *sms)
{
    engine->slotSendStoredSMS( sms);
}

/// KNotify Methods

void kmobiletoolsDevicePart::slotRing( bool ringing)
{
    if(!ringing) return;
    kdDebug() << "KNotify for ring event\n";
        KNotifyClient::event( KMobileTools::KMobiletoolsHelper::instance()->systray()->winId(), "kmobiletools_ring", i18n("Incoming Call") );
}


/*!
    \fn kmobiletoolsDevicePart::slotDial()
 */
void kmobiletoolsDevicePart::slotDial()
{
    if( !m_widget->number_dial->currentText().length()) return;
    if(! engine->isConnected())
    {
        errNotConnected();
        return;
    }
    QString num=m_widget->number_dial->currentText();
    QString name=KMobiletoolsHelper::translateNumber(num);
    if(name==num) name=QString::null;
    (new callDialogImpl(engine, m_widget))->call(num, name );
}

void kmobiletoolsDevicePart::slotDialNumber(const QString &number)
{
    if(b_dialing) return;
    m_widget->number_dial->setCurrentText(number);
    slotDial();
}


/*!
    \fn kmobiletoolsDevicePart::switch2filesystem()
 */
void kmobiletoolsDevicePart::switch2filesystem()
{
    engine->switchToFSMode();
}


/*!
    \fn kmobiletoolsDevicePart::contactsTabChanged()
 */
void kmobiletoolsDevicePart::contactsTabChanged()
{
    kdDebug() << "Contacts tab index: " <<m_widget->contactsTab->currentPageIndex() << endl;
    if(m_widget->contactsTab->currentPageIndex())
        slwidget->searchLine()->setListView(m_widget->phonebookListViewFull);
    else slwidget->searchLine()->setListView(m_widget->phonebookListView);
}
