/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "sms.h"
#include <qstringlist.h>
#include <qdatetime.h>
#include <kdebug.h>
#include <qdir.h>
#include <qfile.h>
#include <klocale.h>
#include <qregexp.h>
#include <kmdcodec.h>
#include "kmobiletoolshelper.h"

SMS::SMS(QObject *parent, const char *name)
 : QObject(parent, name)
{
}

SMS::SMS(const QStringList & numbers, const QString & text, const QDateTime & datetime, QObject *parent, const char *name)
 : QObject(parent, name)
{
    setNumbers(numbers);
    setText(text);
    setDateTime(datetime);
    setFolder(i_folder);
}

SMS::SMS(const QStringList & numbers, const QString & text, QObject *parent, const char *name)
 : QObject(parent, name)
{
    setNumbers(numbers);
    setText(text);
}

void SMS::setDateTime(const QDateTime & datetime)
{
    dt_datetime=datetime;
}

SMS::~SMS()
{
}

QString SMS::getFrom() const
{
    if (i_type==Unsent || i_type==Sent)
    {
        return QString::null;
    } else {
        return QString( sl_numbers.first() );
    }
}

QStringList SMS::getTo() const
{
    if (i_type==Unsent || i_type==Sent)
    {
        return sl_numbers;
    } else {
        return QString::null;
    }
}

bool SMS::operator ==( SMS* compSMS)
{
    return (compSMS->uid() == this->uid());
}

int SMS::getMultiTextCount(const QString &text)
{
    return getMultiTextCount(text.length());
}

int SMS::getMultiTextCount(int length)
{
    if(length <= 160) return 1;
    int textLength=0;
    if(length <= 1404 ) textLength=156; else textLength=154;
    if(length % textLength) return (length / textLength)+1;
    else return (length / textLength);
}

QStringList SMS::getMultiText(const QString &text)
{
    if(text.length()<=160 ) return QStringList(text);
    QStringList out;
    /* Now some notes about total text calculation
    We're adding a separator at the beginning of each message: "n/m:" (without quotes, but WITH the semicolon).
    Now.. if m<10, the separator steal us 4 characters for each sms, so we must split the message in strings of 156 characters.
    So we should check if message length is <=1404 characters.
    If message length > 1404 characters, we should use a "nn/mm:" notation, and now the separarator steal us 6 characters.
    Each message part now should be 154 characters, and so our maximal length now is 15246 characters.
    I just hope that in this world nobody is so mad to send a multi-message made with more than 99 SMS ;-)
    */
    int sepLength, textLength,minLength;
    const QString sep="%1/%2:";
    int totalMessages;
    if(text.length() <= 1404 )
    {
        sepLength=4;
        textLength=156;
        minLength=1;
    } else
    {
        sepLength=6;
        textLength=154;
        minLength=2;
    }
    if(text.length() % textLength) totalMessages=(text.length()/ textLength) +1;
    else totalMessages=text.length() / textLength;
    int j=0;
    for(unsigned int i=0; i<text.length(); i+=textLength)
    {
        j++;
        out+=text.mid( i, textLength )
                .prepend( sep
                        .arg( j, minLength )
                        .arg( totalMessages, minLength )
                        .replace( ' ','0') );
    }
    return out;
}

// Convenience non-static method for the above one
QStringList SMS::getMultiText() const
{
    return getMultiText(s_text);
}

#include "sms.moc"


/*!
    \fn SMS::export(const QString &dir)
 */
bool SMS::exportMD(const QString &dir)
{
    bool retval=false;
    if (i_slot & SIM )
        retval = retval | writeToSlot( dir + QDir::separator() + '.' + i18n("SIM MailDir", "SIM") + ".directory");
    if (i_slot & Phone )
        retval = retval | writeToSlot( dir + QDir::separator() + '.' + i18n("Phone MailDir", "Phone") + ".directory");
    return retval;
}

bool SMS::writeToSlot(const QString &dir)
{
    QString filename=dir+QDir::separator();
    QString text;
    if((i_type & Unsent) || (i_type & Sent) )
    {
        filename+=i18n("Outgoing MailDir", "Outgoing");
        text="To: \"";
        for(QStringList::Iterator it=sl_numbers.begin(); it!=sl_numbers.end(); ++it)
            text+=KMobileTools::KMobiletoolsHelper::translateNumber(*it) + "\" <" + *it + ">\n";
        text+="X-Status: RS\n";
    }
    else
    {
        filename+=i18n("Incoming MailDir", "Incoming");
        text="From: \"" + KMobileTools::KMobiletoolsHelper::translateNumber( getFrom() ) + "\" <" + getFrom() + ">\n";
        text+="X-Status: RC\n";
    }
    QString subject="%1 %2...";
    subject=subject.arg(i18n("SMS/Mail Subject", "[KMobileTools Imported Message]") )
            .arg( KCodecs::quotedPrintableEncode(
                    getText().left( 20+ getText().find( ( QRegExp("[\\s]"), 20 )) ).utf8() )
                );
    subject=subject.replace( QRegExp("([^\\s]*=[\\dA-F]{2,2}[^\\s]*)"), "=?utf-8?q?\\1?=");
    text+=QString("Subject: %1\n").arg(subject);
    text+="Date: " + dt_datetime.toString( "%1, d %2 yyyy hh:mm:ss" )
            .arg( KMobileTools::KMobiletoolsHelper::shortWeekDayNameEng( dt_datetime.date().dayOfWeek() ) )
            .arg( KMobileTools::KMobiletoolsHelper::shortMonthNameEng( dt_datetime.date().month() ) )
            + "\n";
    text+="X-KMobileTools-IntType: " + QString::number(type() ) + '\n';
    text+="X-KMobileTools-TextType: " + SMSTypeString(type() ) + '\n';
    text+="X-KMobileTools-PhoneNumbersTo: " + sl_numbers.join(",") + '\n';
    text+="X-KMobileTools-PhoneNumbersFrom: " + getFrom() + '\n';
    text+="X-KMobileTools-RawSlot: " + rawSlot() + '\n';
    text+="Content-Type: text/plain;\n \tcharset=\"utf-8\"\nContent-Transfer-Encoding: quoted-printable\n";

//     text+="Content-Type: text/plain; charset=utf-8\n";
    text+="\n\n" + KCodecs::quotedPrintableEncode( getText().utf8() )+ '\n';
    filename=filename + QDir::separator() + "cur" + QDir::separator() +
            QString::number(dt_datetime.toTime_t()) + '.' + QString(uid()) + '.' + "kmobiletools";
    kdDebug() << "Writing sms to " << filename << endl;
    QFile file(filename);
    if(! file.open( IO_WriteOnly | IO_Truncate ) ) return false;
    QTextStream stream( &file );
    stream << text.utf8() ;
    file.close();
    return true;
}

/*!
    \fn SMS::exportCSV(const QString &dir)
 */
bool SMS::exportCSV(const QString &dir, const QString &filename)
{
    kdDebug() << "SMS::exportCSV(): " << endl;
    bool retval=false;
    if (i_slot & Phone )
        retval = retval | writeToSlotCSV( dir, filename );
    return retval;
}

bool SMS::writeToSlotCSV(const QString &dir, const QString &filename)
{
    kdDebug() << "SMS::writeToSlotCSV(): " << endl;
    QString text;
    QString filenameOut;

    filenameOut = dir + QDir::separator() + filename;

    if((i_type & Unsent) || (i_type & Sent) )
    {
        text="\"OUTGOING\",";
        for(QStringList::Iterator it=sl_numbers.begin(); it!=sl_numbers.end(); ++it)
            text+="\"" + KMobileTools::KMobiletoolsHelper::translateNumber(*it) + "\",\"" + *it + "\",";
        //filenameOut = dir + QDir::separator() + "outbox_" + filename;
    }
    else
    {
        QString transNumber;
        //transNumber = KMobileTools::KMobiletoolsHelper::translateNumber( getFrom() ).utf8();
        transNumber = KMobileTools::KMobiletoolsHelper::translateNumber( getFrom() );

        //text="\"INCOMING\",\"" + KMobileTools::KMobiletoolsHelper::translateNumber( getFrom() ) + "\",\"" + getFrom() + "\",";
        text="\"INCOMING\",\"" + transNumber  + "\",\"" + getFrom() + "\",";
        //filenameOut = dir + QDir::separator() + "inbox_" + filename;
    }

    text+="\"" + dt_datetime.toString( "%1, d %2 yyyy hh:mm:ss" )
            .arg( KMobileTools::KMobiletoolsHelper::shortWeekDayNameEng( dt_datetime.date().dayOfWeek() ) )
            .arg( KMobileTools::KMobiletoolsHelper::shortMonthNameEng( dt_datetime.date().month() ) )
            + "\",";
    //text+="\"" + KCodecs::quotedPrintableEncode( getText().utf8() ) + "\"";
    //text+="\"" + getText().utf8() + "\"";
    text+="\"" + getText() + "\"";
    kdDebug() << "Writing sms to " << filenameOut << endl;
    QFile file(filenameOut);
    QString lastFile = file.readAll();
    if(! file.open( IO_WriteOnly | IO_Append ) ) return false;
    QTextStream stream( &file );
    //stream << lastFile << text.utf8() << endl;
    stream << lastFile << text << endl;
    file.close();
    return true;
}
