/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "smslist.h"
#include <kdebug.h>
#include <qdir.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "kmobiletools_cfg.h"
#include "kmobiletools_devices.h"

SMSList::SMSList(const char *enginename) : QObject(), QPtrList<SMS>()
{
    resetCount();
    setAutoDelete( false );
    if(enginename) setEngineName(enginename);
}


SMSList::~SMSList()
{
}

void SMSList::append( SMSList *sublist, bool sync)
{
    if(!sublist || !sublist->count()) return;
    QPtrListIterator<SMS> it( *sublist);
    while( it.current() )
    {
        if(sync) { if( QPtrList<SMS>::find(*it)==-1) append(*it); }
        else append(*it);
        ++it;
    }
}


/*!
    \fn SMSList::find(int uid)
 */
int SMSList::find(const QCString &uid)
{
    int found=0;
    SMS *tempSMS=0;
    QPtrListIterator<SMS> it( *this);
    while( (tempSMS=it.current()) !=0 )
    {
        ++it;
        if(tempSMS->uid() == uid)
            return found;
        found++;
    }
    return -1;
}


/*!
    \fn SMSList::sync (SMSList *compList)
 */
void SMSList::sync (SMSList *compList)
{
    SMS *tempSMS;
    // Parsing the current SMSList. If some items are not found in the new one, they'll be removed
    QPtrListIterator<SMS> it (*this);
    while( (!this->isEmpty()) && (( tempSMS=it.current()) != 0)  )
    {
        if( compList->find( tempSMS->uid() ) == -1 )
        {
            emit removed(tempSMS->uid() );
            removeRef(tempSMS);
//             delete tempSMS;
        }
        ++it;
    }
        // Now parsing the new SMSList to find new items to add
    QPtrListIterator<SMS> it2 (*compList);
    while( (tempSMS=it2.current()) != 0 )
    {
        if( this->find( tempSMS->uid() ) == -1 )
        {
            append(tempSMS);
            emit added(tempSMS->uid() );
        }
        ++it2;
    }
}

int SMSList::compareItems( QPtrCollection::Item item1, QPtrCollection::Item item2 )
{
    return ( ((SMS*) item1)->uid()== ((SMS*) item2)->uid() );
}

void SMSList::dump()
{
    SMS *tempSMS;
    int i=0;
    // Parsing the current SMSList. If some items are not found in the new one, they'll be removed
    QPtrListIterator<SMS> it (*this);
    while( (( tempSMS=it.current()) != 0) && !this->isEmpty()  )
    {
        ++it; i++;
        kdDebug() << "SMSList::dump(): " << QString("%1").arg(i,2) << "|" << tempSMS->uid() << "|" << tempSMS->type() << "|" << tempSMS->getText() << endl;
    }
//     kdDebug() << "SMSList::dump(): Unread=" << i_unread << "; Read=" << i_read << "; Sent=" << i_sent << "; Unsent=" << i_unsent << endl;
}

void SMSList::calcSMSNumber()
{
    resetCount();
    SMS *tempSMS;
    QPtrListIterator<SMS> it2 (*this);
    while( (tempSMS=it2.current()) != 0 )
    {
//         kdDebug() << QString("SMS Type: %1").arg(tempSMS->type() ,8,2) << endl;
        switch (tempSMS->type())
        {
            case SMS::Unread:
                if(tempSMS->slot() & SMS::SIM) i_unread_sim++;
                if(tempSMS->slot() & SMS::Phone) i_unread_phone++;
                break;
            case SMS::Read:
                if(tempSMS->slot() & SMS::SIM) i_read_sim++;
                if(tempSMS->slot() & SMS::Phone) i_read_phone++;
                break;
            case SMS::Unsent:
                if(tempSMS->slot() & SMS::SIM) i_unsent_sim++;
                if(tempSMS->slot() & SMS::Phone) i_unsent_phone++;
                break;
            case SMS::Sent:
                if(tempSMS->slot() & SMS::SIM) i_sent_sim++;
                if(tempSMS->slot() & SMS::Phone) i_sent_phone++;
                break;
            case SMS::All:
                //we shouldn't be here...
                break;
        }
    ++it2;
    }
}

int SMSList::count(int smsType, int memSlot)
{
    int result=0;
    if( (smsType & SMS::Unread) )
    {
        if( memSlot & SMS::SIM ) result+= i_unread_sim;
        if( memSlot & SMS::Phone) result+=i_unread_phone;
    }
    if( (smsType & SMS::Read) )
    {
        if( memSlot & SMS::SIM ) result+= i_read_sim;
        if( memSlot & SMS::Phone) result+=i_read_phone;
    }
    if( (smsType & SMS::Unsent) )
    {
        if( memSlot & SMS::SIM ) result+= i_unsent_sim;
        if( memSlot & SMS::Phone) result+=i_unsent_phone;
    }
    if( (smsType & SMS::Sent) )
    {
        if( memSlot & SMS::SIM ) result+= i_sent_sim;
        if( memSlot & SMS::Phone) result+=i_sent_phone;
    }
    return result;
}

#include "smslist.moc"


/*!
    \fn SMSList::saveToMailBox(const QString &engineName)
 */
void SMSList::saveToMailBox(const QString &engineName)
{
    setEngineName(engineName);
    saveToMailBox();
}

/*!
    \fn SMSList::saveToMailBox()
 */
void SMSList::saveToMailBox()
{
    QDir savedir=(KMobileTools::DevicesConfig::prefs(engineName() ))->maildir_path();
    QString dir=savedir.dirName();
    savedir.cdUp();
    dir=savedir.absPath() + QDir::separator() + '.' + dir + ".directory"
            + QDir::separator() + '.' + KMobileTools::DevicesConfig::prefs(s_enginename)->devicename() + ".directory";
    QPtrListIterator<SMS> it(*this);
    SMS *sms;
    while( (sms=it.current()) )
    {
        ++it;
        sms->exportMD(dir);
    }
}

/*!
    \fn SMSList::saveToCSV(const QString &engineName)
 */
int SMSList::saveToCSV(const QString &engineName)
{
    int result;

    setEngineName(engineName);
    result = saveToCSV();
    kdDebug() << "SMSList::saveToCSV(): " << endl;

    return result;
}

/*!
    \fn SMSList::saveToCSV()
 */

/// @TODO Check if we can remove dialog windows out of this class, emitting insteada signal.
int SMSList::saveToCSV()
{
    QString dir=QDir::homeDirPath();
    QPtrListIterator<SMS> it(*this);
    SMS *sms;
    QString saveFile;

    saveFile = KFileDialog::getSaveFileName (QDir::homeDirPath(), "*.csv",  0, i18n("Save file to disk"));

    QString savePath = saveFile;
    QString fileName = saveFile.section( QDir::separator(),-1);
    if ( ! fileName.isEmpty() )
        savePath = saveFile.left(saveFile.length() - (fileName.length() + 1));
        else return 0;

    QDir d = QDir::root();
    if (d.exists(saveFile)) {
        kdDebug() << "SMSList::saveToCSV(): FILE ALREADY EXISTS " << endl;

        int retval;
        retval=KMessageBox::warningContinueCancel(NULL,
                                                  i18n("<qt>The file already exists\nOverwrite the current file?</qt>"),
                                                  "KMobileTools" );

        if(retval == KMessageBox::Continue) {
            d.remove(saveFile);
        }
        else {
            return -1;
        }

    }

    dir =  savePath;

    if (dir == "") {
        dir = QDir::homeDirPath();
    }

    kdDebug() << "SMSList::saveToCSV(): saving CSV file to: " << dir << endl;

    while( (sms=it.current()) )
    {
        ++it;
        sms->exportCSV(dir, fileName);
    }

    return 1;
}

void SMSList::append( SMS *item )
{
    QPtrList<SMS>::append(item);
    connect(item, SIGNAL(updated()), this, SIGNAL(updated()) );
}
