/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* disc.h
 */
#ifndef disc_h
#define disc_h

#include <qstring.h>
#include <qstringlist.h>

class QDomDocument;
class QDomElement;
class QWidget;
class KURL;

namespace KoverArtist
{

/** A disc description. A @ref KoverArtist::Project can have multiple disc
 * descriptions.
 */
class Disc
{
public:
   Disc();
   Disc(const Disc&);
   virtual ~Disc();
   Disc& operator=(const Disc&);

   /** Clear the object. */
   virtual void clear();

   /** Test if the object is empty. */
   virtual bool isEmpty() const;

   /** Returns the title of the disc. */
   QString title() const { return mTitle; }

   /** Set the disc's title to t. */
   void setTitle(const QString& t) { mTitle = t; }

   /** Returns the number of entries the disc has. */
   int count() const { return mEntries.count(); }

   /** Returns the idx-th entry. */
   QString entry(int idx) const { return mEntries[idx]; }

   /** Set the idx-th entry to str. */
   virtual void setEntry(int idx, const QString& str);

   /** Access to the list of entries. */
   const QStringList& entries() const { return mEntries; }
   QStringList& entries() { return mEntries; }

   /** Access to the list of snapshots. Gets filled when importing a
    * dvd project (KMediaFactory and KinoArtist, for example).
    */
   const QStringList& snapShots() const { return mSnapShots; }
   QStringList& snapShots() { return mSnapShots; }

   /** Load the disc from the given url. The window is used for storing
    * authentication information, @see KIO::NetAccess.
    */
   virtual bool load(const KURL& url, QWidget* window=0);

   /** Load the disc from the file fname. */
   virtual bool loadFile(const QString& fname);

   /** Save the disc to the file fname. */
   virtual bool saveFile(const QString& fname) const;

   /** Guess from the file extension of fname if the file could be loaded. */
   static bool canLoad(const QString& fname);

   /** Fill the DOM element de with the object's contents. */
   virtual void toDom(QDomDocument& doc, QDomElement& de) const;

   /** Set the object from the DOM element de. */
   virtual bool fromDom(const QDomElement& de);

protected:
   friend class Project;

   /** Load a KMediaFactory project. */
   virtual bool loadFileKmf(const QString& fname, QDomDocument& doc);

   /** Load a KoverArtist disc file. */
   virtual bool loadFileKoad(const QString& fname, QDomDocument& doc);

   /** Load a KinoArtist disc file. */
   virtual bool loadFileKiap(const QString& fname, QDomDocument& doc);

   /** Load a plaintext file. */
   virtual bool loadFileText(const QString& fname);

   /** Save a KoverArtist disc file. */
   virtual bool saveFileKoad(const QString& fname) const;

   /** Save a plaintext file. */
   virtual bool saveFileText(const QString& fname) const;

private:
   QString mTitle;
   QStringList mEntries, mSnapShots;
};


} //namespace

#endif /*disc_h*/
