/**
 * @file libgalago-gtk/galago-gtk-service-list.c
 *       Contact list widget
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-contact-list.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkContactListPriv
{
	gboolean auto_populate;
	gboolean show_offline;

	GalagoService *service;
	GalagoPerson *person;
	GalagoAccount *contact;

	gboolean populate_lock;
};

enum
{
	COLUMN_DATA,
	COLUMN_ICON,
	COLUMN_TEXT,
	NUM_COLUMNS
};

enum
{
	SELECTION_CHANGED,
	CONTACT_ACTIVATED,
	LAST_SIGNAL
};

static void galago_gtk_contact_list_class_init(GalagoGtkContactListClass *klass);
static void galago_gtk_contact_list_init(GalagoGtkContactList *list);
static void galago_gtk_contact_list_finalize(GObject *obj);
static void galago_gtk_contact_list_destroy(GtkObject *obj);
static void galago_gtk_contact_list_row_activated(GtkTreeView *treeview,
												  GtkTreePath *path,
												  GtkTreeViewColumn *column);

static GtkTreeViewClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_contact_list_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkContactListClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_contact_list_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkContactList),
			0,
			(GInstanceInitFunc)galago_gtk_contact_list_init
		};

		type = g_type_register_static(GTK_TYPE_TREE_VIEW,
									  "GalagoGtkContactList", &info, 0);
	}

	return type;
}

static void
galago_gtk_contact_list_class_init(GalagoGtkContactListClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkTreeViewClass *treeview_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class  = G_OBJECT_CLASS(klass);
	object_class   = GTK_OBJECT_CLASS(klass);
	treeview_class = GTK_TREE_VIEW_CLASS(klass);

	gobject_class->finalize = galago_gtk_contact_list_finalize;

	object_class->destroy = galago_gtk_contact_list_destroy;

	treeview_class->row_activated = galago_gtk_contact_list_row_activated;

	signals[SELECTION_CHANGED] =
		g_signal_new("selection-changed",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkContactListClass,
									 selection_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	signals[CONTACT_ACTIVATED] =
		g_signal_new("contact-activated",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkContactListClass,
									 contact_activated),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);
}

static void
selection_changed_cb(GtkTreeSelection *sel, GalagoGtkContactList *list)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GalagoAccount *contact;

	if (!gtk_tree_selection_get_selected(sel, &model, &iter))
		return;

	gtk_tree_model_get(model, &iter,
					   COLUMN_DATA, &contact,
					   -1);

	galago_gtk_contact_list_set_contact(list, contact);

	g_signal_emit(list, signals[SELECTION_CHANGED], 0);
}

static void
append_contact(GalagoGtkContactList *list, GtkListStore *model,
			   GalagoAccount *contact)
{
	GdkPixbuf *pixbuf;
	GtkTreeIter iter;

	if (!list->priv->show_offline && !galago_account_is_connected(contact))
		return;

	pixbuf = galago_gdk_pixbuf_new_from_account_with_size(contact,
														  GTK_ICON_SIZE_MENU);

	gtk_list_store_append(model, &iter);
	gtk_list_store_set(model, &iter,
					   COLUMN_DATA, contact,
					   COLUMN_ICON, pixbuf,
					   COLUMN_TEXT,
					   galago_account_get_username(contact),
					   -1);

	if (pixbuf != NULL)
		g_object_unref(G_OBJECT(pixbuf));
}

static void
append_contacts_for_service(GalagoGtkContactList *list, GtkListStore *model,
							GalagoService *service)
{
	GList *l;

	for (l = galago_service_get_accounts(service, TRUE);
		 l != NULL;
		 l = l->next)
	{
		append_contact(list, model, (GalagoAccount *)l->data);
	}
}

static void
append_contacts_for_person(GalagoGtkContactList *list, GtkListStore *model,
						   GalagoPerson *person)
{
	GList *l;

	for (l = galago_person_get_accounts(person, TRUE);
		 l != NULL;
		 l = l->next)
	{
		append_contact(list, model, (GalagoAccount *)l->data);
	}
}

static void
populate_contacts_list(GalagoGtkContactList *list)
{
	GtkListStore *model;

	if (list->priv->populate_lock)
		return;

	list->priv->populate_lock = TRUE;

	model = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(list)));

	gtk_list_store_clear(model);

	if (!list->priv->auto_populate)
		goto exit;

	if (list->priv->service != NULL)
	{
		append_contacts_for_service(list, model, list->priv->service);
	}
	else if (list->priv->person != NULL)
	{
		append_contacts_for_person(list, model, list->priv->person);
	}
	else
	{
		GList *l;

		for (l = galago_get_services(GALAGO_REMOTE, TRUE);
			 l != NULL;
			 l = l->next)
		{
			append_contacts_for_service(list, model, (GalagoService *)l->data);
		}
	}

exit:
	list->priv->populate_lock = FALSE;
}

static void
galago_gtk_contact_list_init(GalagoGtkContactList *list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *model;
	GtkTreeSelection *sel;

	list->priv = g_new0(GalagoGtkContactListPriv, 1);
	list->priv->show_offline = TRUE;

	model = gtk_list_store_new(NUM_COLUMNS, G_TYPE_POINTER,
							   GDK_TYPE_PIXBUF, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(model));
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model),
										 COLUMN_TEXT, GTK_SORT_ASCENDING);
	g_object_unref(G_OBJECT(model));

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list), FALSE);
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(list), TRUE);
	gtk_tree_view_set_search_column(GTK_TREE_VIEW(list),
									COLUMN_TEXT);

	column = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "pixbuf", COLUMN_ICON);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "text", COLUMN_TEXT);

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));
	g_signal_connect(G_OBJECT(sel), "changed",
					 G_CALLBACK(selection_changed_cb), list);

	populate_contacts_list(list);
}

static void
galago_gtk_contact_list_finalize(GObject *obj)
{
	GalagoGtkContactList *list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(obj));

	list = GALAGO_GTK_CONTACT_LIST(obj);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_contact_list_destroy(GtkObject *obj)
{
	GalagoGtkContactList *list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(obj));

	list = GALAGO_GTK_CONTACT_LIST(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
galago_gtk_contact_list_row_activated(GtkTreeView *treeview, GtkTreePath *path,
									  GtkTreeViewColumn *column)
{
	GalagoGtkContactList *contact_list;

	contact_list = GALAGO_GTK_CONTACT_LIST(treeview);

	g_signal_emit(contact_list, signals[CONTACT_ACTIVATED], 0,
				  galago_gtk_contact_list_get_contact(contact_list));

	if (parent_class->row_activated != NULL)
		parent_class->row_activated(treeview, path, column);
}

GtkWidget *
galago_gtk_contact_list_new(gboolean auto_populate)
{
	GalagoGtkContactList *contact_list;

	contact_list = g_object_new(GALAGO_GTK_TYPE_CONTACT_LIST, NULL);
	galago_gtk_contact_list_set_auto_populate(contact_list, auto_populate);

	return GTK_WIDGET(contact_list);
}

void
galago_gtk_contact_list_set_auto_populate(GalagoGtkContactList *contact_list,
										  gboolean auto_populate)
{
	g_return_if_fail(contact_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(contact_list));

	if (contact_list->priv->auto_populate == auto_populate)
		return;

	contact_list->priv->auto_populate = auto_populate;

	populate_contacts_list(contact_list);
}

void
galago_gtk_contact_list_set_contact(GalagoGtkContactList *list,
									GalagoAccount *contact)
{
	GtkTreeSelection *sel;

	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list));

	if (list->priv->contact == contact)
		return;

	list->priv->contact = contact;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));

	if (contact == NULL)
	{
		gtk_tree_selection_unselect_all(sel);
	}
	else
	{
		GtkTreeIter iter;
		GtkTreeModel *model;
		GalagoAccount *temp_contact;
		gboolean valid;
		gboolean found = FALSE;

		model = gtk_tree_view_get_model(GTK_TREE_VIEW(list));

		for (valid = gtk_tree_model_get_iter_first(model, &iter);
			 valid;
			 valid = gtk_tree_model_iter_next(model, &iter))
		{
			gtk_tree_model_get(model, &iter,
							   COLUMN_DATA, &temp_contact,
							   -1);

			if (contact == temp_contact)
			{
				gtk_tree_selection_select_iter(sel, &iter);
				found = TRUE;
				break;
			}
		}

		if (!found)
			list->priv->contact = NULL;
	}
}

void
galago_gtk_contact_list_set_service(GalagoGtkContactList *list,
									GalagoService *service)
{
	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list));

	if (list->priv->service == service)
		return;

	list->priv->service = service;
	list->priv->person  = NULL;

	populate_contacts_list(list);
}

void
galago_gtk_contact_list_set_person(GalagoGtkContactList *list,
								   GalagoPerson *person)
{
	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list));

	if (list->priv->person == person)
		return;

	list->priv->person  = person;
	list->priv->service = NULL;

	populate_contacts_list(list);
}

void
galago_gtk_contact_list_set_show_offline(GalagoGtkContactList *list,
										 gboolean show_offline)
{
	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list));

	if (list->priv->show_offline == show_offline)
		return;

	list->priv->show_offline = show_offline;

	populate_contacts_list(list);
}

void
galago_gtk_contact_list_add_contact(GalagoGtkContactList *contact_list,
									GalagoAccount *contact)
{
	g_return_if_fail(contact_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(contact_list));
	g_return_if_fail(!galago_gtk_contact_list_get_auto_populate(contact_list));

	append_contact(contact_list,
		GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(contact_list))),
		contact);
}

void
galago_gtk_contact_list_remove_contact(GalagoGtkContactList *contact_list,
									   GalagoAccount *contact)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GalagoAccount *temp_contact;
	gboolean valid;

	g_return_if_fail(contact_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(contact_list));
	g_return_if_fail(!galago_gtk_contact_list_get_auto_populate(contact_list));

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(contact_list));

	for (valid = gtk_tree_model_get_iter_first(model, &iter);
		 valid;
		 valid = gtk_tree_model_iter_next(model, &iter))
	{
		gtk_tree_model_get(model, &iter,
						   COLUMN_DATA, &temp_contact,
						   -1);

		if (contact == temp_contact)
		{
			gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
			break;
		}
	}
}

void
galago_gtk_contact_list_clear(GalagoGtkContactList *contact_list)
{
	g_return_if_fail(contact_list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_LIST(contact_list));
	g_return_if_fail(!galago_gtk_contact_list_get_auto_populate(contact_list));

	gtk_list_store_clear(GTK_LIST_STORE(
		gtk_tree_view_get_model(GTK_TREE_VIEW(contact_list))));
}

gboolean
galago_gtk_contact_list_get_auto_populate(const GalagoGtkContactList *list)
{
	g_return_val_if_fail(list != NULL,                     FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list), FALSE);

	return list->priv->auto_populate;
}

GalagoAccount *
galago_gtk_contact_list_get_contact(const GalagoGtkContactList *list)
{
	g_return_val_if_fail(list != NULL,                     NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list), NULL);

	return list->priv->contact;
}

GalagoService *
galago_gtk_contact_list_get_service(const GalagoGtkContactList *list)
{
	g_return_val_if_fail(list != NULL,                     NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list), NULL);

	return list->priv->service;
}

GalagoPerson *
galago_gtk_contact_list_get_person(const GalagoGtkContactList *list)
{
	g_return_val_if_fail(list != NULL,                     NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list), NULL);

	return list->priv->person;
}

gboolean
galago_gtk_contact_list_get_show_offline(const GalagoGtkContactList *list)
{
	g_return_val_if_fail(list != NULL,                     FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_LIST(list), FALSE);

	return list->priv->show_offline;
}
